<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\License;
use App\Models\LicenseLog;
use App\Models\Product;
use App\Models\KbCategory;
use App\Models\KbArticle;
use App\Services\LicenseGeneratorService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class ProductController extends Controller
{
    protected $licenseGenerator;

    public function __construct(LicenseGeneratorService $licenseGenerator)
    {
        $this->licenseGenerator = $licenseGenerator;
    }

    /**
     * Get product data from Envato API
     */
    public function getEnvatoProductData(Request $request)
    {
        $request->validate([
            'item_id' => 'required|integer|min:1'
        ]);

        $itemId = $request->input('item_id');

        try {
            $envatoService = app(\App\Services\EnvatoService::class);
            $itemData = $envatoService->getItemInfo($itemId);

            if (!$itemData) {
                return response()->json([
                    'success' => false,
                    'message' => trans('app.Unable to fetch product data from Envato')
                ], 404);
            }

            // Extract relevant data from Envato API response
            $productData = [
                'success' => true,
                'data' => [
                    'envato_item_id' => $itemData['id'] ?? null,
                    'purchase_url_envato' => $itemData['url'] ?? null,
                    'purchase_url_buy' => $itemData['url'] ?? null, // Same as purchase URL for now
                    'support_days' => $this->calculateSupportDays($itemData),
                    'version' => $itemData['version'] ?? null,
                    'price' => $itemData['price_cents'] ? ($itemData['price_cents'] / 100) : null,
                    'name' => $itemData['name'] ?? null,
                    'description' => $itemData['description'] ?? null,
                ]
            ];

            return response()->json($productData);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => trans('app.Error fetching product data: ') . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get user's Envato items for selection
     */
    public function getEnvatoUserItems(Request $request)
    {
        try {
            $envatoService = app(\App\Services\EnvatoService::class);
            $settings = $envatoService->getEnvatoSettings();

            if (empty($settings['username'])) {
                return response()->json([
                    'success' => false,
                    'message' => trans('app.Envato username not configured')
                ], 400);
            }

            $userItems = $envatoService->getUserItems($settings['username']);

            if (!$userItems || !isset($userItems['matches'])) {
                return response()->json([
                    'success' => false,
                    'message' => trans('app.Unable to fetch user items from Envato')
                ], 404);
            }

            $items = collect($userItems['matches'])->map(function ($item) {
                return [
                    'id' => $item['id'],
                    'name' => $item['name'],
                    'url' => $item['url'],
                    'price' => $item['price_cents'] ? ($item['price_cents'] / 100) : 0,
                    'rating' => $item['rating'] ?? null,
                    'sales' => $item['number_of_sales'] ?? 0,
                ];
            });

            return response()->json([
                'success' => true,
                'items' => $items
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => trans('app.Error fetching user items: ') . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Calculate support days from Envato item data
     */
    private function calculateSupportDays($itemData): ?int
    {
        // Envato typically provides 6 months support for most items
        // This can be adjusted based on the actual API response
        if (isset($itemData['attributes'])) {
            foreach ($itemData['attributes'] as $attribute) {
                if (isset($attribute['name']) && $attribute['name'] === 'support') {
                    // Parse support duration (e.g., "6 months", "1 year")
                    $value = strtolower($attribute['value'] ?? '');
                    if (strpos($value, 'month') !== false) {
                        preg_match('/(\d+)/', $value, $matches);
                        return isset($matches[1]) ? (int)$matches[1] * 30 : 180;
                    } elseif (strpos($value, 'year') !== false) {
                        preg_match('/(\d+)/', $value, $matches);
                        return isset($matches[1]) ? (int)$matches[1] * 365 : 365;
                    }
                }
            }
        }

        // Default to 6 months (180 days) if not specified
        return 180;
    }

    /**
     * Get integration code template for product
     */
    private function getIntegrationCodeTemplate(Product $product, string $apiUrl): string
    {
        // Return a minimal placeholder integration file to avoid complex embedded templates here.
        return "<?php\n// Integration placeholder for {$product->slug}\n// API: {$apiUrl}\n";
    }

    /**
     * Display a listing of the resource (admin).
     */
    public function index()
    {
        $productsQuery = Product::with(['category', 'programmingLanguage']);

        // Apply search filter (support both 'q' and 'search')
        $search = request('q', request('search'));
        if (!empty($search)) {
            $productsQuery->where(function ($query) use ($search) {
                $query->where('name', 'like', "%{$search}%")
                      ->orWhere('description', 'like', "%{$search}%");
            });
        }

        // For the main (All Products) section, show only uncategorized products
        $productsQuery->whereNull('category_id');

        // Apply language filter
        if (request('language')) {
            $productsQuery->where('programming_language', request('language'));
        }

        // Apply price filter
        if (request('price_filter') === 'free') {
            $productsQuery->where('price', 0);
        } elseif (request('price_filter') === 'paid') {
            $productsQuery->where('price', '>', 0);
        }

        // Apply sorting
        $sort = request('sort', 'name');
        switch ($sort) {
            case 'price_low':
                $productsQuery->orderBy('price', 'asc');
                break;
            case 'price_high':
                $productsQuery->orderBy('price', 'desc');
                break;
            case 'newest':
                $productsQuery->orderBy('created_at', 'desc');
                break;
            default:
                $productsQuery->orderBy('name', 'asc');
        }

        $products = $productsQuery->paginate(10)->withQueryString();
        $categories = \App\Models\ProductCategory::where('is_active', true)->orderBy('sort_order')->get();
        $programmingLanguages = \App\Models\ProgrammingLanguage::where('is_active', true)->orderBy('sort_order')->get();

        // Provide all products collection for grouped/category displays in the admin index
        $allProducts = Product::with(['category', 'programmingLanguage'])->get();

        return view('admin.products.index', compact('products', 'categories', 'programmingLanguages', 'allProducts'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $categories = \App\Models\ProductCategory::where('is_active', true)->orderBy('sort_order')->get();
        $programmingLanguages = \App\Models\ProgrammingLanguage::where('is_active', true)->orderBy('sort_order')->get();
        $kbCategories = KbCategory::where('is_published', true)->orderBy('name')->get(['id', 'name', 'slug']);
        $kbArticles = KbArticle::where('is_published', true)->with('category:id,name')->orderBy('title')->get(['id', 'title', 'slug', 'kb_category_id']);
        
        return view('admin.products.create', compact('categories', 'programmingLanguages', 'kbCategories', 'kbArticles'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Normalize features (textarea new lines) and tags (comma-separated) to arrays if provided as strings
        $featuresInput = $request->input('features');
        if (is_string($featuresInput)) {
            $request->merge([
                'features' => array_values(array_filter(array_map('trim', preg_split('/\r\n|\r|\n/', $featuresInput))))
            ]);
        }
    
        $tagsInput = $request->input('tags');
        if (is_string($tagsInput)) {
            $request->merge([
                'tags' => array_values(array_filter(array_map('trim', explode(',', $tagsInput))))
            ]);
        }
    
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'slug' => ['nullable', 'string', 'max:255', 'unique:products,slug'],
            'category_id' => ['required', 'exists:product_categories,id'],
            'envato_item_id' => ['nullable', 'string', 'max:50'],
            'purchase_url_envato' => ['nullable', 'url', 'max:1024'],
            'purchase_url_buy' => ['nullable', 'url', 'max:1024'],
            'description' => ['nullable', 'string'],
            'support_days' => ['nullable', 'integer', 'min:0'],
            'price' => ['nullable', 'numeric', 'min:0'],
            'tax_rate' => ['nullable', 'numeric', 'min:0', 'max:100'],
            'renewal_price' => ['nullable', 'numeric', 'min:0'],
            'renewal_period' => ['nullable', 'in:monthly,quarterly,semi-annual,annual,three-years,lifetime'],
            'license_type' => ['required', 'in:single,multi,developer,extended'],
            'duration_days' => ['nullable', 'integer', 'min:0'],
            'auto_renewal' => ['boolean'],
            'renewal_reminder_days' => ['nullable', 'integer', 'min:0'],
            'extended_support_price' => ['nullable', 'numeric', 'min:0'],
            'extended_support_days' => ['nullable', 'integer', 'min:0'],
            'supported_until' => ['nullable', 'date', 'after:now'],
            'extended_supported_until' => ['nullable', 'date', 'after:now', 'before:2100-01-01'],
            'status' => ['nullable', 'in:active,inactive,draft,archived'],
            'stock' => ['nullable', 'integer', 'min:-1'],
            'requires_domain' => ['required', 'in:0,1'],
            'programming_language' => ['required', 'exists:programming_languages,id'],
            'image' => ['nullable', 'image', 'mimes:jpeg,png,jpg,gif', 'max:2048'],
            'gallery_images.*' => ['nullable', 'image', 'mimes:jpeg,png,jpg,gif', 'max:2048'],
            'features' => ['nullable', 'array'],
            'features.*' => ['string', 'max:255'],
            'version' => ['nullable', 'string', 'max:50'],
            'requirements' => ['nullable', 'string'],
            'installation_guide' => ['nullable', 'string'],
            'meta_title' => ['nullable', 'string', 'max:255'],
            'meta_description' => ['nullable', 'string', 'max:500'],
            'tags' => ['nullable', 'array'],
            'tags.*' => ['string', 'max:100'],
            'is_active' => ['boolean'],
            'is_featured' => ['boolean'],
            'is_popular' => ['boolean'],
            'is_downloadable' => ['boolean'],
            'stock_quantity' => ['nullable', 'integer', 'min:-1'],
            'kb_categories' => ['nullable', 'array'],
            'kb_categories.*' => ['integer', 'exists:kb_categories,id'],
            'kb_articles' => ['nullable', 'array'],
            'kb_articles.*' => ['integer', 'exists:kb_articles,id'],
            'kb_access_required' => ['boolean'],
            'kb_access_message' => ['nullable', 'string', 'max:1000'],
            'product_files.*' => ['nullable', 'file', 'max:51200'], // 50MB max
        ]);

        // Handle checkbox values - set to false if not present
        $validated['is_active'] = $request->has('is_active');
        $validated['is_featured'] = $request->has('is_featured');
        $validated['is_popular'] = $request->has('is_popular');
        $validated['is_downloadable'] = $request->has('is_downloadable');
        $validated['kb_access_required'] = $request->has('kb_access_required');
        $validated['auto_renewal'] = $request->has('auto_renewal');
        
        // Convert requires_domain to boolean
        $validated['requires_domain'] = (bool) $validated['requires_domain'];

        $validated['slug'] = $validated['slug'] ?? Str::slug($validated['name']);

        // Handle main image upload
        if ($request->hasFile('image')) {
            $validated['image'] = $request->file('image')->store('products', 'public');
        }

        // Handle gallery images
        if ($request->hasFile('gallery_images')) {
            $galleryPaths = [];
            foreach ($request->file('gallery_images') as $file) {
                $galleryPaths[] = $file->store('products/gallery', 'public');
            }
            $validated['gallery_images'] = $galleryPaths;
        }

        // Handle lifetime renewal period - set extended_supported_until to null for lifetime
        if ($validated['renewal_period'] === 'lifetime') {
            $validated['extended_supported_until'] = null;
        }

        $product = Product::create($validated);

        // Handle product files upload
        if ($request->hasFile('product_files')) {
            try {
                $productFileService = app(\App\Services\ProductFileService::class);
                foreach ($request->file('product_files') as $file) {
                    if ($file->isValid()) {
                        $productFileService->uploadFile($product, $file);
                    }
                }
            } catch (\Exception $e) {
                \Log::error('Product file upload error: ' . $e->getMessage());
                return back()->with('error', 'Error uploading files: ' . $e->getMessage())->withInput();
            }
        }

        // Generate integration file
        $this->generateIntegrationFile($product);

        return redirect()->route('admin.products.edit', $product)->with('success', 'Product created successfully');
    }

    /**
     * Display the specified resource (admin).
     */
    public function show(Product $product)
    {
        // Admin should be redirected to edit page
        return redirect()->route('admin.products.edit', $product);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Product $product)
    {
        // Normalize features (textarea new lines) and tags (comma-separated) to arrays if provided as strings
        $featuresInput = $request->input('features');
        if (is_string($featuresInput)) {
            $request->merge([
                'features' => array_values(array_filter(array_map('trim', preg_split('/\r\n|\r|\n/', $featuresInput))))
            ]);
        }
    
        $tagsInput = $request->input('tags');
        if (is_string($tagsInput)) {
            $request->merge([
                'tags' => array_values(array_filter(array_map('trim', explode(',', $tagsInput))))
            ]);
        }
    
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'slug' => ['nullable', 'string', 'max:255', 'unique:products,slug,' . $product->id],
            'category_id' => ['required', 'exists:product_categories,id'],
            'envato_item_id' => ['nullable', 'string', 'max:50'],
            'purchase_url_envato' => ['nullable', 'url', 'max:1024'],
            'purchase_url_buy' => ['nullable', 'url', 'max:1024'],
            'description' => ['nullable', 'string'],
            'support_days' => ['nullable', 'integer', 'min:0'],
            'price' => ['nullable', 'numeric', 'min:0'],
            'tax_rate' => ['nullable', 'numeric', 'min:0', 'max:100'],
            'renewal_price' => ['nullable', 'numeric', 'min:0'],
            'renewal_period' => ['nullable', 'in:monthly,quarterly,semi-annual,annual,three-years,lifetime'],
            'license_type' => ['required', 'in:single,multi,developer,extended'],
            'duration_days' => ['nullable', 'integer', 'min:0'],
            'auto_renewal' => ['boolean'],
            'renewal_reminder_days' => ['nullable', 'integer', 'min:0'],
            'extended_support_price' => ['nullable', 'numeric', 'min:0'],
            'extended_support_days' => ['nullable', 'integer', 'min:0'],
            'supported_until' => ['nullable', 'date', 'after:now'],
            'extended_supported_until' => ['nullable', 'date', 'after:now', 'before:2100-01-01'],
            'status' => ['nullable', 'in:active,inactive,draft,archived'],
            'stock' => ['nullable', 'integer', 'min:-1'],
            'requires_domain' => ['required', 'in:0,1'],
            'programming_language' => ['required', 'exists:programming_languages,id'],
            'image' => ['nullable', 'image', 'mimes:jpeg,png,jpg,gif', 'max:2048'],
            'gallery_images.*' => ['nullable', 'image', 'mimes:jpeg,png,jpg,gif', 'max:2048'],
            'features' => ['nullable', 'array'],
            'features.*' => ['string'],
            'version' => ['nullable', 'string', 'max:50'],
            'requirements' => ['nullable', 'string'],
            'installation_guide' => ['nullable', 'string'],
            'meta_title' => ['nullable', 'string', 'max:255'],
            'meta_description' => ['nullable', 'string', 'max:500'],
            'tags' => ['nullable', 'array'],
            'tags.*' => ['string'],
            'is_active' => ['boolean'],
            'is_featured' => ['boolean'],
            'is_popular' => ['boolean'],
            'is_downloadable' => ['boolean'],
            'stock_quantity' => ['nullable', 'integer', 'min:-1'],
            'kb_categories' => ['nullable', 'array'],
            'kb_categories.*' => ['integer', 'exists:kb_categories,id'],
            'kb_articles' => ['nullable', 'array'],
            'kb_articles.*' => ['integer', 'exists:kb_articles,id'],
            'kb_access_required' => ['boolean'],
            'kb_access_message' => ['nullable', 'string', 'max:1000'],
            'product_files.*' => ['nullable', 'file', 'max:51200'], // 50MB max
        ]);

        // Handle checkbox values - set to false if not present
        $validated['is_active'] = $request->has('is_active');
        $validated['is_featured'] = $request->has('is_featured');
        $validated['is_popular'] = $request->has('is_popular');
        $validated['is_downloadable'] = $request->has('is_downloadable');
        $validated['kb_access_required'] = $request->has('kb_access_required');
        $validated['auto_renewal'] = $request->has('auto_renewal');
        
        // Convert requires_domain to boolean
        $validated['requires_domain'] = (bool) $validated['requires_domain'];

        // Handle main image upload
        if ($request->hasFile('image')) {
            // Delete old image
            if ($product->image) {
                Storage::disk('public')->delete($product->image);
            }
            $validated['image'] = $request->file('image')->store('products', 'public');
        }

        // Handle gallery images
        if ($request->hasFile('gallery_images')) {
            $galleryPaths = [];
            foreach ($request->file('gallery_images') as $file) {
                $galleryPaths[] = $file->store('products/gallery', 'public');
            }
            $validated['gallery_images'] = $galleryPaths;
        }

        // Handle lifetime renewal period - set extended_supported_until to null for lifetime
        if ($validated['renewal_period'] === 'lifetime') {
            $validated['extended_supported_until'] = null;
        }

        $product->update($validated);

        // Handle product files upload
        if ($request->hasFile('product_files')) {
            try {
                $productFileService = app(\App\Services\ProductFileService::class);
                foreach ($request->file('product_files') as $file) {
                    if ($file->isValid()) {
                        $productFileService->uploadFile($product, $file);
                    }
                }
            } catch (\Exception $e) {
                \Log::error('Product file upload error: ' . $e->getMessage());
                return back()->with('error', 'Error uploading files: ' . $e->getMessage())->withInput();
            }
        }

        // Regenerate file if programming language or envato settings changed
        if ($product->wasChanged(['programming_language', 'envato_item_id', 'name', 'slug'])) {
            $this->generateIntegrationFile($product);
        }

        return back()->with('success', 'Product updated successfully');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Product $product)
    {
        // Delete integration file if exists
        $filePath = "integration/{$product->slug}.php";
        if (Storage::disk('public')->exists($filePath)) {
            Storage::disk('public')->delete($filePath);
        }

        $product->delete();
        return redirect()->route('admin.products.index')->with('success', 'Product deleted');
    }

    /**
     * Get product data for license forms (AJAX endpoint).
     */
    public function getProductData(Product $product)
    {
        return response()->json([
            'id' => $product->id,
            'name' => $product->name,
            'license_type' => $product->license_type,
            'duration_days' => $product->duration_days,
            'support_days' => $product->support_days,
            'price' => $product->price,
            'renewal_price' => $product->renewal_price,
            'renewal_period' => $product->renewal_period,
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Product $product)
    {
        $categories = \App\Models\ProductCategory::where('is_active', true)->orderBy('sort_order')->get();
        $programmingLanguages = \App\Models\ProgrammingLanguage::where('is_active', true)->orderBy('sort_order')->get();
        
        // Load product files
        $product->load('files');
        
        return view('admin.products.edit', compact('product', 'categories', 'programmingLanguages'));
    }

    /**
     * Generate integration file for a product
     */
    private function generateIntegrationFile(Product $product)
    {
        try {
            // Delete old integration file if exists
            $oldFilePath = "integration/{$product->slug}.php";
            if (Storage::disk('public')->exists($oldFilePath)) {
                Storage::disk('public')->delete($oldFilePath);
            }

            // Delete any old files with different extensions based on programming language
            $programmingLanguage = $product->programmingLanguage;
            if ($programmingLanguage) {
                $extensions = $this->getFileExtensionsForLanguage($programmingLanguage->slug);
                foreach ($extensions as $ext) {
                    $oldFileWithExt = "integration/{$product->slug}.{$ext}";
                    if (Storage::disk('public')->exists($oldFileWithExt)) {
                        Storage::disk('public')->delete($oldFileWithExt);
                    }
                }
            }

            // Use the new LicenseGeneratorService
            $filePath = $this->licenseGenerator->generateLicenseFile($product);

            return $filePath;
        } catch (\Exception $e) {
            // Fallback to old method if new service fails
            $apiDomain = rtrim(env('APP_URL', config('app.url')), '/');
            $verificationEndpoint = config('license.verification_endpoint', '/api/license/verify');
            $apiUrl = $apiDomain . '/' . ltrim($verificationEndpoint, '/');
            $integrationCode = $this->getIntegrationCodeTemplate($product, $apiUrl);

            // Save to storage/app/public/integration/
            $filePath = "integration/{$product->slug}.php";
            Storage::disk('public')->put($filePath, $integrationCode);

            // Update product with integration file path
            $product->update([
                'integration_file_path' => $filePath
            ]);

            return $filePath;
        }
    }

    /**
     * Download file for a product
     */
    public function downloadIntegration(Product $product)
    {
        if (!$product->integration_file_path || !Storage::disk('public')->exists($product->integration_file_path)) {
            return redirect()->back()->with('error', 'Integration file not found. Please regenerate it.');
        }

        return Storage::disk('public')->download($product->integration_file_path, "{$product->slug}.php");
    }

    /**
     * Regenerate file for a product
     */
    public function regenerateIntegration(Product $product)
    {
        try {
            // Delete old integration file if exists
            $oldFilePath = "integration/{$product->slug}.php";
            if (Storage::disk('public')->exists($oldFilePath)) {
                Storage::disk('public')->delete($oldFilePath);
            }

            // Delete any old files with different extensions based on programming language
            $programmingLanguage = $product->programmingLanguage;
            if ($programmingLanguage) {
                $extensions = $this->getFileExtensionsForLanguage($programmingLanguage->slug);
                foreach ($extensions as $ext) {
                    $oldFileWithExt = "integration/{$product->slug}.{$ext}";
                    if (Storage::disk('public')->exists($oldFileWithExt)) {
                        Storage::disk('public')->delete($oldFileWithExt);
                    }
                }
            }

            // Generate new integration file
            $this->generateIntegrationFile($product);
            return redirect()->back()->with('success', 'Integration file regenerated successfully.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Failed to Regenerate file: ' . $e->getMessage());
        }
    }

    /**
     * Get file extensions for programming language
     */
    private function getFileExtensionsForLanguage(string $languageSlug): array
    {
        $extensions = [
            'php' => ['php'],
            'laravel' => ['php'],
            'javascript' => ['js'],
            'python' => ['py'],
            'java' => ['java'],
            'csharp' => ['cs'],
            'cpp' => ['cpp', 'h'],
            'wordpress' => ['php'],
            'react' => ['js', 'jsx'],
            'angular' => ['ts'],
            'nodejs' => ['js'],
            'vuejs' => ['js', 'vue'],
            'go' => ['go'],
            'swift' => ['swift'],
            'typescript' => ['ts'],
            'kotlin' => ['kt'],
            'c' => ['c', 'h'],
            'html-css' => ['html', 'css'],
            'flask' => ['py'],
            'django' => ['py'],
            'expressjs' => ['js'],
            'ruby-on-rails' => ['rb'],
            'spring-boot' => ['java'],
            'symfony' => ['php'],
            'aspnet' => ['cs'],
            'html' => ['html'],
            'ruby' => ['rb'],
        ];

        return $extensions[$languageSlug] ?? ['php'];
    }

    /**
     * Generate a test license for the product
     */
    public function generateTestLicense(Request $request, Product $product)
    {
        $validated = $request->validate([
            'domain' => ['required', 'string'],
            'email' => ['required', 'email'],
            'name' => ['nullable', 'string'],
        ]);

        // Find or create user
        $user = \App\Models\User::firstOrCreate(
            ['email' => $validated['email']],
            [
                'name' => $validated['name'] ?? 'Test User',
                'email' => $validated['email'],
                'password' => bcrypt('password123'), // Default password for test users
                'email_verified_at' => now(),
            ]
        );


        // Generate unique purchase code
        $purchaseCode = 'TEST-' . strtoupper(Str::random(16));

        // Create license (license_key will be automatically set to same value as purchase_code)
        $license = License::create([
            'product_id' => $product->id,
            'user_id' => $user->id,
            'purchase_code' => $purchaseCode,
            'status' => 'active',
            'license_type' => 'regular',
            'support_expires_at' => now()->addDays($product->support_days),
            'license_expires_at' => now()->addYear(),
        ]);

        // Add domain
        $license->domains()->create([
            'domain' => $validated['domain'],
        ]);

        return redirect()->back()->with('success', "Test license generated: {$purchaseCode}");
    }

    /**
     * Show license verification logs for a product
     */
    public function logs(Product $product)
    {
        $logs = LicenseLog::with(['license'])
            ->whereHas('license', fn($q) => $q->where('product_id', $product->id))
            ->orWhere('serial', 'like', '%TEST-%') // For test licenses without license_id
            ->latest()
            ->paginate(50);

        return view('admin.products.logs', compact('product', 'logs'));
    }

    /**
     * Get KB categories and articles for product form
     */
    public function getKbData()
    {
        $categories = KbCategory::where('is_published', true)
            ->orderBy('name')
            ->get(['id', 'name', 'slug']);

        $articles = KbArticle::where('is_published', true)
            ->with('category:id,name')
            ->orderBy('title')
            ->get(['id', 'title', 'slug', 'kb_category_id']);

        return response()->json([
            'success' => true,
            'categories' => $categories,
            'articles' => $articles
        ]);
    }

    public function getKbArticles($categoryId)
    {
        $articles = KbArticle::where('kb_category_id', $categoryId)
            ->where('is_published', true)
            ->with('category:id,name')
            ->orderBy('title')
            ->get(['id', 'title', 'slug', 'kb_category_id']);

        return response()->json([
            'success' => true,
            'articles' => $articles
        ]);
    }

}
