<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\ProductFile;
use App\Services\ProductFileService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class ProductFileController extends Controller
{
    protected $productFileService;

    public function __construct(ProductFileService $productFileService)
    {
        $this->productFileService = $productFileService;
    }

    /**
     * Display files for a product
     */
    public function index(Product $product)
    {
        $files = $this->productFileService->getProductFiles($product, false);
        
        return view('admin.products.files.index', compact('product', 'files'));
    }

    /**
     * Store a newly uploaded file
     */
    public function store(Request $request, Product $product): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'file' => 'required|file|max:102400', // 100MB max
            'description' => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $file = $this->productFileService->uploadFile(
                $product,
                $request->file('file'),
                $request->input('description')
            );

            return response()->json([
                'success' => true,
                'message' => 'File uploaded successfully',
                'file' => [
                    'id' => $file->id,
                    'original_name' => $file->original_name,
                    'file_size' => $file->formatted_size,
                    'file_type' => $file->file_type,
                    'description' => $file->description,
                    'download_count' => $file->download_count,
                    'created_at' => $file->created_at->format('Y-m-d H:i:s'),
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('File upload failed', [
                'product_id' => $product->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'File upload failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Download a file (admin only)
     */
    public function download(ProductFile $file)
    {
        try {
            $fileData = $this->productFileService->downloadFile($file, auth()->id());
            
            if (!$fileData) {
                abort(404, 'File not found or access denied');
            }

            return response($fileData['content'])
                ->header('Content-Type', $fileData['mime_type'])
                ->header('Content-Disposition', 'attachment; filename="' . $fileData['filename'] . '"')
                ->header('Content-Length', $fileData['size']);
        } catch (\Exception $e) {
            Log::error('File download failed', [
                'file_id' => $file->id,
                'error' => $e->getMessage()
            ]);

            abort(500, 'Download failed');
        }
    }

    /**
     * Update file status (activate/deactivate)
     */
    public function update(Request $request, ProductFile $file): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'is_active' => 'required|boolean',
            'description' => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $file->update([
                'is_active' => $request->input('is_active'),
                'description' => $request->input('description'),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'File updated successfully',
                'file' => [
                    'id' => $file->id,
                    'is_active' => $file->is_active,
                    'description' => $file->description,
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('File update failed', [
                'file_id' => $file->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'File update failed'
            ], 500);
        }
    }

    /**
     * Delete a file
     */
    public function destroy(ProductFile $file): JsonResponse
    {
        try {
            $success = $this->productFileService->deleteFile($file);

            if ($success) {
                return response()->json([
                    'success' => true,
                    'message' => 'File deleted successfully'
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to delete file'
                ], 500);
            }
        } catch (\Exception $e) {
            Log::error('File deletion failed', [
                'file_id' => $file->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'File deletion failed'
            ], 500);
        }
    }

    /**
     * Get file statistics
     */
    public function statistics(Product $product): JsonResponse
    {
        $files = $product->files;
        
        $stats = [
            'total_files' => $files->count(),
            'active_files' => $files->where('is_active', true)->count(),
            'total_downloads' => $files->sum('download_count'),
            'total_size' => $files->sum('file_size'),
            'formatted_total_size' => $this->formatBytes($files->sum('file_size')),
        ];

        return response()->json([
            'success' => true,
            'statistics' => $stats
        ]);
    }

    /**
     * Format bytes to human readable format
     */
    private function formatBytes($bytes, $precision = 2)
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, $precision) . ' ' . $units[$i];
    }
}