<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\TicketCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class TicketCategoryController extends Controller
{
    public function index()
    {
        $categories = TicketCategory::orderBy('sort_order')->paginate(15);
        return view('admin.ticket-categories.index', compact('categories'));
    }

    public function create()
    {
        return view('admin.ticket-categories.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:ticket_categories,slug',
            'description' => 'nullable|string',
            'color' => 'required|string|max:7',
            'sort_order' => 'required|integer|min:0',
            'is_active' => 'boolean',
            'requires_login' => 'boolean',
            'requires_valid_purchase_code' => 'boolean',
            'meta_title' => 'nullable|string|max:255',
            'meta_keywords' => 'nullable|string|max:500',
            'meta_description' => 'nullable|string|max:500',
            'icon' => 'nullable|string|max:100',
            'priority' => 'nullable|string|in:low,medium,high,urgent',
        ]);

        $validated['slug'] = $validated['slug'] ?: Str::slug($validated['name']);

        TicketCategory::create($validated);

        return redirect()->route('admin.ticket-categories.index')
            ->with('success', 'Ticket category created successfully.');
    }

    public function show(TicketCategory $ticketCategory)
    {
        return view('admin.ticket-categories.show', compact('ticketCategory'));
    }

    public function edit(TicketCategory $ticketCategory)
    {
        return view('admin.ticket-categories.edit', compact('ticketCategory'));
    }

    public function update(Request $request, TicketCategory $ticketCategory)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:ticket_categories,slug,' . $ticketCategory->id,
            'description' => 'nullable|string',
            'color' => 'required|string|max:7',
            'sort_order' => 'required|integer|min:0',
            'is_active' => 'boolean',
            'requires_login' => 'boolean',
            'requires_valid_purchase_code' => 'boolean',
            'meta_title' => 'nullable|string|max:255',
            'meta_keywords' => 'nullable|string|max:500',
            'meta_description' => 'nullable|string|max:500',
            'icon' => 'nullable|string|max:100',
            'priority' => 'nullable|string|in:low,medium,high,urgent',
        ]);

        $validated['slug'] = $validated['slug'] ?: Str::slug($validated['name']);

        $ticketCategory->update($validated);

        return redirect()->route('admin.ticket-categories.index')
            ->with('success', 'Ticket category updated successfully.');
    }

    public function destroy(TicketCategory $ticketCategory)
    {
        $ticketCategory->delete();

        return redirect()->route('admin.ticket-categories.index')
            ->with('success', 'Ticket category deleted successfully.');
    }
}