<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Ticket;
use App\Models\TicketReply;
use App\Models\User;
use App\Models\TicketCategory;
use App\Models\Product;
use App\Models\Invoice;
use App\Services\EmailService;
use Illuminate\Http\Request;

class TicketController extends Controller
{
    protected EmailService $emailService;

    public function __construct(EmailService $emailService)
    {
        $this->emailService = $emailService;
    }

    public function index()
    {
        $tickets = Ticket::with(['user', 'category', 'invoice.product'])->latest()->paginate(10);
        return view('admin.tickets.index', compact('tickets'));
    }

    public function create()
    {
        $users = User::with(['licenses.product'])->get();
        $categories = TicketCategory::active()->ordered()->get();
        $products = Product::all();
        return view('admin.tickets.create', compact('users', 'categories', 'products'));
    }

    public function store(Request $request)
    {
        // Basic ticket validation
        $validated = $request->validate([
            'user_id' => ['required', 'exists:users,id'],
            'category_id' => ['required', 'exists:ticket_categories,id'],
            'subject' => ['required', 'string', 'max:255'],
            'priority' => ['required', 'in:low,medium,high'],
            'content' => ['required', 'string'],
            'create_invoice' => ['sometimes','boolean'],
        ]);

        // Build ticket data (don't pass invoice fields directly)
        $ticketData = [
            'user_id' => $validated['user_id'],
            'category_id' => $validated['category_id'],
            'subject' => $validated['subject'],
            'priority' => $validated['priority'],
            'content' => $validated['content'],
            'status' => 'open',
        ];

        // If admin requested to create an invoice for the user, validate invoice inputs and create
        if ($request->filled('create_invoice') && $request->boolean('create_invoice')) {
            $invoiceProductId = $request->input('invoice_product_id');
            $billingType = $request->input('billing_type', 'one_time');

            // If custom invoice selected, require amount and duration or due_date
            if ($invoiceProductId === 'custom' || empty($invoiceProductId)) {
                $invRules = [
                    'invoice_amount' => ['required','numeric'],
                    // either due date or duration in days should be provided; we'll accept duration
                    'invoice_duration_days' => ['required','integer','min:0'],
                    'invoice_status' => ['nullable','in:pending,paid,overdue,cancelled'],
                    'invoice_notes' => ['nullable','string'],
                ];
                $invData = $request->validate($invRules);

                $amount = $invData['invoice_amount'];
                $duration = $invData['invoice_duration_days'];
                $dueDate = now()->addDays($duration)->toDateString();
                $metadata = [];
                if ($billingType !== 'one_time') {
                    // for custom recurring allow specifying renewal price/period
                    if ($billingType === 'custom_recurring') {
                        $metadata['renewal_price'] = $request->input('invoice_renewal_price') ?: $amount;
                        $metadata['renewal_period_days'] = $request->input('invoice_renewal_period_days') ?: $duration;
                        $metadata['recurrence'] = 'custom';
                    } else {
                        // map standard types to days
                        $map = [
                            'monthly' => 30,
                            'quarterly' => 90,
                            'semi_annual' => 182,
                            'annual' => 365,
                        ];
                        $metadata['recurrence'] = $billingType;
                        $metadata['renewal_period_days'] = $map[$billingType] ?? $duration;
                        $metadata['renewal_price'] = $amount;
                    }
                }
            } else {
                // product-based invoice: ensure product exists
                $product = Product::find($invoiceProductId);
                if (!$product) {
                    return back()->withErrors(['invoice_product_id' => 'Invalid product selected'])->withInput();
                }

                // amount and duration may be provided to override, otherwise use product defaults
                $amount = $request->input('invoice_amount') ?: $product->price;
                $duration = $request->input('invoice_duration_days') ?: $product->duration_days ?: null;
                $dueDate = $request->input('invoice_due_date') ?: ($duration ? now()->addDays($duration)->toDateString() : null);
                $metadata = [];
                if ($billingType && $billingType !== 'one_time') {
                    $map = [
                        'monthly' => 30,
                        'quarterly' => 90,
                        'semi_annual' => 182,
                        'annual' => 365,
                    ];
                    if ($billingType === 'custom_recurring') {
                        $metadata['recurrence'] = 'custom';
                        $metadata['renewal_price'] = $request->input('invoice_renewal_price') ?: $product->renewal_price ?? $amount;
                        $metadata['renewal_period_days'] = $request->input('invoice_renewal_period_days') ?: $product->renewal_period ?: ($duration ?: 30);
                    } else {
                        $metadata['recurrence'] = $billingType;
                        $metadata['renewal_period_days'] = $map[$billingType] ?? ($product->renewal_period ?? $duration);
                        $metadata['renewal_price'] = $product->renewal_price ?? $amount;
                    }
                }
            }

            $invoice = Invoice::create([
                'user_id' => $validated['user_id'],
                'product_id' => is_numeric($invoiceProductId) ? $invoiceProductId : null,
                'amount' => $amount,
                'status' => $request->input('invoice_status') ?? 'pending',
                'due_date' => $dueDate,
                'notes' => $request->input('invoice_notes') ?? null,
                'currency' => config('app.currency', 'USD'),
                'type' => ($billingType && $billingType !== 'one_time') ? 'recurring' : 'one_time',
                'metadata' => $metadata ?? null,
            ]);

            if ($invoice) {
                $ticketData['invoice_id'] = $invoice->id;
            }
        }

        Ticket::create($ticketData);

        return redirect()->route('admin.tickets.index')->with('success', 'Ticket created successfully for user');
    }

    public function show(Ticket $ticket)
    {
        $ticket->load(['user.licenses.product', 'replies.user', 'category', 'invoice.product']);
        return view('admin.tickets.show', compact('ticket'));
    }

    public function edit(Ticket $ticket)
    {
        $categories = TicketCategory::active()->ordered()->get();
        return view('admin.tickets.edit', compact('ticket', 'categories'));
    }

    public function update(Request $request, Ticket $ticket)
    {
        $validated = $request->validate([
            'subject' => ['sometimes', 'string', 'max:255'],
            'priority' => ['sometimes', 'in:low,medium,high'],
            'status' => ['sometimes', 'in:open,pending,resolved,closed'],
            'category_id' => ['sometimes', 'exists:ticket_categories,id'],
            'content' => ['sometimes', 'string'],
        ]);
        
        $ticket->update($validated);
        
        return back()->with('success', 'Ticket updated');
    }

    public function destroy(Ticket $ticket)
    {
        $ticket->delete();
        return redirect()->route('admin.tickets.index')->with('success', 'Ticket deleted');
    }

    public function reply(Request $request, Ticket $ticket)
    {
        $validated = $request->validate([
            'message' => ['required', 'string'],
        ]);
        
        $reply = TicketReply::create([
            'ticket_id' => $ticket->id,
            'user_id' => $request->user()->id,
            'message' => $validated['message'],
        ]);

        // Send email notification to user when admin replies
        try {
            if ($ticket->user) {
                
                $result = $this->emailService->sendTicketReply($ticket->user, [
                    'ticket_id' => $ticket->id,
                    'ticket_subject' => $ticket->subject,
                    'reply_message' => $validated['message'],
                    'replied_by' => $request->user()->name,
                ]);
                
            }
        } catch (\Exception $e) {
        }
        
        return back()->with('success', 'Reply added');
    }

    public function updateStatus(Request $request, Ticket $ticket)
    {
        $validated = $request->validate([
            'status' => ['required', 'in:open,pending,resolved,closed'],
        ]);

        try {
            $oldStatus = $ticket->status;
            $ticket->status = $validated['status'];
            $saved = $ticket->save();

            if (!$saved) {
                // Failed to save ticket status update
                return back()->withErrors(['status' => 'Failed to update ticket status']);
            }

            // Send email notification to user when status is updated
            try {
                if ($ticket->user) {
                    
                    $result = $this->emailService->sendTicketStatusUpdate($ticket->user, [
                        'ticket_id' => $ticket->id,
                        'ticket_subject' => $ticket->subject,
                        'old_status' => $oldStatus,
                        'new_status' => $ticket->status,
                    ]);
                    
                }
            } catch (\Exception $e) {
            }

            return back()->with('success', 'Ticket status updated to ' . ucfirst($ticket->status));
        } catch (\Exception $e) {
            // Exception updating ticket status
            return back()->withErrors(['status' => 'Error updating ticket status: ' . $e->getMessage()]);
        }
    }
}