<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Validation\Rule;
use Spatie\Permission\Models\Role;

class UserController extends Controller
{
    // Middleware is applied at route level

    public function index()
    {
        $users = User::with(['licenses', 'tickets'])
            ->withCount(['licenses', 'tickets'])
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('admin.users.index', compact('users'));
    }

    public function create()
    {
        return view('admin.users.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'role' => 'required|in:user,admin',
            'firstname' => 'nullable|string|max:255',
            'lastname' => 'nullable|string|max:255',
            'companyname' => 'nullable|string|max:255',
            'phonenumber' => 'nullable|string|max:20',
            'address1' => 'nullable|string|max:255',
            'address2' => 'nullable|string|max:255',
            'city' => 'nullable|string|max:255',
            'state' => 'nullable|string|max:255',
            'postcode' => 'nullable|string|max:20',
            'country' => 'nullable|string|max:255',
        ]);

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'email_verified_at' => now(),
            'firstname' => $request->firstname,
            'lastname' => $request->lastname,
            'companyname' => $request->companyname,
            'phonenumber' => $request->phonenumber,
            'address1' => $request->address1,
            'address2' => $request->address2,
            'city' => $request->city,
            'state' => $request->state,
            'postcode' => $request->postcode,
            'country' => $request->country,
        ]);

        // Assign role
        $role = Role::where('name', $request->role)->first();
        if ($role) {
            $user->assignRole($role);
        }

        // Send welcome email if requested
        if ($request->send_welcome_email) {
            // You can implement welcome email here
        }

        return redirect()->route('admin.users.show', $user)
            ->with('success', 'User created successfully.');
    }

    public function show(User $user)
    {
        // Load related simple collections for display
        $user->load(['tickets']);

        // Use unified License scopes so admin views and user views rely on the same logic.
        $licenses = \App\Models\License::with(['product', 'domains'])
            ->forUser($user)
            ->latest()
            ->get();

        return view('admin.users.show', compact('user', 'licenses'));
    }

    public function edit(User $user)
    {
        return view('admin.users.edit', compact('user'));
    }

    public function update(Request $request, User $user)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => ['required', 'string', 'email', 'max:255', Rule::unique('users')->ignore($user->id)],
            'password' => 'nullable|string|min:8|confirmed',
            'role' => 'required|in:user,admin',
            'firstname' => 'nullable|string|max:255',
            'lastname' => 'nullable|string|max:255',
            'companyname' => 'nullable|string|max:255',
            'phonenumber' => 'nullable|string|max:20',
            'address1' => 'nullable|string|max:255',
            'address2' => 'nullable|string|max:255',
            'city' => 'nullable|string|max:255',
            'state' => 'nullable|string|max:255',
            'postcode' => 'nullable|string|max:20',
            'country' => 'nullable|string|max:255',
        ]);

        $user->update([
            'name' => $request->name,
            'email' => $request->email,
            'firstname' => $request->firstname,
            'lastname' => $request->lastname,
            'companyname' => $request->companyname,
            'phonenumber' => $request->phonenumber,
            'address1' => $request->address1,
            'address2' => $request->address2,
            'city' => $request->city,
            'state' => $request->state,
            'postcode' => $request->postcode,
            'country' => $request->country,
        ]);

        if ($request->filled('password')) {
            $user->update([
                'password' => Hash::make($request->password),
            ]);
        }

        // Update role
        $user->syncRoles([$request->role]);

        return redirect()->route('admin.users.show', $user)
            ->with('success', 'User updated successfully.');
    }

    public function destroy(User $user)
    {
        // Prevent deleting the current admin user
        if ($user->id === auth()->id()) {
            return redirect()->route('admin.users.index')
                ->with('error', 'You cannot delete your own account.');
        }

        $user->delete();

        return redirect()->route('admin.users.index')
            ->with('success', 'User deleted successfully.');
    }

    public function toggleAdmin(User $user)
    {
        if ($user->hasRole('admin')) {
            $user->removeRole('admin');
            $user->assignRole('user');
            $message = 'User role changed to regular user.';
        } else {
            $user->removeRole('user');
            $user->assignRole('admin');
            $message = 'User promoted to administrator.';
        }

        return redirect()->back()->with('success', $message);
    }

    public function sendPasswordReset(User $user)
    {
        // Generate password reset token and send email
        // This would typically use Laravel's built-in password reset functionality
        // For now, we'll just return a success message

        return redirect()->back()
            ->with('success', 'Password reset email sent to ' . $user->email);
    }

    /**
     * Get user licenses for API
     */
    public function getUserLicenses($userId)
    {
        try {
            $user = User::with(['licenses.product'])->findOrFail($userId);
            
            $licenses = $user->licenses->map(function ($license) {
                return [
                    'id' => $license->id,
                    'license_key' => $license->license_key,
                    'product_name' => $license->product ? $license->product->name : 'Unknown Product',
                    'product_price' => $license->product ? $license->product->price : 0,
                    'status' => $license->status,
                    'expires_at' => $license->expires_at,
                ];
            });

            return response()->json([
                'success' => true,
                'licenses' => $licenses
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'User not found',
                'licenses' => []
            ], 404);
        }
    }
}