<?php

declare(strict_types=1);

namespace App\Http\Controllers\Api;

use App\Http\Controllers\BaseController;
use App\Models\License;
use App\Models\Product;
use App\Services\EnhancedSecurityService;
use App\Services\EnvatoService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;

/**
 * Enhanced License API Controller
 * 
 * Provides secure, well-structured API endpoints for license management
 * with comprehensive security measures and proper error handling.
 */
class EnhancedLicenseApiController extends BaseController
{
    public function __construct(
        private readonly EnvatoService $envatoService,
        private readonly EnhancedSecurityService $securityService
    ) {
        $this->middleware('throttle:api');
    }

    /**
     * Verify license endpoint with enhanced security
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function verify(Request $request): JsonResponse
    {
        try {
            // Security checks
            $this->performSecurityChecks($request);

            // Rate limiting
            $clientFingerprint = $this->securityService->getClientFingerprint($request);
            if (!$this->securityService->checkRateLimit('license_verification', $clientFingerprint)) {
                $this->logSecurityEvent('Rate limit exceeded', $request, [
                    'action' => 'license_verification',
                    'client_fingerprint' => $clientFingerprint,
                ]);
                
                return $this->errorResponse(
                    'Too many verification attempts. Please try again later.',
                    Response::HTTP_TOO_MANY_REQUESTS
                );
            }

            // Validate request
            $validated = $this->validateRequest($request, [
                'purchase_code' => 'required|string|min:10|max:100|regex:/^[A-Z0-9\-]+$/',
                'product_slug' => 'required|string|max:255|regex:/^[a-z0-9\-]+$/',
                'domain' => 'nullable|string|max:255|regex:/^[a-zA-Z0-9\-\.]+$/',
                'verification_key' => 'nullable|string|max:255',
                'client_info' => 'nullable|array',
                'client_info.version' => 'nullable|string|max:50',
                'client_info.platform' => 'nullable|string|max:50',
            ]);

            // Check authorization
            if (!$this->isAuthorized($request)) {
                $this->logSecurityEvent('Unauthorized API access', $request, [
                    'purchase_code' => $this->securityService->hashForLogging($validated['purchase_code']),
                ]);
                
                return $this->errorResponse('Unauthorized', Response::HTTP_UNAUTHORIZED);
            }

            // Find product
            $product = $this->findProduct($validated['product_slug']);
            if (!$product) {
                return $this->errorResponse('Product not found', Response::HTTP_NOT_FOUND);
            }

            // Verify verification key if provided
            if (isset($validated['verification_key']) && 
                !$this->verifyVerificationKey($product, $validated['verification_key'])) {
                $this->logSecurityEvent('Invalid verification key', $request, [
                    'product_slug' => $validated['product_slug'],
                ]);
                
                return $this->errorResponse('Invalid verification key', Response::HTTP_FORBIDDEN);
            }

            // Process license verification
            $result = $this->processLicenseVerification($product, $validated, $request);

            // Log successful verification
            $this->logSuccessfulVerification($request, $result);

            return $this->jsonResponse($result, 'License verified successfully');

        } catch (ValidationException $e) {
            return $this->errorResponse(
                'Validation failed',
                Response::HTTP_UNPROCESSABLE_ENTITY,
                $e->errors()
            );
        } catch (\Exception $e) {
            return $this->handleException($e, $request, 'License verification');
        }
    }

    /**
     * Register license endpoint
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function register(Request $request): JsonResponse
    {
        try {
            // Security checks
            $this->performSecurityChecks($request);

            // Rate limiting
            $clientFingerprint = $this->securityService->getClientFingerprint($request);
            if (!$this->securityService->checkRateLimit('api_requests', $clientFingerprint)) {
                return $this->errorResponse(
                    'Too many requests. Please try again later.',
                    Response::HTTP_TOO_MANY_REQUESTS
                );
            }

            // Check authorization
            if (!$this->isAuthorized($request)) {
                return $this->errorResponse('Unauthorized', Response::HTTP_UNAUTHORIZED);
            }

            // Validate request
            $validated = $this->validateRequest($request, [
                'purchase_code' => 'required|string|min:10|max:100|regex:/^[A-Z0-9\-]+$/',
                'product_slug' => 'required|string|max:255|regex:/^[a-z0-9\-]+$/',
                'domain' => 'nullable|string|max:255|regex:/^[a-zA-Z0-9\-\.]+$/',
                'envato_data' => 'nullable|array',
                'customer_info' => 'nullable|array',
            ]);

            // Find product
            $product = $this->findProduct($validated['product_slug']);
            if (!$product) {
                return $this->errorResponse('Product not found', Response::HTTP_NOT_FOUND);
            }

            // Check if license already exists
            $existingLicense = License::where('purchase_code', $validated['purchase_code'])
                ->where('product_id', $product->id)
                ->first();

            if ($existingLicense) {
                return $this->jsonResponse([
                    'license_id' => $existingLicense->id,
                    'status' => 'already_exists'
                ], 'License already exists');
            }

            // Create new license
            $license = $this->createLicense($product, $validated);

            // Log registration
            $this->logLicenseRegistration($request, $license);

            return $this->jsonResponse([
                'license_id' => $license->id,
                'status' => 'created'
            ], 'License registered successfully');

        } catch (ValidationException $e) {
            return $this->errorResponse(
                'Validation failed',
                Response::HTTP_UNPROCESSABLE_ENTITY,
                $e->errors()
            );
        } catch (\Exception $e) {
            return $this->handleException($e, $request, 'License registration');
        }
    }

    /**
     * Get license status endpoint
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function status(Request $request): JsonResponse
    {
        try {
            // Security checks
            $this->performSecurityChecks($request);

            // Rate limiting
            $clientFingerprint = $this->securityService->getClientFingerprint($request);
            if (!$this->securityService->checkRateLimit('api_requests', $clientFingerprint)) {
                return $this->errorResponse(
                    'Too many requests. Please try again later.',
                    Response::HTTP_TOO_MANY_REQUESTS
                );
            }

            // Validate request
            $validated = $this->validateRequest($request, [
                'license_key' => 'required|string|max:255',
                'product_slug' => 'required|string|max:255|regex:/^[a-z0-9\-]+$/',
            ]);

            // Find product
            $product = $this->findProduct($validated['product_slug']);
            if (!$product) {
                return $this->errorResponse('Product not found', Response::HTTP_NOT_FOUND);
            }

            // Find license
            $license = License::where('license_key', $validated['license_key'])
                ->where('product_id', $product->id)
                ->first();

            if (!$license) {
                return $this->errorResponse('License not found', Response::HTTP_NOT_FOUND);
            }

            // Check license status
            $isActive = $this->isLicenseActive($license);
            
            $statusData = [
                'license_id' => $license->id,
                'type' => $license->license_type,
                'expires_at' => $license->license_expires_at?->toISOString(),
                'support_expires_at' => $license->support_expires_at?->toISOString(),
                'status' => $license->status,
                'is_active' => $isActive,
            ];

            return $this->jsonResponse($statusData, 'License status retrieved');

        } catch (ValidationException $e) {
            return $this->errorResponse(
                'Validation failed',
                Response::HTTP_UNPROCESSABLE_ENTITY,
                $e->errors()
            );
        } catch (\Exception $e) {
            return $this->handleException($e, $request, 'License status check');
        }
    }

    /**
     * Perform comprehensive security checks
     *
     * @param Request $request
     * @return void
     */
    private function performSecurityChecks(Request $request): void
    {
        // Check for suspicious activity
        $suspicious = $this->securityService->detectSuspiciousActivity($request);
        if (!empty($suspicious)) {
            $this->logSecurityEvent('Suspicious activity detected', $request, [
                'suspicious_patterns' => $suspicious,
            ]);
            abort(Response::HTTP_FORBIDDEN, 'Suspicious activity detected');
        }

        // Check IP blacklist
        if ($this->securityService->isIpBlacklisted($request->ip())) {
            $this->logSecurityEvent('Blacklisted IP access attempt', $request);
            abort(Response::HTTP_FORBIDDEN, 'Access denied');
        }
    }

    /**
     * Check if request is authorized
     *
     * @param Request $request
     * @return bool
     */
    private function isAuthorized(Request $request): bool
    {
        $authHeader = $request->header('Authorization');
        $expectedToken = 'Bearer ' . $this->getApiToken();
        
        return $expectedToken && $authHeader === $expectedToken;
    }

    /**
     * Get API token from settings
     *
     * @return string
     */
    private function getApiToken(): string
    {
        return \App\Helpers\ConfigHelper::getSetting('license_api_token', '', 'LICENSE_API_TOKEN');
    }

    /**
     * Find product by slug
     *
     * @param string $slug
     * @return Product|null
     */
    private function findProduct(string $slug): ?Product
    {
        return Cache::remember("product_slug_{$slug}", 3600, function () use ($slug) {
            return Product::where('slug', $slug)->first();
        });
    }

    /**
     * Verify verification key
     *
     * @param Product $product
     * @param string $verificationKey
     * @return bool
     */
    private function verifyVerificationKey(Product $product, string $verificationKey): bool
    {
        $expectedKey = hash('sha256', $product->id . $product->slug . config('app.key'));
        return hash_equals($expectedKey, $verificationKey);
    }

    /**
     * Process license verification
     *
     * @param Product $product
     * @param array $validated
     * @param Request $request
     * @return array
     */
    private function processLicenseVerification(Product $product, array $validated, Request $request): array
    {
        $purchaseCode = $validated['purchase_code'];
        $domain = $validated['domain'] ?? null;

        // Check database first
        $license = License::where('purchase_code', $purchaseCode)
            ->where('product_id', $product->id)
            ->first();

        if ($license) {
            return $this->verifyExistingLicense($license, $domain, $request);
        }

        // Try Envato API
        return $this->verifyWithEnvato($product, $purchaseCode, $domain, $request);
    }

    /**
     * Verify existing license
     *
     * @param License $license
     * @param string|null $domain
     * @param Request $request
     * @return array
     */
    private function verifyExistingLicense(License $license, ?string $domain, Request $request): array
    {
        if (!$this->isLicenseActive($license)) {
            $this->logSecurityEvent('Inactive license verification attempt', $request, [
                'license_id' => $license->id,
                'status' => $license->status,
            ]);
            
            abort(Response::HTTP_FORBIDDEN, 'License is not active');
        }

        // Handle domain verification
        if ($domain && !$this->verifyDomain($license, $domain)) {
            $this->logSecurityEvent('Unauthorized domain verification attempt', $request, [
                'license_id' => $license->id,
                'domain' => $domain,
            ]);
            
            abort(Response::HTTP_FORBIDDEN, 'Domain not authorized for this license');
        }

        return [
            'license_id' => $license->id,
            'license_type' => $license->license_type,
            'expires_at' => $license->license_expires_at?->toISOString(),
            'support_expires_at' => $license->support_expires_at?->toISOString(),
            'status' => $license->status,
            'verification_method' => 'database',
        ];
    }

    /**
     * Verify with Envato API
     *
     * @param Product $product
     * @param string $purchaseCode
     * @param string|null $domain
     * @param Request $request
     * @return array
     */
    private function verifyWithEnvato(Product $product, string $purchaseCode, ?string $domain, Request $request): array
    {
        $envatoData = $this->envatoService->verifyPurchase($purchaseCode);
        
        if (!$envatoData || !isset($envatoData['item']['id']) || 
            $envatoData['item']['id'] != $product->envato_item_id) {
            
            $this->logSecurityEvent('Invalid Envato verification', $request, [
                'purchase_code' => $this->securityService->hashForLogging($purchaseCode),
                'product_id' => $product->id,
            ]);
            
            abort(Response::HTTP_NOT_FOUND, 'License not found');
        }

        // Create license from Envato data
        $license = $this->createLicenseFromEnvato($product, $purchaseCode, $envatoData);

        return [
            'license_id' => $license->id,
            'license_type' => $license->license_type,
            'expires_at' => $license->license_expires_at?->toISOString(),
            'support_expires_at' => $license->support_expires_at?->toISOString(),
            'status' => $license->status,
            'verification_method' => 'envato_auto_created',
        ];
    }

    /**
     * Check if license is active
     *
     * @param License $license
     * @return bool
     */
    private function isLicenseActive(License $license): bool
    {
        if ($license->status !== 'active') {
            return false;
        }

        if ($license->license_expires_at && $license->license_expires_at->isPast()) {
            return false;
        }

        return true;
    }

    /**
     * Verify domain authorization
     *
     * @param License $license
     * @param string $domain
     * @return bool
     */
    private function verifyDomain(License $license, string $domain): bool
    {
        // Clean domain
        $domain = preg_replace('/^https?:\/\//', '', $domain);
        $domain = preg_replace('/^www\./', '', $domain);

        $authorizedDomains = $license->domains()->where('status', 'active')->get();

        if ($authorizedDomains->isEmpty()) {
            $this->registerDomainForLicense($license, $domain);
            return true;
        }

        foreach ($authorizedDomains as $authorizedDomain) {
            $authDomain = preg_replace('/^https?:\/\//', '', $authorizedDomain->domain);
            $authDomain = preg_replace('/^www\./', '', $authDomain);

            if ($authDomain === $domain) {
                $authorizedDomain->update(['last_used_at' => now()]);
                return true;
            }

            // Check wildcard domains
            if (str_starts_with($authDomain, '*.')) {
                $pattern = str_replace('*.', '', $authDomain);
                if (str_ends_with($domain, $pattern)) {
                    $authorizedDomain->update(['last_used_at' => now()]);
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * Register domain for license
     *
     * @param License $license
     * @param string $domain
     * @return void
     */
    private function registerDomainForLicense(License $license, string $domain): void
    {
        $cleanDomain = preg_replace('/^https?:\/\//', '', $domain);
        $cleanDomain = preg_replace('/^www\./', '', $cleanDomain);

        $existingDomain = $license->domains()
            ->where('domain', $cleanDomain)
            ->first();

        if (!$existingDomain) {
            $license->domains()->create([
                'domain' => $cleanDomain,
                'status' => 'active',
                'added_at' => now(),
                'last_used_at' => now()
            ]);
        } else {
            $existingDomain->update(['last_used_at' => now()]);
        }
    }

    /**
     * Create license from Envato data
     *
     * @param Product $product
     * @param string $purchaseCode
     * @param array $envatoData
     * @return License
     */
    private function createLicenseFromEnvato(Product $product, string $purchaseCode, array $envatoData): License
    {
        return License::create([
            'product_id' => $product->id,
            'purchase_code' => $purchaseCode,
            'license_key' => $purchaseCode,
            'license_type' => $product->license_type ?? 'regular',
            'support_expires_at' => now()->addDays($product->support_days ?? 365),
            'license_expires_at' => $product->license_type === 'extended' ? now()->addYear() : null,
            'status' => 'active',
        ]);
    }

    /**
     * Create new license
     *
     * @param Product $product
     * @param array $validated
     * @return License
     */
    private function createLicense(Product $product, array $validated): License
    {
        $license = License::create([
            'product_id' => $product->id,
            'purchase_code' => $validated['purchase_code'],
            'license_key' => $this->generateLicenseKey(),
            'license_type' => $product->license_type ?? 'regular',
            'support_expires_at' => now()->addDays($product->support_days ?? 365),
            'license_expires_at' => $product->license_type === 'extended' ? now()->addYear() : null,
            'status' => 'active',
        ]);

        if (isset($validated['domain'])) {
            $license->domains()->create([
                'domain' => $validated['domain'],
                'status' => 'active'
            ]);
        }

        return $license;
    }

    /**
     * Generate unique license key
     *
     * @return string
     */
    private function generateLicenseKey(): string
    {
        do {
            $key = strtoupper(substr(md5(uniqid(mt_rand(), true)), 0, 8) . '-' .
                         substr(md5(uniqid(mt_rand(), true)), 0, 8) . '-' .
                         substr(md5(uniqid(mt_rand(), true)), 0, 8) . '-' .
                         substr(md5(uniqid(mt_rand(), true)), 0, 8));
        } while (License::where('license_key', $key)->exists());

        return $key;
    }

    /**
     * Log successful verification
     *
     * @param Request $request
     * @param array $result
     * @return void
     */
    private function logSuccessfulVerification(Request $request, array $result): void
    {
        Log::info('License verification successful', [
            'license_id' => $result['license_id'],
            'verification_method' => $result['verification_method'],
            'ip' => $request->ip(),
            'user_agent' => $request->userAgent(),
        ]);
    }

    /**
     * Log license registration
     *
     * @param Request $request
     * @param License $license
     * @return void
     */
    private function logLicenseRegistration(Request $request, License $license): void
    {
        Log::info('License registered successfully', [
            'license_id' => $license->id,
            'product_id' => $license->product_id,
            'ip' => $request->ip(),
            'user_agent' => $request->userAgent(),
        ]);
    }
}
