<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\License;
use App\Models\Product;
use App\Models\Setting;
use App\Services\EnvatoService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class LicenseApiController extends Controller
{
    protected $envatoService;

    public function __construct(EnvatoService $envatoService)
    {
        $this->envatoService = $envatoService;
    }

    /**
     * Get API token from database settings
     */
    private function getApiToken(): string
    {
        return \App\Helpers\ConfigHelper::getSetting('license_api_token', '', 'LICENSE_API_TOKEN');
    }

    /**
     * Verify license endpoint
     * This endpoint is used by the generated license files
     */
    public function verify(Request $request): JsonResponse
    {
        // Check authorization header
        $authHeader = $request->header('Authorization');
        $expectedToken = 'Bearer ' . $this->getApiToken();
        
        if (!$expectedToken || $authHeader !== $expectedToken) {
            // Log unauthorized attempt
            $this->logApiVerificationAttempt($request, false, 'Unauthorized access attempt', 'api');
            
            return response()->json([
                'valid' => false,
                'message' => 'Unauthorized',
                'error_code' => 'UNAUTHORIZED'
            ], 401);
        }

        $validated = $request->validate([
            'purchase_code' => 'required|string|min:10|max:100',
            'product_slug' => 'required|string|max:255',
            'domain' => 'nullable|string|max:255',
            'verification_key' => 'nullable|string|max:255',
        ]);

        $purchaseCode = $validated['purchase_code'];
        $productSlug = $validated['product_slug'];
        $domain = $validated['domain'] ?? null;
        $verificationKey = $validated['verification_key'] ?? null;

        try {
            // Find product by slug
            $product = Product::where('slug', $productSlug)->first();

            if (!$product) {
                $this->logApiVerificationAttempt($request, false, 'Product not found', 'api');
                
                return response()->json([
                    'valid' => false,
                    'message' => 'Product not found',
                    'error_code' => 'PRODUCT_NOT_FOUND'
                ], 404);
            }

            // Verify verification key if provided
            if ($verificationKey && !$this->verifyVerificationKey($product, $verificationKey)) {
                $this->logApiVerificationAttempt($request, false, 'Invalid verification key', 'api');
                
                return response()->json([
                    'valid' => false,
                    'message' => 'Invalid verification key',
                    'error_code' => 'INVALID_VERIFICATION_KEY'
                ], 403);
            }

            // Step 1: Check if license exists in our database
            $license = License::where('purchase_code', $purchaseCode)
                ->where('product_id', $product->id)
                ->first();

            $databaseValid = false;
            $envatoValid = false;
            $verificationMethod = '';

            // Step 2: Check database license first
            if ($license) {
                // License exists, check its status
                if ($license->status === 'active') {
                    // Check if license has expired
                    if ($license->license_expires_at && $license->license_expires_at->isPast()) {
                        $this->logApiVerificationAttempt($request, false, 'License has expired', 'api');
                        return response()->json([
                            'valid' => false,
                            'message' => 'License has expired',
                            'error_code' => 'LICENSE_EXPIRED',
                            'data' => [
                                'expires_at' => $license->license_expires_at->toISOString()
                            ]
                        ], 403);
                    }
                    
                    $databaseValid = true;
                    $verificationMethod = 'database_only';
                } elseif ($license->status === 'suspended') {
                    $this->logApiVerificationAttempt($request, false, 'License is suspended', 'api');
                    return response()->json([
                        'valid' => false,
                        'message' => 'License is suspended',
                        'error_code' => 'LICENSE_SUSPENDED'
                    ], 403);
                } else {
                    $this->logApiVerificationAttempt($request, false, 'License is not active', 'api');
                    return response()->json([
                        'valid' => false,
                        'message' => 'License is not active',
                        'error_code' => 'LICENSE_INACTIVE'
                    ], 403);
                }
            } else {
                // Step 3: If not in database, try Envato API
                $envatoData = $this->envatoService->verifyPurchase($purchaseCode);
                if ($envatoData && isset($envatoData['item']['id']) && $envatoData['item']['id'] == $product->envato_item_id) {
                    $envatoValid = true;
                    // Create license automatically from Envato
                    $license = $this->createLicenseFromEnvato($product, $purchaseCode, $envatoData);
                    $databaseValid = true;
                    $verificationMethod = 'envato_auto_created';
                } else {
                    // Both invalid - reject and log error
                    $this->logApiVerificationAttempt($request, false, 'License not found', 'api');
                    
                    return response()->json([
                        'valid' => false,
                        'message' => 'License not found',
                        'error_code' => 'LICENSE_NOT_FOUND'
                    ], 404);
                }
            }

            // Step 5: Handle domain registration/verification
            if ($domain) {
                // Check if auto domain registration is enabled
                $autoRegisterDomains = \App\Helpers\ConfigHelper::getSetting('license_auto_register_domains', false);
                $isTestMode = config('app.env') === 'local' || config('app.debug') === true;
                
                if ($autoRegisterDomains || $isTestMode) {
                    // Auto register mode: Register domain automatically
                    $this->registerDomainForLicense($license, $domain);
                    $this->logApiVerificationAttempt($request, true, 'Domain registered automatically', 'api', [
                        'domain' => $domain,
                        'license_id' => $license->id,
                        'mode' => $isTestMode ? 'test' : 'auto_register'
                    ]);
                } else {
                    // Verification mode: Verify domain authorization
                    if (!$this->verifyDomain($license, $domain)) {
                        $this->logApiVerificationAttempt($request, false, 'Domain not authorized for this license', 'api');
                        
                        return response()->json([
                            'valid' => false,
                            'message' => 'Domain not authorized for this license',
                            'error_code' => 'DOMAIN_NOT_AUTHORIZED'
                        ], 403);
                    }
                }
            }

            // Step 6: Log successful verification
            $this->logApiVerificationAttempt($request, true, 'License verified successfully', 'api', [
                'license_id' => $license->id,
                'verification_method' => $verificationMethod,
                'envato_valid' => $envatoValid,
                'database_valid' => $databaseValid,
            ]);

            // Step 7: Log verification in database
            $this->logVerification($license, $domain, $verificationMethod);

            return response()->json([
                'valid' => true,
                'message' => 'License verified successfully',
                'data' => [
                    'license_id' => $license->id,
                    'license_type' => $license->license_type,
                    'expires_at' => $license->license_expires_at?->toISOString(),
                    'support_expires_at' => $license->support_expires_at?->toISOString(),
                    'status' => $license->status,
                    'verification_method' => $verificationMethod,
                    'envato_valid' => $envatoValid,
                    'database_valid' => $databaseValid,
                ]
            ], 200);

        } catch (\Exception $e) {
            $this->logApiVerificationAttempt($request, false, 'Verification failed: ' . $e->getMessage(), 'api');
            
            return response()->json([
                'valid' => false,
                'message' => 'Verification failed: ' . $e->getMessage(),
                'error_code' => 'INTERNAL_ERROR'
            ], 500);
        }
    }

    /**
     * Verify verification key
     */
    private function verifyVerificationKey(Product $product, string $verificationKey): bool
    {
        $expectedKey = hash('sha256', $product->id . $product->slug . config('app.key'));
        return hash_equals($expectedKey, $verificationKey);
    }

    /**
     * Create license from Envato data
     */
    private function createLicenseFromEnvato(Product $product, string $purchaseCode, array $envatoData): License
    {
        $license = License::create([
            'product_id' => $product->id,
            'purchase_code' => $purchaseCode,
            'license_key' => $purchaseCode, // Use same value as purchase_code
            'license_type' => $product->license_type ?? 'regular',
            'support_expires_at' => now()->addDays($product->support_days ?? 365),
            'license_expires_at' => $product->license_type === 'extended' ? now()->addYear() : null,
            'status' => 'active',
        ]);
        
        // Log the license creation from Envato
        $this->logVerification($license, null, 'envato_auto_created');
        
        return $license;
    }

    /**
     * Verify domain authorization
     */
    private function verifyDomain(License $license, string $domain): bool
    {
        // Remove protocol and www
        $domain = preg_replace('/^https?:\/\//', '', $domain);
        $domain = preg_replace('/^www\./', '', $domain);

        $authorizedDomains = $license->domains()->where('status', 'active')->get();

        // If no domains are configured, automatically register the current domain
        if ($authorizedDomains->isEmpty()) {
            $this->registerDomainForLicense($license, $domain);
            return true;
        }

        foreach ($authorizedDomains as $authorizedDomain) {
            $authDomain = preg_replace('/^https?:\/\//', '', $authorizedDomain->domain);
            $authDomain = preg_replace('/^www\./', '', $authDomain);

            if ($authDomain === $domain) {
                // Update last used timestamp
                $authorizedDomain->update(['last_used_at' => now()]);
                // Log verification for exact domain match
                $this->logVerification($license, $domain, 'domain_verification_exact');
                return true;
            }

            // Check wildcard domains
            if (str_starts_with($authDomain, '*.')) {
                $pattern = str_replace('*.', '', $authDomain);
                if (str_ends_with($domain, $pattern)) {
                    // Update last used timestamp
                    $authorizedDomain->update(['last_used_at' => now()]);
                    // Log verification for wildcard domain match
                    $this->logVerification($license, $domain, 'domain_verification_wildcard');
                    return true;
                }
            }
        }

        // Domain not found in authorized domains
        return false;
    }

    /**
     * Register domain for license automatically
     */
    private function registerDomainForLicense(License $license, string $domain): void
    {
        // Clean domain (remove protocol and www)
        $cleanDomain = preg_replace('/^https?:\/\//', '', $domain);
        $cleanDomain = preg_replace('/^www\./', '', $cleanDomain);
        
        // Check if domain already exists for this license
        $existingDomain = $license->domains()
            ->where('domain', $cleanDomain)
            ->first();
            
        if ($existingDomain) {
            // Update last used timestamp
            $existingDomain->update(['last_used_at' => now()]);
            \Log::info('Domain already exists, updated last used timestamp', [
                'license_id' => $license->id,
                'domain' => $cleanDomain,
                'domain_id' => $existingDomain->id
            ]);
            
            // Log verification for existing domain
            $this->logVerification($license, $cleanDomain, 'domain_verification_existing');
        } else {
            // Create new domain record
            $license->domains()->create([
                'domain' => $cleanDomain,
                'status' => 'active',
                'added_at' => now(),
                'last_used_at' => now()
            ]);

            // Log the automatic domain registration
            \Log::info('Domain automatically registered for license', [
                'license_id' => $license->id,
                'purchase_code' => $license->purchase_code,
                'domain' => $cleanDomain,
                'ip' => request()->ip(),
                'user_agent' => request()->userAgent(),
                'mode' => config('app.env') === 'local' || config('app.debug') ? 'test' : 'production'
            ]);
            
            // Log verification for new domain
            $this->logVerification($license, $cleanDomain, 'domain_verification_new');
        }
    }

    /**
     * Log verification error (only once per hour per IP to avoid spam)
     */
    private function logVerificationError(string $purchaseCode, ?string $domain, string $errorType): void
    {
        $ip = request()->ip();
        $cacheKey = "verification_error_logged_{$ip}_{$purchaseCode}_{$errorType}";
        
        // Check if we already logged this error in the last hour
        if (\Cache::has($cacheKey)) {
            return; // Don't log again
        }
        
        // Log the error
        \Log::warning('License verification error', [
            'purchase_code' => $purchaseCode,
            'domain' => $domain,
            'error_type' => $errorType,
            'ip' => $ip,
            'user_agent' => request()->userAgent(),
            'timestamp' => now()
        ]);
        
        // Cache for 1 hour to prevent spam logging
        \Cache::put($cacheKey, true, 3600);
    }

    /**
     * Check if license is active
     */
    private function isLicenseActive(License $license): bool
    {
        if ($license->status !== 'active') {
            return false;
        }

        // Check license expiration
        if ($license->license_expires_at && $license->license_expires_at->isPast()) {
            return false;
        }

        return true;
    }

    /**
     * Generate unique license key
     */
    private function generateLicenseKey(): string
    {
        return strtoupper(substr(md5(uniqid(mt_rand(), true)), 0, 8) . '-' .
                         substr(md5(uniqid(mt_rand(), true)), 0, 8) . '-' .
                         substr(md5(uniqid(mt_rand(), true)), 0, 8) . '-' .
                         substr(md5(uniqid(mt_rand(), true)), 0, 8));
    }

    /**
     * Log API verification attempt
     */
    private function logApiVerificationAttempt(Request $request, bool $isValid, string $message, string $source = 'api', ?array $responseData = null): void
    {
        try {
            $purchaseCode = $request->input('purchase_code', 'unknown');
            $domain = $request->input('domain', 'unknown');
            
            // Log the verification attempt using our logger
            \Log::info('License verification attempt', [
                'purchase_code' => $purchaseCode,
                'domain' => $domain,
                'is_valid' => $isValid,
                'message' => $message,
                'source' => $source,
                'response_data' => $responseData,
                'timestamp' => now()
            ]);
            
        } catch (\Exception $e) {
            // Silent fail - don't break API if logging fails
            \Log::error('Failed to log API verification attempt: ' . $e->getMessage());
        }
    }

    /**
     * Log license verification
     */
    private function logVerification(License $license, ?string $domain, string $method): void
    {
        $license->logs()->create([
            'domain' => $domain ?? 'unknown',
            'ip_address' => request()->ip(),
            'serial' => $license->purchase_code,
            'status' => 'success',
            'user_agent' => request()->userAgent(),
            'request_data' => [
                'verification_method' => $method,
                'domain' => $domain,
                'ip' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ],
            'response_data' => [
                'valid' => true,
                'method' => $method
            ]
        ]);
    }

    /**
     * Register license endpoint
     * This endpoint is used to register licenses from Envato
     */
    public function register(Request $request): JsonResponse
    {
        // Check authorization header
        $authHeader = $request->header('Authorization');
        $expectedToken = 'Bearer ' . $this->getApiToken();
        
        if (!$expectedToken || $authHeader !== $expectedToken) {
            $this->logApiVerificationAttempt($request, false, 'Unauthorized registration attempt', 'api_register');
            
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $validated = $request->validate([
            'purchase_code' => 'required|string|min:10|max:100',
            'product_slug' => 'required|string|max:255',
            'domain' => 'nullable|string|max:255',
            'envato_data' => 'nullable|array',
        ]);

        $purchaseCode = $validated['purchase_code'];
        $productSlug = $validated['product_slug'];
        $domain = $validated['domain'] ?? null;
        $envatoData = $validated['envato_data'] ?? [];

        try {
            // Find product by slug
            $product = Product::where('slug', $productSlug)->first();

            if (!$product) {
                $this->logApiVerificationAttempt($request, false, 'Product not found for registration', 'api_register');
                
                return response()->json([
                    'success' => false,
                    'message' => 'Product not found'
                ], 404);
            }

            // Check if license already exists
            $existingLicense = License::where('purchase_code', $purchaseCode)
                ->where('product_id', $product->id)
                ->first();

            if ($existingLicense) {
                $this->logApiVerificationAttempt($request, true, 'License already exists', 'api_register');
                
                return response()->json([
                    'success' => true,
                    'message' => 'License already exists'
                ]);
            }

            // Create new license
            $license = License::create([
                'product_id' => $product->id,
                'purchase_code' => $purchaseCode,
                'license_key' => $this->generateLicenseKey(),
                'license_type' => $product->license_type ?? 'regular',
                'support_expires_at' => now()->addDays($product->support_days ?? 365),
                'license_expires_at' => $product->license_type === 'extended' ? now()->addYear() : null,
                'status' => 'active',
            ]);

            // Add domain if provided
            if ($domain) {
                $license->domains()->create([
                    'domain' => $domain,
                    'status' => 'active'
                ]);
            }

            // Log the license registration
            $this->logVerification($license, $domain, 'license_registration');

            $this->logApiVerificationAttempt($request, true, 'License registered successfully', 'api_register', [
                'license_id' => $license->id,
                'product_slug' => $productSlug
            ]);

            return response()->json([
                'success' => true,
                'message' => 'License registered successfully',
                'license_id' => $license->id
            ]);

        } catch (\Exception $e) {
            $this->logApiVerificationAttempt($request, false, 'Registration failed: ' . $e->getMessage(), 'api_register');
            
            return response()->json([
                'success' => false,
                'message' => 'Registration failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get license status endpoint
     */
    public function status(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'license_key' => 'required|string',
            'product_slug' => 'required|string',
        ]);

        $licenseKey = $validated['license_key'];
        $productSlug = $validated['product_slug'];

        try {
            $product = Product::where('slug', $productSlug)->first();

            if (!$product) {
                $this->logApiVerificationAttempt($request, false, 'Product not found for status check', 'api_status');
                
                return response()->json([
                    'valid' => false,
                    'message' => 'Product not found'
                ], 404);
            }

            $license = License::where('license_key', $licenseKey)
                ->where('product_id', $product->id)
                ->first();

            if (!$license) {
                $this->logApiVerificationAttempt($request, false, 'License not found for status check', 'api_status');
                
                return response()->json([
                    'valid' => false,
                    'message' => 'License not found'
                ], 404);
            }

            $isActive = $this->isLicenseActive($license);
            
            // Log the status check
            if ($isActive) {
                $this->logVerification($license, null, 'status_check_success');
            }
            
            $this->logApiVerificationAttempt($request, $isActive, $isActive ? 'License status check successful' : 'License status check failed - inactive', 'api_status', [
                'license_id' => $license->id,
                'license_key' => $licenseKey,
                'product_slug' => $productSlug,
                'is_active' => $isActive
            ]);

            return response()->json([
                'valid' => $isActive,
                'license' => [
                    'id' => $license->id,
                    'type' => $license->license_type,
                    'expires_at' => $license->license_expires_at?->toISOString(),
                    'support_expires_at' => $license->support_expires_at?->toISOString(),
                    'status' => $license->status,
                ],
                'product' => [
                    'name' => $product->name,
                    'version' => $product->version,
                ]
            ]);

        } catch (\Exception $e) {
            $this->logApiVerificationAttempt($request, false, 'Status check failed: ' . $e->getMessage(), 'api_status');
            
            return response()->json([
                'valid' => false,
                'message' => 'Status check failed: ' . $e->getMessage()
            ], 500);
        }
    }
}