<?php

namespace App\Http\Controllers;

use App\Models\KbArticle;
use App\Models\KbCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class KbArticleController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $articles = KbArticle::with('category')->latest()->paginate(10);
        $categories = KbCategory::all();
        return view('admin.kb.articles.index', compact('articles', 'categories'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $categories = KbCategory::pluck('name','id');
        return view('admin.kb.articles.create', compact('categories'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'kb_category_id' => ['required','exists:kb_categories,id'],
            'title' => ['required','string','max:255'],
            'slug' => ['nullable','string','max:255','unique:kb_articles,slug'],
            'excerpt' => ['nullable','string'],
            'content' => ['required','string'],
            'image' => ['nullable','image','max:2048'],
            'is_published' => ['sometimes','boolean'],
            'serial' => ['nullable','string','max:255'],
            'requires_serial' => ['sometimes','boolean'],
            'serial_message' => ['nullable','string'],
            'meta_title' => ['nullable','string','max:255'],
            'meta_description' => ['nullable','string','max:500'],
            'meta_keywords' => ['nullable','string','max:255'],
            'allow_comments' => ['boolean'],
            'is_featured' => ['boolean'],
        ]);
        $validated['slug'] = $validated['slug'] ?: Str::slug($validated['title']);
        $validated['is_published'] = $request->boolean('is_published');
        
        // Handle checkbox values - set to false if not present
        $validated['allow_comments'] = $request->has('allow_comments');
        $validated['is_featured'] = $request->has('is_featured');
        $validated['requires_serial'] = $request->has('requires_serial');
        // handle image upload
        if ($request->hasFile('image')) {
            $path = $request->file('image')->store('kb_images', 'public');
            $validated['image'] = $path;
        }

        KbArticle::create($validated);
        return redirect()->route('admin.kb-articles.index')->with('success', 'Article created');
    }

    /**
     * Display the specified resource.
     */
    public function show(KbArticle $kbArticle)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(KbArticle $kbArticle)
    {
        $categories = KbCategory::pluck('name','id');
        return view('admin.kb.articles.edit', ['article'=>$kbArticle, 'categories'=>$categories]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, KbArticle $kbArticle)
    {
        $validated = $request->validate([
            'kb_category_id' => ['required','exists:kb_categories,id'],
            'title' => ['required','string','max:255'],
            'slug' => ['required','string','max:255','unique:kb_articles,slug,'.$kbArticle->id],
            'excerpt' => ['nullable','string'],
            'content' => ['required','string'],
            'image' => ['nullable','image','max:2048'],
            'is_published' => ['sometimes','boolean'],
            'serial' => ['nullable','string','max:255'],
            'requires_serial' => ['sometimes','boolean'],
            'serial_message' => ['nullable','string'],
            'meta_title' => ['nullable','string','max:255'],
            'meta_description' => ['nullable','string','max:500'],
            'meta_keywords' => ['nullable','string','max:255'],
            'allow_comments' => ['boolean'],
            'is_featured' => ['boolean'],
        ]);
        $validated['is_published'] = $request->boolean('is_published');
        
        // Handle checkbox values - set to false if not present
        $validated['allow_comments'] = $request->has('allow_comments');
        $validated['is_featured'] = $request->has('is_featured');
        $validated['requires_serial'] = $request->has('requires_serial');
        
        if ($request->hasFile('image')) {
            $path = $request->file('image')->store('kb_images', 'public');
            $validated['image'] = $path;
        }

        $kbArticle->update($validated);
        return back()->with('success', 'Article updated');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(KbArticle $kbArticle)
    {
        $kbArticle->delete();
        return redirect()->route('admin.kb-articles.index')->with('success', 'Article deleted');
    }
}
