<?php

namespace App\Http\Controllers;

use App\Models\KbArticle;
use App\Models\KbCategory;
use Illuminate\Http\Request;

/**
 * Knowledge Base Controller
 * 
 * This controller handles knowledge base operations.
 */
class KbController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $categories = KbCategory::with('articles')->get();
        $recentArticles = KbArticle::where('status', 'published')
            ->latest()
            ->limit(10)
            ->get();

        return view('kb.index', compact('categories', 'recentArticles'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $categories = KbCategory::all();
        return view('kb.create', compact('categories'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'required|string|max:255|unique:kb_articles,slug',
            'content' => 'required|string',
            'category_id' => 'required|exists:kb_categories,id',
            'status' => 'required|in:draft,published',
            'meta_description' => 'nullable|string|max:160',
            'tags' => 'nullable|string',
        ]);

        $validated['user_id'] = auth()->id();

        KbArticle::create($validated);

        return redirect()->route('kb.index')
            ->with('success', 'Article created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(KbArticle $article)
    {
        $article->load('category');
        
        return view('kb.show', compact('article'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(KbArticle $article)
    {
        $categories = KbCategory::all();
        
        return view('kb.edit', compact('article', 'categories'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, KbArticle $article)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'required|string|max:255|unique:kb_articles,slug,' . $article->id,
            'content' => 'required|string',
            'category_id' => 'required|exists:kb_categories,id',
            'status' => 'required|in:draft,published',
            'meta_description' => 'nullable|string|max:160',
            'tags' => 'nullable|string',
        ]);

        $article->update($validated);

        return redirect()->route('kb.show', $article)
            ->with('success', 'Article updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(KbArticle $article)
    {
        $article->delete();

        return redirect()->route('kb.index')
            ->with('success', 'Article deleted successfully.');
    }

    /**
     * Search articles
     */
    public function search(Request $request)
    {
        $query = $request->get('q');
        
        $articles = KbArticle::where('status', 'published')
            ->where(function($q) use ($query) {
                $q->where('title', 'like', "%{$query}%")
                  ->orWhere('content', 'like', "%{$query}%")
                  ->orWhere('tags', 'like', "%{$query}%");
            })
            ->with('category')
            ->paginate(10);

        return view('kb.search', compact('articles', 'query'));
    }

    /**
     * Show articles by category
     */
    public function category(KbCategory $category)
    {
        $articles = $category->articles()
            ->where('status', 'published')
            ->paginate(10);

        return view('kb.category', compact('category', 'articles'));
    }
}
