<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\ProductFile;
use App\Services\ProductFileService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class ProductFileController extends Controller
{
    protected $productFileService;

    public function __construct(ProductFileService $productFileService)
    {
        $this->productFileService = $productFileService;
    }

    /**
     * Download a product file (user must have valid license and paid invoice)
     */
    public function download(ProductFile $file)
    {
        // Ensure user is authenticated
        if (!auth()->check()) {
            abort(401, 'Authentication required');
        }

        // Check if product is downloadable
        if (!$file->product->is_downloadable) {
            abort(403, 'This product does not support file downloads');
        }

        // Check if file is active
        if (!$file->is_active) {
            abort(404, 'File not available');
        }

        // Check user permissions
        $permissions = $this->productFileService->userCanDownloadFiles($file->product, auth()->id());
        
        if (!$permissions['can_download']) {
            abort(403, $permissions['message']);
        }

        try {
            $fileData = $this->productFileService->downloadFile($file, auth()->id());
            
            if (!$fileData) {
                abort(403, 'File download failed');
            }

            Log::info('User downloaded product file', [
                'user_id' => auth()->id(),
                'file_id' => $file->id,
                'product_id' => $file->product_id,
                'filename' => $fileData['filename']
            ]);

            return response($fileData['content'])
                ->header('Content-Type', $fileData['mime_type'])
                ->header('Content-Disposition', 'attachment; filename="' . $fileData['filename'] . '"')
                ->header('Content-Length', $fileData['size'])
                ->header('Cache-Control', 'no-cache, no-store, must-revalidate')
                ->header('Pragma', 'no-cache')
                ->header('Expires', '0');
        } catch (\Exception $e) {
            Log::error('User file download failed', [
                'user_id' => auth()->id(),
                'file_id' => $file->id,
                'error' => $e->getMessage()
            ]);

            abort(500, 'Download failed');
        }
    }

    /**
     * Get downloadable files for a product (user must have valid license and paid invoice)
     */
    public function index(Product $product)
    {
        // Ensure user is authenticated
        if (!auth()->check()) {
            abort(401, 'Authentication required');
        }

        // Check if product is downloadable
        if (!$product->is_downloadable) {
            abort(403, 'This product does not support file downloads');
        }

        // Check user permissions
        $permissions = $this->productFileService->userCanDownloadFiles($product, auth()->id());
        
        if (!$permissions['can_download']) {
            return view('user.products.files.index', compact('product'))
                ->with('permissions', $permissions)
                ->with('files', collect([]));
        }

        $files = $this->productFileService->getProductFiles($product, true);

        return view('user.products.files.index', compact('product', 'files', 'permissions'));
    }

    /**
     * Download all files as a ZIP archive
     */
    public function downloadAll(Product $product)
    {
        // Ensure user is authenticated
        if (!auth()->check()) {
            abort(401, 'Authentication required');
        }

        // Check if product is downloadable
        if (!$product->is_downloadable) {
            abort(403, 'This product does not support file downloads');
        }

        // Check user permissions
        $permissions = $this->productFileService->userCanDownloadFiles($product, auth()->id());
        
        if (!$permissions['can_download']) {
            abort(403, $permissions['message']);
        }

        $files = $this->productFileService->getProductFiles($product, true);

        if ($files->isEmpty()) {
            abort(404, 'No files available for download');
        }

        try {
            // Create temporary ZIP file
            $zipFileName = $product->slug . '_files_' . now()->format('Y-m-d_H-i-s') . '.zip';
            $zipPath = storage_path('app/temp/' . $zipFileName);
            
            // Ensure temp directory exists
            if (!file_exists(dirname($zipPath))) {
                mkdir(dirname($zipPath), 0755, true);
            }

            $zip = new \ZipArchive();
            if ($zip->open($zipPath, \ZipArchive::CREATE) !== TRUE) {
                abort(500, 'Cannot create ZIP file');
            }

            $addedFiles = 0;
            foreach ($files as $file) {
                $fileData = $this->productFileService->downloadFile($file, auth()->id());
                if ($fileData) {
                    $zip->addFromString($file->original_name, $fileData['content']);
                    $addedFiles++;
                }
            }

            $zip->close();

            if ($addedFiles === 0) {
                unlink($zipPath);
                abort(500, 'No files could be added to ZIP');
            }

            Log::info('User downloaded all product files as ZIP', [
                'user_id' => auth()->id(),
                'product_id' => $product->id,
                'files_count' => $addedFiles
            ]);

            return response()->download($zipPath, $zipFileName)->deleteFileAfterSend(true);
        } catch (\Exception $e) {
            Log::error('ZIP download failed', [
                'user_id' => auth()->id(),
                'product_id' => $product->id,
                'error' => $e->getMessage()
            ]);

            abort(500, 'ZIP creation failed');
        }
    }
}