<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Http\Requests\ProfileUpdateRequest;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Redirect;
use Illuminate\View\View;

class ProfileController extends Controller
{
    /**
     * Display the user's profile.
     */
    public function index(Request $request): View
    {
        $user = $request->user()->load(['licenses.product', 'licenses.domains', 'tickets']);

        return view('profile.index', [
            'user' => $user,
        ]);
    }

    /**
     * Display the user's profile form.
     */
    public function edit(Request $request): View
    {
        return view('profile.index', [
            'user' => $request->user(),
        ]);
    }

    /**
     * Update the user's profile information.
     */
    public function update(ProfileUpdateRequest $request): RedirectResponse
    {
        $request->user()->fill($request->validated());

        if ($request->user()->isDirty('email')) {
            $request->user()->email_verified_at = null;
        }

        $request->user()->save();

        // Redirect to intended URL if exists, otherwise to profile edit
        $intendedUrl = session('url.intended');
        if ($intendedUrl) {
            session()->forget('url.intended');
            return redirect($intendedUrl)->with('success', 'profile-updated');
        }

        return Redirect::route('profile.edit')->with('success', 'profile-updated');
    }

    /**
     * Unlink Envato account from user profile.
     */
    public function unlinkEnvato(Request $request): RedirectResponse
    {
        $user = $request->user();
        
        $user->envato_username = null;
        $user->envato_id = null;
        $user->envato_token = null;
        $user->envato_refresh_token = null;
        $user->envato_token_expires_at = null;
        $user->save();

        return Redirect::route('profile.edit')
            ->with('success', 'envato-unlinked');
    }

    /**
     * Delete the user's account.
     */
    public function destroy(Request $request): RedirectResponse
    {
        $request->validateWithBag('userDeletion', [
            'password' => ['required', 'current_password'],
        ]);

        $user = $request->user();

        Auth::logout();

        $user->delete();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return Redirect::to('/');
    }


    /**
     * Link Envato account to user profile.
     */
    public function linkEnvato(Request $request): RedirectResponse
    {
        $request->validate([
            'envato_username' => 'required|string|max:255',
        ]);

        $user = $request->user();
        $user->update([
            'envato_username' => $request->envato_username,
        ]);

        return redirect()->route('profile.edit')->with('success', 'envato-linked');
    }
}