<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Symfony\Component\HttpFoundation\Response;

class CheckInstallation
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        $installedFile = storage_path('.installed');
        $currentRoute = $request->route() ? $request->route()->getName() : '';
        
        
        // Skip installation check for certain routes (but not install routes if system is installed)
        $skipRoutes = ['up', 'health'];
        
        foreach ($skipRoutes as $skipRoute) {
            if ($currentRoute === $skipRoute) {
                return $next($request);
            }
        }
        
        // If system is not installed and not accessing install routes
        if (!File::exists($installedFile) && !$request->is('install*')) {
            // For AJAX requests, return JSON response
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'System not installed. Please run installation.',
                    'redirect' => route('install.welcome')
                ], 403);
            }
            return redirect()->route('install.welcome');
        }
        
        // If system is installed and trying to access install routes
        if (File::exists($installedFile) && ($request->is('install*') || str_contains($request->path(), 'install'))) {
            // For AJAX requests, return JSON response
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'System is already installed.',
                    'redirect' => route('login')
                ], 403);
            }
            return redirect()->route('login')->with('info', 'System is already installed.');
        }
        
        // If system is installed and trying to access install routes by route name
        if (File::exists($installedFile) && str_starts_with($currentRoute, 'install.')) {
            // For AJAX requests, return JSON response
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'System is already installed.',
                    'redirect' => route('login')
                ], 403);
            }
            return redirect()->route('login')->with('info', 'System is already installed.');
        }
        
        return $next($request);
    }
}
