<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use LicenseProtection\LicenseVerifier;
use Symfony\Component\HttpFoundation\Response;

class LicenseProtection
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        // Skip license check for installation routes
        if ($request->is('install*')) {
            return $next($request);
        }

        // Skip license check for API routes (they have their own protection)
        if ($request->is('api*')) {
            return $next($request);
        }

        // Skip license check for public routes
        if ($request->is('license-status*') || $request->is('kb*') || $request->is('support*')) {
            return $next($request);
        }

        // Check if system is installed
        $installedFile = storage_path('.installed');
        if (!File::exists($installedFile)) {
            return $next($request);
        }

        try {
            // Get license information from database
            $licenseInfo = $this->getLicenseInfo();
            
            if (!$licenseInfo) {
                return $this->handleLicenseError('No license information found');
            }

            // Check if license is expired (optional - you can implement this)
            if ($this->isLicenseExpired($licenseInfo)) {
                return $this->handleLicenseError('License has expired');
            }

            // Verify license periodically (every 24 hours)
            if ($this->shouldVerifyLicense($licenseInfo)) {
                $this->verifyLicensePeriodically($licenseInfo);
            }

        } catch (\Exception $e) {
            // Log the error but don't block the request
            \Log::error('License protection error: ' . $e->getMessage());
        }

        return $next($request);
    }

    /**
     * Get license information from database
     */
    private function getLicenseInfo(): ?array
    {
        try {
            $settings = \App\Models\Setting::where('type', 'license')->get();
            
            if ($settings->isEmpty()) {
                return null;
            }

            $licenseInfo = [];
            foreach ($settings as $setting) {
                $licenseInfo[$setting->key] = $setting->value;
            }

            return $licenseInfo;
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Check if license is expired
     */
    private function isLicenseExpired(array $licenseInfo): bool
    {
        // Implement your license expiration logic here
        // For now, we'll assume licenses don't expire
        return false;
    }

    /**
     * Check if we should verify license periodically
     */
    private function shouldVerifyLicense(array $licenseInfo): bool
    {
        $lastVerification = $licenseInfo['license_last_verification'] ?? null;
        
        if (!$lastVerification) {
            return true;
        }

        $lastVerificationTime = \Carbon\Carbon::parse($lastVerification);
        $now = \Carbon\Carbon::now();
        
        // Verify every 24 hours
        return $now->diffInHours($lastVerificationTime) >= 24;
    }

    /**
     * Verify license periodically
     */
    private function verifyLicensePeriodically(array $licenseInfo): void
    {
        try {
            $licenseVerifier = new LicenseVerifier();
            $result = $licenseVerifier->verifyLicense(
                $licenseInfo['license_purchase_code'],
                $licenseInfo['license_domain']
            );

            // Update last verification time
            \App\Models\Setting::updateOrCreate(
                ['key' => 'license_last_verification', 'type' => 'license'],
                ['value' => now()->toISOString()]
            );

            // If license is invalid, log it but don't block the request immediately
            if (!$result['valid']) {
                \Log::warning('License verification failed during periodic check', [
                    'purchase_code' => substr($licenseInfo['license_purchase_code'], 0, 8) . '...',
                    'domain' => $licenseInfo['license_domain'],
                    'message' => $result['message']
                ]);
            }

        } catch (\Exception $e) {
            \Log::error('Periodic license verification failed: ' . $e->getMessage());
        }
    }

    /**
     * Handle license error
     */
    private function handleLicenseError(string $message): Response
    {
        // For now, we'll just log the error and continue
        // You can implement more strict measures if needed
        \Log::warning('License protection triggered: ' . $message);
        
        // Return a response or redirect to license verification page
        return response()->view('errors.license', [
            'message' => $message
        ], 403);
    }
}
