<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Remove the failed migration from migrations table
        DB::table('migrations')->where('migration', '2025_09_26_041707_update_licenses_license_type_values')->delete();
        
        // Check if max_domains column exists and drop it if it does (from failed migration)
        if (Schema::hasColumn('licenses', 'max_domains')) {
            Schema::table('licenses', function (Blueprint $table) {
                $table->dropColumn('max_domains');
            });
        }
        
        // Now properly add max_domains column (if it doesn't exist)
        if (!Schema::hasColumn('licenses', 'max_domains')) {
            Schema::table('licenses', function (Blueprint $table) {
                $table->integer('max_domains')->default(1);
            });
        }
        
        // Create a temporary column with the new enum values (if it doesn't exist)
        if (!Schema::hasColumn('licenses', 'license_type_temp')) {
            Schema::table('licenses', function (Blueprint $table) {
                $table->enum('license_type_temp', ['single', 'multi', 'developer', 'extended'])->default('single');
            });
        }
        
        // Copy data from old column to new column, converting 'regular' to 'single' (if license_type exists)
        if (Schema::hasColumn('licenses', 'license_type')) {
            DB::statement("UPDATE licenses SET license_type_temp = CASE 
                WHEN license_type = 'regular' THEN 'single'
                WHEN license_type = 'extended' THEN 'extended'
                ELSE 'single'
            END");
            
            // Drop the old column
            Schema::table('licenses', function (Blueprint $table) {
                $table->dropColumn('license_type');
            });
        }
        
        // Rename the new column to the original name.
        // Use a raw ALTER TABLE statement to avoid Blueprint creating malformed default quoting
        // which can produce SQL like default '''single'''.
        if (Schema::hasColumn('licenses', 'license_type_temp')) {
            // MySQL ALTER TABLE CHANGE requires the full column definition; ensure correct ENUM and default
            DB::statement("ALTER TABLE `licenses` CHANGE `license_type_temp` `license_type` ENUM('single','multi','developer','extended') NOT NULL DEFAULT 'single'");
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // Create a temporary column with the old enum values
        Schema::table('licenses', function (Blueprint $table) {
            $table->enum('license_type_old', ['regular', 'extended'])->default('regular');
        });
        
        // Copy data from new column to old column, converting 'single' to 'regular'
        DB::statement("UPDATE licenses SET license_type_old = CASE 
            WHEN license_type = 'single' THEN 'regular'
            WHEN license_type = 'extended' THEN 'extended'
            ELSE 'regular'
        END");
        
        // Drop the new column
        Schema::table('licenses', function (Blueprint $table) {
            $table->dropColumn('license_type');
        });
        
        // Rename the old column back
        Schema::table('licenses', function (Blueprint $table) {
            $table->renameColumn('license_type_old', 'license_type');
        });
        
        // Drop the max_domains column
        Schema::table('licenses', function (Blueprint $table) {
            $table->dropColumn('max_domains');
        });
    }
};