<?php
namespace LithoAddons\Dynamic_Tags;

use Elementor\Core\DynamicTags\Tag as Tag;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Litho dynamic tag for post custom field.
 *
 * @package Litho
 */

// If class `Post_Custom_Field` doesn't exists yet.
if ( ! class_exists( 'LithoAddons\Dynamic_Tags\Post_Custom_Field' ) ) {
	/**
	 * Define `Post_Custom_Field` class.
	 */
	class Post_Custom_Field extends Tag {

		/**
		 * Retrieve the name.
		 *
		 * @access public
		 * @return string name.
		 */
		public function get_name() {
			return 'post-custom-field';
		}

		/**
		 * Retrieve the title.
		 *
		 * @access public
		 *
		 * @return string title.
		 */
		public function get_title() {
			return esc_html__( 'Post Custom Field', 'litho-addons' );
		}

		/**
		 * Retrieve the group.
		 *
		 * @access public
		 *
		 * @return string group.
		 */
		public function get_group() {
			return 'post';
		}

		/**
		 * Retrieve the categories.
		 *
		 * @access public
		 *
		 * @return string categories.
		 */
		public function get_categories() {
			return [
				\Elementor\Modules\DynamicTags\Module::TEXT_CATEGORY,
				\Elementor\Modules\DynamicTags\Module::URL_CATEGORY,
				\Elementor\Modules\DynamicTags\Module::POST_META_CATEGORY,
			];
		}

		/**
		 * Retrieve the panel template settings.
		 *
		 * @access public
		 *
		 * @return string template key.
		 */
		public function get_panel_template_setting_key() {
			return 'key';
		}

		/**
		 * Retrieve settings required.
		 *
		 * @access public
		 *
		 * @return string settings.
		 */
		public function is_settings_required() {
			return true;
		}

		/**
		 * Register post custom fields controls.
		 *
		 * @access protected
		 */
		protected function register_controls() {
			$this->add_control(
				'litho_key',
				[
					'label'     => esc_html__( 'Litho Meta Key', 'litho-addons' ),
					'type'      => 'select',
					'options'   => $this->get_custom_litho_keys_array(),
					'condition' => [
						'key'        => '',
						'custom_key' => '',
					],
				]
			);

			$this->add_control(
				'key',
				[
					'label'     => esc_html__( 'Meta Key', 'litho-addons' ),
					'type'      => 'select',
					'options'   => $this->get_custom_keys_array(),
					'condition' => [
						'litho_key' => '',
					],
				]
			);

			$this->add_control(
				'custom_key',
				[
					'label'       => esc_html__( 'Custom Meta Key', 'litho-addons' ),
					'type'        => 'text',
					'placeholder' => 'key',
					'condition'   => [
						'key'       => '',
						'litho_key' => '',
					],
				]
			);
		}

		/**
		 * Render post custom fields.
		 *
		 * @access public
		 */
		public function render() {
			$select_key = $this->get_settings( 'key' );
			$litho_key  = $this->get_settings( 'litho_key' );
			$custom_key = $this->get_settings( 'custom_key' );

			$key = '';
			if ( ! empty( $litho_key ) ) {
				$key = $litho_key;
			} elseif ( ! empty( $select_key ) ) {
				$key = $select_key;
			} elseif ( ! empty( $custom_key ) ) {
				$key = $custom_key;
			}

			if ( empty( $key ) ) {
				return;
			}

			$post_id = get_the_ID();
			$value   = get_post_meta( $post_id, $key, true );

			if ( ! empty( $value ) ) {
				if ( is_array( $value ) ) {
					foreach ( $value as $val ) {
						echo wp_kses_post( $val );
					}
				} else {
					echo wp_kses_post( $value );
				}
			}
		}

		private function get_custom_keys_array() {
			$custom_keys = get_post_custom_keys();
			$options     = [
				'' => esc_html__( 'Select...', 'litho-addons' ),
			];

			if ( ! empty( $custom_keys ) ) {
				foreach ( $custom_keys as $custom_key ) {
					if ( '_' !== substr( $custom_key, 0, 1 ) ) {
						$options[ $custom_key ] = $custom_key;
					}
				}
			}

			return $options;
		}

		private function get_custom_litho_keys_array() {
			$post_id     = get_the_ID();
			$custom_meta = get_post_meta( $post_id );
			$options     = [
				'' => esc_html__( 'Select...', 'litho-addons' ),
			];

			if ( ! empty( $custom_meta ) && is_array( $custom_meta ) ) {
				foreach ( $custom_meta as $select_key => $meta_value ) {
					if ( strpos( $select_key, 'litho_' ) === 1 ) {
						$options[ $select_key ] = $select_key;
					}
				}
			}
			return $options;
		}
	}
}
