<?php
namespace LithoAddons\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Repeater;
use Elementor\Plugin;
use Elementor\Icons_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Text_Shadow;
use LithoAddons\Classes\Elementor_Templates;
use Elementor\Core\Kits\Documents\Tabs\Global_Typography;

/**
 * Litho widget for dynamic slider.
 *
 * @package Litho
 */

// If class `Dynamic_Slider` doesn't exists yet.
if ( ! class_exists( 'LithoAddons\Widgets\Dynamic_Slider' ) ) {
	/**
	 * Define `Dynamic_Slider` class.
	 */
	class Dynamic_Slider extends Widget_Base {
		/**
		 * Retrieve the widget name.
		 *
		 * @access public
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'litho-dynamic-slider';
		}

		/**
		 * Retrieve the widget title
		 *
		 * @access public
		 * @return string Widget title.
		 */
		public function get_title() {
			return esc_html__( 'Litho Dynamic Slider', 'litho-addons' );
		}

		/**
		 * Retrieve the widget icon.
		 *
		 * @access public
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-slider-push';
		}

		/**
		 * Retrieve the widget categories.
		 *
		 * @access public
		 * @return string Widget categories.
		 */
		public function get_categories() {
			return [
				'litho',
			];
		}
		/**
		 * Get widget keywords.
		 *
		 * Retrieve the list of keywords the widget belongs to.
		 *
		 * @access public
		 * @return array Widget keywords.
		 */
		public function get_keywords() {
			return [
				'slider',
				'slide',
				'text',
				'carousel',
				'swiper',
				'dynamic',
				'custom slider',
				'content slider',
				'dynamic carousel',
			];
		}
		/**
		 * Register dynamic slider widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @access protected
		 */
		protected function register_controls() {
			$this->start_controls_section(
				'litho_section_dynamic_carousel',
				[
					'label' => esc_html__( 'Slides', 'litho-addons' ),
				]
			);

			$repeater = new Repeater();
			$repeater->add_control(
				'litho_item_content_type',
				[
					'label'       => esc_html__( 'Content Type', 'litho-addons' ),
					'type'        => Controls_Manager::SELECT,
					'default'     => 'editor',
					'options'     => [
						'template' => esc_html__( 'Template', 'litho-addons' ),
						'editor'   => esc_html__( 'Editor', 'litho-addons' ),
					],
					'label_block' => true,
				]
			);
			$repeater->add_control(
				'litho_item_template_id',
				[
					'label'       => esc_html__( 'Choose Template', 'litho-addons' ),
					'label_block' => true,
					'type'        => Controls_Manager::SELECT2,
					'default'     => '0',
					'options'     => Elementor_Templates::get_elementor_templates_options(),
					'condition'   => [
						'litho_item_content_type' => 'template',
					],
				]
			);
			$repeater->add_control(
				'litho_carousel_title',
				[
					'label'       => esc_html__( 'Title', 'litho-addons' ),
					'type'        => Controls_Manager::TEXT,
					'dynamic'     => [
						'active' => true,
					],
					'default'     => esc_html__( 'Write Title here', 'litho-addons' ),
					'label_block' => true,
					'condition'   => [
						'litho_item_content_type' => 'editor',
					],
				]
			);
			$repeater->add_control(
				'litho_carousel_subtitle',
				[
					'label'       => esc_html__( 'Subtitle', 'litho-addons' ),
					'type'        => Controls_Manager::TEXT,
					'dynamic'     => [
						'active' => true,
					],
					'default'     => esc_html__( 'Write Subtitle Here', 'litho-addons' ),
					'label_block' => true,
					'condition'   => [
						'litho_item_content_type' => 'editor',
					],
				]
			);
			$repeater->add_control(
				'litho_content',
				[
					'label'       => esc_html__( 'Content', 'litho-addons' ),
					'type'        => Controls_Manager::WYSIWYG,
					'dynamic'     => [
						'active' => true,
					],
					'default'     => esc_html__( 'Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo.', 'litho-addons' ),
					'label_block' => true,
					'condition'   => [
						'litho_item_content_type' => 'editor',
					],
				]
			);
			$this->add_control(
				'litho_carousel_slider',
				[
					'label'       => esc_html__( 'Carousel Items', 'litho-addons' ),
					'label_block' => true,
					'show_label'  => false,
					'type'        => Controls_Manager::REPEATER,
					'fields'      => $repeater->get_controls(),
					'default'     => [
						[
							'litho_carousel_subtitle' => esc_html__( 'Write Subtitle here', 'litho-addons' ),
							'litho_carousel_title'    => esc_html__( 'Write Title Here', 'litho-addons' ),
							'litho_content'           => esc_html__( 'Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo.', 'litho-addons' ),
						],
						[
							'litho_carousel_subtitle' => esc_html__( 'Write Subtitle here', 'litho-addons' ),
							'litho_carousel_title'    => esc_html__( 'Write Title Here', 'litho-addons' ),
							'litho_content'           => esc_html__( 'Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo.', 'litho-addons' ),
						],
					],
				]
			);
			$this->add_control(
				'litho_dynamic_slider_title_size',
				[
					'label'   => esc_html__( 'Title HTML Tag', 'litho-addons' ),
					'type'    => Controls_Manager::SELECT,
					'options' => [
						'h1'   => 'H1',
						'h2'   => 'H2',
						'h3'   => 'H3',
						'h4'   => 'H4',
						'h5'   => 'H5',
						'h6'   => 'H6',
						'div'  => 'div',
						'span' => 'span',
						'p'    => 'p',
					],
					'default' => 'h3',
				]
			);
			$this->end_controls_section();

			$this->start_controls_section(
				'litho_section_carousel_setting',
				[
					'label' => esc_html__( 'Slider Configuration', 'litho-addons' ),
				]
			);
			$this->add_responsive_control(
				'litho_slides_to_show',
				[
					'label'     => esc_html__( 'Slides Per View', 'litho-addons' ),
					'type'      => Controls_Manager::SLIDER,
					'default'   => [
						'size' => 1,
					],
					'range'     => [
						'px' => [
							'min'  => 1,
							'max'  => 10,
							'step' => 1,
						],
					],
					'condition' => [
						'litho_effect' => 'slide',
					],
				]
			);
			$this->add_responsive_control(
				'litho_items_spacing',
				[
					'label'      => esc_html__( 'Items Spacing', 'litho-addons' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => [
						'px',
					],
					'range'      => [
						'px' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default'    => [
						'unit' => 'px',
						'size' => 30,
					],
					'condition'  => [
						'litho_slides_to_show[size]!' => '1',
						'litho_effect'                => 'slide',
					],
				]
			);
			$this->add_control(
				'litho_navigation',
				[
					'label' 		=> esc_html__( 'Navigation', 'litho-addons' ),
					'type' 			=> Controls_Manager::SELECT,
					'default' 		=> 'both',
					'options' 		=> [
						'both' 			=> esc_html__( 'Arrows and Dots', 'litho-addons' ),
						'arrows' 		=> esc_html__( 'Arrows', 'litho-addons' ),
						'dots' 			=> esc_html__( 'Dots', 'litho-addons' ),
						'none' 			=> esc_html__( 'None', 'litho-addons' ),
					],
				]
			);
			$this->add_control(
				'litho_navigation_dynamic_bullets',
				[
					'label' 		=> esc_html__( 'Dynamic Bullets', 'litho-addons' ),
					'type' 			=> Controls_Manager::SWITCHER,
					'default' 		=> '',
					'condition' => [
						'litho_navigation' => [
							'both',
							'dots',
						],
					],
				]
			);
			$this->add_control(
				'litho_autoplay',
				[
					'label'   => esc_html__( 'Enable Autoplay', 'litho-addons' ),
					'type'    => Controls_Manager::SWITCHER,
					'default' => 'no',
				]
			);
			$this->add_control(
				'litho_autoplay_speed',
				[
					'label'     => esc_html__( 'Autoplay Delay', 'litho-addons' ),
					'type'      => Controls_Manager::NUMBER,
					'default'   => 5000,
					'condition' => [
						'litho_autoplay' => 'yes',
					],
				]
			);
			$this->add_control(
				'litho_pause_on_hover',
				[
					'label'     => esc_html__( 'Enable Pause on Hover', 'litho-addons' ),
					'type'      => Controls_Manager::SWITCHER,
					'default'   => 'yes',
					'condition' => [
						'litho_autoplay' => 'yes',
					],
				]
			);
			$this->add_control(
				'litho_infinite',
				[
					'label'   => esc_html__( 'Enable Infinite Loop', 'litho-addons' ),
					'type'    => Controls_Manager::SWITCHER,
					'default' => 'yes',
				]
			);
			$this->add_control(
				'litho_mousewheel',
				[
					'label' => esc_html__( 'Enable Mousewheel', 'litho-addons' ),
					'type'  => Controls_Manager::SWITCHER,
				]
			);
			$this->add_control(
				'litho_speed',
				[
					'label'   => esc_html__( 'Effect Speed', 'litho-addons' ),
					'type'    => Controls_Manager::NUMBER,
					'default' => 500,
				]
			);
			$this->add_control(
				'litho_effect',
				[
					'label'   => esc_html__( 'Effect', 'litho-addons' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'slide',
					'options' => [
						'slide'	=> esc_html__( 'Slide', 'litho-addons' ),
						'fade'	=> esc_html__( 'Fade', 'litho-addons' ),
					],
				]
			);	
			$this->add_control(
				'litho_rtl',
				[
					'label'   => esc_html__( 'RTL', 'litho-addons' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'ltr',
					'options' => [
						''		=> esc_html__( 'Default', 'litho-addons' ),
						'ltr'	=> esc_html__( 'Left', 'litho-addons' ),
						'rtl' 	=> esc_html__( 'Right', 'litho-addons' ),
					],
				]
			);
			$this->add_control(
				'litho_slider_cursor',
				[
					'label'   => esc_html__( 'Cursor', 'litho-addons' ),
					'type'    => Controls_Manager::SELECT,
					'options' => [
						'' 			   => esc_html__( 'Default', 'litho-addons' ),
						'white-cursor' => esc_html__( 'White Cursor', 'litho-addons' ),
						'black-cursor' => esc_html__( 'Black Cursor', 'litho-addons' ),
					],
				]
			);
			$this->end_controls_section();

			$this->start_controls_section(
				'litho_section_arrows_options',
				[
					'label'     => esc_html__( 'Arrows', 'litho-addons' ),
					'condition' => [
						'litho_navigation' => [
							'arrows',
							'both',
						],
					],
				]
			);
			$this->add_control(
				'litho_left_arrow_icon',
				[
					'label'            => esc_html__( 'Left Arrow Icon', 'litho-addons' ),
					'type'             => Controls_Manager::ICONS,
					'label_block'      => true,
					'fa4compatibility' => 'icon',
					'default'          => [
						'value'   => 'fas fa-chevron-left',
						'library' => 'fa-solid',
					],
					'condition'        => [
						'litho_navigation' => [
							'arrows',
							'both',
						],
					],
				]
			);
			$this->add_control(
				'litho_right_arrow_icon',
				[
					'label'            => esc_html__( 'Right Arrow Icon', 'litho-addons' ),
					'type'             => Controls_Manager::ICONS,
					'label_block'      => true,
					'fa4compatibility' => 'icon',
					'default'          => [
						'value'   => 'fas fa-chevron-right',
						'library' => 'fa-solid',
					],
					'condition'        => [
						'litho_navigation' => [
							'arrows',
							'both',
						],
					],
				]
			);
			$this->add_control(
				'litho_navigation_h_align',
				[
					'label'   => esc_html__( 'Horizontal Alignment', 'litho-addons' ),
					'type'    => Controls_Manager::CHOOSE,
					'default' => 'left',
					'options' => [
						'left'   => [
							'title' => esc_html__( 'Left', 'litho-addons' ),
							'icon'  => 'eicon-h-align-left',
						],
						'center' => [
							'title' => esc_html__( 'Center', 'litho-addons' ),
							'icon'  => 'eicon-h-align-center',
						],
						'right'  => [
							'title' => esc_html__( 'Right', 'litho-addons' ),
							'icon'  => 'eicon-h-align-right',
						],
					],
					'condition' => [
						'litho_navigation' => [
							'arrows',
							'both',
						],
					],
				]
			);
			$this->add_control(
				'litho_arrows_icon_shape_style',
				[
					'label'   => esc_html__( 'Shape Style', 'litho-addons' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'nav-icon-circle',
					'options' => [
						''                => esc_html__( 'None', 'litho-addons' ),
						'nav-icon-square' => esc_html__( 'Square', 'litho-addons' ),
						'nav-icon-circle' => esc_html__( 'Round', 'litho-addons' ),
					],
					'condition' => [
						'litho_navigation' => [
							'arrows',
							'both',
						],
					],
				]
			);
			$this->end_controls_section();
			$this->start_controls_section(
				'litho_section_style_image',
				[
					'label' => esc_html__( 'General', 'litho-addons' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				Group_Control_Background::get_type(),
				[
					'name'    => 'litho_content_title_background',
					'exclude' => [
						'image',
						'position',
						'attachment',
						'attachment_alert',
						'repeat',
						'size',
					],
					'selector' => '{{WRAPPER}} .swiper-slider',
				]
			);
			$this->add_responsive_control(
				'litho_slides_horizontal_position',
				[
					'label'   => esc_html__( 'Alignment', 'litho-addons' ),
					'type'    => Controls_Manager::CHOOSE,
					'options' => [
						'left'   => [
							'title' => esc_html__( 'Left', 'litho-addons' ),
							'icon'  => 'eicon-text-align-left',
						],
						'center' => [
							'title' => esc_html__( 'Center', 'litho-addons' ),
							'icon'  => 'eicon-text-align-center',
						],
						'right'  => [
							'title' => esc_html__( 'Right', 'litho-addons' ),
							'icon'  => 'eicon-text-align-right',
						],
					],
					'selectors' => [
						'{{WRAPPER}} .swiper-slide' => 'text-align: {{VALUE}};',
					],
					'default' => 'left',
				]
			);
			$this->add_responsive_control(
				'litho_slides_padding',
				[
					'label'      => esc_html__( 'Padding', 'litho-addons' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [
						'px',
						'em',
						'%',
						'rem',
						'custom',
					],
					'selectors'  => [
						'{{WRAPPER}} .swiper-slider' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'litho_slides_margin',
				[
					'label'      => esc_html__( 'Margin', 'litho-addons' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [
						'px',
						'%',
						'em',
						'rem',
						'custom',
					],
					'selectors'  => [
						'{{WRAPPER}} .swiper-slider' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			$this->start_controls_section(
				'litho_section_style_carousel_title',
				[
					'label' => esc_html__( 'Title', 'litho-addons' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'   => 'litho_title_typography',
					'global' => [
						'default' => Global_Typography::TYPOGRAPHY_SECONDARY,
					],
					'selector' => '{{WRAPPER}} .swiper-slide .title',
				]
			);
			$this->add_control(
				'litho_title_color',
				[
					'label'     => esc_html__( 'Color', 'litho-addons' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .swiper-slide .title, {{WRAPPER}} .swiper-slide .title a' => 'color: {{VALUE}};',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'litho_title_shadow',
					'selector' => '{{WRAPPER}} .swiper-slide .title',
				]
			);
			$this->add_responsive_control(
				'litho_title_spacing',
				[
					'label'      => esc_html__( 'Spacing', 'litho-addons' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => [
						'px',
						'custom',
					],
					'range'      => [
						'px' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'selectors'  => [
						'{{WRAPPER}} .swiper-slide .title' => 'margin-bottom: {{SIZE}}{{UNIT}}',
					],
				]
			);
			$this->end_controls_section();
			$this->start_controls_section(
				'litho_section_style_carousel_subtitle',
				[
					'label' => esc_html__( 'Subtitle', 'litho-addons' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'   => 'litho_subtitle_typography',
					'global' => [
						'default' => Global_Typography::TYPOGRAPHY_SECONDARY,
					],
					'selector' => '{{WRAPPER}} .swiper-slide .subtitle',
				]
			);
			$this->add_control(
				'litho_subtitle_color',
				[
					'label'     => esc_html__( 'Color', 'litho-addons' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .swiper-slide .subtitle, {{WRAPPER}} .swiper-slide .subtitle a' => 'color: {{VALUE}};',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'litho_subtitle_shadow',
					'selector' => '{{WRAPPER}} .swiper-slide .subtitle',
				]
			);
			$this->add_responsive_control(
				'litho_subtitle_spacing',
				[
					'label'      => esc_html__( 'Spacing', 'litho-addons' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => [
						'px',
						'custom',
					],
					'range'      => [
						'px' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'selectors'  => [
						'{{WRAPPER}} .swiper-slide .subtitle:not(:last-child)' => 'margin-bottom: {{SIZE}}{{UNIT}}',
					],
				]
			);
			$this->end_controls_section();
			$this->start_controls_section(
				'litho_section_style_carousel_content',
				[
					'label' => esc_html__( 'Content', 'litho-addons' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'   => 'litho_content_typography',
					'global' => [
						'default' => Global_Typography::TYPOGRAPHY_TEXT,
					],
					'selector' => '{{WRAPPER}} .swiper-slide .content',
				]
			);
			$this->add_control(
				'litho_content_color',
				[
					'label'     => esc_html__( 'Color', 'litho-addons' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .swiper-slide .content, {{WRAPPER}} .swiper-slide .content a' => 'color: {{VALUE}};',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Text_Shadow::get_type(),
				[
					'name'     => 'litho_content_shadow',
					'selector' => '{{WRAPPER}} .swiper-slide .content',
				]
			);
			$this->add_responsive_control(
				'litho_content_spacing',
				[
					'label'      => esc_html__( 'Spacing', 'litho-addons' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => [
						'px',
						'custom',
					],
					'range'      => [
						'px' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'selectors'  => [
						'{{WRAPPER}} .swiper-slide .content' => 'margin-bottom: {{SIZE}}{{UNIT}}',
					],
				]
			);
			$this->add_responsive_control(
				'litho_content_width',
				[
					'label'      => esc_html__( 'Width', 'litho-addons' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => [
						'px',
						'%',
						'em',
						'rem',
						'custom',
					],
					'selectors'  => [
						'{{WRAPPER}} .swiper-slide .content' => 'width: {{SIZE}}{{UNIT}}',
					],
				]
			);
			$this->end_controls_section();
			$this->start_controls_section(
				'litho_section_style_navigation',
				[
					'label'     => esc_html__( 'Navigation', 'litho-addons' ),
					'tab'       => Controls_Manager::TAB_STYLE,
					'condition' => [
						'litho_navigation' => [
							'arrows',
							'dots',
							'both',
						],
					],
				]
			);
			$this->add_control(
				'litho_nav_spacing',
				[
					'label'     => esc_html__( 'Navigation Spacer', 'litho-addons' ),
					'type'      => Controls_Manager::SLIDER,
					'range'     => [
						'px' => [
							'min' => 1,
							'max' => 100,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .swiper' => 'padding-bottom: {{SIZE}}{{UNIT}}',
					],
					'condition' => [
						'litho_navigation' => [
							'arrows',
							'both',
						],
					],
					'default'   => [
						'unit' => 'px',
						'size' => 50,
					],
				]
			);
			$this->add_responsive_control(
				'litho_arrows_box_size',
				[
					'label'     => esc_html__( 'Shape Size', 'litho-addons' ),
					'type'      => Controls_Manager::SLIDER,
					'range'     => [
						'px' => [
							'min' => 1,
							'max' => 100,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .swiper .elementor-swiper-button.elementor-swiper-button-prev, {{WRAPPER}} .swiper .elementor-swiper-button.elementor-swiper-button-next' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
					],
					'condition' => [
						'litho_navigation' => [
							'arrows',
							'both',
						],
					],
					'default'   => [
						'unit' => 'px',
						'size' => 45,
					],
				]
			);
			$this->add_control(
				'litho_arrows_size',
				[
					'label'     => esc_html__( 'Icon Size', 'litho-addons' ),
					'type'      => Controls_Manager::SLIDER,
					'range'     => [
						'px' => [
							'min' => 1,
							'max' => 40,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .swiper .elementor-swiper-button.elementor-swiper-button-prev i, {{WRAPPER}} .swiper .elementor-swiper-button.elementor-swiper-button-next i' => 'font-size: {{SIZE}}{{UNIT}};',
						'{{WRAPPER}} .elementor-widget-container .elementor-swiper-button.elementor-swiper-button-prev svg, {{WRAPPER}} .elementor-widget-container .elementor-swiper-button.elementor-swiper-button-next svg' => 'width: {{SIZE}}{{UNIT}}; height: auto',
					],
					'condition' => [
						'litho_navigation' => [
							'arrows',
							'both',
						],
					],
					'default'   => [
						'unit' => 'px',
						'size' => 16,
					],
				]
			);
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'litho_navigation_box_shadow',
					'selector' => '{{WRAPPER}} .swiper .elementor-swiper-button.elementor-swiper-button-prev, {{WRAPPER}} .swiper .elementor-swiper-button.elementor-swiper-button-next',
					'condition' => [
						'litho_navigation' => [
							'arrows',
							'both',
						],
					],
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'           => 'litho_arrows_box_border_style',
					'selector'       => '{{WRAPPER}} .swiper .elementor-swiper-button.elementor-swiper-button-prev, {{WRAPPER}} .swiper .elementor-swiper-button.elementor-swiper-button-next',
					'condition'      => [
						'litho_navigation' => [
							'arrows',
							'both',
						],
						'litho_arrows_icon_shape_style!' => '',
					],
					'fields_options' => [
						'border' => [
							'label' => esc_html__( 'Border Style', 'litho-addons' ),
						],
					],
				]
			);
			$this->start_controls_tabs( 'litho_arrows_box_style' );
				$this->start_controls_tab(
					'litho_arrows_box_normal_style',
					[
						'label'     => esc_html__( 'Normal', 'litho-addons' ),
						'condition' => [
								'litho_navigation' => [
								'arrows',
								'both',
							],
						],
					]
				);
				$this->add_control(
					'litho_arrows_color',
					[
						'label'     => esc_html__( 'Icon Color', 'litho-addons' ),
						'type'      => Controls_Manager::COLOR,
						'selectors' => [
							'{{WRAPPER}}.elementor-element .swiper .elementor-swiper-button, {{WRAPPER}}.elementor-element .swiper .elementor-swiper-button i' => 'color: {{VALUE}};',
							'{{WRAPPER}}.elementor-element .swiper .elementor-swiper-button svg' => 'fill: {{VALUE}};',
						],
						'condition' => [
							'litho_navigation' => [
								'arrows',
								'both',
							],
						],
					]
				);
				$this->add_group_control(
					Group_Control_Background::get_type(),
					[
						'name'      => 'litho_arrows_background_color',
						'types'     => [ 'classic', 'gradient' ],
						'exclude'   => [
							'image',
							'position',
							'attachment',
							'attachment_alert',
							'repeat',
							'size',
						],
						'selector'  => '{{WRAPPER}} .swiper .elementor-swiper-button.elementor-swiper-button-prev, {{WRAPPER}} .swiper .elementor-swiper-button.elementor-swiper-button-next',
						'condition' => [
							'litho_navigation' => [
								'arrows',
								'both',
							],
							'litho_arrows_icon_shape_style!' => '',
						],
					]
				);
				$this->end_controls_tab();
				$this->start_controls_tab(
					'litho_arrows_box_hover_style',
					[
						'label'     => esc_html__( 'Hover', 'litho-addons' ),
						'condition' => [
							'litho_navigation' => [
								'arrows',
								'both',
							],
						],
					]
				);
				$this->add_control(
					'litho_arrows_hover_color',
					[
						'label'     => esc_html__( 'Icon Color', 'litho-addons' ),
						'type'      => Controls_Manager::COLOR,
						'selectors' => [
							'{{WRAPPER}}.elementor-element .swiper .elementor-swiper-button:hover, {{WRAPPER}}.elementor-element .swiper .elementor-swiper-button:hover i' => 'color: {{VALUE}};',
							'{{WRAPPER}}.elementor-element .swiper .elementor-swiper-button:hover svg, {{WRAPPER}}.elementor-element .swiper .elementor-swiper-button:focus svg' => 'fill: {{VALUE}};',
						],
						'condition' => [
							'litho_navigation' => [
								'arrows',
								'both',
							],
						],
					]
				);
				$this->add_group_control(
					Group_Control_Background::get_type(),
					[
						'name'      => 'litho_arrows_background_hover_color',
						'types'     => [
							'classic',
							'gradient',
						],
						'exclude'   => [
							'image',
							'position',
							'attachment',
							'attachment_alert',
							'repeat',
							'size',
						],
						'selector'  => '{{WRAPPER}} .swiper .elementor-swiper-button.elementor-swiper-button-prev:hover, {{WRAPPER}} .swiper .elementor-swiper-button.elementor-swiper-button-next:hover',
						'condition' => [
							'litho_navigation' => [
								'arrows',
								'both',
							],
							'litho_arrows_icon_shape_style!' => '',
						],
					]
				);
				$this->add_control(
					'litho_arrows_border_hover_color',
					[
						'label'     => esc_html__( 'Border Color', 'litho-addons' ),
						'type'      => Controls_Manager::COLOR,
						'selectors' => [
							'{{WRAPPER}} .swiper .elementor-swiper-button.elementor-swiper-button-prev:hover, {{WRAPPER}} .swiper .elementor-swiper-button.elementor-swiper-button-next:hover' => 'border-color: {{VALUE}};',
						],
						'condition' => [
							'litho_navigation' => [
								'arrows',
								'both',
							],
							'litho_arrows_icon_shape_style!' => '',
						],
					]
				);
				$this->add_control(
					'litho_navigation_hover_opacity',
					[
						'label'     => esc_html__( 'Opacity', 'litho-addons' ),
						'type'      => Controls_Manager::SLIDER,
						'range'     => [
							'px' => [
								'max'  => 1,
								'min'  => 0.10,
								'step' => 0.01,
							],
						],
						'selectors' => [
							'{{WRAPPER}} .swiper .elementor-swiper-button.swiper-button-disabled, {{WRAPPER}} .swiper .elementor-swiper-button.swiper-button-disabled:hover, {{WRAPPER}} .swiper .elementor-swiper-button:hover' => 'opacity: {{SIZE}};',
						],
						'condition' => [
							'litho_navigation' => [
								'arrows',
								'both',
							],
						],
					]
				);
				$this->end_controls_tab();
			$this->end_controls_tabs();
			$this->add_control(
				'litho_heading_style_dots',
				[
					'label' 		=> esc_html__( 'Dots', 'litho-addons' ),
					'type' 			=> Controls_Manager::HEADING,
					'separator' 	=> 'before',
					'condition' 	=> [
						'litho_navigation' => [
							'dots',
							'both',
						],
					],
				]
			);$this->add_control(
				'litho_dots_position',
				[
					'label' 		=> esc_html__( 'Position', 'litho-addons' ),
					'type' 			=> Controls_Manager::SELECT,
					'default' 		=> 'outside',
					'options' 		=> [
						'outside' 		=> esc_html__( 'Outside', 'litho-addons' ),
						'inside' 		=> esc_html__( 'Inside', 'litho-addons' ),
					],
					'prefix_class' 	=> 'elementor-pagination-position-',
					'condition' 	=> [
						'litho_navigation' => [
							'dots',
							'both',
						],
					],
				]
			);
			$this->add_control(
				'litho_dots_spacing',
				[
					'label' 		=> esc_html__( 'Pagination Spacer', 'litho-addons' ),
					'type' 			=> Controls_Manager::SLIDER,
					'range'         => [
						'px'   => [
							'min' => 0,
							'max' => 100,
						],
					],
					'selectors' 	=> [
						'{{WRAPPER}}.elementor-pagination-position-outside .swiper' => 'padding-bottom: {{SIZE}}{{UNIT}}',
					],
					'condition' 	=> [ 
						'litho_navigation' => [
							'dots',
							'both',
						],
						'litho_dots_position'	=> 'outside'
					],
				]
			);
			$this->add_responsive_control(
				'litho_dots_direction_vh',
				[
					'label'      => esc_html__( 'Space Between Pagination', 'litho-addons' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => [
						'px',
						'%',
					],
					'range'      => [
						'px' => [
							'min' => 1,
							'max' => 20,
						],
					],
					'selectors'  => [
						'{{WRAPPER}} .swiper-pagination .swiper-pagination-bullet' => 'margin-right: {{SIZE}}{{UNIT}}; margin-left: {{SIZE}}{{UNIT}};',
					],
					'condition'  => [
						'litho_navigation' 	=> [
							'dots',
							'both',
						],
					],
				]
			);
			$this->start_controls_tabs(
				'litho_dots_tabs',
				[
					'condition' 	=> [
						'litho_navigation' 	=> [
							'dots',
							'both',
						],
					],
				]
			);
			$this->start_controls_tab(
				'litho_dots_normal_tab',
				[
					'label' => esc_html__( 'Normal', 'litho-addons' ),
					'condition' 	=> [
						'litho_navigation' 	=> [
							'dots',
							'both',
						],
					],
				]
			);
			$this->add_control(
				'litho_dots_size',
				[
					'label' 		=> esc_html__( 'Size', 'litho-addons' ),
					'type' 			=> Controls_Manager::SLIDER,
					'range'         => [
						'px'   => [
							'min' => 5,
							'max' => 20,
						],
					],
					'selectors' 	=> [
						'{{WRAPPER}} .swiper-pagination-bullet' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
					],
					'condition' 	=> [
						'litho_navigation' 	=> [
							'dots',
							'both',
						],
					],
				]
			);
			$this->add_control(
				'litho_dots_color',
				[
					'label'     => esc_html__( 'Color', 'litho-addons' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .swiper-pagination .swiper-pagination-bullet:not(.swiper-pagination-bullet-active)' => 'background: {{VALUE}};',
					],
					'condition' 	=> [
						'litho_navigation' 	=> [
							'dots',
							'both',
						],
					],
				]
			);
			$this->end_controls_tab();
			$this->start_controls_tab(
				'litho_dots_active_tab',
				[
					'label' => esc_html__( 'Active', 'litho-addons' ),
					'condition' 	=> [
						'litho_navigation' 	=> [
							'dots',
							'both',
						],
					],
				]
			);
			$this->add_control(
				'litho_dots_active_size',
				[
					'label'     => esc_html__( 'Size', 'litho-addons' ),
					'type'      => Controls_Manager::SLIDER,
					'range'     => [
						'px' => [
							'min' => 5,
							'max' => 20,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .swiper-pagination .swiper-pagination-bullet.swiper-pagination-bullet-active' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
					],
					'condition' 	=> [
						'litho_navigation' 	=> [
							'dots',
							'both',
						],
					],
				]
			);
			$this->add_control(
				'litho_dots_active_color',
				[
					'label'     => esc_html__( 'Color', 'litho-addons' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .swiper-pagination .swiper-pagination-bullet.swiper-pagination-bullet-active' => 'background: {{VALUE}};',
					],
					'condition' 	=> [
						'litho_navigation' 	=> [
							'dots',
							'both',
						],
					],
				]
			);
			$this->end_controls_tab();
			$this->end_controls_tabs();
			$this->end_controls_section();
		}
		/**
		 * Render dynamic slider widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0
		 * @access protected
		 */
		protected function render() {

			$slides       = [];
			$slides_count = '';
			$settings     = $this->get_settings_for_display();

			if ( empty( $settings['litho_carousel_slider'] ) ) {
				return;
			}
			foreach ( $settings['litho_carousel_slider'] as $index => $item ) {

				$index       = ++$index;
				$wrapper_key = 'wrapper_' . $index;

				$this->add_render_attribute(
					$wrapper_key,
					[
						'class' => [
							'elementor-repeater-item-' . $item['_id'],
							'swiper-slide',
							'cover-background',
						],
					]
				);
				ob_start();
				?>
				<div <?php $this->print_render_attribute_string( $wrapper_key ); ?>>
					<div class="swiper-slider">
						<?php
						if ( 'template' === $item['litho_item_content_type'] ) {
							if ( '0' !== $item['litho_item_template_id'] ) {
								$template_content = litho_get_builder_content_for_display( $item['litho_item_template_id'] );
								if ( ! empty( $template_content ) ) {
									if ( Plugin::$instance->editor->is_edit_mode() ) {
										$edit_url = add_query_arg(
											array(
												'elementor' => '',
											),
											get_permalink( $item['litho_item_template_id'] )
										);
										echo sprintf( '<div class="edit-template-with-light-box elementor-template-edit-cover" data-template-edit-link="%s"><i aria-hidden="true" class="eicon-edit"></i><span>%s</span></div>', esc_url( $edit_url ), esc_html__( 'Edit Template', 'litho-addons' ) ); // phpcs:ignore
									}
									echo sprintf( '%s', $template_content ); // phpcs:ignore
								} else {
									echo sprintf( '%s', $this->no_template_content_message() ); // phpcs:ignore
								}
							} else {
								echo sprintf( '%s', $this->no_template_content_message() ); // phpcs:ignore
							}
						} else {
							if ( ! empty( $item['litho_carousel_subtitle'] ) ) {
								?>
								<div class="subtitle">
									<?php echo esc_html( $item['litho_carousel_subtitle'] ); ?>
								</div>
								<?php
							}
							if ( ! empty( $item['litho_carousel_title'] ) ) {
								?>
								<<?php echo $this->get_settings( 'litho_dynamic_slider_title_size' ); // phpcs:ignore ?>  class="title">
									<?php echo esc_html( $item['litho_carousel_title'] ); ?>
								</<?php echo $this->get_settings( 'litho_dynamic_slider_title_size' ); // phpcs:ignore ?>>
								<?php
							}

							if ( ! empty( $item['litho_content'] ) ) {
								?>
								<div class="content">
									<?php echo sprintf( '%s', $item['litho_content'] ); // phpcs:ignore ?>
								</div>
								<?php
							}
						}
						?>
					</div>
				</div>
				<?php
				$slides[] = ob_get_contents();
				ob_end_clean();
			}

			$litho_left_arrow_icon  = '';
			$litho_right_arrow_icon = '';
			$left_icon_migrated     = isset( $settings['__fa4_migrated']['litho_left_arrow_icon'] );
			$right_icon_migrated    = isset( $settings['__fa4_migrated']['litho_right_arrow_icon'] );
			$is_new                 = empty( $settings['icon'] ) && Icons_Manager::is_migration_allowed();

			if ( isset( $settings['litho_left_arrow_icon'] ) && ! empty( $settings['litho_left_arrow_icon'] ) ) {
				if ( $is_new || $left_icon_migrated ) {
					ob_start();
					Icons_Manager::render_icon( $settings['litho_left_arrow_icon'], [ 'aria-hidden' => 'true' ] );
					$litho_left_arrow_icon .= ob_get_clean();
				} elseif ( isset( $settings['litho_left_arrow_icon']['value'] ) && ! empty( $settings['litho_left_arrow_icon']['value'] ) ) {
					$litho_left_arrow_icon .= '<i class="' . esc_attr( $settings['litho_left_arrow_icon']['value'] ) . '" aria-hidden="true"></i>';
				}
			}

			if ( isset( $settings['litho_right_arrow_icon'] ) && ! empty( $settings['litho_right_arrow_icon'] ) ) {
				if ( $is_new || $right_icon_migrated ) {
					ob_start();
					Icons_Manager::render_icon( $settings['litho_right_arrow_icon'], [ 'aria-hidden' => 'true' ] );
					$litho_right_arrow_icon .= ob_get_clean();
				} elseif ( isset( $settings['litho_right_arrow_icon']['value'] ) && ! empty( $settings['litho_right_arrow_icon']['value'] ) ) {
					$litho_right_arrow_icon .= '<i class="' . esc_attr( $settings['litho_right_arrow_icon']['value'] ) . '" aria-hidden="true"></i>';
				}
			}

			if ( empty( $slides ) ) {
				return;
			}
			
			$slides_count                   = count( $settings['litho_carousel_slider'] );
			$litho_slider_cursor            = $this->get_settings( 'litho_slider_cursor' );
			$litho_rtl                     = $this->get_settings( 'litho_rtl' );
			$litho_navigation              = $this->get_settings( 'litho_navigation' );
			$litho_arrows_icon_shape_style = $this->get_settings( 'litho_arrows_icon_shape_style' );
			$litho_navigation_h_align      = $this->get_settings( 'litho_navigation_h_align' );

			$datasettings = array(
				'navigation'                 => $litho_navigation,
				'slide_total'                => $slides_count,
				'loop'                       => $this->get_settings( 'litho_infinite' ),
				'autoplay'                   => $this->get_settings( 'litho_autoplay' ),
				'autoplay_speed'             => $this->get_settings( 'litho_autoplay_speed' ),
				'pause_on_hover'             => $this->get_settings( 'litho_pause_on_hover' ),
				'effect'                     => $this->get_settings( 'litho_effect' ),
				'speed'                      => $this->get_settings( 'litho_speed' ),
				'mousewheel'                 => $this->get_settings( 'litho_mousewheel' ),
				'image_spacing'              => $this->get_settings( 'litho_items_spacing' ),
				'navigation_dynamic_bullets' => $this->get_settings( 'litho_navigation_dynamic_bullets' ),
				'slides_to_show'             => $this->get_settings( 'litho_slides_to_show' ),
				'slides_to_show_mobile'      => $this->get_settings( 'litho_slides_to_show_mobile' ),
				'slides_to_show_tablet'      => $this->get_settings( 'litho_slides_to_show_tablet' ),
			);

			$datasettings = array_merge( $datasettings );
			$litho_effect = $this->get_settings( 'litho_effect' );
			$show_dots    = ( in_array( $litho_navigation, [ 'dots', 'both' ] ) );
			$show_arrows  = ( in_array( $litho_navigation, [ 'arrows', 'both' ] ) );

			$effect = [
				'fade',
				'flip',
				'cube',
				'cards',
				'coverflow',
			];

			if ( '1' === $this->get_settings( 'litho_slides_to_show' )['size'] && in_array( $litho_effect, $effect, true ) ) {
				$datasettings['effect'] = $litho_effect;
			}
			if ( ! empty( $litho_rtl ) ) {
				$this->add_render_attribute(
					'carousel-wrapper',
					'dir',
					$litho_rtl
				);
			}
			
			$this->add_render_attribute(
				[
					'carousel'         => [
						'class' => [
							'swiper-wrapper',
						],
					],
					'carousel-wrapper' => [
						'class'         => [
							'dynamic-slider',
							'swiper',
							'litho-common-swiper',
							$litho_slider_cursor,
						],
						'data-settings' => wp_json_encode( $datasettings ),
					],
				]
			);
			$this->add_render_attribute(
				[
					'carousel-wrapper' => [
						'class' => [
							'slider-vertical',
						],
					],
				],
			);
			if ( ! empty( $litho_navigation_h_align ) ) {
				$this->add_render_attribute(
					[
						'carousel-wrapper' => [
							'class' => [
								'navigation-' . $litho_navigation_h_align,
							],
						],
					]
				);
			}
			if ( '' !== $litho_arrows_icon_shape_style ) {
				$this->add_render_attribute(
					[
						'carousel-wrapper' => [
							'class' => [
								$litho_arrows_icon_shape_style,
							],
						],
					]
				);
			}
			$this->add_render_attribute(
				[
					'carousel-wrapper' => [
						'class' => [
							'elementor-arrows-position-custom',
						],
					],
				]
			);
			?>
			 <div <?php $this->print_render_attribute_string( 'carousel-wrapper' ); ?>>
				<div <?php $this->print_render_attribute_string( 'carousel' ); ?>>
					<?php echo implode( ' ', $slides ); // phpcs:ignore ?>
				</div>
				<?php
				if ( 1 < $slides_count ) {
					if ( $show_arrows ) {
						?>
						<div class="dynamic-slider-nav">
							<div class="elementor-swiper-button elementor-swiper-button-prev">
								<?php
								if ( ! empty( $litho_left_arrow_icon ) ) {
									echo sprintf( '%s', $litho_left_arrow_icon );
								}
								?>
							</div>
							<div class="elementor-swiper-button elementor-swiper-button-next">
								<?php
								if ( ! empty( $litho_right_arrow_icon ) ) {
									echo sprintf( '%s', $litho_right_arrow_icon );
								}
								?>
							</div>
						</div>
						<?php
					}
					if ( $show_dots ) {
						?>
						<div class="swiper-pagination"></div>
						<?php
					}
				}
				?>
			</div>
			<?php
		}
		public function no_template_content_message() {
			
			$message = esc_html__( 'Template is not defined. ', 'litho-addons' );
			$link = add_query_arg(
				array(
					'post_type'     => 'elementor_library',
					'action'        => 'elementor_new_post',
					'_wpnonce'      => wp_create_nonce( 'elementor_action_new_post' ),
					'template_type' => 'section',
				),
				esc_url( admin_url( '/edit.php' ) )
			);

			$new_link = esc_html__( 'Select an existing template or create a ', 'litho-addons' ) . '<a class="elementor-custom-new-template-link elementor-clickable" href="' . $link . '">' . esc_html__( 'new one', 'litho-addons' ) . '</a>';

			return sprintf(
				'<div class="elementor-no-template-message alert alert-warning"><div class="message">%1$s%2$s</div></div>',
				$message,
				( Plugin::instance()->editor->is_edit_mode() || Plugin::instance()->preview->is_preview_mode() ) ? $new_link : ''
			);
		}
	}
}
