<?php
namespace LithoAddons\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Utils;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Typography;

/**
 * Litho widget for image comparison.
 *
 * @package Litho
 */

// If class `Images_Comparison` doesn't exists yet.
if ( ! class_exists( 'LithoAddons\Widgets\Images_Comparison' ) ) {
	/**
	 * Define `Images_Comparison` class.
	 */
	class Images_Comparison extends Widget_Base {
		/**
		 * Retrieve the list of scripts the image comparison widget depended on.
		 *
		 * Used to set scripts dependencies required to run the widget.
		 *
		 * @access public
		 *
		 * @return array Widget scripts dependencies.
		 */
		public function get_script_depends() {
			return [
				'litho-addons-image-comparison',
				'image-compare-viewer',
			];
		}

		/**
		 * Retrieve the widget name.
		 *
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'litho-images-comparison';
		}

		/**
		 * Retrieve the widget title
		 *
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return esc_html__( 'Litho Image Comparison', 'litho-addons' );
		}

		/**
		 * Retrieve the widget icon.
		 *
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-image litho-element-icon';
		}

		/**
		 * Retrieve the widget categories.
		 *
		 * @access public
		 *
		 * @return string Widget categories.
		 */
		public function get_categories() {
			return [
				'litho',
			];
		}

		/**
		 * Get widget keywords.
		 *
		 * Retrieve the list of keywords the widget belongs to.
		 *
		 * @access public
		 *
		 * @return array Widget keywords.
		 */
		public function get_keywords() {
			return [
				'images',
				'photos',
				'visual',
				'comparison',
				'compare',
				'before after',
				'image difference',
			];
		}

		/**
		 * Register image comparison widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @access protected
		 */
		protected function register_controls() {
			$this->start_controls_section(
				'litho_section_image_compare_title',
				[
					'label' => esc_html__( 'General', 'litho-addons' ),
				]
			);

			$this->add_control(
				'litho_before_image',
				[
					'label'       => esc_html__( 'Before Image', 'litho-addons' ),
					'description' => esc_html__( 'Use same size image for before and after for better preview.', 'litho-addons' ),
					'type'        => Controls_Manager::MEDIA,
					'default'     => [
						'url' => Utils::get_placeholder_image_src(),
					],
					'dynamic'     => [
						'active' => true,
					],
				]
			);

			$this->add_control(
				'litho_after_image',
				[
					'label'       => esc_html__( 'After Image', 'litho-addons' ),
					'description' => esc_html__( 'Use same size image for before and after for better preview.', 'litho-addons' ),
					'type'        => Controls_Manager::MEDIA,
					'default'     => [
						'url' => Utils::get_placeholder_image_src(),
					],
					'dynamic'     => [
						'active' => true,
					],
				]
			);
			$this->add_group_control(
				Group_Control_Image_Size::get_type(),
				[
					'name'      => 'litho_thumbnail',
					'default'   => 'full',
					'exclude'   => [
						'custom',
					],
					'separator' => 'none',
				]
			);
			$this->add_control(
				'litho_before_label',
				[
					'label'       => esc_html__( 'Before Label', 'litho-addons' ),
					'type'        => Controls_Manager::TEXT,
					'placeholder' => esc_html__( 'Before Label', 'litho-addons' ),
					'default'     => esc_html__( 'Before', 'litho-addons' ),
					'label_block' => true,
					'dynamic'     => [
						'active' => true,
					],
				]
			);

			$this->add_control(
				'litho_after_label',
				[
					'label'       => esc_html__( 'After Label', 'litho-addons' ),
					'type'        => Controls_Manager::TEXT,
					'placeholder' => esc_html__( 'After Label', 'litho-addons' ),
					'default'     => esc_html__( 'After', 'litho-addons' ),
					'label_block' => true,
					'dynamic'     => [
						'active' => true,
					],
				]
			);

			$this->end_controls_section();

			$this->start_controls_section(
				'litho_section_content_settings',
				[
					'label' => esc_html__( 'Settings', 'litho-addons' ),
					'tab'   => Controls_Manager::TAB_CONTENT,
				]
			);

			$this->add_control(
				'litho_orientation',
				[
					'label'   => esc_html__( 'Orientation', 'litho-addons' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'horizontal',
					'options' => [
						'horizontal' => esc_html__( 'Horizontal', 'litho-addons' ),
						'vertical'   => esc_html__( 'Vertical', 'litho-addons' ),
					],
				]
			);

			$this->add_control(
				'litho_default_offset',
				[
					'label'   => esc_html__( 'Before Image Visiblity', 'litho-addons' ),
					'type'    => Controls_Manager::SLIDER,
					'default' => [
						'size' => 70,
					],
					'range'   => [
						'px' => [
							'max' => 100,
							'min' => 0,
						],
					],
				]
			);

			$this->add_control(
				'litho_no_overlay',
				[
					'label'       => esc_html__( 'Overlay', 'litho-addons' ),
					'description' => esc_html__( 'Do not show the overlay with before and after.', 'litho-addons' ),
					'type'        => Controls_Manager::SWITCHER,
					'default'     => 'yes',
					'separator'   => 'before',
				]
			);

			$this->add_control(
				'litho_on_hover',
				[
					'label'     => esc_html__( 'Show Label on Hover?', 'litho-addons' ),
					'type'      => Controls_Manager::SWITCHER,
					'default'   => 'yes',
					'condition' => [
						'litho_no_overlay' => 'yes',
					],
				]
			);

			$this->add_responsive_control(
				'litho_before_after_vertical_align',
				[
					'label'                => esc_html__( 'Vertical Alignment', 'litho-addons' ),
					'type'                 => Controls_Manager::CHOOSE,
					'toggle'               => false,
					'options'              => [
						'top'    => [
							'title' => esc_html__( 'Top', 'litho-addons' ),
							'icon'  => 'eicon-v-align-top',
						],
						'center' => [
							'title' => esc_html__( 'Center', 'litho-addons' ),
							'icon'  => 'eicon-v-align-middle',
						],
						'bottom' => [
							'title' => esc_html__( 'Bottom', 'litho-addons' ),
							'icon'  => 'eicon-v-align-bottom',
						],
					],
					'default'              => 'center',
					'selectors_dictionary' => [
						'top'    => 'top: 0; bottom: unset;',
						'center' => 'top: 50%; transform: translateY(-50%); bottom: unset;',
						'bottom' => 'bottom: 0;',
					],
					'selectors'            => [
						'{{WRAPPER}} .litho-image-compare .icv__icv--horizontal .icv__label' => '{{VALUE}};',
					],
					'condition'            => [
						'litho_no_overlay'  => 'yes',
						'litho_orientation' => 'horizontal',
					],
				]
			);

			$this->add_responsive_control(
				'litho_before_after_horizontal_align',
				[
					'label'                => esc_html__( 'Horizontal Alignment', 'litho-addons' ),
					'type'                 => Controls_Manager::CHOOSE,
					'toggle'               => false,
					'options'              => [
						'left'   => [
							'title' => esc_html__( 'Left', 'litho-addons' ),
							'icon'  => 'eicon-h-align-left',
						],
						'center' => [
							'title' => esc_html__( 'Center', 'litho-addons' ),
							'icon'  => 'eicon-h-align-center',
						],
						'right'  => [
							'title' => esc_html__( 'Right', 'litho-addons' ),
							'icon'  => 'eicon-h-align-right',
						],
					],
					'default'              => 'center',
					'selectors_dictionary' => [
						'left'   => 'left: 0; right: auto;',
						'center' => 'left: 50%; transform: translateX(-50%);',
						'right'  => 'right: 0; left: auto;',
					],
					'selectors'            => [
						'{{WRAPPER}} .litho-image-compare .icv__label.vertical' => '{{VALUE}};',
					],
					'condition'            => [
						'litho_no_overlay'  => 'yes',
						'litho_orientation' => 'vertical',
					],
				]
			);

			$this->add_control(
				'litho_move_slider_on_hover',
				[
					'label'       => esc_html__( 'Slide on Hover', 'litho-addons' ),
					'description' => esc_html__( 'Move slider on mouse hover?', 'litho-addons' ),
					'type'        => Controls_Manager::SWITCHER,
					'separator'   => 'before',
				]
			);

			$this->add_control(
				'litho_add_circle',
				[
					'label'     => esc_html__( 'Add Circle In Bar?', 'litho-addons' ),
					'type'      => Controls_Manager::SWITCHER,
					'separator' => 'before',
				]
			);

			$this->add_control(
				'litho_add_circle_blur',
				[
					'label'     => esc_html__( 'add Circle Blur?', 'litho-addons' ),
					'type'      => Controls_Manager::SWITCHER,
					'condition' => [
						'litho_add_circle' => 'yes',
					],
				]
			);

			$this->add_control(
				'litho_add_circle_shadow',
				[
					'label'     => esc_html__( 'Circle Shadow?', 'litho-addons' ),
					'type'      => Controls_Manager::SWITCHER,
					'condition' => [
						'litho_add_circle' => 'yes',
					],
				]
			);

			$this->add_control(
				'litho_smoothing',
				[
					'label'     => esc_html__( 'Smoothing?', 'litho-addons' ),
					'type'      => Controls_Manager::SWITCHER,
					'default'   => 'yes',
					'separator' => 'before',
				]
			);

			$this->add_control(
				'litho_smoothing_amount',
				[
					'label'     => esc_html__( 'Smoothing Amount', 'litho-addons' ),
					'type'      => Controls_Manager::SLIDER,
					'default'   => [
						'size' => 400,
					],
					'range'     => [
						'px' => [
							'max'  => 1000,
							'min'  => 100,
							'step' => 10,
						],
					],
					'condition' => [
						'litho_smoothing' => 'yes',
					],
				]
			);
			$this->end_controls_section();

			$this->start_controls_section(
				'litho_section_style',
				[
					'label' => esc_html__( 'Style', 'litho-addons' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);

			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'     => 'litho_after_before_typography',
					'label'    => esc_html__( 'Typography', 'litho-addons' ),
					'selector' => '{{WRAPPER}} .litho-image-compare .icv__label',
				]
			);

			$this->start_controls_tabs( 'litho_tabs_image_compare_style' );

				$this->start_controls_tab(
					'tlitho_ab_image_compare_before_style',
					[
						'label' => esc_html__( 'Before', 'litho-addons' ),
					]
				);

				$this->add_control(
					'litho_before_background',
					[
						'label'     => esc_html__( 'Background', 'litho-addons' ),
						'type'      => Controls_Manager::COLOR,
						'selectors' => [
							'{{WRAPPER}} .litho-image-compare .icv__label.icv__label-before' => 'background-color: {{VALUE}};',
						],
					]
				);

				$this->add_control(
					'litho_before_color',
					[
						'label'     => esc_html__( 'Color', 'litho-addons' ),
						'type'      => Controls_Manager::COLOR,
						'selectors' => [
							'{{WRAPPER}} .litho-image-compare .icv__label.icv__label-before' => 'color: {{VALUE}};',
						],
					]
				);

				$this->end_controls_tab();

				$this->start_controls_tab(
					'litho_tab_image_compare_after_style',
					[
						'label' => esc_html__( 'After', 'litho-addons' ),
					]
				);

				$this->add_control(
					'litho_after_background',
					[
						'label'     => esc_html__( 'Background', 'litho-addons' ),
						'type'      => Controls_Manager::COLOR,
						'selectors' => [
							'{{WRAPPER}} .litho-image-compare .icv__label.icv__label-after' => 'background-color: {{VALUE}};',
						],
					]
				);

				$this->add_control(
					'litho_after_color',
					[
						'label'     => esc_html__( 'Color', 'litho-addons' ),
						'type'      => Controls_Manager::COLOR,
						'selectors' => [
							'{{WRAPPER}} .litho-image-compare .icv__label.icv__label-after' => 'color: {{VALUE}};',
						],
					]
				);

				$this->end_controls_tab();

				$this->start_controls_tab(
					'litho_tab_image_compare_bar_style',
					[
						'label' => esc_html__( 'Bar', 'litho-addons' ),
					]
				);

				$this->add_control(
					'litho_bar_color',
					[
						'label'   => esc_html__( 'Bar Color', 'litho-addons' ),
						'type'    => Controls_Manager::COLOR,
						'default' => '#fff',
					]
				);

				$this->end_controls_tab();

			$this->end_controls_tabs();

			$this->add_responsive_control(
				'litho_after_before_padding',
				[
					'label'      => esc_html__( 'Padding', 'litho-addons' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'selectors'  => [
						'{{WRAPPER}} .litho-image-compare .icv__label' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
					'separator'  => 'before',
				]
			);

			$this->add_control(
				'litho_after_before_radius',
				[
					'label'      => esc_html__( 'Border Radius', 'litho-addons' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%' ],
					'selectors'  => [
						'{{WRAPPER}} .litho-image-compare .icv__label' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'litho_overlay_color',
				[
					'label'     => esc_html__( 'Overlay Color', 'litho-addons' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .litho-image-compare .litho-image-compare-overlay:before' => 'background: {{VALUE}};',
					],
					'condition' => [
						'litho_no_overlay' => 'yes',
					],
				]
			);
			$this->end_controls_section();
		}

		/**
		 * Render image comparison widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @access protected
		 */
		protected function render() {
			$settings = $this->get_settings_for_display();

			if ( $settings['litho_default_offset']['size'] < 1 ) {
				$settings['litho_default_offset']['size'] = $settings['litho_default_offset']['size'] * 100;
			}

			$this->add_render_attribute(
				[
					'image-compare' => [
						'id'            => 'image-compare-' . $this->get_id(),
						'class'         => [ 'image-compare' ],
						'data-settings' => [
							wp_json_encode(
								array_filter(
									[
										'id'               => 'image-compare-' . $this->get_id(),
										'default_offset_pct' => $settings['litho_default_offset']['size'],
										'orientation'      => ( 'horizontal' === $settings['litho_orientation'] ) ? false : true,
										'before_label'     => esc_attr( $settings['litho_before_label'] ),
										'after_label'      => esc_attr( $settings['litho_after_label'] ),
										'no_overlay'       => ( 'yes' === $settings['litho_no_overlay'] ) ? true : false,
										'on_hover'         => ( 'yes' === $settings['litho_on_hover'] ) ? true : false,
										'move_slider_on_hover' => ( 'yes' === $settings['litho_move_slider_on_hover'] ) ? true : false,
										'add_circle'       => ( 'yes' === $settings['litho_add_circle'] ) ? true : false,
										'add_circle_blur'  => ( 'yes' === $settings['litho_add_circle_blur'] ) ? true : false,
										'add_circle_shadow' => ( 'yes' === $settings['litho_add_circle_shadow'] ) ? true : false,
										'smoothing'        => ( 'yes' === $settings['litho_smoothing'] ) ? true : false,
										'smoothing_amount' => $settings['litho_smoothing_amount']['size'],
										'bar_color'        => $settings['litho_bar_color'],
									]
								)
							),
						],
					],
				]
			);

			if ( 'yes' === $settings['litho_no_overlay'] ) {
				$this->add_render_attribute(
					'image-compare',
					'class',
					'litho-image-compare-overlay',
				);
			}
			?>
			<div class="litho-image-compare">
				<div <?php $this->print_render_attribute_string( 'image-compare' ); ?>>
					<?php
					if ( ! empty( $settings['litho_before_image']['id'] ) && ! wp_attachment_is_image( $settings['litho_before_image']['id'] ) ) {
						$settings['litho_before_image']['id'] = '';
					}
					if ( isset( $settings['litho_before_image'] ) && isset( $settings['litho_before_image']['id'] ) && ! empty( $settings['litho_before_image']['id'] ) ) {
						litho_get_attachment_html( $settings['litho_before_image']['id'], $settings['litho_before_image']['url'], $settings['litho_thumbnail_size'] ); // phpcs:ignore
					} elseif ( isset( $settings['litho_before_image'] ) && isset( $settings['litho_before_image']['url'] ) && ! empty( $settings['litho_before_image']['url'] ) ) {
						litho_get_attachment_html( $settings['litho_before_image']['id'], $settings['litho_before_image']['url'], $settings['litho_thumbnail_size'] ); // phpcs:ignore
					}

					if ( ! empty( $settings['litho_after_image']['id'] ) && ! wp_attachment_is_image( $settings['litho_after_image']['id'] ) ) {
						$settings['litho_after_image']['id'] = '';
					}
					if ( isset( $settings['litho_after_image'] ) && isset( $settings['litho_after_image']['id'] ) && ! empty( $settings['litho_after_image']['id'] ) ) {
						litho_get_attachment_html( $settings['litho_after_image']['id'], $settings['litho_after_image']['url'], $settings['litho_thumbnail_size'] ); // phpcs:ignore
					} elseif ( isset( $settings['litho_after_image'] ) && isset( $settings['litho_after_image']['url'] ) && ! empty( $settings['litho_after_image']['url'] ) ) {
						litho_get_attachment_html( $settings['litho_after_image']['id'], $settings['litho_after_image']['url'], $settings['litho_thumbnail_size'] ); // phpcs:ignore
					}
					?>
				</div>
			</div>
			<?php
		}
	}
}
