<?php
namespace LithoAddons\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Core\Kits\Documents\Tabs\Global_Colors;

/**
 * Litho widget for inline SVG.
 *
 * @package Litho
 */

// If class 'Inline_SVG' doesn't exists yet.
if ( ! class_exists( 'LithoAddons\Widgets\Inline_SVG' ) ) {
	/**
	 * Define 'Inline_SVG' class.
	 */
	class Inline_SVG extends Widget_Base {

		/**
		 * Get widget name.
		 *
		 * Retrieve inline SVG widget name.
		 */
		public function get_name() {
			return 'litho-inline-svg';
		}

		/**
		 * Get widget title.
		 * Retrieve inline SVG widget title.
		 *
		 * @access public
		 * @return string Widget title.
		 */
		public function get_title() {
			return esc_html__( 'Litho Inline SVG', 'litho-addons' );
		}

		/**
		 * Get widget icon.
		 * Retrieve inline SVG widget icon.
		 *
		 * @access public
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-notification';
		}

		/**
		 * Retrieve the widget categories.
		 *
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [
				'litho',
			];
		}

		/**
		 * Get widget keywords.
		 *
		 * Retrieve the list of keywords the widget belongs to.
		 *
		 * @access public
		 *
		 * @return array Widget keywords.
		 */
		public function get_keywords() {
			return [
				'SVG',
				'inline',
				'icon',
				'vector',
				'custom svg',
				'inline graphic',
			];
		}

		/**
		 * Register inline SVG widget controls.
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @access protected
		 */
		protected function register_controls() {
			$this->start_controls_section(
				'litho_section_svg',
				[
					'label' => esc_html__( 'General', 'litho-addons' ),
				]
			);
			$this->add_control(
				'litho_custom_svg',
				[
					'label'      => esc_html__( 'SVG Code', 'litho-addons' ),
					'type'       => Controls_Manager::TEXTAREA,
					'show_label' => true,
					'dynamic'    => [
						'active' => true,
					],
					'rows'       => 10,
				]
			);
			$this->add_control(
				'litho_inline_svg_link',
				[
					'label'       => esc_html__( 'External Link', 'litho-addons' ),
					'type'        => Controls_Manager::URL,
					'dynamic'     => [
						'active' => true,
					],
					'placeholder' => esc_html__( 'https://your-link.com', 'litho-addons' ),
				]
			);
			$this->end_controls_section();

			$this->start_controls_section(
				'litho_section_style_icon',
				[
					'label' => esc_html__( 'General', 'litho-addons' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_responsive_control(
				'litho_inline_svg_text_align',
				[
					'label'     => esc_html__( 'Alignment', 'litho-addons' ),
					'type'      => Controls_Manager::CHOOSE,
					'options'   => [
						'left'    => [
							'title' => esc_html__( 'Left', 'litho-addons' ),
							'icon'  => 'eicon-text-align-left',
						],
						'center'  => [
							'title' => esc_html__( 'Center', 'litho-addons' ),
							'icon'  => 'eicon-text-align-center',
						],
						'right'   => [
							'title' => esc_html__( 'Right', 'litho-addons' ),
							'icon'  => 'eicon-text-align-right',
						],
						'justify' => [
							'title' => esc_html__( 'Justified', 'litho-addons' ),
							'icon'  => 'eicon-text-align-justify',
						],
					],
					'default'   => 'center',
					'selectors' => [
						'{{WRAPPER}} .custom-svg-icon' => 'text-align: {{VALUE}};',
					],
				]
			);
			$this->add_control(
				'litho_inline_svg_color',
				[
					'label'     => esc_html__( 'Color', 'litho-addons' ),
					'type'      => Controls_Manager::COLOR,
					'global'    => [
						'default' => Global_Colors::COLOR_PRIMARY,
					],
					'selectors' => [
						'{{WRAPPER}} .custom-svg-icon svg' => 'fill: {{VALUE}};',
					],
				]
			);
			$this->add_responsive_control(
				'eael_inline_svg_size',
				[
					'label'      => esc_html__( 'Size', 'litho-addons' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => [
						'px',
						'%',
					],
					'range'      => [
						'px' => [
							'min'  => 1,
							'max'  => 500,
							'step' => 1,
						],
					],
					'default'    => [
						'size' => 100,
					],
					'selectors'  => [
						'{{WRAPPER}} .custom-svg-icon svg' => 'width:{{SIZE}}{{UNIT}}; height:{{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
		}

		/**
		 * Render inline svg widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @access protected
		 */
		protected function render() {

			$icon_tag = '';
			$settings = $this->get_settings_for_display();

			$this->add_render_attribute(
				'litho_custom_svg',
				'class',
				[
					'custom-svg-icon',
				]
			);
			$icon_tag = 'a';
			$this->add_link_attributes( 'litho_inline_svg_link', $settings['litho_inline_svg_link'] );
			$link_attributes = $this->get_render_attribute_string( 'litho_inline_svg_link' );

			if ( ! empty( $settings['litho_custom_svg'] ) ) {
				?>
				<div <?php $this->print_render_attribute_string( 'litho_custom_svg' ); ?>>
					<?php
					if ( ! empty( $settings['litho_inline_svg_link']['url'] ) ) {
						?>
						<<?php echo implode( ' ', [ $icon_tag, $link_attributes ] ); // phpcs:ignore ?>>
						<?php
					}
					?>
					<?php echo sprintf( '%s', $settings['litho_custom_svg'] );// phpcs:ignore ?>
					</<?php echo $icon_tag; // phpcs:ignore ?>>		
				</div>
				<?php
			}
		}
	}
}
