<?php
namespace LithoAddons\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Icons_Manager;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Typography;
use Elementor\Core\Kits\Documents\Tabs\Global_Typography;
use LithoAddons\Controls\Dynamic_Select;
use LithoAddons\Controls\Groups\Text_Gradient_Background;

/**
 * Litho widget for Litho Loop Builder.
 *
 * @package Litho
 *
 * @since   1.0
 */

 // If class `Loop_Builder` doesn't exists yet.
if ( ! class_exists( 'LithoAddons\Widgets\Loop_Builder' ) ) {
	/**
	 * Define `Loop_Builder` class.
	 */
	class Loop_Builder extends Widget_Base {
		/**
		 * Retrieve the widget name.
		 *
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'litho-loop-builder';
		}

		/**
		 * Retrieve the widget title.
		 *
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return esc_html__( 'Litho Loop Builder', 'litho-addons' );
		}

		/**
		 * Retrieve the widget icon.
		 *
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-loop-builder';
		}

		/**
		 * Retrieve the widget categories.
		 *
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [
				'litho',
			];
		}

		/**
		 * Get widget keywords.
		 *
		 * Retrieve the list of keywords the widget belongs to.
		 *
		 * @access public
		 *
		 * @return array Widget keywords.
		 */
		public function get_keywords() {
			return [
				'loop',
				'item',
				'dynamic',
				'list',
				'grid',
				'archive',
				'blog',
				'news',
				'article',
				'recent',
				'related',
				'repeater',
				'products',
				'posts',
				'portfolio',
				'cpt',
				'query',
				'custom',
				'builder',
				'source',
				'meta',
				'global',
				'multiple',
				'taxonomy',
				'term',
				'category',
				'tags',
				'author',
				'parameter',
				'argument',
				'current',
				'manual',
				'selection',
			];
		}

		/**
		 * Retrieve the list of scripts the Query Loop widget depended on.
		 *
		 * Used to set scripts dependencies required to run the widget.
		 *
		 * @access public
		 *
		 * @return array Widget scripts dependencies.
		 */
		public function get_script_depends() {
			$loop_builder_scripts = [];
			if ( litho_load_javascript_by_key( 'imagesloaded' ) ) {
				$loop_builder_scripts[] = 'imagesloaded';
			}

			if ( litho_load_javascript_by_key( 'isotope' ) ) {
				$loop_builder_scripts[] = 'isotope';
			}

			if ( litho_load_javascript_by_key( 'infinite-scroll' ) ) {
				$loop_builder_scripts[] = 'infinite-scroll';
			}

			if ( litho_load_javascript_by_key( 'jquery.fitvids' ) ) {
				$loop_builder_scripts[] = 'jquery.fitvids';
			}

			if ( litho_load_javascript_by_key( 'swiper' ) ) {
				$loop_builder_scripts[] = 'swiper';
			}

			$loop_builder_scripts = [
				'litho-blog-list-widget',
			];

			return $loop_builder_scripts;
		}

		/**
		 * Retrieve the list of styles the Query Loop widget depended on.
		 *
		 * Used to set styles dependencies required to run the widget.
		 *
		 * @access public
		 *
		 * @return array Widget styles dependencies.
		 */
		public function get_style_depends() {
			$loop_builder_styles = [];

			if ( litho_load_stylesheet_by_key( 'swiper' ) ) {
				$loop_builder_scripts[] = 'swiper';
			}

			$loop_builder_styles = [
				'litho-blog-list-widget',
			];

			return $loop_builder_styles;
		}

		/**
		 * Render Query Loop widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @access protected
		 */
		protected function register_controls() {

			$this->start_controls_section(
				'litho_section_loop_item_content',
				[
					'label' => esc_html__( 'General', 'litho-addons' ),
				]
			);
			$this->add_responsive_control(
				'litho_column_settings',
				[
					'label'   => esc_html__( 'Number of Columns', 'litho-addons' ),
					'type'    => Controls_Manager::SLIDER,
					'default' => [
						'size' => 3,
					],
					'separator' => 'before',
					'range'     => [
						'px' => [
							'min'  => 1,
							'max'  => 6,
							'step' => 1,
						],
					],
				]
			);
			$this->add_responsive_control(
				'litho_columns_gap',
				[
					'label'      => esc_html__( 'Columns Gap', 'litho-addons' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => [
						'px',
					],
					'default'    => [
						'size' => 15,
					],
					'range'      => [
						'px' => [
							'min'  => 0,
							'max'  => 100,
							'step' => 1,
						],
					],
					'selectors'  => [
						'{{WRAPPER}} ul li.grid-gutter'  => 'padding-left: {{SIZE}}{{UNIT}}; padding-right: {{SIZE}}{{UNIT}};',
						'{{WRAPPER}} ul.litho-loop-item' => 'margin-left: -{{SIZE}}{{UNIT}}; margin-right: -{{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'litho_post_bottom_spacing',
				[
					'label'      => esc_html__( 'Bottom Gap', 'litho-addons' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => [
						'px',
					],
					'range'      => [
						'px' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'selectors'  => [
						'{{WRAPPER}} ul li.grid-gutter' => 'margin-bottom: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'litho_items_per_page',
				[
					'label'   => esc_html__( 'Number of Items to Show', 'litho-addons' ),
					'type'    => Controls_Manager::NUMBER,
					'dynamic' => [
						'active' => true,
					],
					'default' => 12,
				]
			);
			$this->add_control(
				'litho_enable_masonry',
				[
					'label'        => esc_html__( 'Enable Masonry Layout', 'litho-addons' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => esc_html__( 'Yes', 'litho-addons' ),
					'label_off'    => esc_html__( 'No', 'litho-addons' ),
					'default'      => '',
					'return_value' => 'yes',
				]
			);
			$this->end_controls_section();
			$this->start_controls_section(
				'litho_post_content_data',
				[
					'label' => esc_html__( 'Data', 'litho-addons' ),
				]
			);

			$litho_post_types = [];
			if ( function_exists( 'litho_get_post_types' ) ) {
				$litho_post_types = litho_get_post_types();
			}

			$litho_extra_types = [
				'manual_selection' => esc_html__( 'Manual Selection', 'litho-addons' ),
				'current_query'    => esc_html__( 'Current Query', 'litho-addons' ),
			];

			$litho_post_types = array_merge( $litho_post_types, $litho_extra_types );

			$this->add_control(
				'litho_post_type_source',
				[
					'label'   => esc_html__( 'Source Type', 'litho-addons' ),
					'type'    => Controls_Manager::SELECT,
					'options' => $litho_post_types,
					'default' => 'post',
				]
			);

			$this->add_control(
				'litho_orderby',
				[
					'label'   => esc_html__( 'Order By', 'litho-addons' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'date',
					'options' => [
						'date'          => esc_html__( 'Date', 'litho-addons' ),
						'title'         => esc_html__( 'Title', 'litho-addons' ),
						'menu_order'    => esc_html__( 'Menu order', 'litho-addons' ),
						'last_modified' => esc_html__( 'Last Modified', 'litho-addons' ),
						'comment_count' => esc_html__( 'Comment count', 'litho-addons' ),
						'random'        => esc_html__( 'Random', 'litho-addons' ),
					],
				]
			);

			$this->add_control(
				'litho_order',
				[
					'label'   => esc_html__( 'Sort By', 'litho-addons' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'DESC',
					'options' => [
						'DESC' => esc_html__( 'Descending', 'litho-addons' ),
						'ASC'  => esc_html__( 'Ascending', 'litho-addons' ),
					],
				]
			);

			$this->add_control(
				'litho_posts_selected_ids',
				[
					'label'       => esc_html__( 'Search & Select', 'litho-addons' ),
					'type'        => Dynamic_Select::TYPE,
					'multiple'    => true,
					'label_block' => true,
					'query_args'  => [
						'query' => 'posts',
					],
					'condition'   => [
						'litho_post_type_source' => 'manual_selection',
					],
				]
			);

			$this->start_controls_tabs(
				'litho_args_tabs',
			);

			$this->start_controls_tab(
				'litho_include_by_tab',
				[
					'label'     => esc_html__( 'Include By', 'litho-addons' ),
					'condition' => [
						'litho_post_type_source!' => [
							'manual_selection',
							'current_query',
						],
					],
				]
			);

			$this->add_control(
				'litho_include_by',
				[
					'label'       => esc_html__( 'Include By', 'litho-addons' ),
					'type'        => Controls_Manager::SELECT2,
					'label_block' => true,
					'multiple'    => true,
					'options'     => [
						'authors' => esc_html__( 'Authors', 'litho-addons' ),
						'terms'   => esc_html__( 'Terms', 'litho-addons' ),
					],
					'condition'   => [
						'litho_post_type_source!' => [
							'manual_selection',
							'current_query',
						],
					],
				]
			);
			$this->add_control(
				'litho_posts_include_author_ids',
				[
					'label'       => esc_html__( 'Authors', 'litho-addons' ),
					'type'        => Dynamic_Select::TYPE,
					'multiple'    => true,
					'label_block' => true,
					'query_args'  => [
						'query' => 'authors',
					],
					'condition'   => [
						'litho_include_by'        => 'authors',
						'litho_post_type_source!' => [
							'manual_selection',
							'current_query',
						],
					],
				]
			);

			$this->add_control(
				'litho_posts_include_term_ids',
				[
					'label'       => esc_html__( 'Terms', 'litho-addons' ),
					'description' => esc_html__( 'Terms are items in a taxonomy. The available taxonomies are: Categories, Tags, Formats and custom taxonomies.', 'litho-addons' ),
					'type'        => Dynamic_Select::TYPE,
					'multiple'    => true,
					'label_block' => true,
					'placeholder' => esc_html__( 'Type and select terms', 'litho-addons' ),
					'query_args'  => [
						'query'        => 'terms',
						'widget_props' => [
							'post_type' => 'litho_post_type_source',
						],
					],
					'condition'   => [
						'litho_include_by'        => 'terms',
						'litho_post_type_source!' => [
							'manual_selection',
							'current_query',
						],
					],
				]
			);

			$this->end_controls_tab();

			$this->start_controls_tab(
				'litho_exclude_by_tab',
				[
					'label'     => esc_html__( 'Exclude By', 'litho-addons' ),
					'condition' => [
						'litho_post_type_source!' => [
							'manual_selection',
							'current_query',
						],
					],
				]
			);

			$this->add_control(
				'litho_exclude_by',
				[
					'label'       => esc_html__( 'Exclude By', 'litho-addons' ),
					'type'        => Controls_Manager::SELECT2,
					'multiple'    => true,
					'label_block' => true,
					'options'     => [
						'authors'          => esc_html__( 'Authors', 'litho-addons' ),
						'current_post'     => esc_html__( 'Current Post', 'litho-addons' ),
						'manual_selection' => esc_html__( 'Manual Selection', 'litho-addons' ),
						'terms'            => esc_html__( 'Terms', 'litho-addons' ),
					],
					'condition'   => [
						'litho_post_type_source!' => [
							'manual_selection',
							'current_query',
						],
					],
				]
			);
			$this->add_control(
				'litho_posts_exclude_ids',
				[
					'label'       => esc_html__( 'Search & Select', 'litho-addons' ),
					'type'        => Dynamic_Select::TYPE,
					'multiple'    => true,
					'label_block' => true,
					'query_args'  => [
						'query'        => 'posts',
						'widget_props' => [
							'post_type' => 'litho_post_type_source',
						],
					],
					'condition'   => [
						'litho_exclude_by'        => 'manual_selection',
						'litho_post_type_source!' => [
							'manual_selection',
							'current_query',
						],
					],
				]
			);
			$this->add_control(
				'litho_posts_exclude_author_ids',
				[
					'label'       => esc_html__( 'Authors', 'litho-addons' ),
					'type'        => Dynamic_Select::TYPE,
					'multiple'    => true,
					'label_block' => true,
					'query_args'  => [
						'query' => 'authors',
					],
					'condition'   => [
						'litho_exclude_by'        => 'authors',
						'litho_post_type_source!' => [
							'manual_selection',
							'current_query',
						],
					],
				]
			);
			$this->add_control(
				'litho_posts_exclude_term_ids',
				[
					'label'       => esc_html__( 'Terms', 'litho-addons' ),
					'description' => esc_html__( 'Terms are items in a taxonomy. The available taxonomies are: Categories, Tags, Formats and custom taxonomies.', 'litho-addons' ),
					'type'        => Dynamic_Select::TYPE,
					'multiple'    => true,
					'label_block' => true,
					'placeholder' => esc_html__( 'Type and select terms', 'litho-addons' ),
					'query_args'  => [
						'query'        => 'terms',
						'widget_props' => [
							'post_type' => 'litho_post_type_source',
						],
					],
					'condition'   => [
						'litho_exclude_by'        => 'terms',
						'litho_post_type_source!' => [
							'manual_selection',
							'current_query',
						],
					],
				]
			);
			$this->end_controls_tab();
			$this->end_controls_tabs();
			$this->add_control(
				'litho_query_loop_offset',
				[
					'label'   => esc_html__( 'Offset', 'litho-addons' ),
					'type'    => Controls_Manager::NUMBER,
					'dynamic' => [
						'active' => true,
					],
					'condition' => [
						'litho_post_type_source!' => 'current_query',
					],
				]
			);
			$this->add_control(
				'query_id',
				[
					'label'       => esc_html__( 'Query ID', 'litho-addons' ),
					'description' => esc_html__( 'Give your Query a custom unique id to allow server side filtering', 'litho-addons' ),
					'type'        => Controls_Manager::TEXT,
					'separator'   => 'before',
				]
			);
			$this->end_controls_section();

			$this->start_controls_section(
				'litho_post_content_settings',
				[
					'label' => esc_html__( 'Content Settings', 'litho-addons' ),
				]
			);

			$this->add_control(
				'litho_show_post_thumbnail',
				[
					'label'        => esc_html__( 'Enable Thumbnail', 'litho-addons' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => esc_html__( 'Yes', 'litho-addons' ),
					'label_off'    => esc_html__( 'No', 'litho-addons' ),
					'return_value' => 'yes',
					'default'      => 'yes',
				]
			);

			$this->add_control(
				'litho_thumbnail',
				[
					'label'          => esc_html__( 'Image Size', 'litho-addons' ),
					'type'           => Controls_Manager::SELECT,
					'default'        => 'full',
					'options'        => ( function_exists( 'litho_get_thumbnail_image_sizes' ) ) ? litho_get_thumbnail_image_sizes() : [],
					'style_transfer' => true,
					'condition'      => [
						'litho_show_post_thumbnail' => 'yes',
					],
				]
			);

			$this->add_control(
				'litho_show_post_category',
				[
					'label'        => esc_html__( 'Enable Category', 'litho-addons' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => esc_html__( 'Yes', 'litho-addons' ),
					'label_off'    => esc_html__( 'No', 'litho-addons' ),
					'return_value' => 'yes',
					'default'      => 'yes',
				]
			);

			$this->add_control(
				'litho_show_post_title',
				[
					'label'        => esc_html__( 'Enable Title', 'litho-addons' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => esc_html__( 'Yes', 'litho-addons' ),
					'label_off'    => esc_html__( 'No', 'litho-addons' ),
					'return_value' => 'yes',
					'default'      => 'yes',
				]
			);
			$this->add_control(
				'litho_show_post_excerpt',
				[
					'label'        => esc_html__( 'Enable Excerpt', 'litho-addons' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => esc_html__( 'Yes', 'litho-addons' ),
					'label_off'    => esc_html__( 'No', 'litho-addons' ),
					'return_value' => 'yes',
					'default'      => 'yes',
				]
			);
			$this->add_control(
				'litho_post_excerpt_length',
				[
					'label'   => esc_html__( 'Excerpt Length', 'litho-addons' ),
					'type'    => Controls_Manager::NUMBER,
					'dynamic' => [
						'active' => true,
					],
					'min'       => 1,
					'default'   => 18,
					'condition' => [
						'litho_show_post_excerpt' => 'yes',
					],
				]
			);
			$this->add_control(
				'litho_show_post_author',
				[
					'label'        => esc_html__( 'Enable Author', 'litho-addons' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => esc_html__( 'Yes', 'litho-addons' ),
					'label_off'    => esc_html__( 'No', 'litho-addons' ),
					'return_value' => 'yes',
					'default'      => 'yes',
				]
			);
			$this->add_control(
				'litho_show_post_author_image',
				[
					'label'        => esc_html__( 'Enable Author Avtar', 'litho-addons' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => esc_html__( 'Yes', 'litho-addons' ),
					'label_off'    => esc_html__( 'No', 'litho-addons' ),
					'return_value' => 'yes',
					'condition'    => [
						'litho_show_post_author' => 'yes',
					],
				]
			);
			$this->add_control(
				'litho_show_post_author_text',
				[
					'label'   => esc_html__( 'Author Before Text', 'litho-addons' ),
					'type'    => Controls_Manager::TEXT,
					'dynamic' => [
						'active' => true,
					],
					'default'   => esc_html__( 'By&nbsp;', 'litho-addons' ),
					'condition' => [
						'litho_show_post_author' => 'yes',
					],
				]
			);
			$this->add_control(
				'litho_show_post_date',
				[
					'label'        => esc_html__( 'Enable Date', 'litho-addons' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => esc_html__( 'Yes', 'litho-addons' ),
					'label_off'    => esc_html__( 'No', 'litho-addons' ),
					'return_value' => 'yes',
					'default'      => 'yes',
				]
			);
			$this->add_control(
				'litho_post_icon',
				[
					'label'        => esc_html__( 'Enable Date Icon', 'litho-addons' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => esc_html__( 'Yes', 'litho-addons' ),
					'label_off'    => esc_html__( 'No', 'litho-addons' ),
					'return_value' => 'yes',
					'condition'    => [
						'litho_show_post_date' => 'yes',
					],
				]
			);
			$this->add_control(
				'litho_post_date_format',
				[
					'label'   => esc_html__( 'Post Date Format', 'litho-addons' ),
					'type'    => Controls_Manager::TEXT,
					'dynamic' => [
						'active' => true,
					],
					'description' => sprintf(
						'%1$s <a target="_blank" href="%2$s" rel="noopener noreferrer">%3$s</a> %4$s',
						esc_html__( 'Date format should be like F j, Y', 'litho-addons' ),
						esc_url( 'https://wordpress.org/support/article/formatting-date-and-time/#format-string-examples' ),
						esc_html__( 'click here', 'litho-addons' ),
						esc_html__( 'to see other date formats.', 'litho-addons' )
					),
					'condition'   => [
						'litho_show_post_date' => 'yes',
					],
				]
			);
			$this->add_control(
				'litho_show_post_like',
				[
					'label'        => esc_html__( 'Enable Like Count', 'litho-addons' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => esc_html__( 'Yes', 'litho-addons' ),
					'label_off'    => esc_html__( 'No', 'litho-addons' ),
					'return_value' => 'yes',
				],
			);
			$this->add_control(
				'litho_show_post_like_text',
				[
					'label'        => esc_html__( 'Enable Like Label', 'litho-addons' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => esc_html__( 'Yes', 'litho-addons' ),
					'label_off'    => esc_html__( 'No', 'litho-addons' ),
					'return_value' => 'inline-block',
					'default'      => '',
					'selectors'    => [
						'{{WRAPPER}} .blog-like span.posts-like' => 'display: {{VALUE}} !important;',
					],
					'condition'    => [
						'litho_show_post_like' => 'yes',
					],
				]
			);
			$this->add_control(
				'litho_show_post_comments',
				array_merge(
					[
						'label'        => esc_html__( 'Enable Comments Count', 'litho-addons' ),
						'type'         => Controls_Manager::SWITCHER,
						'label_on'     => esc_html__( 'Yes', 'litho-addons' ),
						'label_off'    => esc_html__( 'No', 'litho-addons' ),
						'return_value' => 'yes',
					],
				)
			);
			$this->add_control(
				'litho_show_post_comments_text',
				array_merge(
					[
						'label'        => esc_html__( 'Enable Comments Label', 'litho-addons' ),
						'type'         => Controls_Manager::SWITCHER,
						'label_on'     => esc_html__( 'Yes', 'litho-addons' ),
						'label_off'    => esc_html__( 'No', 'litho-addons' ),
						'return_value' => 'inline-block',
						'default'      => '',
						'selectors'    => [
							'{{WRAPPER}} .comment-link span.comment-text' => 'display: {{VALUE}} !important;',
						],
						'condition'    => [
							'litho_show_post_comments' => 'yes',
						],
					],
				)
			);
			$this->add_control(
				'litho_show_post_read_more_button',
				[
					'label'        => esc_html__( 'Enable Button', 'litho-addons' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => esc_html__( 'Yes', 'litho-addons' ),
					'label_off'    => esc_html__( 'No', 'litho-addons' ),
					'default'      => 'yes',
					'return_value' => 'yes',
				]
			);
			$this->add_control(
				'litho_read_more_text',
				[
					'label'   => esc_html__( 'Button Text', 'litho-addons' ),
					'type'    => Controls_Manager::TEXT,
					'dynamic' => [
						'active' => true,
					],
					'default'   => esc_html__( 'Read More', 'litho-addons' ),
					'condition' => [
						'litho_show_post_read_more_button' => 'yes',
					],
				]
			);
			$this->add_control(
				'litho_separator',
				[
					'label'        => esc_html__( 'Separator', 'litho-addons' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => esc_html__( 'Yes', 'litho-addons' ),
					'label_off'    => esc_html__( 'No', 'litho-addons' ),
					'return_value' => 'yes',
					'default'      => 'yes',
					'condition'    => [
						'litho_show_post_read_more_button' => 'yes',
					],
				]
			);
			$this->add_control(
				'litho_ignore_sticky_posts',
				[
					'label'        => esc_html__( 'Ignore Sticky Posts', 'litho-addons' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => esc_html__( 'Yes', 'litho-addons' ),
					'label_off'    => esc_html__( 'No', 'litho-addons' ),
					'return_value' => 'yes',
					'default'      => 'yes',
					'condition'    => [
						'litho_post_type_source' => 'post',
					],
				]
			);
			$this->end_controls_section();

			$this->start_controls_section(
				'litho_section_pagination',
				[
					'label' => esc_html__( 'Pagination', 'litho-addons' ),
				]
			);
			$this->add_control(
				'litho_pagination',
				[
					'label'   => esc_html__( 'Pagination', 'litho-addons' ),
					'type'    => Controls_Manager::SELECT,
					'default' => '',
					'options' => [
						''                           => esc_html__( 'None', 'litho-addons' ),
						'number-pagination'          => esc_html__( 'Number', 'litho-addons' ),
						'next-prev-pagination'       => esc_html__( 'Next / Previous', 'litho-addons' ),
						'infinite-scroll-pagination' => esc_html__( 'Infinite Scroll', 'litho-addons' ),
						'load-more-pagination'       => esc_html__( 'Load More', 'litho-addons' ),
					],
				]
			);
			$this->add_control(
				'litho_pagination_next_label',
				[
					'label'   => esc_html__( 'Next Label', 'litho-addons' ),
					'type'    => Controls_Manager::TEXT,
					'dynamic' => [
						'active' => true,
					],
					'default'   => esc_html__( 'NEXT', 'litho-addons' ),
					'condition' => [
						'litho_pagination' => [
							'number-pagination',
							'next-prev-pagination',
						],
					],
				]
			);
			$this->add_control(
				'litho_pagination_next_icon',
				[
					'label'            => esc_html__( 'Next Icon', 'litho-addons' ),
					'type'             => Controls_Manager::ICONS,
					'fa4compatibility' => 'icon',
					'default'          => [
						'value'   => 'feather icon-feather-arrow-right',
						'library' => 'fa-solid',
					],
					'recommended'            => [
						'fa-solid'   => [
							'angle-right',
							'caret-square-right',
						],
						'fa-regular' => [
							'caret-square-right',
						],
					],
					'skin'                   => 'inline',
					'exclude_inline_options' => 'svg',
					'label_block'            => false,
					'condition'              => [
						'litho_pagination' => [
							'number-pagination',
							'next-prev-pagination',
						],
					],
				]
			);
			$this->add_control(
				'litho_pagination_prev_label',
				[
					'label'   => esc_html__( 'Previous Label', 'litho-addons' ),
					'type'    => Controls_Manager::TEXT,
					'dynamic' => [
						'active' => true,
					],
					'default'   => esc_html__( 'PREV', 'litho-addons' ),
					'condition' => [
						'litho_pagination' => [
							'number-pagination',
							'next-prev-pagination',
						],
					],
				]
			);
			$this->add_control(
				'litho_pagination_prev_icon',
				[
					'label'            => esc_html__( 'Previous Icon', 'litho-addons' ),
					'type'             => Controls_Manager::ICONS,
					'fa4compatibility' => 'icon',
					'default'          => [
						'value'   => 'feather icon-feather-arrow-left',
						'library' => 'fa-solid',
					],
					'recommended'            => [
						'fa-solid'   => [
							'angle-left',
							'caret-square-left',
						],
						'fa-regular' => [
							'caret-square-left',
						],
					],
					'skin'                   => 'inline',
					'exclude_inline_options' => 'svg',
					'label_block'            => false,
					'condition'              => [
						'litho_pagination' => [
							'number-pagination',
							'next-prev-pagination',
						],
					],
				]
			);
			$this->add_control(
				'litho_pagination_load_more_button_label',
				[
					'label'   => esc_html__( 'Button Label', 'litho-addons' ),
					'type'    => Controls_Manager::TEXT,
					'dynamic' => [
						'active' => true,
					],
					'default'     => esc_html__( 'Load more', 'litho-addons' ),
					'render_type' => 'template',
					'condition'   => [
						'litho_pagination' => 'load-more-pagination',
					],
				]
			);
			$this->end_controls_section();

			$this->start_controls_section(
				'litho_section_loop_item_general_style',
				[
					'label' => esc_html__( 'General', 'litho-addons' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_group_control(
				Group_Control_Background::get_type(),
				[
					'name'    => 'litho_loop_item_content_box_color',
					'exclude' => [
						'image',
						'position',
						'attachment',
						'attachment_alert',
						'repeat',
						'size',
					],
					'selector' => '{{WRAPPER}} .blog-post',
				]
			);
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				[
					'name'     => 'litho_loop_item_post_shadow',
					'selector' => '{{WRAPPER}} .blog-post',
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'     => 'litho_loop_item_post_border',
					'selector' => '{{WRAPPER}} .blog-post',
				]
			);
			$this->add_responsive_control(
				'litho_loop_item_post_border_radius',
				[
					'label'      => esc_html__( 'Border Radius', 'litho-addons' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [
						'px',
						'%',
						'custom',
					],
					'selectors'  => [
						'{{WRAPPER}} .blog-post' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'litho_loop_item_content_box_padding',
				[
					'label'      => esc_html__( 'Padding', 'litho-addons' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [
						'px',
						'%',
						'em',
						'rem',
						'custom',
					],
					'selectors'  => [
						'{{WRAPPER}} .grid-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();

			$this->start_controls_section(
				'litho_section_blog_list_image_style',
				[
					'label'      => esc_html__( 'Thumbnail', 'litho-addons' ),
					'tab'        => Controls_Manager::TAB_STYLE,
					'show_label' => false,
					'condition'  => [
						'litho_show_post_thumbnail' => 'yes'
					],
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'     => 'litho_image_border',
					'selector' => '{{WRAPPER}} .blog-post-images',
				]
			);
			$this->add_responsive_control(
				'litho_image_border_radius',
				[
					'label'      => esc_html__( 'Border Radius', 'litho-addons' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [
						'px',
						'%',
						'custom',
					],
					'selectors'     => [
						'{{WRAPPER}} .blog-post-images' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);

			$this->end_controls_section();

			$this->start_controls_section(
				'litho_section_loop_item_title_style',
				[
					'label'     => esc_html__( 'Title', 'litho-addons' ),
					'tab'       => Controls_Manager::TAB_STYLE,
					'condition' => [
						'litho_show_post_title' => 'yes',
					]
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'   => 'litho_loop_item_title_typography',
					'global' => [
						'default' => Global_Typography::TYPOGRAPHY_SECONDARY,
					],
					'selector' => '{{WRAPPER}} .entry-title',
				]
			);

			$this->start_controls_tabs(
				'litho_loop_item_title_tabs',
			);

			$this->start_controls_tab(
				'litho_loop_item_title_normal_tab',
				[
					'label' => esc_html__( 'Normal', 'litho-addons' ),
				]
			);

			$this->add_control(
				'litho_loop_item_title_color',
				[
					'label'     => esc_html__( 'Color', 'litho-addons' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .entry-title' => 'color: {{VALUE}};',
					],
				]
			);

			$this->end_controls_tab();

			$this->start_controls_tab(
				'litho_loop_item_title_hover_tab',
				[
					'label' => esc_html__( 'Hover', 'litho-addons' ),
				]
			);
			$this->add_control(
				'litho_loop_item_title_hover_color',
				[
					'label'     => esc_html__( 'Color', 'litho-addons' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .entry-title:hover' => 'color: {{VALUE}};',
					],
				]
			);

			$this->end_controls_tab();

			$this->end_controls_tabs();

			$this->add_responsive_control(
				'litho_loop_item_title_width',
				[
					'label'      => esc_html__( 'Width', 'litho-addons' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => [
						'px',
						'%',
						'custom',
					],
					'range'      => [
						'px' => [
							'min' => 1,
							'max' => 200,
						],
						'%'  => [
							'min' => 1,
							'max' => 100,
						],
					],
					'default'    => [
						'unit' => '%',
						'size' => 100,
					],
					'selectors'  => [
						'{{WRAPPER}} .entry-title' => 'width: {{SIZE}}{{UNIT}}',
					],
				]
			);
			$this->add_responsive_control(
				'litho_loop_item_title_min_height',
				[
					'label'      => esc_html__( 'Min Height', 'litho-addons' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => [
						'px',
						'custom',
					],
					'range'      => [
						'px' => [
							'min' => 1,
							'max' => 150,
						],
					],
					'selectors'  => [
						'{{WRAPPER}} .entry-title' => 'min-height: {{SIZE}}{{UNIT}}',
					],
				]
			);
			$this->end_controls_section();

			$this->start_controls_section(
				'litho_section_loop_item_content_style',
				[
					'label'      => esc_html__( 'Excerpt', 'litho-addons' ),
					'tab'        => Controls_Manager::TAB_STYLE,
					'show_label' => false,
					'condition'  => [
						'litho_show_post_excerpt' => 'yes',
					]
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'   => 'litho_loop_item_content_typography',
					'global' => [
						'default' => Global_Typography::TYPOGRAPHY_TEXT,
					],
					'selector' => '{{WRAPPER}} .entry-content',
				]
			);
			$this->add_control(
				'litho_loop_item_content_color',
				[
					'label'     => esc_html__( 'Color', 'litho-addons' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .entry-content' => 'color: {{VALUE}};',
					],
				]
			);
			$this->add_responsive_control(
				'litho_loop_item_content_width',
				[
					'label'      => esc_html__( 'Width', 'litho-addons' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => [
						'px',
						'%',
						'custom',
					],
					'range'      => [
						'px' => [
							'min' => 18,
							'max' => 200,
						],
						'%'  => [
							'min' => 1,
							'max' => 100,
						],
					],
					'default'    => [
						'unit' => '%',
					],
					'selectors'  => [
						'{{WRAPPER}} .entry-content' => 'width: {{SIZE}}{{UNIT}}',
					],
				]
			);
			$this->add_responsive_control(
				'litho_loop_item_content_margin',
				[
					'label'      => esc_html__( 'Spacer', 'litho-addons' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => [
						'px',
						'custom',
					],
					'range'      => [
						'px' => [
							'min' => 0,
							'max' => 50,
						],
					],
					'selectors'  => [
						'{{WRAPPER}} .entry-content' => 'margin-bottom: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();

			$this->start_controls_section(
				'litho_loop_item_date_style_heading',
				[
					'label' => esc_html__( 'Date', 'litho-addons' ),
					'tab'   => Controls_Manager::TAB_STYLE,
					'condition' 	=> [
						'litho_show_post_date' => 'yes',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'   => 'litho_loop_item_date_typography',
					'global' => [
						'default' => Global_Typography::TYPOGRAPHY_PRIMARY,
					],
					'selector' => '{{WRAPPER}} .post-date',
				]
			);
			$this->add_control(
				'litho_loop_item_date_color',
				[
					'label'     => esc_html__( 'Color', 'litho-addons' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .post-date' => 'color: {{VALUE}};',
					],
				]
			);
			$this->add_responsive_control(
				'litho_blog_list_meta_date_margin',
				[
					'label'         => esc_html__( 'Margin', 'litho-addons' ),
					'type'          => Controls_Manager::DIMENSIONS,
					'size_units'    => [
						'px',
						'%',
						'em',
						'rem',
						'custom',
					],
					'selectors'     => [
						'{{WRAPPER}} .post-date' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();

			$this->start_controls_section(
				'litho_loop_item_author_style',
				[
					'label' => esc_html__( 'Author', 'litho-addons' ),
					'tab'   => Controls_Manager::TAB_STYLE,
					'condition'    => [
						'litho_show_post_author' => 'yes',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'   => 'litho_loop_item_author_typography',
					'global' => [
						'default' => Global_Typography::TYPOGRAPHY_PRIMARY,
					],
					'selector' => '{{WRAPPER}} .author-name a',
				]
			);

			$this->start_controls_tabs(
				'litho_loop_item_author_tabs',
			);

			$this->start_controls_tab(
				'litho_loop_item_author_normal_tab',
				[
					'label' => esc_html__( 'Normal', 'litho-addons' ),
				]
			);

			$this->add_control(
				'litho_loop_item_author_color',
				[
					'label'     => esc_html__( 'Color', 'litho-addons' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .author-name a' => 'color: {{VALUE}};',
					],
				]
			);

			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'     => 'litho_loop_item_author_border',
					'selector' => '{{WRAPPER}} .author-name a',
				]
			);

			$this->end_controls_tab();

			$this->start_controls_tab(
				'litho_loop_item_author_hover_tab',
				[
					'label' => esc_html__( 'Hover', 'litho-addons' ),
				]
			);

			$this->add_control(
				'litho_loop_item_author_hover_color',
				[
					'label'     => esc_html__( 'Color', 'litho-addons' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .author-name a:hover' => 'color: {{VALUE}};',
					],
				]
			);

			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'     => 'litho_loop_item_author_hover_border',
					'selector' => '{{WRAPPER}} .author-name a:hover',
				]
			);

			$this->end_controls_tab();

			$this->end_controls_tabs();

			$this->add_responsive_control(
				'litho_blog_list_meta_author_margin',
				[
					'label'         => esc_html__( 'Margin', 'litho-addons' ),
					'type'          => Controls_Manager::DIMENSIONS,
					'size_units'    => [
						'px',
						'%',
						'custom',
					],
					'selectors'     => [
						'{{WRAPPER}} .author-name' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);

			$this->end_controls_section();

			$this->start_controls_section(
				'litho_loop_item_categories_style',
				[
					'label'     => esc_html__( 'Category', 'litho-addons' ),
					'tab'       => Controls_Manager::TAB_STYLE,
					'condition' => [
						'litho_post_type_source!' => [
							'page',
						],
						'litho_show_post_category' => 'yes', 
					],
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'   => 'litho_loop_item_categories_typography',
					'global' => [
						'default' => Global_Typography::TYPOGRAPHY_SECONDARY,
					],
					'selector' => '{{WRAPPER}} .blog-category a',
				]
			);

			$this->start_controls_tabs(
				'litho_loop_item_categories_tabs'
			);

			$this->start_controls_tab(
				'litho_loop_item_categories_normal_tab',
				[
					'label' => esc_html__( 'Normal', 'litho-addons' ),
				],
			);

			$this->add_group_control(
				Group_Control_Background::get_type(),
				[
					'name'    => 'litho_loop_itemcategories_bg__color',
					'exclude' => [
						'image',
						'position',
						'attachment',
						'attachment_alert',
						'repeat',
						'size',
					],
					'selector' => '{{WRAPPER}} .blog-category a',
				]
			);

			$this->add_control(
				'litho_loop_item_categories_color',
				[
					'label'     => esc_html__( 'Text Color', 'litho-addons' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .blog-category a' => 'color: {{VALUE}};',
					],
				]
			);

			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'     => 'litho_loop_item_categories_border',
					'selector' => '{{WRAPPER}} .blog-category a',
				]
			);

			$this->add_responsive_control(
				'litho_loop_item_categories_border_radius',
				[
					'label'      => esc_html__( 'Border Radius', 'litho-addons' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [
						'px',
						'%',
						'custom',
					],
					'selectors'  => [
						'{{WRAPPER}} .blog-category a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);

			$this->end_controls_tab();

			$this->start_controls_tab(
				'litho_loop_item_categories_hover_tab',
				[
					'label' => esc_html__( 'Hover', 'litho-addons' ),
				],
			);

			$this->add_group_control(
				Group_Control_Background::get_type(),
				[
					'name'    => 'litho_loop_item_categories_hover_bg_color',
					'exclude' => [
						'image',
						'position',
						'attachment',
						'attachment_alert',
						'repeat',
						'size',
					],
					'selector' => '{{WRAPPER}} .blog-category a:hover',
				]
			);

			$this->add_control(
				'litho_loop_item_categories_hover_color',
				[
					'label'     => esc_html__( 'Text Color', 'litho-addons' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .blog-category a:hover' => 'color: {{VALUE}};',
					],
				]
			);
			$this->add_control(
				'litho_loop_item_categories_hover_border',
				[
					'label'     => esc_html__( 'Border Color', 'litho-addons' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .blog-category a:hover' => 'border-color: {{VALUE}};',
					],
				]
			);

			$this->end_controls_tab();

			$this->end_controls_tabs();

			$this->add_responsive_control(
				'litho_loop_item_categories_padding',
				[
					'label'      => esc_html__( 'Padding', 'litho-addons' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [
						'px',
						'em',
						'%',
						'rem',
						'custom',
					],
					'selectors'  => [
						'{{WRAPPER}} .blog-category a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();

			$this->start_controls_section(
				'litho_post_meta_likes_style_heading',
				[
					'label'     => esc_html__( 'Like', 'litho-addons' ),
					'tab'       => Controls_Manager::TAB_STYLE,
					'condition' => [
						'litho_post_type_source!' => [
							'page',
						],
						'litho_show_post_like' => 'yes',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'   => 'litho_blog_list_meta_likes_typography',
					'global' => [
						'default' => Global_Typography::TYPOGRAPHY_PRIMARY,
					],
					'selector'  => '{{WRAPPER}} .post-meta-like a',
					'condition' => [
						'litho_show_post_like' => 'yes',
					],
				]
			);
			$this->start_controls_tabs( 'litho_blog_list_meta_likes_tabs' );
				$this->start_controls_tab( 'litho_blog_list_meta_likes_normal_tab', 
					[ 
						'label'     => esc_html__( 'Normal', 'litho-addons' ),
						'condition' => [
							'litho_show_post_like' => 'yes',
						],
					]
				);
					$this->add_control(
						'litho_blog_list_meta_likes_color',
						[
							'label'     => esc_html__( 'Color', 'litho-addons' ),
							'type'      => Controls_Manager::COLOR,
							'selectors' => [
								'{{WRAPPER}} .post-meta-like a' => 'color: {{VALUE}};',
							],
							'condition' 	=> [
								'litho_show_post_like' => 'yes',
							],
						]
					);
				$this->end_controls_tab();
				$this->start_controls_tab( 'litho_blog_list_meta_likes_hover_tab', 
					[ 
						'label'     => esc_html__( 'Hover', 'litho-addons' ),
						'condition' => [
							'litho_show_post_like' => 'yes',
						],
					]
				);
					$this->add_control(
						'litho_blog_list_meta_likes_hover_color',
						[
							'label'     => esc_html__( 'Color', 'litho-addons' ),
							'type'      => Controls_Manager::COLOR,
							'selectors' => [
								'{{WRAPPER}} .post-meta-like a:hover' => 'color: {{VALUE}};',
							],
							'condition' 	=> [
								'litho_show_post_like' => 'yes',
							],
						]
					);
					$this->end_controls_tab();
				$this->end_controls_tabs();
			$this->end_controls_section();

			$this->start_controls_section(
				'litho_post_meta_comment_style_heading',
				[
					'label'     => esc_html__( 'Comment', 'litho-addons' ),
					'tab'       => Controls_Manager::TAB_STYLE,
					'condition' => [
						'litho_post_type_source!' => [
							'page',
						],
						'litho_show_post_comments' => 'yes',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'   => 'litho_blog_list_meta_comment_typography',
					'global' => [
						'default' => Global_Typography::TYPOGRAPHY_PRIMARY,
					],
					'selector'  => '{{WRAPPER}} .post-meta-comments a',
					'condition' => [
						'litho_show_post_comments' => 'yes',
					],
				]
			);
			$this->start_controls_tabs( 'litho_blog_list_meta_comment_tabs' );
				$this->start_controls_tab( 'litho_blog_list_meta_comment_normal_tab', 
					[ 
						'label'     => esc_html__( 'Normal', 'litho-addons' ),
						'condition' => [
							'litho_show_post_comments' => 'yes',
						],
					]
				);
					$this->add_control(
						'litho_blog_list_meta_comment_color',
						[
							'label'     => esc_html__( 'Color', 'litho-addons' ),
							'type'      => Controls_Manager::COLOR,
							'selectors' => [
								'{{WRAPPER}} .post-meta-comments a' => 'color: {{VALUE}};',
							],
							'condition' 	=> [
								'litho_show_post_comments' => 'yes',
							],
						]
					);
				$this->end_controls_tab();
				$this->start_controls_tab( 'litho_blog_list_meta_comment_hover_tab', 
					[ 
						'label'     => esc_html__( 'Hover', 'litho-addons' ),
						'condition' => [
							'litho_show_post_comments' => 'yes',
						],
					]
				);
					$this->add_control(
						'litho_blog_list_meta_comment_hover_color',
						[
							'label'     => esc_html__( 'Color', 'litho-addons' ),
							'type'      => Controls_Manager::COLOR,
							'selectors' => [
								'{{WRAPPER}} .post-meta-comments a:hover' => 'color: {{VALUE}};',
							],
							'condition' 	=> [
								'litho_show_post_comments' => 'yes',
							],
						]
					);
				$this->end_controls_tab();
			$this->end_controls_tabs();
			$this->end_controls_section();

			$this->start_controls_section(
				'litho_section_blog_list_cta_button',
				[
					'label'     => esc_html__( 'Read More', 'litho-addons' ),
					'tab'       => Controls_Manager::TAB_STYLE,
					'condition' => [
						'litho_show_post_read_more_button' => 'yes',
					],
				]
			);
			$this->add_responsive_control(
				'litho_post_button_width',
				[
					'label'      => esc_html__( 'Width', 'litho-addons' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => [
						'px',
						'em',
						'%',
						'custom',
					],
					'default' 		=> [
						'unit' => 'px',
					],
					'tablet_default' => [
						'unit' => 'px',
					],
					'mobile_default' => [
						'unit' => 'px',
					],
					'range' 		=> [
						'px' => [
							'min' => 1,
							'max' => 1000,
						],
					],
					'selectors' 	=> [
						'{{WRAPPER}} a.elementor-button:not(.btn-custom-effect), {{WRAPPER}} a.elementor-button.btn-custom-effect:not(.hvr-btn-expand-ltr), {{WRAPPER}} a.elementor-button.hvr-btn-expand-ltr:before, {{WRAPPER}} .elementor-gradient-button' => 'width: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'litho_post_button_height',
				[
					'label'      => esc_html__( 'Height', 'litho-addons' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => [
						'px',
						'em',
					],
					'default' 		=> [
						'unit' => 'px',
					],
					'tablet_default' => [
						'unit' => 'px',
					],
					'mobile_default' => [
						'unit' => 'px',
					],
					'range' 		=> [
						'px' => [
							'min' => 1,
							'max' => 1000,
						],
					],
					'selectors' 	=> [
						'{{WRAPPER}} .elementor-button, {{WRAPPER}} .elementor-gradient-button' => 'height: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'   => 'litho_post_button_typography',
					'global' => [
						'default' => Global_Typography::TYPOGRAPHY_PRIMARY,
					],
					'selector' => '{{WRAPPER}} .elementor-button, {{WRAPPER}} .elementor-gradient-button',
				]
			);
			$this->start_controls_tabs( 'litho_tabs_button' );
				$this->start_controls_tab(
					'litho_tabs_button_normal',
					[
						'label' => esc_html__( 'Normal', 'litho-addons' ),
					]
				);
				$this->add_group_control(
					Text_Gradient_Background::get_type(),
					[
						'name'     => 'litho_post_button_text_color',
						'selector' => '{{WRAPPER}} a.elementor-gradient-button .elementor-gradient-button-text, {{WRAPPER}} .elementor-gradient-button .elementor-gradient-button-text, {{WRAPPER}} a.elementor-gradient-button .elementor-gradient-button-icon i, {{WRAPPER}} .elementor-gradient-button .elementor-gradient-button-icon i',
						
					]
				);
				$this->add_control(
					'litho_post_button_border_heading',
					[
						'label'     => esc_html__( 'Border', 'litho-addons' ),
						'type'      => Controls_Manager::HEADING,
						'separator' => 'before',
					]
				);
				$this->add_group_control(
					Group_Control_Border::get_type(),
					[
						'name'     => 'litho_post_button_border',
						'selector' => '{{WRAPPER}} .elementor-button, {{WRAPPER}} .elementor-gradient-button',
					]
				);
				$this->add_control(
					'litho_post_button_border_radius',
					[
						'label'      => esc_html__( 'Border Radius', 'litho-addons' ),
						'type'       => Controls_Manager::DIMENSIONS,
						'size_units' => [
							'px',
							'%',
							'custom',
						],
						'default' 		=> [
							'unit' => 'px',
						],
						'selectors' 	=> [
							'{{WRAPPER}} a.elementor-button:not(.btn-custom-effect), {{WRAPPER}} a.elementor-button.btn-custom-effect:not(.hvr-btn-expand-ltr), {{WRAPPER}} a.elementor-button.hvr-btn-expand-ltr:before, {{WRAPPER}} .elementor-gradient-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
						],
					]
				);
				$this->add_group_control(
					Group_Control_Box_Shadow::get_type(),
					[
						'name'     => 'litho_post_button_box_shadow',
						'selector' => '{{WRAPPER}} .elementor-button, {{WRAPPER}} .elementor-gradient-button',
					]
				);
				$this->end_controls_tab();

				$this->start_controls_tab(
					'litho_tabs_button_hover',
					[
						'label' => esc_html__( 'Hover', 'litho-addons' ),
					]
				);
				$this->add_group_control(
					Text_Gradient_Background::get_type(),
					[
						'name'     => 'litho_post_button_hover_text_color',
						'selector' => '{{WRAPPER}} a.elementor-gradient-button:hover .elementor-gradient-button-text, {{WRAPPER}} .elementor-gradient-button:hover .elementor-gradient-button-text, {{WRAPPER}} a.elementor-gradient-button:hover .elementor-gradient-button-icon i, {{WRAPPER}} .elementor-gradient-button:hover .elementor-gradient-button-icon i',
					]
				);
				$this->end_controls_tab();
			$this->end_controls_tabs();
			$this->add_responsive_control(
				'litho_blog_list_button_padding',
				[
					'label'      => esc_html__( 'Padding', 'litho-addons' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [
						'px',
						'%',
						'em',
						'rem',
						'custom',
					],
					'selectors'     => [
						'{{WRAPPER}} .elementor-button, {{WRAPPER}} .elementor-gradient-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'litho_post_button_spacing',
				[
					'label'      => esc_html__( 'Margin', 'litho-addons' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [
						'px',
						'%',
						'em',
						'rem',
						'custom',
					],
					'selectors' 	=> [
						'{{WRAPPER}} .elementor-button, {{WRAPPER}} .elementor-gradient-button' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
			$this->start_controls_section(
				'litho_section_blog_list_separator_style',
				[
					'label'      => esc_html__( 'Separator', 'litho-addons' ),
					'tab'        => Controls_Manager::TAB_STYLE,
					'show_label' => false,
				]
			);
			$this->add_group_control(
				Group_Control_Background::get_type(),
				[
					'name'  => 'litho_separator_color',
					'types' => [
						'classic',
						'gradient',
					],
					'exclude'		=> [
						'image',
						'position',
						'attachment',
						'attachment_alert',
						'repeat',
						'size',
					],
					'fields_options' => [
						'background' 	=> [
							'label' => esc_html__( 'Color', 'litho-addons' ),
						],
					],
					'selector' => '{{WRAPPER}} .horizontal-separator',
				]
			);
			$this->add_control(
				'litho_separator_thickness',
				[
					'label'      => esc_html__( 'Separator Thickness', 'litho-addons' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => [
						'px',
					],
					'range'         => [
						'px'   => [
							'min' => 1,
							'max' => 10,
						],
					],
					'selectors'     => [
						'{{WRAPPER}} .horizontal-separator' => 'height: {{SIZE}}{{UNIT}};',
					]
				]
			);
			$this->add_responsive_control(
				'litho_separator_length',
				[
					'label'      => esc_html__( 'Separator Length', 'litho-addons' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => [
						'px',
						'%',
						'custom',
					],
					'range'         => [
						'px'   => [
							'min' => 1,
							'max' => 200,
						],
						'%'   => [
							'min' => 1,
							'max' => 100,
						],
					],
					'selectors'     => [
						'{{WRAPPER}} .horizontal-separator' => 'width: {{SIZE}}{{UNIT}};',
					]
				]
			);
			$this->add_responsive_control(
				'litho_separator_spacing',
				[
					'label'      => esc_html__( 'Separator Spacing', 'litho-addons' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [
						'px',
						'%',
						'em',
						'rem',
						'custom',
					],
					'selectors'     => [
						'{{WRAPPER}} .horizontal-separator' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					]
				]
			);
			$this->end_controls_section();
			$this->start_controls_section(
				'litho_section_pagination_style',
				[
					'label'      => esc_html__( 'Pagination', 'litho-addons' ),
					'tab'        => Controls_Manager::TAB_STYLE,
					'show_label' => false,
					'conditions' => [
						'relation' => 'or',
						'terms'    => [
							[
								'relation' => 'and',
								'terms'    => [
									[
										'name'     => 'litho_pagination',
										'operator' => '===',
										'value'    => 'number-pagination',
									],
								],
							],
							[
								'relation' => 'and',
								'terms'    => [
									[
										'name'     => 'litho_pagination',
										'operator' => '===',
										'value'    => 'next-prev-pagination',
									],
								],
							],
							[
								'relation' => 'and',
								'terms'    => [
									[
										'name'     => 'litho_pagination',
										'operator' => '===',
										'value'    => 'load-more-pagination',
									],
								],
							],
						],
					],
				]
			);
			$this->add_responsive_control(
				'litho_pagination_alignment',
				[
					'label'       => esc_html__( 'Alignment', 'litho-addons' ),
					'type'        => Controls_Manager::CHOOSE,
					'label_block' => false,
					'default'     => 'center',
					'options'     => [
						'flex-start' => [
							'title' => esc_html__( 'Left', 'litho-addons' ),
							'icon'  => 'eicon-text-align-left',
						],
						'center'     => [
							'title' => esc_html__( 'Center', 'litho-addons' ),
							'icon'  => 'eicon-text-align-center',
						],
						'flex-end'   => [
							'title' => esc_html__( 'Right', 'litho-addons' ),
							'icon'  => 'eicon-text-align-right',
						],
					],
					'selectors'   => [
						'{{WRAPPER}} .litho-pagination' => 'display: flex; justify-content: {{VALUE}};',
					],
					'condition'   => [
						'litho_pagination' => 'number-pagination',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'   => 'litho_pagination_typography',
					'global' => [
						'default' => Global_Typography::TYPOGRAPHY_PRIMARY,
					],
					'selector'  => '{{WRAPPER}} .page-numbers li .page-numbers, {{WRAPPER}} .new-post a , {{WRAPPER}} .old-post a',
					'condition' => [
						'litho_pagination' => [
							'number-pagination',
							'next-prev-pagination',
						],
					],
				]
			);
			$this->start_controls_tabs(
				'litho_pagination_tabs',
			);
				$this->start_controls_tab(
					'litho_pagination_normal_tab',
					[
						'label'     => esc_html__( 'Normal', 'litho-addons' ),
						'condition' => [
							'litho_pagination' => [
								'number-pagination',
								'next-prev-pagination',
							],
						],
					]
				);
				$this->add_control(
					'litho_pagination_color',
					[
						'label'     => esc_html__( 'Color', 'litho-addons' ),
						'type'      => Controls_Manager::COLOR,
						'selectors' => [
							'{{WRAPPER}} .page-numbers li .page-numbers , {{WRAPPER}} .new-post a , {{WRAPPER}} .old-post a' => 'color: {{VALUE}};',
						],
						'condition' => [
							'litho_pagination' => [
								'number-pagination',
								'next-prev-pagination',
							],
						],
					]
				);
				$this->end_controls_tab();
				$this->start_controls_tab(
					'litho_pagination_hover_tab',
					[
						'label'     => esc_html__( 'Hover', 'litho-addons' ),
						'condition' => [
							'litho_pagination' => [
								'number-pagination',
								'next-prev-pagination',
							],
						],
					],
				);
				$this->add_control(
					'litho_pagination_hover_color',
					[
						'label'     => esc_html__( 'Color', 'litho-addons' ),
						'type'      => Controls_Manager::COLOR,
						'selectors' => [
							'{{WRAPPER}} .page-numbers li .page-numbers:hover, {{WRAPPER}} .new-post a:hover , {{WRAPPER}} .old-post a:hover' => 'color: {{VALUE}};',
						],
						'condition' => [
							'litho_pagination' => [
								'number-pagination',
								'next-prev-pagination',
							],
						],
					]
				);
				$this->add_responsive_control(
					'litho_pagination_bg_hover_icon_color',
					[
						'label'     => esc_html__( 'Icon Color', 'litho-addons' ),
						'type'      => Controls_Manager::COLOR,
						'selectors' => [
							'{{WRAPPER}} .litho-pagination .page-numbers li a.next:hover, {{WRAPPER}} .litho-pagination .page-numbers li a.prev:hover' => 'color: {{VALUE}};',
						],
						'condition' => [
							'litho_pagination' => [
								'number-pagination',
							],
						],
					]
				);
				$this->add_responsive_control(
					'litho_pagination_bg_hover_color',
					[
						'label'     => esc_html__( 'Background Color', 'litho-addons' ),
						'type'      => Controls_Manager::COLOR,
						'selectors' => [
							'{{WRAPPER}} .page-numbers li .page-numbers:hover, {{WRAPPER}} .new-post a:hover , {{WRAPPER}} .old-post a:hover' => 'background-color: {{VALUE}};',
						],
					]
				);
				$this->end_controls_tab();
				$this->start_controls_tab(
					'litho_pagination_active_tab',
					[
						'label'     => esc_html__( 'Active', 'litho-addons' ),
						'condition' => [
							'litho_pagination' => 'number-pagination',
						],
					],
				);
				$this->add_control(
					'litho_pagination_active_color',
					[
						'label'     => esc_html__( 'Color', 'litho-addons' ),
						'type'      => Controls_Manager::COLOR,
						'selectors' => [
							'{{WRAPPER}} .page-numbers li .page-numbers.current' => 'color: {{VALUE}};',
						],
						'condition' => [
							'litho_pagination' => 'number-pagination',
						],
					]
				);
				$this->add_responsive_control(
					'litho_pagination_bg_active_color',
					[
						'label'     => esc_html__( 'Background Color', 'litho-addons' ),
						'type'      => Controls_Manager::COLOR,
						'selectors' => [
							'{{WRAPPER}} .litho-pagination .page-numbers li .page-numbers.current' => 'background-color: {{VALUE}};',
						],
					]
				);
				$this->end_controls_tab();
			$this->end_controls_tabs();
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'      => 'litho_pagination_border',
					'selector'  => '{{WRAPPER}} .post-pagination, {{WRAPPER}} .litho-pagination',
					'condition' => [
						'litho_pagination' => 'next-prev-pagination',
					],
				]
			);
			$this->add_responsive_control(
				'litho_pagination_icon_size',
				[
					'label'      => esc_html__( 'Icon Size', 'litho-addons' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => [
						'px',
						'%',
					],
					'range'      => [
						'px' => [
							'min' => 1,
							'max' => 100,
						],
					],
					'selectors'  => [
						'{{WRAPPER}} .page-numbers li a i' => 'font-size: {{SIZE}}{{UNIT}};',
					],
					'condition'  => [
						'litho_pagination' => 'number-pagination',
					],
				],
			);
			$this->add_responsive_control(
				'litho_pagination_space',
				[
					'label'      => esc_html__( 'Space Between', 'litho-addons' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => [
						'px',
						'%',
					],
					'range'      => [
						'px' => [
							'min' => 1,
							'max' => 100,
						],
					],
					'selectors'  => [
						'{{WRAPPER}} .page-numbers li' => 'margin-right: {{SIZE}}{{UNIT}};',
					],
					'condition'  => [
						'litho_pagination' => 'number-pagination',
					],
				],
			);
			$this->add_responsive_control(
				'litho_pagination_margin',
				[
					'label'      => esc_html__( 'Top Space', 'litho-addons' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => [
						'px',
						'%',
					],
					'range'      => [
						'px' => [
							'min' => 1,
							'max' => 200,
						],
					],
					'selectors'  => [
						'{{WRAPPER}} .litho-pagination, {{WRAPPER}} .blog-pagination' => 'margin-top: {{SIZE}}{{UNIT}};',
					],
					'condition'  => [
						'litho_pagination' => [
							'number-pagination',
							'next-prev-pagination',
						],
					],
				]
			);

			  // load more button style.
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'   => 'litho_typography',
					'global' => [
						'default' => Global_Typography::TYPOGRAPHY_PRIMARY,
					],
					'selector'  => '{{WRAPPER}} .blog-pagination .view-more-button',
					'condition' => [
						'litho_pagination' => 'load-more-pagination'
					]
				]
			);
			$this->start_controls_tabs( 'litho_tabs_button_style' );
			$this->start_controls_tab(
				'litho_tab_button_normal',
				[
					'label'     => esc_html__( 'Normal', 'litho-addons' ),
					'condition' => [
						'litho_pagination' => 'load-more-pagination'
					]
				]
			);
			$this->add_control(
				'litho_button_text_color',
				[
					'label'     => esc_html__( 'Text Color', 'litho-addons' ),
					'type'      => Controls_Manager::COLOR,
					'default'   => '',
					'selectors' => [
						'{{WRAPPER}} .blog-pagination .view-more-button' => 'fill: {{VALUE}}; color: {{VALUE}};',
					],
					'condition' 	=> [
						'litho_pagination' => 'load-more-pagination'
					]
				]
			);
			$this->add_group_control(
				Group_Control_Background::get_type(),
				[
					'name'  => 'litho_background_color',
					'types' => [
						'classic',
						'gradient',
					],
					'exclude'           => [
						'image',
						'position',
						'attachment',
						'attachment_alert',
						'repeat',
						'size',
					],
					'selector'  => '{{WRAPPER}} .blog-pagination .view-more-button',
					'condition' => [
						'litho_pagination' => 'load-more-pagination'
					]
				]
			);
			$this->end_controls_tab();
			$this->start_controls_tab(
				'litho_tab_button_hover',
				[
					'label'     => esc_html__( 'Hover', 'litho-addons' ),
					'condition' => [
						'litho_pagination' => 'load-more-pagination'
					]
				]
			);
			$this->add_control(
				'litho_hover_color',
				[
					'label'     => esc_html__( 'Text Color', 'litho-addons' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .blog-pagination .view-more-button:hover, {{WRAPPER}} .blog-pagination .view-more-button:focus'         => 'color: {{VALUE}};',
						'{{WRAPPER}} .blog-pagination .view-more-button:hover svg, {{WRAPPER}} .blog-pagination .view-more-button:focus svg' => 'fill: {{VALUE}};',
					],
					'condition' 	=> [
						'litho_pagination' => 'load-more-pagination'
					]
				]
			);
			$this->add_group_control(
				Group_Control_Background::get_type(),
				[
					'name'  => 'litho_button_background_hover_color',
					'types' => [
						'classic',
						'gradient',
					],
					'exclude'           => [
						'image',
						'position',
						'attachment',
						'attachment_alert',
						'repeat',
						'size',
					],
					'selector'  => '{{WRAPPER}} .blog-pagination .view-more-button:hover, {{WRAPPER}} .blog-pagination .view-more-button:focus',
					'condition' => [
						'litho_pagination' => 'load-more-pagination'
					]
				]
			);

			$this->add_control(
				'litho_button_hover_border_color',
				[
					'label'     => esc_html__( 'Border Color', 'litho-addons' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .blog-pagination .view-more-button:hover, {{WRAPPER}} .blog-pagination .view-more-button:focus' => 'border-color: {{VALUE}};',
					],
					'condition' 	=> [
						'litho_border_border!' => '',
						'litho_pagination'     => 'load-more-pagination'
					]
				]
			);
			$this->add_control(
				'litho_load_more_btn_hover_animation',
				[
					'label'     => esc_html__( 'Hover Animation', 'litho-addons' ),
					'type'      => Controls_Manager::HOVER_ANIMATION,
					'condition' => [
						'litho_pagination' => 'load-more-pagination'
					]
				]
			);
			$this->add_control(
				'litho_load_more_button_hover_transition',
				[
					'label' => esc_html__( 'Transition Duration', 'litho-addons' ),
					'type'  => Controls_Manager::SLIDER,
					'range' => [
						'px'        => [
							'max'  => 3,
							'step' => 0.1,
						],
					],
					'render_type' => 'ui',
					'selectors'   => [
						'{{WRAPPER}} .blog-pagination .view-more-button' => 'transition-duration: {{SIZE}}s',
					],
					'condition' 	=> [
						'litho_pagination' => 'load-more-pagination'
					]
				]
			);
			$this->end_controls_tab();
			$this->end_controls_tabs();
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name'      => 'litho_border',
					'selector'  => '{{WRAPPER}} .blog-pagination .view-more-button',
					'condition' => [
						'litho_pagination' => 'load-more-pagination'
					],
					'fields_options' => [
						'border' 	=> [
							'separator' => 'before',
						],
					],
				]
			);
			$this->add_control(
				'litho_button_border_radius',
				[
					'label'      => esc_html__( 'Border Radius', 'litho-addons' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [
						'px',
						'%',
						'custom',
					],
					'selectors'  => [
						'{{WRAPPER}} .blog-pagination .view-more-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
					'condition' 	=> [
						'litho_pagination' => 'load-more-pagination'
					]
				]
			);
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				[
					'name'      => 'litho_button_box_shadow',
					'selector'  => '{{WRAPPER}} .blog-pagination .view-more-button',
					'condition' => [
						'litho_pagination' => 'load-more-pagination'
					]
				]
			);
			$this->add_responsive_control(
				'litho_text_padding',
				[
					'label'      => esc_html__( 'Padding', 'litho-addons' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => [
						'px',
						'em',
						'%',
						'custom',
					],
					'selectors'  => [
						'{{WRAPPER}} .blog-pagination .view-more-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
					'condition' 	=> [
						'litho_pagination' => 'load-more-pagination'
					]
				]
			);
			$this->end_controls_section();
		}

		/**
		 * Render Query Loop widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0
		 * @access protected
		 */
		protected function render() {
			global $litho_blog_unique_id;

			$settings = $this->get_settings_for_display();

			$litho_show_post_title  = $this->get_settings( 'litho_show_post_title' );
			$litho_post_type_source = $this->get_settings( 'litho_post_type_source' );
			/* Column Settings */
			$litho_column_class      = array();
			$litho_column_class[]    = ! empty( $settings['litho_column_settings'] ) ? 'grid-' . $settings['litho_column_settings']['size'] . 'col' : 'grid-3col';
			$litho_column_class[]    = ! empty( $settings['litho_column_settings_laptop'] ) ? 'xl-grid-' . $settings['litho_column_settings_laptop']['size'] . 'col' : '';
			$litho_column_class[]    = ! empty( $settings['litho_column_settings_tablet_extra'] ) ? 'lg-grid-' . $settings['litho_column_settings_tablet_extra']['size'] . 'col' : '';
			$litho_column_class[]    = ! empty( $settings['litho_column_settings_tablet'] ) ? 'md-grid-' . $settings['litho_column_settings_tablet']['size'] . 'col' : '';
			$litho_column_class[]    = ! empty( $settings['litho_column_settings_mobile_extra'] ) ? 'sm-grid-' . $settings['litho_column_settings_mobile_extra']['size'] . 'col' : '';
			$litho_column_class[]    = ! empty( $settings['litho_column_settings_mobile'] ) ? 'xs-grid-' . $settings['litho_column_settings_mobile']['size'] . 'col' : '';
			$litho_column_class      = array_filter( $litho_column_class );
			$litho_column_class_list = implode( ' ', $litho_column_class );
			// END No. of Column.

			$litho_post_date_format           = ( isset( $settings['litho_post_date_format'] ) && $settings['litho_post_date_format'] ) ? $settings['litho_post_date_format'] : '';
			$litho_thumbnail                  = ( isset( $settings['litho_thumbnail'] ) && $settings['litho_thumbnail'] ) ? $settings['litho_thumbnail'] : 'full';
			$litho_post_excerpt_length        = ( isset( $settings['litho_post_excerpt_length'] ) && $settings['litho_post_excerpt_length'] ) ? $settings['litho_post_excerpt_length'] : '';
			$litho_enable_masonry             = ( isset( $settings['litho_enable_masonry'] ) && $settings['litho_enable_masonry'] ) ? $settings['litho_enable_masonry'] : '';
			$litho_show_post_author_text      = $this->get_settings( 'litho_show_post_author_text' );
			$litho_show_post_author           = ( isset( $settings['litho_show_post_author'] ) && $settings['litho_show_post_author'] ) ? $settings['litho_show_post_author'] : '';
			$litho_show_post_date             = ( isset( $settings['litho_show_post_date'] ) && $settings['litho_show_post_date'] ) ? $settings['litho_show_post_date'] : '';
			$litho_show_post_thumbnail        = ( isset( $settings['litho_show_post_thumbnail'] ) && $settings['litho_show_post_thumbnail'] ) ? $settings['litho_show_post_thumbnail'] : '';
			$litho_show_post_excerpt          = ( isset( $settings['litho_show_post_excerpt'] ) && $settings['litho_show_post_excerpt'] ) ? $settings['litho_show_post_excerpt'] : '';
			$litho_show_post_read_more_button = ( isset( $settings['litho_show_post_read_more_button'] ) && $settings['litho_show_post_read_more_button'] ) ? $settings['litho_show_post_read_more_button'] : '';
			$litho_read_more_text             = ( isset( $settings['litho_read_more_text'] ) && $settings['litho_read_more_text'] ) ? $settings['litho_read_more_text'] : '';
			$litho_show_post_thumbnail_icon   = ( isset( $settings['litho_show_post_thumbnail_icon'] ) && $settings['litho_show_post_thumbnail_icon'] ) ? $settings['litho_show_post_thumbnail_icon'] : '';
			$litho_show_post_category         = ( isset( $settings['litho_show_post_category'] ) && $settings['litho_show_post_category'] ) ? $settings['litho_show_post_category'] : '';
			$litho_post_icon                  = ( isset( $settings['litho_post_icon'] ) && $settings['litho_post_icon'] ) ? $settings['litho_post_icon'] : '';
			$litho_show_post_like             = ( isset( $settings['litho_show_post_like'] ) && $settings['litho_show_post_like'] ) ? $settings['litho_show_post_like'] : '';
			$litho_show_post_comments         = ( isset( $settings['litho_show_post_comments'] ) && $settings['litho_show_post_comments'] ) ? $settings['litho_show_post_comments'] : '';
			$litho_show_post_author_image     = ( isset( $settings['litho_show_post_author_image'] ) && $settings['litho_show_post_author_image'] ) ? $settings['litho_show_post_author_image'] : '';
			$litho_separator                  = ( isset( $settings['litho_separator'] ) && $settings['litho_separator'] ) ? $settings['litho_separator'] : '';

			// Pagination.
			$litho_pagination = ( isset( $settings['litho_pagination'] ) && $settings['litho_pagination'] ) ? $settings['litho_pagination'] : '';

			// Check if blog id and class.
			$litho_blog_unique_id = ! empty( $litho_blog_unique_id ) ? $litho_blog_unique_id : 1;
			$litho_blog_id        = 'litho-blog-loop';
			$litho_blog_id       .= '-' . $litho_blog_unique_id;
			++$litho_blog_unique_id;

			$datasettings = array(
				'pagination_type' => $litho_pagination,
			);

			$this->add_render_attribute(
				'wrapper',
				[
					'data-uniqueid'      => $litho_blog_id,
					'class'              => [
						$litho_blog_id,
						'grid',
						'litho-blog-list',
						'blog-classic',
						'blog-grid',
						$litho_column_class_list,
						$litho_pagination,
					],
					'data-blog-settings' => wp_json_encode( $datasettings ),
				]
			);

			if ( 'yes' === $litho_enable_masonry ) {
				$this->add_render_attribute(
					'wrapper',
					'class',
					'grid-masonry'
				);
			} else {
				$this->add_render_attribute(
					'wrapper',
					'class',
					'no-masonry'
				);
			}

			$query_args = $this->litho_get_query_args();
			$the_query  = new \WP_Query( $query_args );

			if ( $the_query->have_posts() ) {
				?>
				<ul <?php $this->print_render_attribute_string( 'wrapper' ); ?>>
					<li class="grid-sizer d-none p-0 m-0"></li>
					<?php
					$index = 0;
					while ( $the_query->have_posts() ) {
						$the_query->the_post();

						$cat_slug_cls       = [];
						$inner_wrapper_key  = 'inner_wrapper_' . $index;
						$blog_post_key      = 'blog_post_' . $index;
						$button_key         = 'button_' . $index;
						$button_link_key    = 'button_link_' . $index;
						$button_content_key = 'button_content_wrapper_' . $index;
						$button_text_key    = 'button_text_' . $index;

						$this->add_render_attribute( $button_key, 'class', [ 'elementor-gradient-button-wrapper' ] );

						$this->add_render_attribute(
							$button_link_key,
							[
								'href'  => get_permalink(),
								'class' => [
									'elementor-gradient-button-link',
									'elementor-gradient-button',
									'blog-button',
								],
								'role'  => 'button'
							]
						);
						$this->add_render_attribute( $button_content_key, 'class', [ 'elementor-gradient-button-content-wrapper' ] );

						$this->add_render_attribute( $button_text_key, 'class', [ 'elementor-gradient-button-text' ] );

						$cat_slug_class_list = implode( ' ', $cat_slug_cls );
						$post_class_list     = get_post_class( array( 'grid-item', 'grid-gutter', $cat_slug_class_list ) );

						$this->add_render_attribute(
							$inner_wrapper_key,
							[
								'class' => $post_class_list,
							]
						);

						$this->add_render_attribute(
							$blog_post_key,
							[
								'class' => [
									'blog-post',
								],
							]
						);

						$thumbnail_id = get_post_thumbnail_id( get_the_ID() );
						$image_key    = 'image_' . $index;
						$img_alt      = '';
						$img_alt      = get_post_meta( $thumbnail_id, '_wp_attachment_image_alt', true );

						if ( empty( $img_alt ) ) {
							$img_alt = esc_attr( get_the_title( $thumbnail_id ) );
						}
						$this->add_render_attribute( $image_key, 'class', 'image-link' );
						$this->add_render_attribute( $image_key, 'aria-label', $img_alt );
						?>
						<li <?php $this->print_render_attribute_string( $inner_wrapper_key ); ?>>
							<div class="blog-post">
								<?php
								if ( ! post_password_required() && 'yes' === $litho_show_post_thumbnail && has_post_thumbnail() ) {
									?>
									<div class="blog-post-images">
										<?php
										if ( has_post_thumbnail() ) {
											?>
											<a href="<?php the_permalink(); ?>" <?php $this->print_render_attribute_string( $image_key ); ?>>
												<?php
												$thumbnail_id = get_post_thumbnail_id( get_the_ID() );
												$alt          = get_post_meta( $thumbnail_id, '_wp_attachment_image_alt', true );
												if ( '' === $alt ) {
													$alt = get_the_title();
												}
												echo get_the_post_thumbnail( get_the_ID(), $litho_thumbnail, array( 'alt' => $alt ) );
												if ( 'yes' === $litho_show_post_thumbnail_icon ) {
													$post_format           = get_post_format( get_the_ID() );
													$blog_lightbox_gallery = get_post_meta( get_the_ID(), '_litho_lightbox_image_single', true );
													$video_type_single     = get_post_meta( get_the_ID(), '_litho_video_type_single', true );

													if ( 'gallery' === $post_format && '1' === $blog_lightbox_gallery ) {

														?><span class="post-icon post-type-<?php echo esc_attr( $post_format ); ?>"></span><?php

														} elseif ( 'gallery' === $post_format ) {

														?><span class="post-icon post-type-<?php echo esc_attr( $post_format ); ?>-slider"></span><?php

														} elseif ( 'video' === $post_format && 'self' === $video_type_single ) {

														?><span class="post-icon post-type-<?php echo esc_attr( $post_format ); ?>-html5"></span><?php

														} elseif ( 'video' === $post_format ) {

														?><span class="post-icon post-type-<?php echo esc_attr( $post_format ); ?>"></span><?php

														} elseif ( 'audio' === $post_format ) {

														?><span class="post-icon post-type-<?php echo esc_attr( $post_format ); ?>"></span><?php

														} elseif ( 'quote' === $post_format ) {

														?><span class="post-icon post-type-<?php echo esc_attr( $post_format ); ?>"></span><?php
													}
												}
												?>
											</a>
											<?php
										}
										?>
									</div>
									<?php
								}
								?>
								<div class="post-details">                                  
									<?php
									if ( 'yes' === $litho_show_post_author || 'yes' === $litho_show_post_category || 'yes' === $litho_show_post_date ) {
										?>
										<div class="post-meta-wrapper d-flex align-items-center post-meta">
											<span class="post-author-meta">
												<?php
												if ( 'yes' === $litho_show_post_author && get_the_author() ) {
													if ( 'yes' === $litho_show_post_author_image ) {
														echo get_avatar( get_the_author_meta( 'ID' ), 30 );
													}
													?>
													<span class="author-name">
														<?php echo esc_html( $litho_show_post_author_text ); ?>
														<a href="<?php echo esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ); ?>">
															<?php echo esc_html( get_the_author() ); ?>
														</a>
													</span>
													<?php
												}
												if ( 'yes' === $litho_show_post_category ) {
													?>
													<span class="blog-category alt-font">
														<?php
														$terms = ! empty( $litho_post_terms ) ? $litho_post_terms : get_the_terms( get_the_ID(), 'category' );
														if ( ! empty( $terms ) && ! is_wp_error( $terms ) ) {
															$first_term = array_shift( $terms );
															?>
															<a href="<?php echo esc_url( get_term_link( $first_term ) ); ?>">
																<?php echo esc_html( $first_term->name ); ?>
															</a>
															<?php
														}
														?>
													</span>
													<?php
												}
												if ( 'yes' === $litho_show_post_date ) {
													if ( 'yes' === $litho_post_icon ) {
														?>
														<i class="feather icon-feather-calendar"></i>
														<?php
													}
													?>
													<span class="post-date published"><?php echo esc_html( get_the_date( $litho_post_date_format, get_the_ID() ) ); ?></span>
													<?php
												}
												?>
											</span>
										</div>
										<?php
									}
									if ( 'yes' === $litho_show_post_title ) {
										?>
										<a href="<?php the_permalink(); ?>" class="entry-title"><?php the_title(); ?></a>
										<?php
									}
									if ( 'yes' === $litho_show_post_excerpt ) {
										$show_excerpt_grid = ! empty( $litho_post_excerpt_length ) ? litho_get_the_excerpt_theme( $litho_post_excerpt_length ) : litho_get_the_excerpt_theme( 15 );
										if ( ! empty( $show_excerpt_grid ) ) {
											?>
											<p class="entry-content"><?php echo wp_kses_post( $show_excerpt_grid ); ?></p>
											<?php
										}
									}
									if ( 'yes' === $litho_show_post_read_more_button || 'yes' === $litho_show_post_like || 'yes' === $litho_show_post_comments && ( comments_open() || get_comments_number() ) ) {
										?>
										<div class="blog-post-meta-wrapper d-flex align-items-center ">
											<?php
											if ( 'yes' === $litho_separator && 'yes' === $litho_show_post_read_more_button ) {
												?>
												<div class="blog-post-button-wrapper ">
													<?php
													if ( 'yes' === $litho_separator ) {
														?>
														<div class="horizontal-separator"></div>
														<?php
													}
													if ( 'yes' === $litho_show_post_read_more_button ) {
														?>
														<div <?php $this->print_render_attribute_string( $button_key ); // phpcs:ignore ?>>
															<a <?php $this->print_render_attribute_string( $button_link_key ); // phpcs:ignore ?>>
																<span <?php $this->print_render_attribute_string( $button_content_key ); // phpcs:ignore ?>>
																	<span <?php $this->print_render_attribute_string( $button_text_key ); // phpcs:ignore ?>>
																		<?php echo esc_html( $litho_read_more_text ); ?>
																	</span>
																</span>
															</a>
														</div>
														<?php
													}
													?>
												</div>
												<?php
											}
											if ( 'yes' === $litho_show_post_like || 'yes' === $litho_show_post_comments && ( comments_open() || get_comments_number() ) ) {
												?>
												<div class="post-meta">
													<?php
													if ( 'yes' === $litho_show_post_like && function_exists( 'litho_get_simple_likes_button' ) ) {
														?>
														<span class="post-meta-like">
															<?php echo litho_get_simple_likes_button( get_the_ID() ); // phpcs:ignore ?>
														</span>
														<?php
													}
													if ( 'yes' === $litho_show_post_comments && ( comments_open() || get_comments_number() ) ) {
														?>
														<span class="post-meta-comments">
															<?php echo comments_popup_link( '<i class="fa-regular fa-comment"></i><span class="comment-count">0</span> <span class="comment-text">'.esc_html__( 'Comment', 'litho-addons' ).'</span>', '<i class="fa-regular fa-comment"></i><span class="comment-count">1</span> <span class="comment-text">'.esc_html__( 'Comment', 'litho-addons' ).'</span>', '<i class="fa-regular fa-comment"></i><span class="comment-count">%</span> <span class="comment-text">'.esc_html__( 'Comments', 'litho-addons' ).'</span>', 'comment-link' );
															?>
														</span>
														<?php
													}
													?>
												</div>
												<?php
											}
											?>
										</div>
										<?php
									}
									?>
								</div>                                  
							</div>
						</li>
						<?php
						++$index;
					}
					wp_reset_postdata();
					?>
				</ul>
				<?php
				get_next_posts_page_link( $the_query->max_num_pages );
				$this->litho_post_pagination( $the_query, $settings );
			}
		}

		public function litho_post_pagination( $wpQuery ) {
			$litho_pagination_prev_icon_attr         = '';
			$litho_pagination_next_icon_attr         = '';
			$settings                                = $this->get_settings_for_display();
			$litho_pagination                        = ( isset( $settings['litho_pagination'] ) && $settings['litho_pagination'] ) ? $settings['litho_pagination'] : '';
			$litho_pagination_next_label             = ( isset( $settings['litho_pagination_next_label'] ) && $settings['litho_pagination_next_label'] ) ? $settings['litho_pagination_next_label'] : '';
			$litho_pagination_prev_label             = ( isset( $settings['litho_pagination_prev_label'] ) && $settings['litho_pagination_prev_label'] ) ? $settings['litho_pagination_prev_label'] : '';
			$litho_pagination_load_more_button_label = ( isset( $settings['litho_pagination_load_more_button_label'] ) && $settings['litho_pagination_load_more_button_label'] ) ? $settings['litho_pagination_load_more_button_label'] : esc_html__( 'Load more', 'litho-addons' );

			if ( $litho_pagination ) {
				ob_start();
					Icons_Manager::render_icon( $settings['litho_pagination_prev_icon'], [
						'aria-hidden' => 'true',
					] );
					$litho_pagination_prev_icon_attr .= ob_get_contents();
				ob_end_clean();
				$litho_pagination_prev_icon_attr .= $litho_pagination_prev_label;

				$litho_pagination_next_icon_attr .= $litho_pagination_next_label;
				ob_start();
					Icons_Manager::render_icon( $settings['litho_pagination_next_icon'], [
						'aria-hidden' => 'true',
					] );
					$litho_pagination_next_icon_attr .= ob_get_contents();
				ob_end_clean();
			}

			switch ( $litho_pagination ) {
				case 'number-pagination':
					$current = ( $wpQuery->query_vars['paged'] > 1 ) ? $wpQuery->query_vars['paged'] : 1;
					if ( $wpQuery->max_num_pages > 1 ) {
						?>
						<div class="col-12 litho-pagination">
							<div class="pagination align-items-center">
								<?php
								// phpcs:ignore
								echo paginate_links( array(
									'base'      => esc_url_raw( str_replace( 999999999, '%#%', get_pagenum_link( 999999999, false ) ) ),
									'format'    => '',
									'add_args'  => '',
									'current'   => $current,
									'total'     => $wpQuery->max_num_pages,
									'prev_text' => $litho_pagination_prev_icon_attr . '<span class="screen-reader-text">' . esc_html__( 'Prev',  'litho-addons' ) . '</span>',
									'next_text' => $litho_pagination_next_icon_attr . '<span class="screen-reader-text">' . esc_html__( 'Next',  'litho-addons' ) . '</span>',
									'type'      => 'list',
									'end_size'  => 2,
									'mid_size'  => 2
								) );
								?>
							</div>
						</div>
						<?php
					}
					break;
				case 'next-prev-pagination':
					if ( $wpQuery->max_num_pages > 1 ) {
						?>
						<div class="blog-pagination col-12">
							<div class="new-post">
								<?php
								if ( is_elementor_activated() && \Elementor\Plugin::$instance->editor->is_edit_mode() ) {
									?>
									<a href="#">
										<i aria-hidden="true" class="feather icon-feather-arrow-left"></i>
										<?php echo esc_html__( 'PREV', 'litho-addons' ); ?>
									</a>
									<?php
								} else {
									echo get_previous_posts_link( $litho_pagination_prev_icon_attr ); // phpcs:ignore
								}
								?>
							</div>
							<div class="old-post">
								<?php
								if ( is_elementor_activated() && \Elementor\Plugin::$instance->editor->is_edit_mode() ) {
									?>
									<a href="#">
										<?php echo esc_html__( 'NEXT', 'litho-addons' ); ?>
										<i aria-hidden="true" class="feather icon-feather-arrow-right"></i>
									</a>
									<?php
								} else {
									echo get_next_posts_link( $litho_pagination_next_icon_attr, $wpQuery->max_num_pages ); // phpcs:ignore
								}
								?>
							</div>
						</div>
						<?php
					}
					break;
				case 'infinite-scroll-pagination':
					?>
					<div class="blog-pagination blog-infinite-scroll-pagination col-12 post-infinite-scroll" data-pagination="<?php echo esc_attr( $wpQuery->max_num_pages ); ?>">
						<div class="page-load-status text-center">
							<p class="infinite-scroll-request">
								<?php
								/**
								 * Apply filters to modify loading GIF
								 */
								$litho_pagination_loading_gif = apply_filters( 'litho_pagination_loading_gif', esc_url( LITHO_THEME_IMAGES_URI . '/loading-infinite-scroll.svg' ) );
								?>
								<img src="<?php echo $litho_pagination_loading_gif; // phpcs:ignore ?>" alt="<?php echo esc_attr__( 'loading', 'litho-addons' ); ?>">
							</p>
						</div>
						<div class="old-post d-none">
							<?php
							if ( get_next_posts_link( '', $wpQuery->max_num_pages ) ) {
								next_posts_link( esc_html__( 'Next', 'litho-addons' ).'<i aria-hidden="true" class="fa-solid fa-angle-right"></i>', $wpQuery->max_num_pages );
							}
							?>
						</div>
					</div>
					<?php
					break;
				case 'load-more-pagination':
					$this->add_render_attribute( 'pagination_main', [
						'class'           => [ 'blog-pagination blog-infinite-scroll-pagination', 'litho-blog-load-more' ],
						'data-pagination' => $wpQuery->max_num_pages
					] );
					?>
					<div <?php $this->print_render_attribute_string( 'pagination_main' ); // phpcs:ignore ?>>
						<div class="old-post d-none">
							<?php
							if ( get_next_posts_link( '', $wpQuery->max_num_pages ) ) {
								next_posts_link( esc_html__( 'Next', 'litho-addons' ).'<i aria-hidden="true" class="fa-solid fa-angle-right"></i>', $wpQuery->max_num_pages );
							}
							?>
						</div>
						<div class="load-more-btn text-center">
							<button class="btn view-more-button">
								<?php echo esc_html( $litho_pagination_load_more_button_label ); ?>
							</button>
						</div>
					</div>
					<?php
					break;
			}
		}

		/**
		 * Returns the Litho category list.
		 *
		 * @param int    $id         The ID to retrieve categories for.
		 * @param bool   $separator  Whether to include a separator in the list. Default is true.
		 * @param string $count      The number of categories to retrieve. Default is '10'.
		 *
		 * @return array The list of categories.
		 */
		public function litho_get_category_list( $id, $separator = true, $count = '10' ) {
			$term_cat                = [];
			$litho_post_type_source = $this->get_settings( 'litho_post_type_source' );
			if ( 'portfolio' === $litho_post_type_source ) {
				$terms = get_the_terms( $id, 'portfolio-category' );
			} elseif ( 'product' === $litho_post_type_source ) {
				$terms = get_the_terms( $id, 'product_cat' );
			}
			foreach ( array_slice( $terms, 0, intval( $count ) ) as $term ) {
				$term_link  = get_term_link( $term->term_id );
				$term_cat[] = '<a rel="category tag" href="' . esc_url( $term_link ) . '">' . esc_html( $term->name ) . '</a>';
			}

			return implode( $separator ? ', ' : '', $term_cat );
		}

		/**
		 * Returns the query arguments.
		 *
		 * @return array The query arguments array.
		 */
		public function litho_get_query_args() {

			$settings = $this->get_settings_for_display();

			$litho_post_type_source    = $this->get_settings( 'litho_post_type_source' );
			$litho_posts_exclude_ids   = $this->get_settings( 'litho_posts_exclude_ids' );
			$litho_ignore_sticky_posts = $this->get_settings( 'litho_ignore_sticky_posts' );
			$litho_items_per_page      = $this->get_settings( 'litho_items_per_page' );
			$litho_query_loop_offset   = $this->get_settings( 'litho_query_loop_offset' );

			if ( get_query_var( 'paged' ) ) {
				$paged = get_query_var( 'paged' );
			} elseif ( get_query_var( 'page' ) ) {
				$paged = get_query_var( 'page' );
			} else {
				$paged = 1;
			}

			$args = $this->set_meta_query_args();

			$args['post_status']      = 'publish';
			$args['suppress_filters'] = false;
			$exclude_by               = $this->get_group_control_query_param_by( 'exclude' );

			if ( 0 < $litho_query_loop_offset ) {
				$args['offset_to_fix'] = $litho_query_loop_offset;
			}

			$args['posts_per_page'] = intval( $litho_items_per_page ); // phpcs:ignore
			$args['paged']          = $paged;

			if ( 'post' === $litho_post_type_source && 'yes' === $litho_ignore_sticky_posts ) {
				$args['ignore_sticky_posts'] = true;
				if ( in_array( 'current_post', $exclude_by, true ) ) {
					$args['post__not_in'] = [ get_the_ID() ];
				}
			}

			if ( 'manual_selection' === $litho_post_type_source ) {
				$selected_ids      = $this->get_settings_for_display( 'litho_posts_selected_ids' );
				$selected_ids      = wp_parse_id_list( $selected_ids );
				$args['post_type'] = 'any';
				if ( ! empty( $selected_ids ) ) {
					$args['post__in'] = $selected_ids;
				}

				$args['ignore_sticky_posts'] = true;
			} elseif ( 'current_query' === $litho_post_type_source ) {
				$args = $GLOBALS['wp_query']->query_vars;
				$args = apply_filters( 'litho_query_loop_current_query', $args );
			} elseif ( 'related_post_type' === $litho_post_type_source ) {
				$post_id           = get_queried_object_id();
				$related_post_id   = is_singular() && ( 0 !== $post_id ) ? $post_id : null;
				$args['post_type'] = get_post_type( $related_post_id );
				$exclude_by        = $this->get_group_control_query_param_by( 'exclude' );

				if ( in_array( 'current_post', $exclude_by, true ) ) {
					$args['post__not_in'] = [ get_the_ID() ];
				}

				$args = $this->get_author_args( $args, $settings, $related_post_id );
				$args = $this->get_terms_args( $args, $settings );

				$args['ignore_sticky_posts'] = true;

				$args = apply_filters( 'litho_query_loop_related_query', $args );
			} else {
				$args['post_type'] = $litho_post_type_source;
				$exclude_by        = $this->get_group_control_query_param_by( 'exclude' );

				$current_post = [];
				if ( in_array( 'current_post', $exclude_by, true ) && is_singular() ) {
					$current_post = [ get_the_ID() ];
				}

				if ( in_array( 'manual_selection', $exclude_by, true ) ) {
					$exclude_ids          = $litho_posts_exclude_ids;
					$args['post__not_in'] = array_merge( $current_post, wp_parse_id_list( $exclude_ids ) );
				}

				$args = $this->get_author_args( $args, $settings );
				$args = $this->get_terms_args( $args, $settings );
			}

			if ( $this->get_settings_for_display( 'query_id' ) ) {
				add_action( 'pre_get_posts', [ $this, 'pre_get_posts_query_filter' ] );
			}

			// fixing custom offset.
			// https://codex.wordpress.org/Making_Custom_Queries_using_Offset_and_Pagination.
			add_action( 'pre_get_posts', [ $this, 'fix_query_offset' ], 1 );
			add_filter( 'found_posts', [ $this, 'prefix_adjust_offset_pagination' ], 1, 2 );

			return $args;
		}

		/**
		 * Returns the meta query arguments.
		 *
		 * @return array The meta query arguments array.
		 */
		private function set_meta_query_args() {
			$litho_post_type_source = $this->get_settings( 'litho_post_type_source' );
			$args                   = [];

			if ( 'current_query' === $litho_post_type_source ) {
				return [];
			}

			$args['order']   = $this->get_settings_for_display( 'litho_order' );
			$args['orderby'] = $this->get_settings_for_display( 'litho_orderby' );

			return $args;
		}

		/**
		 * Gets query parameters based on the provided arguments.
		 *
		 * @param string $by The parameter to filter by.
		 *
		 * @return array|mixed The query parameters or a specific value based on the input.
		 */
		private function get_group_control_query_param_by( $by = 'exclude' ) {
			$map_by = [
				'exclude' => 'litho_exclude_by',
				'include' => 'litho_include_by',
			];

			$setting = $this->get_settings_for_display( $map_by[ $by ] );

			return ( ! empty( $setting ) ? $setting : [] );
		}
		/**
		 * Returns an array of author arguments.
		 *
		 * @param array        $args The arguments array.
		 * @param array        $settings The settings array.
		 * @param WP_Post|null $post The post object or null if not provided. Default is null.
		 *
		 * @return array The modified author arguments array.
		 */
		private function get_author_args( $args, $settings, $post = null ) {
			$include_by    = $this->get_group_control_query_param_by( 'include' );
			$exclude_by    = $this->get_group_control_query_param_by( 'exclude' );
			$include_users = [];
			$exclude_users = [];

			if ( in_array( 'authors', $include_by, true ) ) {
				$include_users = wp_parse_id_list( $settings['litho_posts_include_author_ids'] );
			} elseif ( $post ) {
				$include_users = get_post_field( 'post_author', $post );
			}

			if ( in_array( 'authors', $exclude_by, true ) ) {
				$exclude_users = wp_parse_id_list( $settings['litho_posts_exclude_author_ids'] );
				$include_users = array_diff( $include_users, $exclude_users );
			}

			if ( ! empty( $include_users ) ) {
				$args['author__in'] = $include_users;
			}

			if ( ! empty( $exclude_users ) ) {
				$args['author__not_in'] = $exclude_users;
			}

			return $args;
		}
		/**
		 * Returns an array of terms arguments.
		 *
		 * @param array $args The arguments array.
		 * @param array $settings The settings array.
		 *
		 * @return array The modified terms arguments array.
		 */
		private function get_terms_args( $args, $settings ) {
			$include_by    = $this->get_group_control_query_param_by( 'include' );
			$exclude_by    = $this->get_group_control_query_param_by( 'exclude' );
			$include_terms = [];
			$terms_query   = [];

			if ( in_array( 'terms', $include_by, true ) ) {
				$include_terms = wp_parse_id_list( $settings['litho_posts_include_term_ids'] );
			}

			if ( in_array( 'terms', $exclude_by, true ) ) {
				$exclude_terms = wp_parse_id_list( $settings['litho_posts_exclude_term_ids'] );
				$include_terms = array_diff( $include_terms, $exclude_terms );
			}

			if ( ! empty( $include_terms ) ) {
				$tax_terms_map = $this->map_group_control_query( $include_terms );

				foreach ( $tax_terms_map as $tax => $terms ) {
					$terms_query[] = [
						'taxonomy' => $tax,
						'field'    => 'term_id',
						'terms'    => $terms,
						'operator' => 'IN',
					];
				}
			}

			if ( ! empty( $exclude_terms ) ) {
				$tax_terms_map = $this->map_group_control_query( $exclude_terms );

				foreach ( $tax_terms_map as $tax => $terms ) {
					$terms_query[] = [
						'taxonomy' => $tax,
						'field'    => 'term_id',
						'terms'    => $terms,
						'operator' => 'NOT IN',
					];
				}
			}

			if ( ! empty( $terms_query ) ) {
				$args['tax_query']             = $terms_query; // phpcs:ignore
				$args['tax_query']['relation'] = 'AND';
			}

			return $args;
		}
		/**
		 * Returns the map group control.
		 *
		 * Initializes the $term_ids array to store term IDs.
		 *
		 * @param array $term_ids Array of term ids.
		 */
		private function map_group_control_query( $term_ids = [] ) {

			$terms = get_terms(
				[
					'term_taxonomy_id' => $term_ids,
					'hide_empty'       => false,
				]
			);

			$tax_terms_map = [];

			if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
				foreach ( $terms as $term ) {
					$taxonomy                     = $term->taxonomy;
					$tax_terms_map[ $taxonomy ][] = $term->term_id;
				}
			}

			return $tax_terms_map;
		}

		/**
		 * @param WP_Query $query fix the offset.
		 */
		public function fix_query_offset( &$query ) {
			if ( isset( $query->query_vars['offset_to_fix'] ) ) {
				if ( $query->is_paged ) {
					$page_offset = $query->query_vars['offset_to_fix'] + ( ( $query->query_vars['paged'] - 1 ) * $query->query_vars['posts_per_page'] );
					$query->set( 'offset', $page_offset );
				} else {
					$query->set( 'offset', $query->query_vars['offset_to_fix'] );
				}
			}
		}

		/**
		 * Adjusted number of found posts after applying the offset
		 *
		 * @param mixed $found_posts The original total number of found posts.
		 *
		 * @param mixed $query Query object.
		 */
		public function prefix_adjust_offset_pagination( $found_posts, $query ) {
			if ( isset( $query->query_vars['offset_to_fix'] ) ) {
				$offset_to_fix = intval( $query->query_vars['offset_to_fix'] );

				if ( $offset_to_fix ) {
					$found_posts -= $offset_to_fix;
				}
			}

			return $found_posts;
		}

		/**
		 * @param WP_Query $wp_query fix the offset.
		 */
		public function pre_get_posts_query_filter( $wp_query ) {
			if ( $this ) {
				$query_id = $this->get_settings_for_display( 'query_id' );
				do_action( "litho_query_loop/{$query_id}", $wp_query, $this );
			}
		}
	}
}
