<?php
namespace LithoAddons\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Litho widget for lottie.
 *
 * @package Litho
 */

use Elementor\Controls_Manager;
use Elementor\Group_Control_Css_Filter;
use Elementor\Group_Control_Typography;

// If class `Lottie` doesn't exists yet.
if ( ! class_exists( 'LithoAddons\Widgets\Lottie' ) ) {
	/**
	 * Define `Lottie` class.
	 */
	class Lottie extends \Elementor\Widget_Base {
		
		/**
		 * Retrieve the list of scripts the lottie widget depended on.
		 *
		 * Used to set scripts dependencies required to run the widget.
		 *
		 * @access public
		 *
		 * @return array Widget scripts dependencies.
		 */
		public function get_script_depends() {
			$lottie_scripts = [
				'lottie',
				'lottie-animation',
			];

			return $lottie_scripts;
		}

		/**
		 * Retrieve the widget name.
		 *
		 * @access public
		 *
		 * @return string Widget name.
		 */
		public function get_name() {
			return 'litho-lottie';
		}

		/**
		 * Retrieve the widget title
		 *
		 * @access public
		 *
		 * @return string Widget title.
		 */
		public function get_title() {
			return esc_html__( 'Litho Lottie Animation', 'litho-addons' );
		}

		/**
		 * Retrieve the widget icon.
		 *
		 * @access public
		 *
		 * @return string Widget icon.
		 */
		public function get_icon() {
			return 'eicon-lottie';
		}

		/**
		 * Retrieve the widget categories.
		 *
		 * @access public
		 *
		 * @return array Widget categories.
		 */
		public function get_categories() {
			return [
				'litho',
			];
		}

		/**
		 * Get widget keywords.
		 *
		 * Retrieve the list of keywords the widget belongs to.
		 *
		 * @access public
		 *
		 * @return array Widget keywords.
		 */
		public function get_keywords() {
			return [
				'lottie',
				'JSON',
				'vector',
				'motion graphics',
				'creative animations',
				'looping animation',
				'web animation',
				'json animation',
			];
		}

		/**
		 * Register lottie widget controls.
		 *
		 * Adds different input fields to allow the user to change and customize the widget settings.
		 *
		 * @access protected
		 */
		protected function register_controls() {
			$this->start_controls_section(
				'litho_content_section',
				[
					'label' => esc_html__( 'Lottie Animation', 'litho-addons' ),
					'tab'   => Controls_Manager::TAB_CONTENT,
				]
			);
			$this->add_control(
				'litho_lottie_source',
				[
					'label'   => esc_html__( 'Source', 'litho-addons' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'media_file',
					'options' => [
						'media_file'   => esc_html__( 'Media File', 'litho-addons' ),
						'external_url' => esc_html__( 'External URL', 'litho-addons' ),
					],
				]
			);
			$this->add_control(
				'litho_lottie_media',
				[
					'label'       => esc_html__( 'Upload JSON File', 'litho-addons' ),
					'type'        => Controls_Manager::MEDIA,
					'dynamic'     => [
						'active' => true,
					],
					'media_types' => [ 'application/json' ],
					'condition'   => [
						'litho_lottie_source' => 'media_file',
					],
				]
			);
			$this->add_control(
				'litho_animation_url',
				[
					'label'       => esc_html__( 'External URL', 'litho-addons' ),
					'type'        => Controls_Manager::URL,
					'dynamic'     => [
						'active' => true,
					],
					'placeholder' => esc_html__( 'https://your-link.com', 'litho-addons' ),
					'condition'   => [
						'litho_lottie_source' => 'external_url',
					],
				]
			);
			$this->add_control(
				'litho_caption_source',
				[
					'label'     => esc_html__( 'Caption', 'litho-addons' ),
					'type'      => Controls_Manager::SELECT,
					'default'   => 'none',
					'options'   => [
						'none'    => esc_html__( 'None', 'litho-addons' ),
						'title'   => esc_html__( 'Title', 'litho-addons' ),
						'caption' => esc_html__( 'Caption', 'litho-addons' ),
						'custom'  => esc_html__( 'Custom', 'litho-addons' ),
					],
					'condition' => [
						'litho_lottie_source!'     => 'external_url',
						'litho_lottie_media[url]!' => '',
					],
				]
			);
			$this->add_control(
				'litho_caption',
				[
					'label'       => esc_html__( 'Custom Caption', 'litho-addons' ),
					'type'        => Controls_Manager::TEXT,
					'render_type' => 'none',
					'conditions'  => [
						'relation' => 'or',
						'terms'    => [
							[
								'name'  => 'litho_caption_source',
								'value' => 'custom',
							],
							[
								'name'  => 'litho_lottie_source',
								'value' => 'external_url',
							],
						],
					],
					'dynamic'     => [
						'active' => true,
					],
				]
			);
			$this->add_control(
				'litho_link_to',
				[
					'label'       => esc_html__( 'Link', 'litho-addons' ),
					'type'        => Controls_Manager::SELECT,
					'render_type' => 'none',
					'default'     => 'none',
					'options'     => [
						'none'   => esc_html__( 'None', 'litho-addons' ),
						'custom' => esc_html__( 'Custom URL', 'litho-addons' ),
					],
				]
			);
			$this->add_control(
				'litho_custom_link',
				[
					'label'       => esc_html__( 'Link', 'litho-addons' ),
					'type'        => Controls_Manager::URL,
					'render_type' => 'none',
					'placeholder' => esc_html__( 'Enter your URL', 'litho-addons' ),
					'condition'   => [
						'litho_link_to' => 'custom',
					],
					'dynamic'     => [
						'active' => true,
					],
					'default'     => [
						'url' => '',
					],
					'show_label'  => false,
				]
			);
			$this->end_controls_section();

			$this->start_controls_section(
				'settings',
				[
					'label' => esc_html__( 'Settings', 'litho-addons' ),
				]
			);
			$this->add_control(
				'litho_trigger',
				[
					'label'   => esc_html__( 'Trigger', 'litho-addons' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'arriving_to_viewport',
					'options' => [
						'arriving_to_viewport' => esc_html__( 'Viewport', 'litho-addons' ),
						'on_click'             => esc_html__( 'On Click', 'litho-addons' ),
						'on_hover'             => esc_html__( 'On Hover', 'litho-addons' ),
						'bind_to_scroll'       => esc_html__( 'Scroll', 'litho-addons' ),
						'none'                 => esc_html__( 'None', 'litho-addons' ),
					],
				]
			);
			$this->add_control(
				'litho_renderer',
				[
					'label'     => esc_html__( 'Render Type', 'litho-addons' ),
					'type'      => Controls_Manager::SELECT,
					'default'   => 'svg',
					'options'   => [
						'svg'    => esc_html__( 'SVG', 'litho-addons' ),
						'canvas' => esc_html__( 'Canvas', 'litho-addons' ),
					],
					'condition' => [
						'litho_trigger!' => 'none',
					],
				]
			);
			$this->add_control(
				'litho_effects_relative_to',
				[
					'label'       => esc_html__( 'Effects Relative To', 'litho-addons' ),
					'type'        => Controls_Manager::SELECT,
					'render_type' => 'none',
					'condition'   => [
						'litho_trigger' => 'bind_to_scroll',
					],
					'default'     => 'viewport',
					'options'     => [
						'viewport' => esc_html__( 'Viewport', 'litho-addons' ),
						'page'     => esc_html__( 'Entire Page', 'litho-addons' ),
					],
				]
			);
			$this->add_control(
				'litho_on_hover_out',
				[
					'label'       => esc_html__( 'On Hover Out', 'litho-addons' ),
					'type'        => Controls_Manager::SELECT,
					'render_type' => 'none',
					'condition'   => [
						'litho_trigger' => 'on_hover',
					],
					'default'     => 'default',
					'options'     => [
						'default' => esc_html__( 'Default', 'litho-addons' ),
						'reverse' => esc_html__( 'Reverse', 'litho-addons' ),
						'pause'   => esc_html__( 'Pause', 'litho-addons' ),
					],
				]
			);
			$this->add_control(
				'litho_viewport',
				[
					'label'       => esc_html__( 'Viewport', 'litho-addons' ),
					'type'        => Controls_Manager::SLIDER,
					'render_type' => 'none',
					'conditions'  => [
						'relation' => 'or',
						'terms'    => [
							[
								'name'     => 'litho_trigger',
								'operator' => '===',
								'value'    => 'arriving_to_viewport',
							],
							[
								'name'     => 'litho_trigger',
								'operator' => '===',
								'value'    => 'bind_to_scroll',
							],
						],
					],
					'default'     => [
						'sizes' => [
							'start' => 0,
							'end'   => 100,
						],
						'unit'  => '%',
					],
					'labels'      => [
						esc_html__( 'Bottom', 'litho-addons' ),
						esc_html__( 'Top', 'litho-addons' ),
					],
					'scales'      => 1,
					'handles'     => 'range',
				]
			);
			$this->add_control(
				'litho_lottie_speed',
				[
					'label'       => esc_html__( 'Play Speed', 'litho-addons' ),
					'type'        => Controls_Manager::SLIDER,
					'render_type' => 'none',
					'condition'   => [
						'litho_trigger!' => [
							'bind_to_scroll',
							'none',
						],
					],
					'default'     => [
						'size' => 1,
					],
					'range'       => [
						'px' => [
							'min'  => 0.1,
							'max'  => 5,
							'step' => 0.1,
						],
					],
					'dynamic'     => [
						'active' => true,
					],
				]
			);
			$this->add_control(
				'litho_start_point',
				[
					'label'       => esc_html__( 'Start Point', 'litho-addons' ),
					'type'        => Controls_Manager::SLIDER,
					'render_type' => 'none',
					'default'     => [
						'size' => 0,
						'unit' => '%',
					],
					'size_units'  => [
						'%',
					],
					'conditions'  => [
						'relation' => 'or',
						'terms'    => [
							[
								'name'     => 'litho_trigger',
								'operator' => '===',
								'value'    => 'on_click',
							],
						],
					],
				]
			);
			$this->add_control(
				'litho_end_point',
				[
					'label'       => esc_html__( 'End Point', 'litho-addons' ),
					'type'        => Controls_Manager::SLIDER,
					'render_type' => 'none',
					'default'     => [
						'size' => 100,
						'unit' => '%',
					],
					'size_units'  => [
						'%',
					],
					'conditions'  => [
						'relation' => 'or',
						'terms'    => [
							[
								'name'     => 'litho_trigger',
								'operator' => '===',
								'value'    => 'on_click',
							],
						],
					],
				]
			);
			$this->add_control(
				'litho_reverse_animation',
				[
					'label'        => esc_html__( 'Reverse', 'litho-addons' ),
					'type'         => Controls_Manager::SWITCHER,
					'render_type'  => 'none',
					'conditions'   => [
						'relation' => 'and',
						'terms'    => [
							[
								'name'     => 'litho_trigger',
								'operator' => '!==',
								'value'    => 'bind_to_scroll',
							],
							[
								'name'     => 'litho_trigger',
								'operator' => '!==',
								'value'    => 'on_hover',
							],
							[
								'name'     => 'litho_trigger',
								'operator' => '!==',
								'value'    => 'none',
							],
						],
					],
					'return_value' => 'yes',
					'default'      => '',
				]
			);
			$this->add_control(
				'litho_lottie_loop',
				[
					'label'        => esc_html__( 'Loop', 'litho-addons' ),
					'type'         => Controls_Manager::SWITCHER,
					'render_type'  => 'none',
					'condition'    => [
						'litho_trigger!' => [
							'bind_to_scroll',
							'none',
						],
					],
					'return_value' => 'yes',
					'default'      => '',
				]
			);
			$this->add_control(
				'litho_number_of_times',
				[
					'label'       => esc_html__( 'Times', 'litho-addons' ),
					'type'        => Controls_Manager::NUMBER,
					'render_type' => 'none',
					'conditions'  => [
						'relation' => 'and',
						'terms'    => [
							[
								'name'     => 'litho_trigger',
								'operator' => '!==',
								'value'    => 'bind_to_scroll',
							],
							[
								'name'     => 'litho_lottie_loop',
								'operator' => '===',
								'value'    => 'yes',
							],
							[
								'name'     => 'litho_trigger',
								'operator' => '!==',
								'value'    => 'none',
							],
						],
					],
					'min'         => 0,
					'step'        => 1,
				]
			);
			$this->end_controls_section();

			$this->start_controls_section(
				'litho_lottie_style',
				[
					'label' => esc_html__( 'Lottie', 'litho-addons' ),
					'tab'   => Controls_Manager::TAB_STYLE,
				]
			);
			$this->add_responsive_control(
				'litho_lottie_align',
				[
					'label'        => esc_html__( 'Alignment', 'litho-addons' ),
					'type'         => Controls_Manager::CHOOSE,
					'options'      => [
						'left'   => [
							'title' => esc_html__( 'Left', 'litho-addons' ),
							'icon'  => 'eicon-text-align-left',
						],
						'center' => [
							'title' => esc_html__( 'Center', 'litho-addons' ),
							'icon'  => 'eicon-text-align-center',
						],
						'right'  => [
							'title' => esc_html__( 'Right', 'litho-addons' ),
							'icon'  => 'eicon-text-align-right',
						],
					],
					'prefix_class' => 'elementor%s-align-',
					'default'      => 'center',
				]
			);
			$this->add_responsive_control(
				'litho_lottie_width',
				[
					'label'          => esc_html__( 'Width', 'litho-addons' ),
					'type'           => Controls_Manager::SLIDER,
					'size_units'     => [
						'px',
						'%',
						'em',
						'rem',
						'vw',
						'custom',
					],
					'default'        => [
						'unit' => '%',
					],
					'tablet_default' => [
						'unit' => '%',
					],
					'mobile_default' => [
						'unit' => '%',
					],
					'range'          => [
						'%'   => [
							'min' => 1,
							'max' => 100,
						],
						'px'  => [
							'min' => 1,
							'max' => 1000,
						],
						'em'  => [
							'max' => 100,
						],
						'rem' => [
							'max' => 100,
						],
						'vw'  => [
							'min' => 1,
							'max' => 100,
						],
					],
					'selectors'      => [
						'{{WRAPPER}} .lottie-animation-wrapper' => 'width: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'litho_lottie_space',
				[
					'label'          => esc_html__( 'Max Width', 'litho-addons' ),
					'type'           => Controls_Manager::SLIDER,
					'size_units'     => [
						'px',
						'%',
						'em',
						'rem',
						'vw',
						'custom',
					],
					'default'        => [
						'unit' => '%',
					],
					'tablet_default' => [
						'unit' => '%',
					],
					'mobile_default' => [
						'unit' => '%',
					],
					'range'          => [
						'%'   => [
							'min' => 1,
							'max' => 100,
						],
						'px'  => [
							'min' => 1,
							'max' => 1000,
						],
						'em'  => [
							'max' => 100,
						],
						'rem' => [
							'max' => 100,
						],
						'vw'  => [
							'min' => 1,
							'max' => 100,
						],
					],
					'selectors'      => [
						'{{WRAPPER}} .lottie-animation-wrapper' => 'max-width: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->start_controls_tabs(
				'image_effects',
			);
			$this->start_controls_tab(
				'litho_lottie_normal',
				[
					'label' => esc_html__( 'Normal', 'litho-addons' ),
				]
			);
			$this->add_control(
				'litho_lottie_opacity',
				[
					'label'     => esc_html__( 'Opacity', 'litho-addons' ),
					'type'      => Controls_Manager::SLIDER,
					'range'     => [
						'px' => [
							'max'  => 1,
							'min'  => 0.10,
							'step' => 0.01,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .lottie-animation-wrapper' => 'opacity: {{SIZE}};',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Css_Filter::get_type(),
				[
					'name'     => 'litho_lottie_css_filters',
					'selector' => '{{WRAPPER}} .lottie-animation-wrapper',
				]
			);
			$this->end_controls_tab();

			$this->start_controls_tab(
				'litho_lottie_hover',
				[
					'label' => esc_html__( 'Hover', 'litho-addons' ),
				]
			);
			$this->add_control(
				'litho_lottie_opacity_hover',
				[
					'label'     => esc_html__( 'Opacity', 'litho-addons' ),
					'type'      => Controls_Manager::SLIDER,
					'range'     => [
						'px' => [
							'max'  => 1,
							'min'  => 0.10,
							'step' => 0.01,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .lottie-animation-wrapper:hover' => 'opacity: {{SIZE}};',
					],
				]
			);
			$this->add_control(
				'litho_lottie_hover_transition',
				[
					'label'     => esc_html__( 'Transition Duration', 'litho-addons' ) . ' (s)',
					'type'      => Controls_Manager::SLIDER,
					'range'     => [
						'px' => [
							'min'  => 0,
							'max'  => 3,
							'step' => 0.1,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .lottie-animation-wrapper' => 'transition: {{SIZE}}s',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Css_Filter::get_type(),
				[
					'name'     => 'litho_lottie_css_filters_hover',
					'selector' => '{{WRAPPER}} .lottie-animation-wrapper:hover',
				]
			);
			$this->end_controls_tab();
			$this->end_controls_tabs();
			$this->end_controls_section();

			$this->start_controls_section(
				'litho_section_style_caption',
				[
					'label'     => esc_html__( 'Caption', 'litho-addons' ),
					'tab'       => Controls_Manager::TAB_STYLE,
					'condition' => [
						'litho_caption_source!' => 'none',
					],
				]
			);
			$this->add_control(
				'litho_lottie_caption_align',
				[
					'label'     => esc_html__( 'Alignment', 'litho-addons' ),
					'type'      => Controls_Manager::CHOOSE,
					'options'   => [
						'left'   => [
							'title' => esc_html__( 'Left', 'litho-addons' ),
							'icon'  => 'eicon-text-align-left',
						],
						'center' => [
							'title' => esc_html__( 'Center', 'litho-addons' ),
							'icon'  => 'eicon-text-align-center',
						],
						'right'  => [
							'title' => esc_html__( 'Right', 'litho-addons' ),
							'icon'  => 'eicon-text-align-right',
						],
					],
					'default'   => 'center',
					'selectors' => [
						'{{WRAPPER}} .lottie-caption' => 'text-align: {{VALUE}};',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name'     => 'litho_lottie_caption_typography',
					'selector' => '{{WRAPPER}} .lottie-caption',
				]
			);
			$this->add_control(
				'litho_lottie_text_color',
				[
					'label'     => esc_html__( 'Color', 'litho-addons' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .lottie-caption' => 'color: {{VALUE}};',
					],
				]
			);
			$this->add_responsive_control(
				'litho_lottie_caption_space',
				[
					'label'      => esc_html__( 'Spacing', 'litho-addons' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => [
						'px',
						'em',
						'rem',
						'custom',
					],
					'range'      => [
						'px'  => [
							'max' => 100,
						],
						'em'  => [
							'max' => 10,
						],
						'rem' => [
							'max' => 10,
						],
					],
					'selectors'  => [
						'{{WRAPPER}} .lottie-caption' => 'margin-top: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->end_controls_section();
		}
		
        /**
		 * Render lottie widget output on the frontend.
		 *
		 * Written in PHP and used to generate the final HTML.
		 *
		 * @since 1.0
		 *
		 * @access protected
		 */
		protected function render() {

			$settings       = $this->get_settings_for_display();
			$lottie_caption = '';
			$external_url   = ( isset( $settings['litho_animation_url']['url'] ) && $settings['litho_animation_url']['url'] ) ? $settings['litho_animation_url']['url'] : '';
			$media_url      = ( isset( $settings['litho_lottie_media']['url'] ) && $settings['litho_lottie_media']['url'] ) ? $settings['litho_lottie_media']['url'] : '';
			$source_url     = ( isset( $settings['litho_lottie_source'] ) && $settings['litho_lottie_source'] ) ? $settings['litho_lottie_source'] : '';
			$loop           = 'yes' === $settings['litho_lottie_loop'] ? 'true' : 'false';
			$speed          = ( isset( $settings['litho_lottie_speed']['size'] ) && $settings['litho_lottie_speed']['size'] ) ? $settings['litho_lottie_speed']['size'] : '';
			$animation_url  = 'external_url' === $source_url ? $external_url : $media_url;
			$caption        = ( isset( $settings['litho_caption_source'] ) && $settings['litho_caption_source'] ) ? $settings['litho_caption_source'] : '';
			$trigger        = ( isset( $settings['litho_trigger'] ) && $settings['litho_trigger'] ) ? $settings['litho_trigger'] : '';
			$renderer       = ( isset( $settings['litho_renderer'] ) && $settings['litho_renderer'] ) ? $settings['litho_renderer'] : '';
			$viewport_start = ( isset( $settings['litho_viewport']['sizes']['start'] ) && $settings['litho_viewport']['sizes']['start'] ) ? $settings['litho_viewport']['sizes']['start'] : '';
			$viewport_end   = ( isset( $settings['litho_viewport']['sizes']['end'] ) && $settings['litho_viewport']['sizes']['end'] ) ? $settings['litho_viewport']['sizes']['end'] : '';
			$times          = ( isset( $settings['litho_number_of_times'] ) && $settings['litho_number_of_times'] ) ? $settings['litho_number_of_times'] : '';
			$start_point    = ( isset( $settings['litho_start_point']['size'] ) && $settings['litho_start_point']['size'] ) ? $settings['litho_start_point']['size'] : '';
			$end_point      = ( isset( $settings['litho_end_point']['size'] ) && $settings['litho_end_point']['size'] ) ? $settings['litho_end_point']['size'] : '';
			$relative_to    = ( isset( $settings['litho_effects_relative_to'] ) && $settings['litho_effects_relative_to'] ) ? $settings['litho_effects_relative_to'] : '';
			$reverse        = ( isset( $settings['litho_reverse_animation'] ) && $settings['litho_reverse_animation'] ) ? $settings['litho_reverse_animation'] : '';
			$hover_out      = ( isset( $settings['litho_on_hover_out'] ) && $settings['litho_on_hover_out'] ) ? $settings['litho_on_hover_out'] : '';

			$lottieconfig = array(
				'lottie_loop'    => $loop,
				'speed'          => $speed,
				'trigger'        => $trigger,
				'renderer'       => $renderer,
				'viewport_start' => $viewport_start,
				'viewport_end'   => $viewport_end,
				'number_of_time' => $times,
				'start_point'    => $start_point,
				'end_point'      => $end_point,
				'relative_to'    => $relative_to,
				'reverse'        => $reverse,
				'hover_out'      => $hover_out,
			);

			if ( ! empty( $settings['litho_custom_link']['url'] ) && 'custom' === $settings['litho_link_to'] ) {
				$this->add_link_attributes( 'url', $settings['litho_custom_link'] );
			}

			if ( ( ( 'media_file' === $source_url && 'none' !== $caption ) && 'custom' === $caption ) || ( 'external_url' === $source_url && '' !== $settings['litho_caption'] ) ) {
				$lottie_caption = $settings['litho_caption'];
			} elseif ( 'caption' === $settings['litho_caption_source'] ) {
				$lottie_caption = wp_get_attachment_caption( $settings['litho_lottie_media']['id'] );
			} elseif ( 'title' === $settings['litho_caption_source'] ) {
				$lottie_caption = get_the_title( $settings['litho_lottie_media']['id'] );
			}

			$this->add_render_attribute(
				[
					'wrapper' => [
						'class'         => [
							'lottie-animation-wrapper',
						],
						'data-settings' => wp_json_encode( $lottieconfig ),
					],
				]
			);

			?>
			<div <?php $this->print_render_attribute_string( 'wrapper' ); ?>>
				<?php
				if ( ! empty( $settings['litho_custom_link']['url'] ) ) {
					?>
					<a <?php $this->print_render_attribute_string( 'url' ); ?>>
					<?php
				}
				?>
				<div class="lottie-animation" data-animation-url="<?php echo esc_url( $animation_url ); ?>"></div>
				<?php
				if ( ! empty( $settings['litho_custom_link']['url'] ) ) {
					?>
					</a>
					<?php
				}
				?>
				<div class="lottie-caption"><?php echo esc_html( $lottie_caption ); ?></div>
			</div>
			<?php
		}
    }
}
