<?php
namespace LithoAddons\Section_builder\Classes;

/**
 * Section builder admin.
 *
 * @package Litho
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// If class `Section_Builder_Admin` doesn't exists yet.
if ( ! class_exists( 'Section_Builder_Admin' ) ) {

	/**
	 * Define Section_Builder_Admin classe
	 */
	class Section_Builder_Admin {

		/**
		 * Define Post type slug
		 */
		public $post_type = 'sectionbuilder';

		public function __construct() {
			add_action( 'init', array( $this, 'litho_sectionbuilder_post_type_init' ) );
			add_action( 'template_redirect', array( $this, 'litho_sectionbuilder_block_template_frontend' ) );
			add_filter( 'single_template', array( $this, 'litho_sectionbuilder_load_canvas_template' ) );
			add_filter( 'manage_sectionbuilder_posts_columns', array( $this, 'litho_sectionbuilder_set_columns_fields' ) );
			add_action( 'manage_sectionbuilder_posts_custom_column', array( $this, 'litho_sectionbuilder_render_column_fields' ), 10, 2 );
			add_filter( 'manage_edit-sectionbuilder_sortable_columns', array( $this, 'litho_sectionbuilder_sortable_columns' ) );
			add_filter( 'views_edit-sectionbuilder', array( $this, 'litho_sectionbuilder_admin_print_tabs' ) );
			add_filter( 'pre_get_posts', array( $this, 'litho_sectionbuilder_make_sortable' ) );
		}

		/**
		 * Print custom admin navigation tabs for Section Builder template types.
		 *
		 * @param array $views The existing views (not used here, but returned for compatibility).
		 * @return array
		 */
		public function litho_sectionbuilder_admin_print_tabs( $views ) {
			if ( is_admin() && isset( $_GET['post_type'] ) && sanitize_key( $_GET['post_type'] ) === $this->post_type ) { // phpcs:ignore.
				$template_type = array(
					'all'               => __( 'Saved Templates', 'litho-addons' ),
					'mini-header'       => __( 'Mini Header', 'litho-addons' ),
					'header'            => __( 'Header', 'litho-addons' ),
					'footer'            => __( 'Footer', 'litho-addons' ),
					'archive'           => __( 'Archive', 'litho-addons' ),
					'archive-portfolio' => __( 'Archive Portfolio', 'litho-addons' ),
					'custom-title'      => __( 'Page Title', 'litho-addons' ),
					'promo_popup'       => __( 'Promo popup', 'litho-addons' ),
					'side_icon'         => __( 'Side Icon', 'litho-addons' ),
				);
				?>
				<div class="litho-nav-tabs nav-tab-wrapper">
					<?php
					$current_type = isset( $_GET['template_type'] ) ? sanitize_key( $_GET['template_type'] ) : 'all'; // phpcs:ignore.

					foreach ( $template_type as $key => $label ) {
						$is_active   = ( $key === $current_type ) ? ' nav-tab-active' : '';

						$url = esc_url( add_query_arg( array(
							'post_type'     => $this->post_type,
							'template_type' => $key,
						), 'edit.php' ) );
						

						printf(
							'<a href="%1$s" class="nav-tab%2$s">%3$s</a>',
							$url,
							esc_attr( $is_active ),
							esc_html( $label )
						);
					}
					?>
				</div>
				<?php
				return $views;
			}
		}

		/**
		 * Registers the 'Section Builder' custom post type.
		 *
		 * This post type is used to create and manage reusable sections or templates,
		 * typically built with Elementor. These templates are hidden from navigation
		 * menus and search results, and meant only for internal use.
		 */
		public function litho_sectionbuilder_post_type_init() {
			$labels = array(
				'name'               => __( 'Section Builder', 'litho-addons' ),
				'singular_name'      => __( 'Section Builder', 'litho-addons' ),
				'menu_name'          => __( 'Section Builder', 'litho-addons' ),
				'name_admin_bar'     => __( 'Section Builder', 'litho-addons' ),
				'add_new'            => __( 'Add New', 'litho-addons' ),
				'add_new_item'       => __( 'Add New Template', 'litho-addons' ),
				'new_item'           => __( 'New Template', 'litho-addons' ),
				'edit_item'          => __( 'Edit Template', 'litho-addons' ),
				'view_item'          => __( 'View Template', 'litho-addons' ),
				'all_items'          => __( 'All Templates', 'litho-addons' ),
				'search_items'       => __( 'Search Template', 'litho-addons' ),
				'parent_item_colon'  => __( 'Parent Template:', 'litho-addons' ),
				'not_found'          => __( 'No template found.', 'litho-addons' ),
				'not_found_in_trash' => __( 'No template found in trash.', 'litho-addons' ),
			);

			$args = array(
				'labels'              => $labels,
				'public'              => true,
				'show_ui'             => true,
				'show_in_menu'        => true,
				'show_in_nav_menus'   => false,
				'capability_type'     => 'post',
				'hierarchical'        => false,
				'menu_position'       => 20,
				'menu_icon'           => 'dashicons-editor-kitchensink',
				'exclude_from_search' => true,
				'rewrite'             => false, // Prevents public URLs.
				'supports'            => array( 'title', 'elementor', 'author' ),
			);

			register_post_type( 'sectionbuilder', $args );
		}

		/**
		 * Prevent frontend access to Section Builder templates for non-editors.
		 *
		 * Redirects users who try to view a Section Builder post on the frontend
		 * (e.g., via a direct URL) unless they have permission to edit posts.
		 */
		public function litho_sectionbuilder_block_template_frontend() {
			if ( is_singular( $this->post_type ) && ! current_user_can( 'edit_posts' ) ) {
				wp_redirect( site_url(), 301 );
				die;
			}
		}

		/**
		 * Load Elementor's Canvas template for Section Builder single view.
		 *
		 * If the current post type matches the Section Builder type,
		 * this function returns the Elementor canvas template path to render a blank layout.
		 *
		 * @param string $single_template The path to the current single template.
		 * @return string Modified template path.
		 */
		public function litho_sectionbuilder_load_canvas_template( $single_template ) {
			global $post;

			if ( isset( $post->post_type ) && $this->post_type === $post->post_type ) {

				// Check Elementor 2.0+ canvas path first.
				$canvas_template = ELEMENTOR_PATH . '/modules/page-templates/templates/canvas.php';

				if ( file_exists( $canvas_template ) ) {
					return $canvas_template;
				}

				// Fallback to older Elementor canvas path.
				return ELEMENTOR_PATH . '/includes/page-templates/canvas.php';
			}

			return $single_template;
		}

		/**
		 * Customize admin columns for the Section Builder post type.
		 *
		 * Adds conditional columns like 'Archive type' and 'Sticky type'
		 * based on the selected template type in the URL.
		 *
		 * @param array $columns Existing column list.
		 * @return array Modified column list.
		 */
		public function litho_sectionbuilder_set_columns_fields( $columns ) {
			// Preserve the original 'Date' column to re-insert it later.	
			$date_column = $columns['date'];
			unset( $columns['date'] );

			// Add 'Section Type' column.
			$columns['template_type'] = esc_html__( 'Section type', 'litho-addons' );

			// Conditionally add 'Archive type' column.
			if ( isset( $_GET['template_type'] ) && ( 'archive' === $_GET['template_type'] || 'archive-portfolio' === $_GET['template_type'] ) ) { // phpcs:ignore.
				$columns['archive_type'] = esc_html__( 'Archive type', 'litho-addons' );
			}

			// Conditionally add 'Sticky type' column.
			if ( isset( $_GET['template_type'] ) && ( 'mini-header' === $_GET['template_type'] || 'header' === $_GET['template_type'] ) ) { // phpcs:ignore.
				$columns['sticky_type'] = esc_html__( 'Sticky type', 'litho-addons' );
			}

			// Re-add the 'Date' column at the end.
			$columns['date'] = $date_column;

			return $columns;
		}

		/**
		 * Render custom column values for the Section Builder admin table.
		 *
		 * @param string $column  Column name.
		 * @param int    $post_id Post ID.
		 */
		public function litho_sectionbuilder_render_column_fields( $column, $post_id ) {
			switch ( $column ) {
				case 'template_type':
					$template_type  = get_post_meta( $post_id, '_litho_section_builder_template', true );
					$template_label = ( $template_type ) ? litho_get_template_type_by_key( $template_type ) : '-';

					if ( ! empty( $template_type ) ) {
						$out[] = sprintf(
							'<a href="%s">%s</a>',
							esc_url(
								add_query_arg(
									array(
										'post_type'     => $this->post_type,
										'template_type' => $template_type,
									),
									'edit.php'
								)
							),
							$template_label
						);
						echo join( ', ', $out ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
					} else {
						printf( '%s', esc_html__( 'N/A', 'litho-addons' ) );
					}
					break;
				case 'sticky_type':
					$template_type = get_post_meta( $post_id, '_litho_section_builder_template', true );

					if ( 'mini-header' === $template_type ) {

						$header_sticky = get_post_meta( $post_id, '_litho_mini_header_sticky_type', true );

					} else {

						$header_sticky = get_post_meta( $post_id, '_litho_header_sticky_type', true );
					}

					if ( $header_sticky && ( 'mini-header' === $template_type || 'header' === $template_type ) ) {

						$sticky_label = litho_get_header_sticky_type_by_key( $header_sticky );

					} else {

						$sticky_label = esc_html__( 'N/A', 'litho-addons' );
					}

					echo sprintf( '%s', $sticky_label ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
					break;
				case 'archive_type':
					$template_type                   = get_post_meta( $post_id, '_litho_section_builder_template', true );
					$template_archive_type           = get_post_meta( $post_id, '_litho_template_archive_style', true );
					$template_archive_portfolio_type = get_post_meta( $post_id, '_litho_template_archive_portfolio_style', true );

					if ( $template_archive_type && 'archive' === $template_type ) {
						$archive_label_arr = array();

						foreach ( litho_get_archive_style_by_key() as $key => $option ) {
							if ( is_array( $template_archive_type ) && in_array( $key, $template_archive_type ) ) {
								$archive_label_arr[] = $option;
							}
						}
						$archive_label = implode( ', ', $archive_label_arr );
					} elseif ( $template_archive_portfolio_type && 'archive-portfolio' === $template_type ) {
						$archive_label_arr = array();

						foreach ( litho_get_archive_style_by_key() as $key => $option ) {
							if ( is_array( $template_archive_portfolio_type ) && in_array( $key, $template_archive_portfolio_type ) ) {
								$archive_label_arr[] = $option;
							}
						}
						$archive_label = implode( ', ', $archive_label_arr );

					} else {
						$archive_label = esc_html__( 'N/A', 'litho-addons' );
					}

					echo sprintf( '%s', $archive_label ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
					break;
			}
		}

		/**
		 * Makes the 'template_type' column sortable by meta key in the Section Builder admin listing.
		 *
		 * Also handles filtering via $_GET['template_type'] for admin filtering UI.
		 *
		 * @param WP_Query $query The current query object.
		 */
		public function litho_sectionbuilder_make_sortable( $query ) {
			global $pagenow;

			// Run only in admin on the post listing screen.
			if ( ! is_admin() && ( 'edit.php' !== $pagenow ) ) {
				return;
			}

			if ( isset( $_GET['template_type'] ) && ! empty( $_GET['template_type'] ) && 'all' !== $_GET['template_type'] ) { // phpcs:ignore
				$query->query_vars['meta_key']   = '_litho_section_builder_template';// phpcs:ignore
				$query->query_vars['meta_value'] = $_GET['template_type']; // phpcs:ignore

			} else {
				if ( 'sectionbuilder_template_type' === $query->get( 'orderby' ) ) {
					$query->set( 'meta_key', '_litho_section_builder_template' );
					$query->set( 'orderby', 'meta_value' );
				}
			}
		}

		/**
		 * Adds a sortable custom column to the Section Builder post list table.
		 *
		 * This function registers the 'template_type' column as sortable in the
		 * WordPress admin list view for a custom post type (e.g., section builder templates).
		 *
		 * @param array $cols Existing sortable columns.
		 * @return array Modified array of sortable columns.
		 */
		public function litho_sectionbuilder_sortable_columns( $cols ) {
			$cols['template_type'] = 'sectionbuilder_template_type';
			return $cols;
		}

	} // End of Class.

} // End of Class Exists.
