<?php
/**
 * Litho Performance Manager
 *
 * @package Litho
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

// If class `Litho_Performance_Manager` doesn't exists yet.
if ( ! class_exists( 'Litho_Performance_Manager' ) ) {
	/**
	 * Define Litho_Performance_Manager class.
	 */
	class Litho_Performance_Manager {
		/**
		 * Constructor
		 */
		public function __construct() {
			$litho_remove_url_query_string = get_theme_mod( 'litho_remove_url_query_string', '0' );
			$litho_disable_xmlrpc          = get_theme_mod( 'litho_disable_xmlrpc', '0' );
			$litho_remove_rsd_link         = get_theme_mod( 'litho_remove_rsd_link', '0' );
			$litho_remove_shortlink        = get_theme_mod( 'litho_remove_shortlink', '0' );
			$litho_remove_wp_ver_generator = get_theme_mod( 'litho_remove_wp_version_generator', '0' );

			add_action( 'wp', array( $this, 'litho_remove_style_wp_emojis' ) );
			add_action( 'init', array( $this, 'litho_remove_feed_links' ) );
			add_action( 'wp_head', array( $this, 'litho_preload_resources' ), -5 );
			add_action( 'wp_enqueue_scripts', array( $this, 'litho_remove_gutenberg_styles' ), 20 );
			add_action( 'wp_default_scripts', array( $this, 'litho_disable_jquery_migrate' ) );
			add_action( 'wp_print_styles', array( $this, 'litho_remove_dashicons' ), 100 );
			add_action( 'pre_ping', array( $this, 'litho_remove_self_ping' ) );
			add_filter( 'jpeg_quality', array( $this, 'litho_set_jpeg_quality' ) );
			add_filter( 'wp_editor_set_quality', array( $this, 'litho_set_jpeg_quality' ) );
			add_filter( 'big_image_size_threshold', array( $this, 'litho_set_big_image_size_threshold' ) );
			add_filter( 'wp_revisions_to_keep', array( $this, 'litho_set_revisions_to_keep' ), 10, 2 );
			add_filter( 'heartbeat_settings', array( $this, 'litho_set_heartbeat_time_interval' ) );
			add_filter( 'wp_resource_hints', array( $this, 'litho_add_prefetch_dns_urls' ), 9, 2 );
			add_filter( 'wp_resource_hints', array( $this, 'litho_add_preconnect_urls' ), 10, 2 );

			// Disable WooCommerce cart fragments.
			add_action( 'wp_enqueue_scripts', array( $this, 'litho_remove_cart_fragments' ), 11 );

			if ( '1' === $litho_remove_url_query_string && ! is_admin() ) {
				add_filter( 'script_loader_src', array( $this, 'litho_remove_query_strings' ), 15, 1 );
				add_filter( 'style_loader_src', array( $this, 'litho_remove_query_strings' ), 15, 1 );
			}

			if ( '1' === $litho_disable_xmlrpc ) {
				add_filter( 'xmlrpc_enabled', '__return_false' );
			}

			if ( '1' === $litho_remove_rsd_link ) {
				remove_action( 'wp_head', 'rsd_link' );
				remove_action( 'wp_head', 'wlwmanifest_link' );
			}

			if ( '1' === $litho_remove_shortlink ) {
				remove_action( 'wp_head', 'wp_shortlink_wp_head' );
			}

			if ( '1' === $litho_remove_wp_ver_generator ) {
				remove_action( 'wp_head', 'wp_generator' );
			}

			add_action(
				'elementor/frontend/after_enqueue_styles',
				function () {
					// Check if we're in the admin area or the current user has permission to manage options.
					if ( is_admin() || current_user_can( 'manage_options' ) ) {
						return;
					}

					$litho_elementor_icons_css = litho_option( 'litho_elementor_icons_css_file', '0' );
					if ( '1' === $litho_elementor_icons_css ) {
						wp_dequeue_style( 'elementor-icons' );
					}
				}
			);
		}

		/**
		 * Remove RSS feed links from <head> if the "Disable RSS Feeds" option is enabled in the Customizer.
		 *
		 * This helps clean up the page head and prevents bots or users from discovering disabled RSS endpoints.
		 */
		public function litho_remove_feed_links() {
			$litho_disable_rss_feeds = get_theme_mod( 'litho_disable_rss_feeds', '0' );
			if ( '1' === $litho_disable_rss_feeds ) {
				remove_action( 'wp_head', 'feed_links', 2 );
				remove_action( 'wp_head', 'feed_links_extra', 3 );

				// Block RSS feed endpoints.
				foreach ( array( 'do_feed', 'do_feed_rdf', 'do_feed_rss', 'do_feed_rss2', 'do_feed_atom' ) as $hook ) {
					add_action( $hook, array( $this, 'litho_disable_rss_output' ), 1 );
				}
			}
		}

		/**
		 * Outputs a message and terminates the request to disable RSS feeds.
		 */
		public function litho_disable_rss_output() {
			wp_die(
				__( 'RSS feeds are disabled on this site.', 'litho' ),
				__( 'Feed Disabled', 'litho-addons' ),
				array( 'response' => 403 )
			);
		}

		/**
		 * JPG image quality control
		 *
		 * @param string $quality JPG image quality.
		 *
		 * @since 1.0
		 */
		public function litho_set_jpeg_quality( $quality ) {
			$litho_jpg_image_quality = get_theme_mod( 'litho_jpg_image_quality', '' );

			// Return the custom quality if set, otherwise return the default quality.
			return $litho_jpg_image_quality ? (int) $litho_jpg_image_quality : $quality;
		}

		/**
		 * Control Image Size Threshold
		 *
		 * @since 1.0
		 */
		public function litho_set_big_image_size_threshold() {
			$threshold = (int) get_theme_mod( 'litho_big_image_size_threshold', 2560 );
			$threshold = 0 === $threshold ? false : $threshold;

			return $threshold;
		}

		/**
		 * Litho remove WP emojis.
		 */
		public function litho_remove_style_wp_emojis() {
			$litho_wp_emojis = get_theme_mod( 'litho_wp_emojis', '0' );

			// Check if the user has enabled emojis.
			if ( '1' === $litho_wp_emojis ) {
				// Remove emoji scripts and styles.
				remove_action( 'wp_head', 'print_emoji_detection_script', 7 );
				remove_action( 'admin_print_scripts', 'print_emoji_detection_script' );
				remove_action( 'wp_print_styles', 'print_emoji_styles' );
				remove_action( 'admin_print_styles', 'print_emoji_styles' );
				remove_filter( 'the_content_feed', 'wp_staticize_emoji' );
				remove_filter( 'comment_text_rss', 'wp_staticize_emoji' );
				remove_filter( 'wp_mail', 'wp_staticize_emoji_for_email' );
				// Disable TinyMCE emoji plugin.
				add_filter( 'tiny_mce_plugins', array( $this, 'litho_disable_emojis_tinymce' ) );
				// Remove DNS prefetch for emojis.
				add_filter( 'wp_resource_hints', array( $this, 'litho_disable_emojis_remove_dns_prefetch' ), 10, 2 );
			}
		}

		/**
		 * Disable Emojis in TinyMCE Editor.
		 *
		 * @param array $plugins Emojis tinymce.
		 */
		public function litho_disable_emojis_tinymce( $plugins ) {
			// Remove the wpemoji plugin if present.
			return is_array( $plugins ) ? array_diff( $plugins, array( 'wpemoji' ) ) : [];
		}

		/**
		 * Remove the WordPress emoji CDN hostname from DNS prefetching hints.
		 *
		 * @param  array  $urls URLs to print for resource hints.
		 * @param  string $relation_type The relation type the URLs are printed for.
		 * @return array  Difference betwen the two arrays.
		 */
		public function litho_disable_emojis_remove_dns_prefetch( $urls, $relation_type ) {
			if ( 'dns-prefetch' === $relation_type ) {
				$emoji_svg_url_bit = 'https://s.w.org/images/core/emoji/';
				foreach ( $urls as $key => $url ) {
					if ( false !== strpos( $url, $emoji_svg_url_bit ) ) {
						unset( $urls[ $key ] );
					}
				}
			}

			return $urls;
		}

		/**
		 * Remove Gutenberg global and inline styles based on theme settings.
		 */
		public static function litho_remove_gutenberg_styles() {
			$litho_enable_remove_global_styles = get_theme_mod( 'litho_enable_remove_global_styles', '0' );

			// Check if global styles should be removed.
			if ( '1' === $litho_enable_remove_global_styles ) {
				// Remove global and classic theme styles.
				wp_deregister_style( 'global-styles' );
				wp_dequeue_style( 'global-styles' );
				wp_deregister_style( 'classic-theme-styles' );
				wp_dequeue_style( 'classic-theme-styles' );
			}

			$disabled_styles = get_theme_mod( 'litho_disable_gutenberg_styles', '' );
			$disabled_styles = explode( ',', $disabled_styles );

			if ( is_array( $disabled_styles ) ) {
				foreach ( $disabled_styles as $style ) {
					wp_deregister_style( $style );
					wp_dequeue_style( $style );
				}
			}
		}

		/**
		 * Disable jQuery Migrate if the corresponding theme option is enabled.
		 *
		 * @param WP_Scripts $scripts WP_Scripts object.
		 * @return void
		 */
		public function litho_disable_jquery_migrate( $scripts ) {
			// Check if the setting to disable jQuery Migrate is enabled.
			$litho_disable_jq_migrate = get_theme_mod( 'litho_disable_jq_migrate', '0' );

			if ( '1' !== $litho_disable_jq_migrate ) {
				return; // Exit if the option is not enabled.
			}

			// Only apply this on the front end.
			if ( ! is_admin() && isset( $scripts->registered['jquery'] ) ) {
				$script = $scripts->registered['jquery'];

				// Remove jQuery Migrate from the jQuery dependencies if it exists.
				if ( $script->deps ) {
					$script->deps = array_diff( $script->deps, array( 'jquery-migrate' ) );
				}
			}
		}

		/**
		 * Disable Dashicons if the corresponding theme option is enabled.
		 */
		public function litho_remove_dashicons() {
			// Check if the setting to disable dashicons is enabled.
			$litho_disable_dashicons = get_theme_mod( 'litho_disable_dashicons', '0' );

			// Exit if Dashicons should not be disabled.
			if ( '1' !== $litho_disable_dashicons ) {
				return;
			}

			// Only remove Dashicons if the admin bar is not showing and not in customizer preview.
			if ( ! is_admin_bar_showing() && ! is_customize_preview() ) {
				wp_dequeue_style( 'dashicons' );
				wp_deregister_style( 'dashicons' );
			}
		}

		/**
		 * Disables pingbacks from the site by removing self-pings.
		 *
		 * @param array $links Array of URLs that may include self-pings.
		 */
		public function litho_remove_self_ping( &$links ) {
			// Get the setting to disable self-pings.
			$litho_disable_self_pings = get_theme_mod( 'litho_disable_self_pings', '0' );

			// Exit if self-pings should not be disabled.
			if ( '1' !== $litho_disable_self_pings ) {
				return;
			}

			// Get the home URL.
			$home = get_option( 'home' );

			// Remove self-ping links.
			foreach ( $links as $l => $link ) {
				if ( str_starts_with( $link, $home ) ) {
					unset( $links[ $l ] );
				}
			}
		}

		/**
		 * Set Heartbeat Interval.
		 *
		 * @param array $settings Heartbeat settings.
		 * @return array Modified heartbeat settings.
		 */
		public function litho_set_heartbeat_time_interval( $settings ) {
			$litho_control_heartbeat  = get_theme_mod( 'litho_control_heartbeat', '0' );
			$litho_heartbeat_interval = (int) get_theme_mod( 'litho_heartbeat_interval', 60 );

			// If heartbeat control is enabled, set the interval.
			if ( '1' === $litho_control_heartbeat ) {
				$settings['interval'] = $litho_heartbeat_interval;
			}

			return $settings;
		}

		/**
		 * Prefetch DNS URLs.
		 *
		 * @param array  $hints Array of resources and their attributes.
		 * @param string $relation_type The relation type the URLs are printed for.
		 *                              One of 'dns-prefetch', 'preconnect', 'prefetch', or 'prerender'.
		 * @return array Modified hints array with added DNS prefetch URLs.
		 */
		public function litho_add_prefetch_dns_urls( $hints, $relation_type ) {
			// Only proceed if the relation type is 'dns-prefetch'.
			if ( 'dns-prefetch' !== $relation_type ) {
				return $hints;
			}

			// Retrieve the user-defined DNS URLs from the theme mod.
			$litho_prefetch_dns_urls = get_theme_mod( 'litho_prefetch_dns_urls', '' );

			// Decode the JSON data to an array.
			$urls = json_decode( $litho_prefetch_dns_urls, true );

			// Ensure the data is an array and iterate through it.
			if ( is_array( $urls ) ) {
				foreach ( $urls as $resource ) {
					// Validate each URL before adding it to the hints array.
					if ( isset( $resource['url'] ) && ! empty( $resource['url'] ) ) {
						$validated_url = esc_url( $resource['url'] );
						if ( ! empty( $validated_url ) ) {
							$hints[] = $validated_url;
						}
					}
				}
			}

			return $hints;
		}

		/**
		 * Preconnect URLs.
		 *
		 * This function adds URLs for preconnecting based on user-defined settings.
		 * The URLs should be provided in a newline-separated format, optionally with a "crossorigin" attribute.
		 *
		 * @param array  $hints Array of resources and their attributes.
		 * @param string $relation_type The relation type the URLs are printed for.
		 *                              One of 'dns-prefetch', 'preconnect', 'prefetch', or 'prerender'.
		 * @return array Modified hints array with added preconnect URLs.
		 */
		public function litho_add_preconnect_urls( $hints, $relation_type ) {
			// Only proceed if the relation type is 'preconnect'.
			if ( 'preconnect' !== $relation_type ) {
				return $hints;
			}

			// Retrieve the user-defined preconnect URLs.
			$litho_preconnect_urls = get_theme_mod( 'litho_preconnect_urls', '' );

			// Decode the JSON data to an array.
			$urls = json_decode( $litho_preconnect_urls, true );

			// Ensure the data is an array and iterate through it.
			if ( is_array( $urls ) ) {
				foreach ( $urls as $resource ) {
					// Validate each URL before adding it to the hints array.
					if ( isset( $resource['url'] ) && ! empty( $resource['url'] ) ) {
						$hint = [
							'href' => esc_url( $resource['url'] ),
						];

						// Add crossorigin attribute if set.
						if ( isset( $resource['isChecked'] ) && $resource['isChecked'] ) {
							$hint[] = 'crossorigin';
						}

						$hints[] = $hint;
					}
				}
			}

			return $hints;
		}

		/**
		 * Preload Resources like CSS, JS, Fonts URLs.
		 */
		public function litho_preload_resources() {
			$preload_resources_theme = get_theme_mod( 'litho_preload_resources', wp_json_encode( [] ) );
			$resources_array_theme   = json_decode( $preload_resources_theme, true );

			// Early return if no valid resources.
			if ( empty( $resources_array_theme ) ) {
				return;
			}

			$preload_tags        = [];
			$processed_resources = [];

			// Detect the current device type.
			$is_mobile = isset( $_SERVER['HTTP_USER_AGENT'] ) && preg_match( '/(android|webos|iphone|ipad|ipod|blackberry|iemobile|opera mini)/i', $_SERVER['HTTP_USER_AGENT'] ); // phpcs:ignore
			$current_device = $is_mobile ? 'mobile' : 'desktop';

			// Loop through all combined resources.
			foreach ( $resources_array_theme as $resource ) {
				$url    = '';
				$type   = '';
				$device = 'all'; // Default to all devices.

				// Check if resource is from customizer or meta box.
				if ( isset( $resource['url'], $resource['type'], $resource['device'] ) ) {
					$url    = esc_url( $resource['url'] );
					$type   = esc_html( $resource['type'] );
					$device = esc_html( $resource['device'] );
				} elseif ( isset( $resource['select'], $resource['textarea'], $resource['select_device'] ) ) {
					$url    = esc_url( trim( $resource['textarea'] ) );
					$type   = esc_attr( $resource['select'] );
					$device = esc_attr( $resource['select_device'] );
				}

				// Skip if the resource is for a different device.
				if ( 'all' !== $device && $device !== $current_device ) {
					continue;
				}

				// Generate a unique key for each resource based on URL and type.
				$resource_key = $url . '|' . $type;

				// Add the preload tag only if it hasn't been processed.
				if ( ! empty( $url ) && ! in_array( $resource_key, $processed_resources, true ) ) {
					$processed_resources[] = $resource_key;

					// Generate the appropriate preload tag based on the resource type.
					switch ( $type ) {
						case 'document':
							$preload_tags[] = "<link rel='preload' href='$url' as='document' />\n";
							break;
						case 'font':
							$preload_tags[] = "<link rel='preload' href='$url' as='font' type='font/woff2' crossorigin='anonymous' />\n";
							break;
						case 'image':
							$preload_tags[] = "<link rel='preload' href='$url' as='image' />\n";
							break;
						case 'script':
							$preload_tags[] = "<link rel='preload' href='$url' as='script' />\n";
							break;
						case 'style':
							$preload_tags[] = "<link rel='preload' href='$url' as='style' />\n";
							break;
					}
				}
			}

			echo implode( '', $preload_tags ); // phpcs:ignore
		}

		/**
		 * Set the post revisions limit.
		 *
		 * @param int     $num  Number of revisions to store.
		 * @param WP_Post $post Post object.
		 * @return int
		 **/
		public function litho_set_revisions_to_keep( $num, $post ) {
			$litho_post_revisions = get_theme_mod( 'litho_post_revisions', '' );

			// If no custom revisions limit is set, return the default.
			if ( '' === $litho_post_revisions ) {
				return $num;
			}

			$revisions_limit = (int) $litho_post_revisions;

			return ( $revisions_limit >= 0 ) ? $revisions_limit : $num;
		}

		/**
		 * Parse the url of script/style tags to remove the version query string.
		 *
		 * @param string $url  Script loader source path.
		 *
		 * @return string
		 */
		public function litho_remove_query_strings( $url ) {
			$parts = preg_split( '/\?ver|\?timestamp/', $url );
			return $parts[0];
		}

		/**
		 * Optimize WooCommerce cart fragments.
		 */
		public function litho_remove_cart_fragments() {
			// Do not run on admin pages.
			if ( is_admin() ) {
				return;
			}

			if ( ! is_woocommerce_activated() ) {
				return;
			}

			$litho_disable_cart_fragments = get_theme_mod( 'litho_disable_cart_fragments', '0' );
			if ( '1' === $litho_disable_cart_fragments ) {
				if ( ! is_cart() && ! is_checkout() ) {
					wp_dequeue_script( 'wc-cart-fragments' );
					wp_deregister_script( 'wc-cart-fragments' );
				}
			}
		}
	}

	new Litho_Performance_Manager();
}
