<?php
/**
 * Litho Helper
 *
 * Contains all the helping functions
 *
 * @package Litho
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/* Check if Elementor is active.*/
if ( ! function_exists( 'is_elementor_activated' ) ) {
	function is_elementor_activated() {
		return defined( 'ELEMENTOR_VERSION' ) ? true : false;
	}
}

/* Check if WooCommerce is Active.*/
if ( ! function_exists( 'is_woocommerce_activated' ) ) {
	function is_woocommerce_activated() {
		return class_exists( 'WooCommerce' ) ? true : false;
	}
}

/* Check if Contact Form 7 is Active. */
if ( ! function_exists( 'is_contact_form_7_activated' ) ) {
	function is_contact_form_7_activated() {
		return class_exists( 'WPCF7' ) ? true : false;
	}
}

// Check if mailchimp form is active.
if ( ! function_exists( 'is_mailchimp_form_activated' ) ) {
	function is_mailchimp_form_activated() {
		return class_exists( 'MC4WP_MailChimp' ) ? true : false;
	}
}

// Check if revolution slider is active.
if ( ! function_exists( 'is_revolution_slider_activated' ) ) {
	function is_revolution_slider_activated() {
		return class_exists( 'UniteFunctionsRev' ) ? true : false;
	}
}

// To get register sidebar list in metabox.
if ( ! function_exists( 'litho_register_sidebar_array' ) ) {
	function litho_register_sidebar_array() {
		global $wp_registered_sidebars;
		$sidebar_array = array();
		if ( ! empty( $wp_registered_sidebars ) && is_array( $wp_registered_sidebars ) ) {
			$sidebar_array['default'] = esc_html__( 'Default', 'litho-addons' );
			$sidebar_array['none']    = esc_html__( 'No Sidebar', 'litho-addons' );
			foreach ( $wp_registered_sidebars as $sidebar ) {
				$sidebar_array[ $sidebar['id'] ] = $sidebar['name'];
			}
		}
		return $sidebar_array;
	}
}

if ( ! class_exists( 'Litho_filesystem' ) ) {
	/**
	 * Filesystem
	 */
	final class Litho_filesystem {
		public static function init_filesystem() {
			// The WordPress filesystem.
			global $wp_filesystem;

			$credentials = [];

			if ( ! defined( 'FS_METHOD' ) ) {
				define( 'FS_METHOD', 'direct' );
			}

			$method = defined( 'FS_METHOD' ) ? FS_METHOD : false;

			if ( 'ftpext' === $method ) {
				// If defined, set it to that, Else, set to NULL.
				$credentials['hostname'] = defined( 'FTP_HOST' ) ? preg_replace( '|\w+://|', '', FTP_HOST ) : null;
				$credentials['username'] = defined( 'FTP_USER' ) ? FTP_USER : null;
				$credentials['password'] = defined( 'FTP_PASS' ) ? FTP_PASS : null;

				// Set FTP port.
				if ( strpos( $credentials['hostname'], ':' ) && null !== $credentials['hostname'] ) {
					list( $credentials['hostname'], $credentials['port'] ) = explode( ':', $credentials['hostname'], 2 );
					if ( ! is_numeric( $credentials['port'] ) ) {
						unset( $credentials['port'] );
					}
				} else {
					unset( $credentials['port'] );
				}
				// Set connection type.
				if ( ( defined( 'FTP_SSL' ) && FTP_SSL ) && 'ftpext' === $method ) {
					$credentials['connection_type'] = 'ftps';
				} elseif ( ! array_filter( $credentials ) ) {
					$credentials['connection_type'] = null;
				} else {
					$credentials['connection_type'] = 'ftp';
				}
			}

			if ( empty( $wp_filesystem ) ) {
				require_once wp_normalize_path( ABSPATH . '/wp-admin/includes/file.php' );
				WP_Filesystem( $credentials );
			}

			return $wp_filesystem;
		}
	}
}

if ( ! function_exists( 'litho_allow_mime_types' ) ) {
	/**
	 * Allow additional MIME types for upload.
	 *
	 * This function enables support for SVG, TTF, WOFF, WOFF2, and CSV file uploads
	 * based on the `litho_svg_support` theme setting.
	 *
	 * @param array $mimes Existing allowed MIME types.
	 * @return array Modified MIME types list.
	 */
	function litho_allow_mime_types( $mimes ) {
		$litho_svg_support = get_theme_mod( 'litho_svg_support', '0' );

		if ( '1' === $litho_svg_support ) {
			$mimes['svg']   = 'image/svg+xml';
			$mimes['svgz']  = 'image/svg+xml';
			$mimes['ttf']   = 'font/ttf';
			$mimes['woff']  = 'font/woff';
			$mimes['woff2'] = 'font/woff2';
			$mimes['csv']   = 'text/csv';
		}
		
		return $mimes;
	}
}
add_filter( 'upload_mimes', 'litho_allow_mime_types' );

if ( ! function_exists( 'litho_customize_save' ) ) {
	/**
	 * Clean up temporary font directory after Customizer save.
	 *
	 * Deletes the temporary font folder created during customization,
	 * if it resides within the expected uploads/litho-fonts path.
	 *
	 * Hooked into 'customize_save' to trigger cleanup after saving customizer options.
	 */
	function litho_customize_save() {
		$filesystem = Litho_filesystem::init_filesystem();
		$upload_dir = wp_upload_dir();
		$base_dir   = wp_normalize_path( untrailingslashit( $upload_dir['basedir'] ) );
		$fonts_dir  = $base_dir . '/litho-fonts';
		$temp_dir   = $fonts_dir . '/litho-temp-fonts';

		// Ensure directory exists and is inside uploads/litho-fonts
		if ( is_dir( $temp_dir ) && strpos( $temp_dir, $fonts_dir ) === 0 ) {
			$filesystem->delete( $temp_dir, FS_CHMOD_DIR );
		}
	}
}
add_action( 'customize_save', 'litho_customize_save' );

if ( ! function_exists( 'litho_customize_preview_init' ) ) {
	/**
	 * Create directories for custom fonts during Customizer preview initialization.
	 *
	 * This function processes fonts defined in the theme mods, moves them from a temporary
	 * location to their permanent directory in the uploads/litho-fonts folder, and
	 * deletes the temporary font directories afterward.
	 *
	 * Hooked to 'customize_controls_init' to run early in Customizer UI setup.
	 */
	function litho_customize_preview_init() {
		$custom_fonts_json = get_theme_mod( 'litho_custom_fonts', '' );
		$custom_fonts      = ! empty( $custom_fonts_json ) ? json_decode( $custom_fonts_json ) : [];

		if ( empty( $custom_fonts ) ) {
			return;
		}

		if ( ! is_array( $custom_fonts ) ) {
			return;
		}

		$filesystem = Litho_filesystem::init_filesystem();
		$upload_dir = wp_upload_dir();
		$base_dir   = untrailingslashit( wp_normalize_path( $upload_dir['basedir'] ) );
		$fonts_root = $base_dir . '/litho-fonts';
		$temp_root  = $fonts_root . '/litho-temp-fonts';

		foreach ( $custom_fonts as $font ) {
			if ( ! empty( $font[0] ) ) {
				$font_slug   = str_replace( ' ', '-', $font[0] );
				$target_dir  = $fonts_root . '/' . $font_slug;
				$source_dir  = $temp_root . '/' . $font_slug;

				// Ensure target directory exists
				if ( ! file_exists( $target_dir ) ) {
					wp_mkdir_p( $target_dir );
				}

				// Skip if source font directory doesn't exist
				if ( ! is_dir( $source_dir ) ) {
					continue;
				}

				// Copy fonts from temp to final directory
				copy_dir( $source_dir, $target_dir );

				// Clean up temp font folder
				if ( file_exists( $source_dir ) ) {
					$filesystem->delete( $source_dir, FS_CHMOD_DIR );
				}
			}
		}
	}
}
add_action( 'customize_controls_init', 'litho_customize_preview_init' );

// Customize custom fonts uploads Ajax callback
if ( ! function_exists( 'litho_upload_custom_font_action_data' ) ) {
	function litho_upload_custom_font_action_data() {
		/* Verify nonce */
		check_ajax_referer( 'litho_font_nonce', '_ajax_nonce' );

		/* Check current user permission */
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( esc_html__( 'Unauthorized request', 'litho-addons' ) );
		}

		// File check
		if ( ! isset( $_FILES['file'] ) || empty( $_FILES['file'] ) ) {
			wp_send_json_error( esc_html__( 'No file uploaded.', 'litho-addons' ) );
		}

		$file      = $_FILES['file'];
		$filename  = sanitize_file_name( $file['name'] );		
		$ext       = strtolower( pathinfo( $filename, PATHINFO_EXTENSION ) );
		$font_type = ( isset( $_POST['font_type'] ) && ! empty( $_POST['font_type'] ) ) ? $_POST['font_type'] : esc_html__( 'Current', 'litho-addons' );

		// Validate file extension against expected type
		if ( $ext !== $font_type ) {
			wp_send_json_error( sprintf(
				'%1$s %2$s %3$s',
				esc_html__( 'The file you are trying to upload is not a', 'litho-addons' ),
				esc_html( $font_type ),
				esc_html__( 'file. Please try again.', 'litho-addons' )
			) );
		}

		// Customize upload location
		add_filter( 'upload_dir', 'litho_custom_font_upload_dir' );
		$upload_result = wp_handle_upload( $file, [ 'test_form' => false ] );
		remove_filter( 'upload_dir', 'litho_custom_font_upload_dir' );

		// Check for upload errors
		if ( isset( $upload_result['error'] ) ) {
			wp_send_json_error( esc_html( $upload_result['error'] ) );
		}

		wp_send_json_success( [
			'message' => sprintf(
				'%1$s %2$s %3$s',
				esc_html__( 'Your', 'litho-addons' ),
				esc_html( $font_type ),
				esc_html__( 'file was uploaded.', 'litho-addons' )
			),
			'url' => esc_url_raw( $upload_result['url'] ),
		] );
	}
}
add_action( 'wp_ajax_litho_upload_custom_font_action_data', 'litho_upload_custom_font_action_data' );

if ( ! function_exists( 'litho_custom_font_upload_dir' ) ) {
	/**
	 * Customize the upload directory for custom fonts.
	 *
	 * This function modifies the upload path to store uploaded font files
	 * under a specific subdirectory based on the font family name.
	 *
	 * Example path: /wp-content/uploads/litho-fonts/FontFamily/
	 *
	 * @param array $path The original upload path array.
	 * @return array Modified upload path.
	 */
	function litho_custom_font_upload_dir( $path ) {
		// Return early if there was an upload error already.
		if ( ! empty( $path['error'] ) ) {
			return $path;
		}

		// Check if the font family is passed in the request.
		if ( isset( $_POST['fontFamily'] ) && ! empty( $_POST['fontFamily'] ) ) { // phpcs:ignore
			// Sanitize fontFamily input: allow only safe characters.
			$font_family = sanitize_file_name( $_POST['fontFamily'] ); // removes unsafe chars.
			$font_family = str_replace( ' ', '-', $font_family );

			// Validate directory name pattern (letters, numbers, dash, underscore).
			if ( ! preg_match( '/^[a-zA-Z0-9\-_]+$/', $font_family ) ) {
				return $path; // invalid directory name, fallback.
			}

			// Define custom subdirectory inside uploads folder.
			$customdir   = '/litho-fonts/' . $font_family; // phpcs:ignore

			$path['path'] = str_replace( $path['subdir'], '', $path['path'] );
			$path['url']  = str_replace( $path['subdir'], '', $path['url'] );
			$path['subdir'] = $customdir; // phpcs:ignore
			$path['path']  .= $customdir; // phpcs:ignore
			$path['url']   .= $customdir; // phpcs:ignore
		}

		return $path;
	}
}

if ( ! function_exists( 'is_section_builder_archive_template' ) ) {
	/**
	 * Check if the current context is using a Section Builder template of type "archive".
	 *
	 * This function determines whether the current page context (in frontend or Elementor edit/preview mode)
	 * is displaying an archive-related template created with the Section Builder.
	 *
	 * @return bool True if it's an archive template or relevant archive page, false otherwise.
	 */
	function is_section_builder_archive_template() {
		global $post;

		if ( is_category() || is_archive() || is_author() || is_tag() || is_search() || is_home() ) {
			return true;
		}

		// If post object is not available or invalid, it's not a Section Builder context.
		if ( empty( $post->ID ) || ! isset( $post->ID )) {
			return false;
		}

		$litho_get_template_type = litho_post_meta_by_id( $post->ID, 'litho_section_builder_template' );

		if ( 'sectionbuilder' === $post->post_type && ( \Elementor\Plugin::$instance->editor->is_edit_mode() || \Elementor\Plugin::$instance->preview->is_preview_mode() ) ) {

			if ( 'archive' === $litho_get_template_type ) {
				return true;
			} else {
				return false;
			}
		} else {
			return true;
		}
	}
}

if ( ! function_exists( 'is_section_builder_archive_portfolio_template' ) ) {
	/**
	 * Check if the current context is using a Section Builder template of type "archive-portfolio".
	 *
	 * This function determines whether the current page is related to portfolio archives or
	 * if the template loaded in Elementor editor/preview is set as a portfolio archive template.
	 *
	 * @return bool True if it's a portfolio archive or related template, false otherwise.
	 */
	function is_section_builder_archive_portfolio_template() {
		global $post;

		// Return true for portfolio archive conditions.
		if ( is_tax( 'portfolio-category' ) || is_tax( 'portfolio-tags' ) || is_post_type_archive( 'portfolio' ) ) {
			return true;
		}

		// If post object is not available or invalid, it's not a Section Builder context.
		if ( empty( $post->ID ) || ! isset( $post->ID )) {
			return false;
		}

		$litho_get_template_type = litho_post_meta_by_id( $post->ID, 'litho_section_builder_template' );

		if ( 'sectionbuilder' === $post->post_type && ( \Elementor\Plugin::$instance->editor->is_edit_mode() || \Elementor\Plugin::$instance->preview->is_preview_mode() ) ) {

			if ( 'archive-portfolio' === $litho_get_template_type ) {
				return true;
			} else {
				return false;
			}
		} else {
			return true;
		}
	}
}

if ( ! function_exists( 'is_section_builder_page_title_template' ) ) {
	/**
	 * Determine if the current context should use a Section Builder template of type "custom-title".
	 *
	 * This function checks whether the current page is a typical archive or taxonomy,
	 * or whether the Elementor editor/preview is loading a Section Builder template of type "custom-title".
	 *
	 * @return bool True if custom title template should be used, false otherwise.
	 */
	function is_section_builder_page_title_template() {
		global $post;

		// Return true for archive-related templates and WooCommerce pages.
		if ( is_category() || is_archive() || is_author() || is_tag() || is_search() || is_home() || is_tax( 'portfolio-category' ) || is_tax( 'portfolio-tags' ) || is_post_type_archive( 'portfolio' ) || ( is_woocommerce_activated() && ( is_shop() || is_tax( 'product_cat' ) || is_tax( 'product_tag' ) || is_tax( 'product_brand' ) ) ) ) {
			return true;
		}

		// If post object is not available or invalid, it's not a Section Builder context.
		if ( empty( $post->ID ) || ! isset( $post->ID )) {
			return false;
		}

		$litho_get_template_type = litho_post_meta_by_id( $post->ID, 'litho_section_builder_template' );

		if ( 'sectionbuilder' === $post->post_type && ( \Elementor\Plugin::$instance->editor->is_edit_mode() || \Elementor\Plugin::$instance->preview->is_preview_mode() ) ) {

			if ( 'custom-title' === $litho_get_template_type ) {
				return true;
			} else {
				return false;
			}
		} else {
			return true;
		}
	}
}

if ( ! function_exists( 'litho_get_template_type_by_key' ) ) {
	/**
	 * Retrieve Section Builder template type details.
	 *
	 * This function returns either the entire template type array or a specific template type detail,
	 * based on the provided key.
	 *
	 * @param string $template_type Optional. Template type key to fetch (e.g., 'custom-title').
	 * @return mixed Array of all template types or a single template type detail if key exists.
	 */
	function litho_get_template_type_by_key( $template_type = '' ) {
		$template_type_data = litho_section_builder_template_types();

		$template_type_data = ! empty( $template_type ) && isset( $template_type_data[ $template_type ] ) ? $template_type_data[ $template_type ] : $template_type_data;

		return $template_type_data;
	}
}

if ( ! function_exists( 'litho_section_builder_template_types' ) ) {
	/**
	 * Get all available Section Builder template types.
	 *
	 * Returns an associative array of Section Builder template type keys and their
	 * human-readable, translatable labels.
	 *
	 * @return array List of template types (key => label).
	 */
	function litho_section_builder_template_types() {
		return array(
			'mini-header'       => esc_html__( 'Mini Header', 'litho-addons' ),
			'header'            => esc_html__( 'Header', 'litho-addons' ),
			'footer'            => esc_html__( 'Footer', 'litho-addons' ),
			'archive'           => esc_html__( 'Archive', 'litho-addons' ),
			'archive-portfolio' => esc_html__( 'Archive Portfolio', 'litho-addons' ),
			'custom-title'      => esc_html__( 'Page Title', 'litho-addons' ),
			'promo_popup'       => esc_html__( 'Promo popup', 'litho-addons' ),
			'side_icon'         => esc_html__( 'Side Icon', 'litho-addons' ),
		);
	}
}

if ( ! function_exists( 'litho_get_header_sticky_type_by_key' ) ) {
	/**
	 * Get header sticky type label or full list.
	 *
	 * Returns the human-readable label for a given sticky header type key.
	 * If no key is provided, returns the full array of available sticky types.
	 *
	 * @param string $header_sticky_type Optional. Sticky type key (e.g. 'shrink-nav'). Default empty.
	 * @return string|array Sticky type label or array of all sticky types.
	 */
	function litho_get_header_sticky_type_by_key( $header_sticky_type = '' ) {
		$types = array(
			'appear-down-scroll' => esc_html__( 'Sticky on down scroll', 'litho-addons' ),
			'appear-up-scroll'   => esc_html__( 'Sticky on up scroll', 'litho-addons' ),
			'shrink-nav'         => esc_html__( 'Shrink', 'litho-addons' ),
			'no-sticky'          => esc_html__( 'Non sticky', 'litho-addons' ),
		);

		return $header_sticky_type && isset( $types[ $header_sticky_type ] )
			? $types[ $header_sticky_type ]
			: $types;
	}
}

if ( ! function_exists( 'litho_get_header_style_by_key' ) ) {
	/**
	 * Get header layout style label or the full list.
	 *
	 * Returns the display label for a given header style key.
	 * If no key is provided, the function returns the entire array of styles.
	 *
	 * @param string $header_style Optional. Header style key (e.g. 'left-menu-modern'). Default empty.
	 * @return string|array Header style label or array of all styles.
	 */
	function litho_get_header_style_by_key( $header_style = '' ) {
		$styles = array(
			'standard'          => esc_html__( 'Standard', 'litho-addons' ),
			'left-menu-classic' => esc_html__( 'Left Menu Classic', 'litho-addons' ),
			'left-menu-modern'  => esc_html__( 'Left Menu Modern', 'litho-addons' ),
		);

		return $header_style && isset( $styles[ $header_style ] )
			? $styles[ $header_style ]
			: $styles;
	}
}

// Return mini header sticky type.
if ( ! function_exists( 'litho_get_mini_header_sticky_type_by_key' ) ) {
	/**
	 * Get mini header sticky type label or the full list.
	 *
	 * Returns the display label for a given mini header sticky type key.
	 * If no key is provided, returns the full array of available options.
	 *
	 * @param string $mini_header_sticky_type Optional. Sticky type key (e.g. 'appear-up-scroll'). Default empty.
	 * @return string|array Sticky type label or array of all sticky types.
	 */
	function litho_get_mini_header_sticky_type_by_key( $mini_header_sticky_type = '' ) {
		$sticky_types = array(
			'appear-down-scroll' => esc_html__( 'Sticky on down scroll', 'litho-addons' ),
			'appear-up-scroll'   => esc_html__( 'Sticky on up scroll', 'litho-addons' ),
			'no-sticky'          => esc_html__( 'Non sticky', 'litho-addons' ),
		);

		return $mini_header_sticky_type && isset( $sticky_types[ $mini_header_sticky_type ] )
			? $sticky_types[ $mini_header_sticky_type ]
			: $sticky_types;
	}
}

// Return footer sticky type.
if ( ! function_exists( 'litho_get_footer_sticky_type_by_key' ) ) {
	/**
	 * Get footer sticky type label or the full list.
	 *
	 * Returns the display label for a given footer sticky type key.
	 * If no key is provided, returns the full array of available options.
	 *
	 * @param string $footer_sticky_type Optional. Sticky type key (e.g. 'sticky'). Default empty.
	 * @return string|array Sticky type label or array of all sticky types.
	 */
	function litho_get_footer_sticky_type_by_key( $footer_sticky_type = '' ) {
		$sticky_types = array(
			'no-sticky' => esc_html__( 'Non sticky', 'litho-addons' ),
			'sticky'    => esc_html__( 'Sticky', 'litho-addons' ),
		);
		return $footer_sticky_type && isset( $sticky_types[ $footer_sticky_type ] )
			? $sticky_types[ $footer_sticky_type ]
			: $sticky_types;
	}
}

if ( ! function_exists( 'litho_get_archive_style_by_key' ) ) {
	/**
	 * Get archive style label or the full list of styles.
	 *
	 * Returns the human-readable label for a specific archive style key.
	 * If no key is provided, returns the complete list of archive style options.
	 *
	 * @param string $archive_style Optional. Archive style key (e.g., 'category-archives'). Default empty.
	 * @return string|array Archive style label or array of all archive styles.
	 */
	function litho_get_archive_style_by_key( $archive_style = '' ) {
		$styles = array(
			'general'           => esc_html__( 'General', 'litho-addons' ),
			'category-archives' => esc_html__( 'Category', 'litho-addons' ),
			'tag-archives'      => esc_html__( 'Tag', 'litho-addons' ),
			'author-archives'   => esc_html__( 'Author', 'litho-addons' ),
			'search-archives'   => esc_html__( 'Search', 'litho-addons' ),
		);
		
		return $archive_style && isset( $styles[ $archive_style ] )
			? $styles[ $archive_style ]
			: $styles;
	}
}

if ( ! function_exists( 'litho_get_archive_portfolio_style_by_key' ) ) {
	/**
	 * Get portfolio archive style label or the full list of styles.
	 *
	 * Returns the label for a specific portfolio archive style key.
	 * If no key is provided, returns all portfolio archive styles.
	 *
	 * @param string $archive_portfolio_style Optional. Archive style key (e.g., 'portfolio-cat-archives'). Default empty.
	 * @return string|array Portfolio archive style label or array of all styles.
	 */
	function litho_get_archive_portfolio_style_by_key( $archive_portfolio_style = '' ) {
		$styles = array(
			'general'                 => esc_html__( 'General', 'litho-addons' ),
			'portfolio-cat-archives'  => esc_html__( 'Portfolio Category', 'litho-addons' ),
			'portfolio-tags-archives' => esc_html__( 'Portfolio Tags', 'litho-addons' ),
			'portfolio-archives'      => esc_html__( 'Portfolio Archive', 'litho-addons' ),
		);

		return $archive_portfolio_style && isset( $styles[ $archive_portfolio_style ] )
			? $styles[ $archive_portfolio_style ]
			: $styles;
	}
}

if ( ! function_exists( 'litho_get_builder_section_data' ) ) {
	/**
	 * Get a list of Section Builder templates by section type.
	 *
	 * @param string  $section_type Required. The value of the '_litho_section_builder_template' meta key to filter templates.
	 * @param boolean $meta         Optional. Whether to include a 'Default' option (for meta fields). Default false.
	 * @param boolean $general      Optional. Whether to include a 'General' option (for general UI). Default false.
	 * @return array                Associative array of template ID => title, or empty if none found.
	 */
	function litho_get_builder_section_data( $section_type = '', $meta = false, $general = false ) {
		// Return early if the template type is empty.
		if ( empty( $section_type ) ) {
			return [];
		}

		// Base options depending on context.
		$builder_section_data = [];

		if ( $meta ) {
			$builder_section_data['default'] = esc_html__( 'Default', 'litho' );
		} elseif ( $general ) {
			$builder_section_data[''] = esc_html__( 'General', 'litho' );
		} else {
			$builder_section_data[''] = esc_html__( 'Select', 'litho' );
		}

		// Prepare the meta query based on the template type.
		$litho_filter_section = [
			'key'     => '_litho_section_builder_template',
			'value'   => $section_type, // phpcs:ignore
			'compare' => '=',
		];

		$posts = get_posts( [
			'post_type'      => 'sectionbuilder',
			'post_status'    => 'publish',
			'posts_per_page' => -1,
			'no_found_rows'  => true,
			'meta_query'     => [
				$litho_filter_section,
			],
		] );

		// Add each found post's ID and title to the return array.
		if ( ! empty( $posts ) ) {
			foreach ( $posts as $post ) {
				$builder_section_data[ $post->ID ] = esc_html( $post->post_title );
			}
		}

		return $builder_section_data;
	}
}

if ( ! function_exists( 'is_sectionbuilder_screen' ) ) {
	/**
	 * Check if the current admin screen is for the Section Builder post type.
	 *
	 * @return bool True if editing or listing 'sectionbuilder' post type, false otherwise.
	 */
	function is_sectionbuilder_screen() {
		global $pagenow, $typenow;

		if ( 'sectionbuilder' === $typenow && ( 'edit.php' === $pagenow || 'post.php' === $pagenow ) ) {
			return true;
		}
		return false;
	}
}

if ( ! function_exists( 'litho_save_mega_menu_settings' ) ) {
	/**
	 * AJAX callback to save mega menu settings for a specific menu item.
	 *
	 * @return void Sends a JSON response on success or error.
	 */
	function litho_save_mega_menu_settings() {

		// Ensure the user has permission to manage options.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error(
				array(
					'message' => esc_html__( 'You are not allowed to do this.', 'litho-addons' ),
				)
			);
		}

		$current_menu_itemt_id = ( isset( $_POST['current_menu_itemt_id'] ) ) ? absint( $_POST['current_menu_itemt_id'] ) : false;
		if ( ! $current_menu_itemt_id ) {
			wp_send_json_error(
				array(
					'message' => esc_html__( 'Invalid menu item ID.', 'litho-addons' ),
				)
			);
		}

		$enable_mega_submenu = isset( $_POST['enable_mega_submenu'] ) ? $_POST['enable_mega_submenu'] : false; // phpcs:ignore
		update_post_meta( $current_menu_itemt_id, '_enable_mega_submenu', $enable_mega_submenu );

		// List of fields to update.
		$fields_to_update = array(
			'_menu_item_icon'          => isset( $_POST['menu-item-icon'] ) && ! empty( $_POST['menu-item-icon'] ) ? $_POST['menu-item-icon'] : NULL, // phpcs:ignore
			'_menu_item_icon_position' => isset( $_POST['menu-item-icon-position'] ) && ! empty( $_POST['menu-item-icon-position'] ) ? $_POST['menu-item-icon-position'] : NULL, // phpcs:ignore
			'_menu_item_icon_color'    => isset( $_POST['menu-item-icon-color'] ) && ! empty( $_POST['menu-item-icon-color'] ) ? $_POST['menu-item-icon-color'] : NULL, // phpcs:ignore
		);

		// Update post meta for each field.
		foreach ( $fields_to_update as $meta_key => $meta_value ) {
			if ( NULL !== $meta_value ) {
				update_post_meta( $current_menu_itemt_id, $meta_key, $meta_value );
			}
		}

		wp_send_json_success(
			array(
				'message' => esc_html__( 'Success!', 'litho-addons' ),
			)
		);
		
		die();
	}
}
add_action( 'wp_ajax_litho_save_mega_menu_settings', 'litho_save_mega_menu_settings' );

if ( ! function_exists( 'litho_section_builder_lightbox' ) ) {
	/**
	 * AJAX callback to create a new SectionBuilder template and return its Elementor edit URL.
	 */
	function litho_section_builder_lightbox() {
		check_ajax_referer( 'litho_builder_nonce', 'security' );

		// Validate required parameter
		if ( ! isset( $_REQUEST['sectionbuilder_template_type'] ) && empty( $_REQUEST['sectionbuilder_template_type'] ) ) {
			return;
		}

		$template_type              = $_REQUEST['sectionbuilder_template_type'];
		$template_style             = ( isset( $_REQUEST['sectionbuilder_template_style'] ) && ! empty( $_REQUEST['sectionbuilder_template_style'] ) ) ? $_REQUEST['sectionbuilder_template_style'] : 'standard';
		$template_title             = ( isset( $_REQUEST['sectionbuilder_template_title'] ) && ! empty( $_REQUEST['sectionbuilder_template_title'] ) ) ? $_REQUEST['sectionbuilder_template_title'] : '';
		$template_archive           = ( isset( $_REQUEST['sectionbuilder_template_archive'] ) && ! empty( $_REQUEST['sectionbuilder_template_archive'] ) ) ? $_REQUEST['sectionbuilder_template_archive'] : '';
		$template_archive_portfolio = ( isset( $_REQUEST['sectionbuilder_template_archive_portfolio'] ) && ! empty( $_REQUEST['sectionbuilder_template_archive_portfolio'] ) ) ? $_REQUEST['sectionbuilder_template_archive_portfolio'] : '';

		$meta_input = array(
			'_litho_section_builder_template' => $template_type
		);

		if ( 'header' === $template_type ) {
			$meta_input['_litho_template_header_style'] = $template_style;
		}
		if ( 'archive' === $template_type ) {
			$meta_input['_litho_template_archive_style'] = $template_archive;
		}

		if ( 'archive-portfolio' === $template_archive_portfolio ) {
			$meta_input['_litho_template_archive_portfolio_style'] = $template_archive_portfolio;
		}

		// Insert new SectionBuilder post.
		$post_id = wp_insert_post(
			array(
				'post_title' => $template_title,
				'post_type'  => 'sectionbuilder',
				'meta_input' => $meta_input,
			),
			true
		);

		// If no title provided, auto-generate one using post ID.
		if ( empty( $template_title ) ) {
			$post_data = array(
				'ID'          => $post_id,
				'post_title'  => sprintf(
					'%1$s %2$s %3$s%4$s',
					esc_html__( 'Section', 'litho-addons' ),
					ucfirst( esc_html( $template_type ) ),
					'#',
					esc_html( $post_id )
				)
			);
			wp_update_post( $post_data );
		}

		$edit_url = add_query_arg(
			array(
				'post'   => $post_id,
				'action' => 'elementor',
			),
			admin_url( 'edit.php' )
		);

		printf( $edit_url );
		die();
	}
}
add_action( 'wp_ajax_litho_section_builder_lightbox', 'litho_section_builder_lightbox' );

if ( ! function_exists( 'litho_admin_new_template_lightbox' ) ) {
	/**
	 * Section Builder: Show popup to add new template in admin.
	 */
	function litho_admin_new_template_lightbox() {
		if ( ! is_sectionbuilder_screen() ) {
			return;
		}
		
		global $post, $pagenow;

		$litho_current_template          = '';
		$litho_current_header            = '';
		$litho_current_archive           = [];
		$litho_current_archive_portfolio = [];

		if ( 'post.php' === $pagenow && isset( $post->ID ) ) {
			$litho_current_template          = get_post_meta( $post->ID, '_litho_section_builder_template', true );
			$litho_current_header            = get_post_meta( $post->ID, '_litho_template_header_style', true );
			$litho_current_archive           = get_post_meta( $post->ID, '_litho_template_archive_style', true );
			$litho_current_archive_portfolio = get_post_meta( $post->ID, '_litho_template_archive_portfolio_style', true );
		}
		?>
		<div class="sectionbuilder-outer">
			<div class="sectionbuilder-inner">
				<button class="close"><?php echo esc_html__( 'X', 'litho-addons' ); ?></button>
				<form class="sectionbuilder-new-template-form">
					<div class="sectionbuilder-new-template-form-title">
						<?php echo esc_html__( 'Choose Template Type', 'litho-addons' ); ?>
					</div>
					<div class="sectionbuilder-form-field">
						<div class="template-form-field">
							<label for="sectionbuilder-template-type" class="sectionbuilder-new-template-form-label">
								<?php echo esc_html__( 'Select the type of template', 'litho-addons' ); ?>
							</label>
							<div class="sectionbuilder-form-field-select-wrapper">
								<select id="sectionbuilder-template-type" class="sectionbuilder-form-field-select sectionbuilder-dropdown" name="template_type" required="required" autocomplete="off">
								<?php
								$litho_types = litho_section_builder_template_types();
								foreach ( $litho_types as $key => $value ) {
									if ( ! empty( $litho_current_template ) && 'post.php' === $pagenow ) {
										$selected = ( $litho_current_template === $key ) ? ' selected="selected"' : '';
									} else {
										$selected = ( 'header' === $key ) ? ' selected="selected"' : '';
									}
									?>
									<option value="<?php echo esc_attr( $key ); ?>"<?php echo esc_attr( $selected ); ?>><?php echo esc_attr( $value ); ?></option>
									<?php
								}
								?>
								</select>
							</div>
						</div>
						<div class="header-form-field input-field-control">
							<label for="sectionbuilder-template-style" class="sectionbuilder-new-template-form-label">
								<?php echo esc_html__( 'Select the style of header', 'litho-addons' ); ?>
							</label>
							<div class="sectionbuilder-form-field-select-wrapper">
								<select id="sectionbuilder-template-style" class="sectionbuilder-form-field-select-style sectionbuilder-dropdown" name="template_style" required="required">
								<?php
								$litho_header_style = litho_get_header_style_by_key();
								foreach ( $litho_header_style as $key => $value ) {
									if ( ! empty( $litho_current_header ) && 'post.php' === $pagenow ) {
										$selected = ( $litho_current_header === $key ) ? ' selected="selected"' : '';
									} else {
										$selected = ( 'standard' === $key ) ? ' selected="selected"' : '';
									}
									?>
									<option value="<?php echo esc_attr( $key ); ?>"<?php echo esc_attr( $selected ); ?>><?php echo esc_attr( $value ); ?></option>
									<?php
								}
								?>
								</select>
							</div>
						</div>
						<div class="archive-form-field input-field-control" style="display: none;">
							<label for="sectionbuilder-archive-style" class="sectionbuilder-new-template-form-label">
								<?php echo esc_html__( 'Select archive', 'litho-addons' ); ?>
							</label>
							<div class="sectionbuilder-form-field-select-wrapper">
								<select id="sectionbuilder-archive-style" class="sectionbuilder-form-field-archive-style litho-dropdown-select2 sectionbuilder-dropdown" name="archive_style[]" multiple="multiple" style="width:99%;max-width:25em;">
									<?php
									$litho_archive_style = litho_get_archive_style_by_key();
									foreach ( $litho_archive_style as $key => $value ) {
										if ( ! empty( $litho_current_archive ) && is_array( $litho_current_archive ) && 'post.php' === $pagenow ) {
											$selected = ( in_array( $key, $litho_current_archive ) ) ? ' selected="selected"' : '';
										} else {
											$selected = ( 'general' === $key ) ? ' selected="selected"' : '';
										}
										?>
										<option value="<?php echo esc_attr( $key ); ?>"<?php echo esc_attr( $selected ); ?>><?php echo esc_attr( $value ); ?></option>
										<?php
									}
									?>
								</select>
							</div>
						</div>
						<div class="archive-portfolio-form-field input-field-control" style="display: none;">
							<label for="sectionbuilder-archive-portfolio-style" class="sectionbuilder-new-template-form-label">
								<?php echo esc_html__( 'Select archive portfolio', 'litho-addons' ); ?>
							</label>
							<div class="sectionbuilder-form-field-select-wrapper">
								<select id="sectionbuilder-archive-portfolio-style" class="sectionbuilder-form-field-archive-portfolio-style litho-dropdown-select2 sectionbuilder-dropdown" name="archive_portfolio_style[]" multiple="multiple" style="width:99%;max-width:25em;">
									<?php
									$litho_archive_portfolio_style = litho_get_archive_portfolio_style_by_key();
									foreach ( $litho_archive_portfolio_style as $key => $value ) {
										if ( ! empty( $litho_current_archive_portfolio ) && is_array( $litho_current_archive_portfolio ) && 'post.php' === $pagenow ) {
											$selected = ( in_array( $key, $litho_current_archive_portfolio ) ) ? ' selected="selected"' : '';
										} else {
											$selected = ( 'general' === $key ) ? ' selected="selected"' : '';
										}
										?>
										<option value="<?php echo esc_attr( $key ); ?>"<?php echo esc_attr( $selected ); ?>><?php echo esc_attr( $value ); ?></option>
										<?php
									}
									?>
								</select>
							</div>
						</div>
					</div>
					<div class="sectionbuilder-form-field">
						<label for="sectionbuilder-new-template-form-post-title" class="sectionbuilder-new-template-form-label">
							<?php echo esc_html__( 'Name your template', 'litho-addons' ); ?>
						</label>
						<input type="text" placeholder="<?php echo esc_attr__( 'Enter template name (optional)', 'litho-addons' ); ?>" id="sectionbuilder-new-template-form-post-title" class="sectionbuilder-new-template-form-post-title" name="sectionbuilder-new-template-post-title">
					</div>
					<button id="sectionbuilder-form-submit" class="create-template-button"><?php echo esc_html__( 'Create Template', 'litho-addons' ); ?></button>
				</form>
			</div>
		</div>
		<?php
	}
}
add_action( 'admin_footer', 'litho_admin_new_template_lightbox' );

if ( ! function_exists( 'litho_add_portfolio_meta_box' ) ) {
	/**
	 * Register a custom meta box for alternate image in Portfolio post type.
	 */
	function litho_add_portfolio_meta_box() {
		add_meta_box(
			'litho-portfolio-alternate-image',
			esc_html__( 'Alternate image', 'litho-addons' ),
			'litho_portfolio_alternate_image_content',
			'portfolio',
			'side',
			'high'
		);
	}
}
add_action( 'add_meta_boxes', 'litho_add_portfolio_meta_box' );

if ( ! function_exists( 'litho_portfolio_alternate_image_content' ) ) {
	/**
	 * Renders the meta box UI for selecting an alternate image on Portfolio post type.
	 *
	 * @param WP_Post $post The current post object.
	 */
	function litho_portfolio_alternate_image_content( $post ) {
		$field_id         = 'portfolio_alternate_image';
		$alternate_img_id = get_post_meta( $post->ID, '_litho_portfolio_alternate_image_single', true );
		$nonce            = wp_create_nonce( $field_id . $post->ID );

		if ( $alternate_img_id ) {
			$link_title         = wp_get_attachment_image( $alternate_img_id, 'full', false, array( 'style' => 'width:100%; height:auto;' ) );
			$hide_remove_button = '';
		} else {
			$alternate_img_id   = -1;
			$link_title         = esc_html__( 'Add alternate portfolio image', 'litho-addons' );
			$hide_remove_button = 'display: none;';
		}
		?>
		<p class="hide-if-no-js portfolio-alternate-image-container-<?php echo esc_attr( $field_id ); ?>">
			<a href="#" class="portfolio-alternate-add-media portfolio-alternate-media-edit portfolio-alternate-media-edit-<?php echo esc_attr( $field_id ); ?>" data-title="<?php echo esc_html__( 'Alternate image', 'litho-addons' ); ?>" data-button="<?php echo esc_html__( 'Use as alternate portfolio image', 'litho-addons' ); ?>" data-id="<?php echo esc_attr( $field_id ); ?>" data-nonce="<?php echo esc_attr( $nonce ); ?>" data-postid="<?php echo esc_attr( $post->ID ); ?>" style="display: inline-block;">
				<?php echo wp_kses_post( $link_title ); ?>
			</a>
		</p>
		<p class="hide-if-no-js howto" style="<?php echo esc_attr( $hide_remove_button ); ?>"><?php echo esc_html__( 'Click the image to edit or update', 'litho-addons' ); ?></p>

		<p class="hide-if-no-js hide-if-no-image-<?php echo esc_attr( $field_id ); ?>" style="<?php echo esc_attr( $hide_remove_button ); ?>">
			<a href="#" class="portfolio-alternate-media-delete portfolio-alternate-media-delete-<?php echo esc_attr( $field_id ); ?>" data-title="<?php echo esc_html__( 'Alternate image', 'litho-addons' ); ?>" data-button="<?php echo esc_html__( 'Use as alternate portfolio image', 'litho-addons' ); ?>" data-id="<?php echo esc_attr( $field_id ); ?>" data-nonce="<?php echo esc_attr( $nonce ); ?>" data-postid="<?php echo esc_attr( $post->ID ); ?>" data-label_set="<?php echo esc_html__( 'Add alternate portfolio image', 'litho-addons' ); ?>">
				<?php echo esc_html__( 'Remove alternate portfolio image', 'litho-addons' ); ?>
			</a>
		</p>
		<?php
	}
}

if ( ! function_exists( 'litho_ajax_set_portfolio_alternate_image' ) ) {
	/**
	 * AJAX handler to set or update the alternate image for a portfolio post.
	 *
	 */
	function litho_ajax_set_portfolio_alternate_image() {
		$alt_img_id = isset( $_POST['alt_img_id'] ) ? absint( $_POST['alt_img_id'] ) : 0;
		$postid     = isset( $_POST['postid'] ) ? absint( $_POST['postid'] ) : 0;
		$field_id   = isset( $_POST['id'] ) ? $_POST['id'] : '';

		// Verify nonce.
		check_ajax_referer( $field_id . $postid, 'sec' );

		if ( $alt_img_id && wp_attachment_is_image( $alt_img_id ) ) {
			update_post_meta( $postid, '_litho_portfolio_alternate_image_single', $alt_img_id );

			echo wp_get_attachment_image(
				$alt_img_id,
				'full',
				false,
				array(
					'style' => 'width:100%; height:auto;',
				)
			);
			
		}

		wp_die();
	}
}
add_action( 'wp_ajax_set_portfolio_alternate_image', 'litho_ajax_set_portfolio_alternate_image' );

if ( ! function_exists( 'litho_ajax_remove_portfolio_alternate_image' ) ) {
	/**
	 * AJAX handler to set or remove the alternate image for a portfolio post.
	 *
	 */
	function litho_ajax_remove_portfolio_alternate_image() {
		$postid    = isset( $_POST['postid'] ) ? absint( $_POST['postid'] ) : 0;
		$label_set = isset( $_POST['label_set'] ) ? $_POST['label_set'] : '';
		$field_id  = isset( $_POST['id'] ) ? $_POST['id'] : '';

		// Verify nonce.
		check_ajax_referer( $field_id . $postid, 'sec' );

		// Delete the alternate image meta.
		delete_post_meta( $postid, '_litho_portfolio_alternate_image_single' );

		// Send the fallback label (e.g., "Add image" button label).
		echo esc_attr( $label_set );

		wp_die();
	}
}
add_action( 'wp_ajax_remove_portfolio_alternate_image', 'litho_ajax_remove_portfolio_alternate_image' );

if ( ! function_exists( 'litho_addressbar_color_wp_head_meta' ) ) {
	/**
	 * Outputs meta tags for customizing the address bar color in supported browsers
	 * and sets compatibility for Internet Explorer.
	 */
	function litho_addressbar_color_wp_head_meta() {

		$litho_addressbar_color = get_theme_mod( 'litho_addressbar_color', '' );

		if ( ! empty( $litho_addressbar_color ) ) {

			//this is for Chrome, Firefox OS, Opera
			echo '<meta name="theme-color" content="' . $litho_addressbar_color . '">'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			//Windows Phone **
			echo '<meta name="msapplication-navbutton-color" content="' . $litho_addressbar_color . '">'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		}

		if ( isset( $_SERVER['HTTP_USER_AGENT'] ) && ( strpos( $_SERVER['HTTP_USER_AGENT'], 'MSIE' ) !== false ) ) {
			echo '<meta http-equiv="X-UA-Compatible" content="IE=edge,chrome=1" />';
		}
	}
}
add_action( 'wp_head', 'litho_addressbar_color_wp_head_meta' );

if ( ! function_exists( 'litho_custom_hover_animation_effect' ) ) {
	/**
	 * Returns an array of supported custom hover animation effects for buttons.
	 *
	 * @return array List of CSS class names for hover effects.
	 */
	function litho_custom_hover_animation_effect() {
		$effects = array(
			'btn-slide-up-bg',
			'btn-slide-down-bg',
			'btn-slide-left-bg',
			'btn-slide-right-bg',
			'btn-expand-ltr',
		);

		/**
		 * Filter the list of hover animation effects.
		 *
		 * @param array $effects Array of animation effect class names.
		 */
		return apply_filters( 'litho_button_hover_animation_effect', $effects );
	}
}

if ( ! function_exists( 'litho_post_meta' ) ) {
	/**
	 * Return post meta value with standardized key format.
	 *
	 * @param string $option  The meta key suffix (without prefix and _single).
	 *
	 * @return mixed|null Meta value or null if not found.
	 */
	function litho_post_meta( $option ) {
		global $post;

		if ( empty( $post->ID ) ) {
			return;
		}

		$meta_key = '_' . $option . '_single';

		return get_post_meta( $post->ID, $meta_key, true );
	}
}

if ( ! function_exists( 'litho_post_meta_by_id' ) ) {
	/**
	 * Retrieves post meta value by post ID and meta key suffix.
	 *
	 * Automatically prefixes the meta key with an underscore for consistency.
	 *
	 * @param int    $post_id    The ID of the post.
	 * @param string $key_suffix The unique part of the meta key (without underscore).
	 *
	 * @return mixed|null The meta value if found, otherwise null.
	 */
	function litho_post_meta_by_id( $post_id, $option ) {
		if ( ! $post_id ) {
			return;
		}

		$meta_key = '_' . $option;

		return get_post_meta( $post_id, $meta_key, true );
	}
}

if ( ! function_exists( 'litho_get_post_types' ) ) {
	/**
	 * Get public post types for Litho, excluding certain types.
	 *
	 * @param array $args Additional arguments to customize the post type query.
	 */
	function litho_get_post_types( $args = [] ) {
		// Default arguments to filter post types that show in navigation menus.
		$post_type_args = [
			// Default is the value $public.
			'show_in_nav_menus' => true,
		];

		// Keep for backwards compatibility.
		if ( ! empty( $args['post_type'] ) ) {
			$post_type_args['name'] = $args['post_type'];
			unset( $args['post_type'] );
		}

		// Merge passed arguments with the default ones.
		$post_type_args = wp_parse_args( $post_type_args, $args );

		// Get all post types based on the query arguments.
		$litho_post_types = get_post_types( $post_type_args, 'objects' );

		$post_types = [];
		foreach ( $litho_post_types as $post_type => $object ) {
			if ( 'e-landing-page' !== $post_type ) {
				$post_types[ $post_type ] = $object->label;
			}
		}

		/**
		 * Apply filters for get all post types so user can add its post types.
		 *
		 * @since 1.0
		 */
		return apply_filters( 'litho_get_public_post_types', $post_types );
	}
}

if ( ! function_exists( 'litho_post_category_array' ) ) {
	/**
	 * Retrieve all WordPress post categories as an associative array.
	 *
	 * Returns an array of category slugs as keys and their corresponding names as values.
	 * This is useful for dropdowns, filters, and taxonomy selectors.
	 *
	 * @return array Associative array of category slug => name.
	 */
	function litho_post_category_array() {
		$args = array(
			'hide_empty' => true,
			'orderby'    => 'name',
			'order'      => 'ASC',
		);

		$categories = get_categories( $args );

		$categories_array = [];
		if ( ! empty( $categories ) && ! is_wp_error( $categories ) ) {
			foreach ( $categories as $category ) {
				$categories_array[ $category->slug ] = $category->name;
			}
		}

		return $categories_array;
	}
}

if ( ! function_exists( 'litho_single_post_meta_category' ) ) {
	/**
	 * Output post categories in a list item, optionally with an icon.
	 *
	 * Displays categories for standard 'post' type only.
	 * The category output is generated via `litho_post_category()` and can be filtered.
	 *
	 * @param int  $id   Post ID.
	 * @param bool $icon Whether to prepend a folder icon.
	 */
	function litho_single_post_meta_category( $id, $icon = false ) {
		if ( '' == $id ) {
			return;
		}

		if ( 'post' !== get_post_type( $id ) ) {
			return;
		}

		
		$litho_term_limit = apply_filters( 'litho_single_post_category_limit', '40' );
		$category_list    = litho_post_category( $id, true, $litho_term_limit );

		$icon_html = $icon ? '<i class="feather icon-feather-folder text-fast-blue"></i>' : '';

		if ( $category_list ) {
			printf( '<li>%1$s%2$s</li>', $icon_html, $category_list );
		}
	}
}

if ( ! function_exists( 'litho_post_exists' ) ) {
	/**
	 * Check if a post exists and is valid.
	 *
	 * @param int|string $id Post ID.
	 * @return bool True if the post exists, false otherwise.
	 */
	function litho_post_exists( $id ) {
		if ( '' == $id ) {
			return false;
		}

		if ( is_string( get_post_status( $id ) ) ) {
			return true;
		}
		return false;
	}
}

if ( ! function_exists( 'litho_post_category' ) ) {
	/**
	 * Get formatted post categories or tags as linked items.
	 *
	 * @param int    $id        Post ID.
	 * @param bool   $separator Whether to separate terms with comma.
	 * @param int    $count     Max number of terms to return.
	 * @param string $terms     Term type: 'category' or 'tag'.
	 * @return string           HTML string of term links.
	 */
	function litho_post_category( $id, $separator = true, $count = '10', $terms = 'category' ) {
		if ( '' == $id ) {
			return '';
		}

		if ( 'post' !== get_post_type( $id ) ) {
			return '';
		}

		$post_term_arr = [];
		if ( 'category' === $terms ) {
			$categories       = get_the_category( $id );
			$category_counter = 0;
			if ( ! empty( $categories ) ) {
				foreach ( $categories as $cat ) {
					if ( $count == $category_counter ) {
						break;
					}

					$cat_link        = get_category_link( $cat->cat_ID );
					$post_term_arr[] = '<a rel="category tag" href="' . esc_url( $cat_link ) . '">' . esc_html( $cat->name ) . '</a>';
					$category_counter++;
				}
			}
		} else {
			$tags         = get_the_tags( $id );
			$tags_counter = 0;
			if ( ! empty( $tags ) ) {
				foreach ( $tags as $tag ) {
					if ( $count == $tags_counter ) {
						break;
					}
					$tag_link        = get_tag_link( $tag->term_id );
					$post_term_arr[] = '<a rel="category tag" href="' . esc_url( $tag_link ) . '">' . esc_html( $tag->name ) . '</a>';
					$tags_counter++;
				}
			}
		}

		if ( empty( $post_term_arr ) ) {
			return '';
		}

		$post_terms = $separator ? implode( ', ', $post_term_arr ) : implode( '', $post_term_arr );
		
		return $post_terms;
	}
}

if ( ! function_exists( 'litho_post_tags_array' ) ) {
	/**
	 * Returns an associative array of post tag slugs => names.
	 *
	 * Useful for generating tag filter options or selects.
	 *
	 * @return array Associative array of tag slugs as keys and tag names as values.
	 */
	function litho_post_tags_array() {
		$tags = get_terms(
			array(
				'taxonomy'   => 'post_tag',
				'hide_empty' => true,
				'orderby'    => 'name',
				'order'      => 'ASC',
			)
		);

		$tags_array = [];
		if ( ! empty( $tags ) && ! is_wp_error( $tags ) ) {
			foreach ( $tags as $tag ) {
				$tags_array[ $tag->slug ] = $tag->name;
			}
		}

		return $tags_array;
	}
}

if ( ! function_exists( 'litho_post_array' ) ) {
	/**
	 * Returns an associative array of post IDs => post titles.
	 *
	 * Retrieves all published posts of specified types for use in dropdowns or filters.
	 * Defaults to `post`, `page`, and `portfolio`, but can be modified via a filter.
	 *
	 * @return array Associative array where keys are post IDs and values are post titles.
	 */
	function litho_post_array() {
		// Allowed post types (can be filtered).
		$post_types = apply_filters( 'litho_exclude_post_type_list', [ 'post', 'page', 'portfolio' ] );

		$query_args = array(
			'post_type'      => $post_types, // phpcs:ignore
			'posts_per_page' => -1,
			'post_status'    => 'publish',
			'no_found_rows'  => true,
			'fields'         => 'ids',
		);

		$post_ids = get_posts( $query_args );

		$post_array = [];
		if ( ! empty( $post_ids ) ) {
			foreach ( $post_ids as $post_id ) {
				$title = get_the_title( $post_id );

				$post_array[ $post_id ] = $title ? $title : esc_html__( '(no title)', 'litho-addons' );
			}
		}
		return $post_array;
	}
}

if ( ! function_exists( 'litho_portfolio_array' ) ) {
	/**
	 * Returns an associative array of published portfolio post IDs and their titles.
	 *
	 * Uses a direct database query via $wpdb for better performance in large datasets.
	 *
	 * @return array Associative array of [ post_id => post_title ].
	 */
	function litho_portfolio_array() {
		global $wpdb;

		// phpcs:ignore
		$results = $wpdb->get_results( "
			SELECT ID, post_title
			FROM {$wpdb->posts}
			WHERE post_type = 'portfolio'
			AND post_status = 'publish'
			ORDER BY post_title ASC
		", OBJECT_K ); // phpcs:ignore

		$post_array = [];
		foreach ( $results as $id => $row ) {
			$post_array[ $id ] = ! empty( $row->post_title ) ? $row->post_title : esc_html__( '(no title)', 'litho-addons' );
		}

		return $post_array;
	}
}

if ( ! function_exists( 'litho_portfolio_category_array' ) ) {
	/**
	 * Returns an associative array of portfolio category slugs and names.
	 *
	 * Fetches all terms from the 'portfolio-category' taxonomy.
	 *
	 * @return array Associative array of [ slug => name ] for each category.
	 */
	function litho_portfolio_category_array() {
		$categories = get_terms(
			array(
				'taxonomy'   => 'portfolio-category',
				'hide_empty' => true,
			)
		);

		$categories_array = [];
		if ( ! empty( $categories ) && ! is_wp_error( $categories ) ) {
			foreach ( $categories as $category ) {
				$categories_array[ $category->slug ] = $category->name;
			}
		}
		return $categories_array;
	}
}

if ( ! function_exists( 'litho_portfolio_tags_array' ) ) {
	/**
	 * Returns an associative array of portfolio tags slugs and names.
	 *
	 * Fetches all terms from the 'portfolio-tags' taxonomy.
	 *
	 * @return array Associative array of [ slug => name ] for each tags.
	 */
	function litho_portfolio_tags_array() {
		$tags = get_terms(
			array(
				'taxonomy'   => 'portfolio-tags',
				'hide_empty' => true,
			)
		);

		$tags_array = [];
		if ( ! empty( $tags ) && ! is_wp_error( $tags ) ) {
			foreach ( $tags as $tag ) {
				$tags_array[ $tag->slug ] = $tag->name;
			}
		}
		return $tags_array;
	}
}

if ( ! function_exists( 'litho_product_category_array' ) ) {
	/**
	 * Returns an associative array of product category slugs and names.
	 *
	 * Fetches all terms from the 'product-category' taxonomy.
	 *
	 * @return array Associative array of [ slug => name ] for each category.
	 */
	function litho_product_category_array() {
		$categories = get_terms(
			array(
				'taxonomy'   => 'product_cat',
				'hide_empty' => true,
			)
		);

		$categories_array = [];
		if ( ! empty( $categories ) && ! is_wp_error( $categories ) ) {
			foreach ( $categories as $category ) {
				$categories_array[ $category->slug ] = $category->name;
			}
		}
		return $categories_array;
	}
}

if ( ! function_exists( 'litho_product_tags_array' ) ) {
	/**
	 * Returns an associative array of product tags slugs and names.
	 *
	 * Fetches all terms from the 'product-tags' taxonomy.
	 *
	 * @return array Associative array of [ slug => name ] for each tags.
	 */
	function litho_product_tags_array() {
		$tags = get_terms(
			array(
				'taxonomy'   => 'product_tag',
				'hide_empty' => true,
			)
		);

		$tags_array = [];
		if ( ! empty( $tags ) && ! is_wp_error( $tags ) ) {
			foreach ( $tags as $tag ) {
				$tags_array[ $tag->slug ] = $tag->name;
			}
		}
		return $tags_array;
	}
}

if ( ! function_exists( 'litho_option' ) ) {
	/**
	 * Get option value with fallback to theme mod.
	 *
	 * Priority: Post meta → Theme mod → Default.
	 *
	 * @param string $option         Option key.
	 * @param mixed  $default_value  Default value to return if not found.
	 * @return mixed                 Final option value.
	 */
	function litho_option( $option, $default_value ) {
		global $post;

		$litho_option_value = '';

		// Return theme mod directly for 404 page.
		if ( is_404() ) {
			$litho_option_value = get_theme_mod( $option, $default_value );
		} else {
			if ( ( ! ( is_category() || is_archive() || is_author() || is_tag() || is_search() || is_home() || is_tax( 'portfolio-category' ) || is_tax( 'portfolio-tags' ) || is_post_type_archive( 'portfolio' ) ) || ( is_woocommerce_activated() && is_shop() ) ) && isset( $post->ID ) ) {

				// Meta Prefix.
				$meta_prefix = '_';

				if ( is_woocommerce_activated() && is_shop() ) {
					$page_id = wc_get_page_id( 'shop' );
					$option  = str_replace( '_product_archive_', '_page_', $option );
					$value   = get_post_meta( $page_id, $meta_prefix . $option . '_single', true );

				} else {
					$value = get_post_meta( $post->ID, $meta_prefix . $option . '_single', true );
				}

				if ( is_string( $value ) && ( strlen( $value ) > 0 || is_array( $value ) ) && ( 'default' != $value ) ) {
					if ( strtolower( $value ) == '.' ) {
						$litho_option_value = '';
					} else {
						$litho_option_value = $value;
					}
				} else {
					$litho_option_value = get_theme_mod( $option, $default_value );
				}
			} else {
				$litho_option_value = get_theme_mod( $option, $default_value );
			}
		}

		return $litho_option_value;
	}
}

if ( ! function_exists( 'litho_builder_customize_option' ) ) {
	/**
	 * Get customized option value based on current context (404, WooCommerce, Portfolio, Blog, etc.)
	 *
	 * @param string  $option         Option name (without suffix).
	 * @param mixed   $default_value  Fallback value if no custom option found.
	 * @param boolean $general_option If true, fetches the global theme mod directly.
	 * @return mixed                  Option value based on context or fallback.
	 */
	function litho_builder_customize_option( $option, $default_value, $general_option = false ) {
		// Return global option if explicitly requested.
		if ( $general_option ) {
			return get_theme_mod( $option, $default_value );
		}

		$context_suffix = '';

		if ( is_404() ) {
			$context_suffix = '_404_page';
		} elseif ( is_woocommerce_activated() && ( is_product_category() || is_product_tag() || is_tax( 'product_brand' ) || is_shop() ) ) { // if 
				$context_suffix = '_product_archive';
		} elseif ( is_woocommerce_activated() && is_product() ) { // if WooCommerce plugin is activated and WooCommerce product page
				$context_suffix = '_single_product';
		} elseif ( is_tax( 'portfolio-category' ) || is_tax( 'portfolio-tags' ) || is_post_type_archive( 'portfolio' ) ) {
			$context_suffix = '_portfolio_archive';
		} elseif ( is_singular( 'portfolio' ) ) {
			$context_suffix = '_single_portfolio';
		} elseif ( is_search() || is_category() || is_archive() || is_tag() ) {
			$context_suffix = '_archive';
		} elseif ( is_home() ) {
			$context_suffix = '_default';
		} elseif ( is_single() ) {
			$context_suffix = '_single_post';
		} elseif ( is_page() ) {
			$context_suffix = '_single_page';
		}

		$contextual_option = $option . $context_suffix;

		$value = get_theme_mod( $contextual_option, '' );

		// Fallback if no context-specific value found.
		if ( $value === '' ) {
			$value = get_theme_mod( $option, $default_value );
		}

		return $value;
	}
}

if ( ! function_exists( 'litho_builder_option' ) ) {
	/**
	 * Retrieve theme or post meta option with contextual logic.
	 *
	 * @param string  $option          Option key to fetch.
	 * @param mixed   $default_value   Default fallback value.
	 * @param boolean $general_option  Force global theme option (skip post meta).
	 * @return mixed                   Final resolved value.
	 */
	function litho_builder_option( $option, $default_value, $general_option = false ) {
		global $post;

		$litho_option_value = '';

		if ( is_404() ) {
			if ( $general_option ) {
				return get_theme_mod( $option, $default_value );
			}

			$litho_option_value = get_theme_mod( $option . '_404_page', $default_value );

		} else {
			if ( ( ! ( is_category() || is_archive() || is_author() || is_tag() || is_search() || is_home() || is_tax( 'portfolio-category' ) || is_tax( 'portfolio-tags' ) || is_post_type_archive( 'portfolio' ) ) || ( is_woocommerce_activated() && is_shop() ) ) && isset( $post->ID ) ) {
				// Meta Prefix.
				$meta_prefix = '_';

				if ( is_woocommerce_activated() && is_shop() ) {
					$page_id = wc_get_page_id( 'shop' );
					$option  = str_replace( '_product_archive_', '_page_', $option );
				} else {
					$page_id = $post->ID;
				}

				$meta_key = "{$meta_prefix}{$option}_single";
				$value    = get_post_meta( $page_id, $meta_key, true );

				if ( is_string( $value ) && ( strlen( $value ) > 0 || is_array( $value ) ) && ( 'default' != $value ) ) {
					$litho_option_value = ( strtolower( $value ) == '.' ) ? '' : $value;
				} else {
					$litho_option_value = litho_builder_customize_option( $option, $default_value, $general_option );
				}
			} else {
				$litho_option_value = litho_builder_customize_option( $option, $default_value, $general_option );
			}
		}

		return $litho_option_value;
	}
}

/* Check For Product Brand, Product Tag, Product Category, Category & Tag Title */
if ( ! function_exists( 'litho_taxonomy_title_option' ) ) {
	/**
	 * Retrieve a taxonomy-specific theme option with fallback to customizer setting.
	 *
	 * Supports taxonomy types:
	 * - Portfolio: portfolio-category, portfolio-tags
	 * - WooCommerce: product_cat, product_tag, product_brand
	 * - WordPress: category, tag
	 *
	 * @param string $option         Meta key or theme mod option name.
	 * @param mixed  $default_value  Fallback value if option not found or set to 'default'.
	 * @return mixed                 Option value or default fallback.
	 */
	function litho_taxonomy_title_option( $option, $default_value ) {

		$litho_option_value = '';
		$litho_t_id         = ( is_tax( 'portfolio-category' ) || is_tax( 'portfolio-tags' ) || ( is_woocommerce_activated() && ( is_tax( 'product_cat' ) || is_tax( 'product_tag' ) || is_tax( 'product_brand' ) ) ) || is_category() || is_tag() ) ? get_queried_object()->term_id : get_query_var( 'cat' );

		$value = get_term_meta( $litho_t_id, $option, true );

		if ( strlen( $value ) > 0 && ( $value != 'default' ) && ( is_category() || is_tag() || is_tax( 'portfolio-category' ) || is_tax( 'portfolio-tags' ) || ( is_woocommerce_activated() && ( is_tax( 'product_cat' ) || is_tax( 'product_tag' ) || is_tax( 'product_brand' ) ) ) ) && ! ( is_author() || is_search() ) ) {
			$litho_option_value = $value;
		} else {
			$litho_option_value = get_theme_mod( $option, $default_value );
		}

		return $litho_option_value;
	}
}

if ( ! function_exists( 'litho_option_image_alt' ) ) {

	/**
	 * Return Image Alt text
	 *
	 * @param int $attachment_id Attachment ID of the image.
	 * @return array Image alt text array or empty array.
	 */
	function litho_option_image_alt( $attachment_id ) {
		// Return early if not an image.
		if ( wp_attachment_is_image( $attachment_id ) == false ) {
			return [];
		}

		/* Check image alt is on / off */
		$litho_image_alt = get_theme_mod( 'litho_image_alt', '1' );

		if ( $attachment_id && $litho_image_alt == 1 ) {
			/* Get attachment metadata by attachment id */
			$litho_image_meta = array(
				'alt' => get_post_meta( $attachment_id, '_wp_attachment_image_alt', true ),
			);

			return $litho_image_meta;
		} else {
			return [];
		}
	}
}

if ( ! function_exists( 'litho_option_image_title' ) ) {
	/**
	 * Return Image Title text
	 *
	 * @param int $attachment_id Attachment ID of the image.
	 * @return array Image title text array or empty array.
	 */
	function litho_option_image_title( $attachment_id ) {
		// Return early if not an image.
		if ( wp_attachment_is_image( $attachment_id ) == false ) {
			return [];
		}

		/* Check image title is on / off */
		$litho_image_title = get_theme_mod( 'litho_image_title', '0' );

		if ( $attachment_id && ( $litho_image_title == 1 ) ) {
			$litho_image_meta = array(
				'title' => esc_attr( get_the_title( $attachment_id ) ),
			);

			return $litho_image_meta;
		} else {
			return [];
		}
	}
}

if ( ! function_exists( 'litho_get_intermediate_image_sizes' ) ) {
	/**
	 * Get all registered image sizes including WordPress defaults and custom sizes.
	 *
	 * @return array List of image size names (e.g., 'thumbnail', 'medium', 'custom-size').
	 */
	function litho_get_intermediate_image_sizes() {
		// Default WordPress sizes.
		$image_sizes = array(
			'full',
			'thumbnail',
			'medium',
			'medium_large',
			'large',
		);
		
		// Include custom registered sizes if WordPress version is 4.7+.
		if ( version_compare( get_bloginfo( 'version' ), '4.7', '>=' ) ) {
			$additional_sizes = wp_get_additional_image_sizes();
			if ( ! empty( $additional_sizes ) ) {
				$image_sizes = array_merge( $image_sizes, array_keys( $additional_sizes ) );
			}
		}

		// Allow other plugins/themes to modify image size list.
		return apply_filters( 'intermediate_image_sizes', array_unique( $image_sizes ) );
	}
}

if ( ! function_exists( 'litho_get_image_sizes' ) ) {
	/**
	 * Get all registered image sizes including WordPress defaults and custom sizes.
	 *
	 * @return array Associative array of image sizes with width, height, and crop settings.
	 */
	function litho_get_image_sizes() {
		global $_wp_additional_image_sizes;

		$sizes = [];

		$default_sizes = array(
			'full',
			'thumbnail',
			'medium',
			'medium_large',
			'large',
		);

		foreach ( get_intermediate_image_sizes() as $size ) {
			if ( in_array( $size, $default_sizes, true ) ) {
				$sizes[ $size ] = array(
					'width'  => (int) get_option( "{$size}_size_w" ),
					'height' => (int) get_option( "{$size}_size_h" ),
					'crop'   => (bool) get_option( "{$size}_crop" ),
				);
			} elseif ( isset( $_wp_additional_image_sizes[ $size ] ) ) {
				$sizes[ $size ] = array(
					'width'  => (int) $_wp_additional_image_sizes[ $size ]['width'],
					'height' => (int) $_wp_additional_image_sizes[ $size ]['height'],
					'crop'   => (bool) $_wp_additional_image_sizes[ $size ]['crop'],
				);
			}
		}
		
		return $sizes;
	}
}

if ( ! function_exists( 'litho_get_image_size' ) ) {
	function litho_get_image_size( $size ) {
		$sizes = litho_get_image_sizes();

		if ( isset( $sizes[ $size ] ) ) {
			return $sizes[ $size ];
		}

		return false;
	}
}

/* For Get image srcset and sizes */
if ( ! function_exists( 'litho_get_image_srcset_sizes' ) ) {

	/**
	 * Get responsive image attributes (srcset and sizes) for an attachment image.
	 *
	 * @param int    $attachment_id Attachment ID of the image.
	 * @param string $size          Registered image size (e.g. 'thumbnail', 'medium', 'large', 'full', or custom). Default 'full'.
	 *
	 * @return string HTML attributes including srcset and sizes, or empty string if unavailable.
	 */
	function litho_get_image_srcset_sizes( $attachment_id, $size ) {

		if ( ! $attachment_id ) {
			return '';
		}

		$attributes = '';

		$srcset = wp_get_attachment_image_srcset( $attachment_id, $size );
		if ( $srcset ) {
			$attributes .= ' srcset="' . esc_attr( $srcset ) . '"';
		}

		$sizes = wp_get_attachment_image_sizes( $attachment_id, $size );
		if ( $sizes ) {
			$attributes .= ' sizes="' . esc_attr( $sizes ) . '"';
		}

		return $attributes;
	}
}

if ( ! function_exists( 'litho_get_thumbnail_image_sizes' ) ) {
	/**
	 * Get a list of available image sizes with human-readable labels and dimensions.
	 *
	 * @return array Associative array of image size keys => formatted size name with dimensions.
	 */
	function litho_get_thumbnail_image_sizes() {

		$sizes       = [];
		$image_sizes = litho_get_intermediate_image_sizes();

		if ( empty( $image_sizes ) ) {
			return $sizes;
		}

		foreach ( $image_sizes as $value => $size_name ) {
			$image_data = litho_get_image_size( $size_name );

			if ( isset( $image_data['width'] ) ) {
				if ( $image_data['width'] == 0 ) {
					$width = esc_html__( 'Auto', 'litho-addons' );
				} else {
					$width = $image_data['width'] . 'px';
				}
			} else {
				$width = esc_html__( 'Auto', 'litho-addons' );
			}

			if ( isset( $image_data['height'] ) ) {
				if ( $image_data['height'] == 0 ) {
					$height = esc_html__( 'Auto', 'litho-addons' );
				} else {
					$height = $image_data['height'] . 'px';
				}
			} else {
				$height = esc_html__( 'Auto', 'litho-addons' );
			}

			if ( 'full' === $size_name ) {
				$label = esc_html__( 'Original Full Size', 'litho-addons' );
			} else {
				$label = ucwords( str_replace( [ '_', '-' ], ' ', $size_name ) );
				$label .= ' (' . esc_html( $width ) . ' × ' . esc_html( $height ) . ')';
			}
			
			$sizes[ esc_attr( $size_name ) ] = $label;
		}

		return $sizes;
	}
}

if ( ! function_exists( 'litho_get_the_post_content' ) ) {
	/**
	 * Retrieve and filter the current post content using 'the_content' filter.
	 *
	 * @return string Filtered post content with shortcodes and formatting applied.
	 */
	function litho_get_the_post_content() {
		return apply_filters( 'the_content', get_the_content() );
	}
}

if ( ! function_exists( 'litho_get_the_excerpt_theme' ) ) {
	/**
	 * Get post excerpt by custom length using Litho_Excerpt class.
	 *
	 * @param int $length Number of words to limit the excerpt.
	 * @return string Formatted post excerpt.
	 */
	function litho_get_the_excerpt_theme( $length ) {
		if ( ! class_exists( 'Litho_Excerpt' ) ) {
			return '';
		}

		return Litho_Excerpt::litho_get_by_length( $length );
	}
}

if ( ! function_exists( 'litho_extract_shortcode_contents' ) ) {
	/**
	 * Extracts plain text content from shortcodes for use in excerpts.
	 *
	 * @param array $m The matched shortcode components from `preg_replace_callback`.
	 * @return string Extracted plain text content.
	 */
	function litho_extract_shortcode_contents( $m ) {
		global $shortcode_tags;

		// Setup the array of all registered shortcodes.
		$shortcodes = array_keys( $shortcode_tags );

		$no_space_shortcodes = array( 'dropcap' );
		$omitted_shortcodes  = array( 'slide' );

		// Extract contents from all shortcodes recursively.
		if ( in_array( $m[2], $shortcodes, true ) && ! in_array( $m[2], $omitted_shortcodes, true ) ) {

			$pattern = get_shortcode_regex();
			
			// Add space after shortcode content unless specifically restricted.
			$space = in_array( $m[2], $no_space_shortcodes, true ) ? '' : ' ';

			$content = preg_replace_callback( "/$pattern/s", 'litho_extract_shortcode_contents', rtrim( $m[5] ) . $space );
			return $content;
		}

		// Handle escaped shortcode syntax like [[foo]].
		if ( $m[1] == '[' && $m[6] == ']' ) {
			return substr( $m[0], 1, -1 );
		}
		
		// Return shortcode start and end wrapper if no match.
		return $m[1] . $m[6];
	}
}

if ( ! function_exists( 'litho_woocommerce_pagination_args' ) ) {
	/**
	 * Customize WooCommerce pagination arrows with icons and accessible text.
	 *
	 * @param array $args Default pagination arguments.
	 * @return array Modified pagination arguments.
	 */
	function litho_woocommerce_pagination_args( $args ) {

		$args['prev_text'] = '<i aria-hidden="true" class="feather icon-feather-arrow-left"></i><span class="screen-reader-text">' . esc_html__( 'Previous', 'litho-addons' ) . '</span>';
		$args['next_text'] = '<i aria-hidden="true" class="feather icon-feather-arrow-right"></i><span class="screen-reader-text">' . esc_html__( 'Next', 'litho-addons' ) . '</span>';

		return $args;

	}
}
add_filter( 'woocommerce_pagination_args', 'litho_woocommerce_pagination_args' );

if ( ! function_exists( 'litho_get_pagination' ) ) {
	/**
	 * Outputs accessible and styled pagination links for archive or blog pages.
	 *
	 * This function checks if pagination is needed and uses WordPress's
	 * paginate_links() function to output a pagination list with previous/next
	 * icons and screen reader accessibility.
	 *
	 * @return void
	 */
	function litho_get_pagination() {
		global $wp_query;

		// Determine the current page number.
		$current = ( $wp_query->query_vars['paged'] > 1 ) ? $wp_query->query_vars['paged'] : 1;

		// Only output pagination if there's more than one page.
		if ( $wp_query->max_num_pages > 1 ) {
			?>
			<div class="col-12 litho-pagination">
				<div class="pagination align-items-center">
				<?php
				echo paginate_links(
					array(
						'base'      => esc_url_raw( str_replace( 999999999, '%#%', get_pagenum_link( 999999999, false ) ) ),
						'format'    => '',
						'add_args'  => '',
						'current'   => $current, // phpcs:ignore
						'total'     => $wp_query->max_num_pages, // phpcs:ignore
						'prev_text' => '<i aria-hidden="true" class="feather icon-feather-arrow-left"></i><span class="screen-reader-text">' . esc_html__( 'Prev',  'litho-addons' ) . '</span>',
						'next_text' => '<i aria-hidden="true" class="feather icon-feather-arrow-right"></i><span class="screen-reader-text">' . esc_html__( 'Next',  'litho-addons' ) . '</span>',
						'type'      => 'list',
						'end_size'  => 2,
						'mid_size'  => 2,
					)
				);
				?>
				</div>
			</div>
			<?php
		}
	}
}

if ( ! function_exists( 'litho_get_builder_content_for_display' ) ) {
	/**
	 * Retrieves and returns Elementor template content by ID.
	 *
	 * @param int $template_id The ID of the Elementor template.
	 * @return string Template content or empty string if unavailable.
	 */
	function litho_get_builder_content_for_display( $template_id ) {
		// Bail early if Elementor is not available.
		if ( ! class_exists( 'Elementor\Plugin' ) ) {
			return '';
		}

		// Return early if template ID is invalid.
		if ( empty( $template_id ) ) {
			return '';
		}

		$template_content = \Elementor\Plugin::$instance->frontend->get_builder_content_for_display( $template_id );

		return $template_content;
	}
}

if ( ! function_exists( 'litho_mobile_nav_body_attributes' ) ) {
	/**
	 * Adds mobile navigation-related data attributes to the <body> tag.
	 *
	 * @param array $attributes Existing body tag attributes.
	 * @return array Modified attributes with mobile nav data.
	 */
	function litho_mobile_nav_body_attributes( $attributes ) {
		// Check if custom header is enabled.
		$litho_enable_header_general = litho_builder_customize_option( 'litho_enable_header_general', '1' );
		$litho_enable_header         = litho_builder_option( 'litho_enable_header', '1', $litho_enable_header_general );
		$litho_header_section_id     = litho_builder_option( 'litho_header_section', '', $litho_enable_header_general );

		// Fallback to theme mod if post doesn't exist.
		if ( ! litho_post_exists( $litho_header_section_id ) ) {
			$litho_header_section_id = get_theme_mod( 'litho_header_section' );
		}

		// Retrieve header settings.
		$litho_header_style                 = get_post_meta( $litho_header_section_id, '_litho_template_header_style', true );
		$litho_header_style                 = ( ! empty( $litho_header_style ) ) ? $litho_header_style : 'standard';
		$litho_header_mobile_menu_style     = get_post_meta( $litho_header_section_id, '_litho_header_mobile_menu_style', true );
		$litho_header_mobile_menu_alignment = get_post_meta( $litho_header_section_id, '_litho_header_mobile_menu_trigger_alignment', true );
		$litho_header_mobile_menu_bg_color  = get_post_meta( $litho_header_section_id, '_litho_header_mobile_menu_bg_color', true );

		// Default Values.
		if ( 'standard' === $litho_header_style ) {
			$attributes['data-mobile-nav-style']             = ( ! empty( $litho_header_mobile_menu_style ) ) ? $litho_header_mobile_menu_style : 'classic';
			$attributes['data-mobile-nav-trigger-alignment'] = ( ! empty( $litho_header_mobile_menu_alignment ) ) ? $litho_header_mobile_menu_alignment : 'left';

			if ( $litho_header_mobile_menu_bg_color ) {
				$attributes['data-mobile-nav-bg-color'] = $litho_header_mobile_menu_bg_color;
			}
		}
		return $attributes;
	}
}
add_filter( 'litho_attr_body', 'litho_mobile_nav_body_attributes', 10 );

if ( ! function_exists( 'litho_mobile_nav_body_attributes_elementor_preview' ) ) {
	/**
	 * Adds mobile navigation-related data attributes to <body> during Elementor preview
	 * of the custom post type 'sectionbuilder'. Only applied if header style is 'standard'.
	 *
	 */
	function litho_mobile_nav_body_attributes_elementor_preview() {
		// Only apply on 'sectionbuilder' single preview pages.
		if ( is_singular( 'sectionbuilder' ) ) {
			$litho_enable_header_general        = litho_builder_customize_option( 'litho_enable_header_general', '1' );
			$litho_enable_header                = litho_builder_option( 'litho_enable_header', '1', $litho_enable_header_general );
			$litho_header_section_id            = litho_builder_option( 'litho_header_section', '', $litho_enable_header_general );
			$litho_header_style                 = get_post_meta( $litho_header_section_id, '_litho_template_header_style', true );
			$litho_header_style                 = ( ! empty( $litho_header_style ) ) ? $litho_header_style : 'standard';
			$litho_header_mobile_menu_style     = get_post_meta( $litho_header_section_id, '_litho_header_mobile_menu_style', true );
			$litho_header_mobile_menu_alignment = get_post_meta( $litho_header_section_id, '_litho_header_mobile_menu_trigger_alignment', true );
			$litho_header_mobile_menu_bg_color  = get_post_meta( $litho_header_section_id, '_litho_header_mobile_menu_bg_color', true );

			$litho_header_mobile_menu_style     = ( ! empty( $litho_header_mobile_menu_style ) ) ? $litho_header_mobile_menu_style : 'classic';
			$litho_header_mobile_menu_alignment = ( ! empty( $litho_header_mobile_menu_alignment ) ) ? $litho_header_mobile_menu_alignment : 'left';

			if ( 'standard' === $litho_header_style ) {
				?>
				<script type="text/javascript">
					( function( $ ) {
						setTimeout( function () {
							$( 'body' ).attr( {
								'data-mobile-nav-style': '<?php echo sprintf( '%s', $litho_header_mobile_menu_style ); // phpcs:ignore ?>',
								'data-mobile-nav-trigger-alignment': '<?php echo sprintf( '%s', $litho_header_mobile_menu_alignment ); // phpcs:ignore ?>',
								'data-mobile-nav-bg-color': '<?php echo sprintf( '%s', $litho_header_mobile_menu_bg_color ); // phpcs:ignore ?>'
							});
						}, 1000 );
					})( jQuery );
				</script>
				<?php
			}
		}
	}
}
add_action( 'wp_head', 'litho_mobile_nav_body_attributes_elementor_preview' );

if ( ! function_exists( 'litho_single_post_share_shortcode' ) ) {
	/**
	 * Add [litho_single_post_share] Shortcode.
	 */
	function litho_single_post_share_shortcode() {
		global $post;

		if ( ! $post ) {
			return false;
		}

		$output             = '';
		$litho_enable_share = litho_option( 'litho_enable_share', '1' );
		$post_social_share  = litho_option( 'litho_single_post_social_sharing', array( 'facebook', '1', 'Facebook', 'twitter', '1', 'Twitter', 'linkedin', '1', 'Linkedin', 'pinterest', '1', 'Pinterest' ) );
		$permalink          = get_permalink( $post->ID );
		$featured_image     = urlencode( wp_get_attachment_url( get_post_thumbnail_id( $post->ID ) ) );
		$post_title         = rawurlencode( get_the_title( $post->ID ) );
 
		ob_start();	
		if ( 1 == $litho_enable_share && ! empty( $post_social_share ) ) {
			?>
			<div class="social-icon-style-3 medium-icon blog-details-social-sharing">
				<ul>
					<?php
					$i = 0;

					$count = count( $post_social_share );
					foreach ( $post_social_share as $key => $value ) {
						if ( $i < $count ) {
							if ( $post_social_share[ $i+1 ] == '1' ) {
								switch ( $post_social_share[ $i ] ) {
									case 'facebook':
										?>
										<li><a class="social-sharing-icon facebook-f" href="//www.facebook.com/sharer.php?u=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" rel="nofollow" target="_blank" title="<?php echo esc_attr( $post_title ); ?>"><i class="fa-brands fa-facebook-f"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'twitter':
										?>
										<li><a class="social-sharing-icon twitter" href="//twitter.com/share?url=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;"  rel="nofollow" target="_blank" title="<?php echo esc_attr( $post_title ); ?>"><i class="fa-brands fa-x-twitter"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'linkedin':
										?>
										<li><a class="social-sharing-icon linkedin-in" href="//linkedin.com/shareArticle?mini=true&amp;url=<?php echo esc_url( $permalink ); ?>&amp;title=<?php echo esc_attr( $post_title ); ?>" target="_blank" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;"  rel="nofollow" title="<?php echo esc_attr( $post_title ); ?>"><i class="fa-brands fa-linkedin-in"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'pinterest':
										?>
										<li><a class="social-sharing-icon pinterest-p" href="//pinterest.com/pin/create/button/?url=<?php echo esc_url( $permalink ); ?>&amp;media=<?php echo $featured_image; ?>&amp;description=<?php echo esc_attr ( $post_title ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" rel="nofollow" target="_blank" title="<?php echo esc_attr( $post_title ); ?>"><i class="fa-brands fa-pinterest-p"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'reddit':
										?>
										<li><a class="social-sharing-icon reddit" href="//reddit.com/submit?url=<?php echo esc_url( $permalink ); ?>&amp;title=<?php echo esc_attr( $post_title ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-reddit"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'stumbleupon':
										?>
										<li><a class="social-sharing-icon stumbleupon" href="http://www.stumbleupon.com/submit?url=<?php echo esc_url( $permalink ); ?>&amp;title=<?php echo esc_attr( $post_title ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-stumbleupon"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'digg':
										?>
										<li><a class="social-sharing-icon digg" href="//www.digg.com/submit?url=<?php echo esc_url( $permalink ); ?>&amp;title=<?php echo esc_attr( $post_title ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-digg"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'vk':
										?>
										<li><a class="social-sharing-icon vk" href="//vk.com/share.php?url=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-vk"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'xing':
										?>
										<li><a class="social-sharing-icon xing" href="//www.xing.com/app/user?op=share&url=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-xing"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'telegram':
										?>
										<li><a class="social-sharing-icon telegram-plane" href="//t.me/share/url?url=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-telegram-plane"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'ok':
										?>
										<li><a class="social-sharing-icon odnoklassniki" href="//connect.ok.ru/offer?url=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-odnoklassniki"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'viber':
										?>
										<li><a class="social-sharing-icon viber" href="//viber://forward?text=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-viber"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'whatsapp':
										?>
										<li><a class="social-sharing-icon whatsapp" href="//api.whatsapp.com/send?text=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-whatsapp"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'skype':
										?>
										<li><a class="social-sharing-icon skype" href="//web.skype.com/share?source=button&url=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-skype"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
								}
							}
							$i = $i + 3;
						}
					}
					?>
				</ul>
			</div>
			<?php
		}

		$output = ob_get_contents();
		ob_end_clean();
		return $output;
	}
}
add_shortcode( 'litho_single_post_share', 'litho_single_post_share_shortcode' );

if ( ! function_exists( 'litho_single_portfolio_share_shortcode' ) ) {
	/**
	 * Add [litho_single_portfolio_share] Shortcode.
	 */
	function litho_single_portfolio_share_shortcode() {
		global $post;

		if ( ! $post ) {
			return false;
		}

		$output                 = '';
		$enable_portfolio_share = litho_option( 'litho_hide_single_portfolio_share', '1' );
		$portfolio_social_share = litho_option( 'litho_single_portfolio_social_sharing', '' );
		$permalink              = get_permalink( $post->ID );
		$featured_image         = urlencode( wp_get_attachment_url( get_post_thumbnail_id( $post->ID ) ) );
		$post_title             = rawurlencode( get_the_title( $post->ID ) );

		if ( 1 == $enable_portfolio_share && ! empty( $portfolio_social_share ) ) {
			?>
			<div class="social-icon-style-3 medium-icon blog-details-social-sharing">
				<ul>
					<?php
					$i = 0;

					$count = count( $portfolio_social_share );
					foreach ( $portfolio_social_share as $key => $value ) {
						if ( $i < $count ) {
							if ( '1' == $portfolio_social_share[ $i+1 ] ) {
								switch ( $portfolio_social_share[ $i ] ) {
									case 'facebook':
										?>
										<li><a class="social-sharing-icon facebook-f" href="//www.facebook.com/sharer.php?u=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" rel="nofollow" target="_blank" title="<?php echo esc_attr( $post_title ); ?>"><i class="fa-brands fa-facebook-f"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'twitter':
										?>
										<li><a class="social-sharing-icon twitter" href="//twitter.com/share?url=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;"  rel="nofollow" target="_blank" title="<?php echo esc_attr( $post_title ); ?>"><i class="fa-brands fa-x-twitter"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'linkedin':
										?>
										<li><a class="social-sharing-icon linkedin-in" href="//linkedin.com/shareArticle?mini=true&amp;url=<?php echo esc_url( $permalink ); ?>&amp;title=<?php echo esc_attr( $post_title ); ?>" target="_blank" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;"  rel="nofollow" title="<?php echo esc_attr( $post_title ); ?>"><i class="fa-brands fa-linkedin-in"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'pinterest':
										?>
										<li><a class="social-sharing-icon pinterest-p" href="//pinterest.com/pin/create/button/?url=<?php echo esc_url( $permalink ); ?>&amp;media=<?php echo $featured_image; ?>&amp;description=<?php echo esc_attr ( $post_title ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" rel="nofollow" target="_blank" title="<?php echo esc_attr( $post_title ); ?>"><i class="fa-brands fa-pinterest-p"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'reddit':
										?>
										<li><a class="social-sharing-icon reddit" href="//reddit.com/submit?url=<?php echo esc_url( $permalink ); ?>&amp;title=<?php echo esc_attr( $post_title ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-reddit"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'stumbleupon':
										?>
										<li><a class="social-sharing-icon stumbleupon" href="http://www.stumbleupon.com/submit?url=<?php echo esc_url( $permalink ); ?>&amp;title=<?php echo esc_attr( $post_title ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-stumbleupon"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'digg':
										?>
										<li><a class="social-sharing-icon digg" href="//www.digg.com/submit?url=<?php echo esc_url( $permalink ); ?>&amp;title=<?php echo esc_attr( $post_title ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-digg"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'vk':
										?>
										<li><a class="social-sharing-icon vk" href="//vk.com/share.php?url=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-vk"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'xing':
										?>
										<li><a class="social-sharing-icon xing" href="//www.xing.com/app/user?op=share&url=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-xing"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'telegram':
										?>
										<li><a class="social-sharing-icon telegram-plane" href="//t.me/share/url?url=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-telegram-plane"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'ok':
										?>
										<li><a class="social-sharing-icon odnoklassniki" href="//connect.ok.ru/offer?url=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-odnoklassniki"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'viber':
										?>
										<li><a class="social-sharing-icon viber" href="//viber://forward?text=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-viber"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'whatsapp':
										?>
										<li><a class="social-sharing-icon whatsapp" href="//api.whatsapp.com/send?text=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-whatsapp"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'skype':
										?>
										<li><a class="social-sharing-icon skype" href="//web.skype.com/share?source=button&url=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-skype"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
								}
							}
							$i = $i + 3;
						}
					}
					?>
				</ul>
			</div>
			<?php
		}

		$output = ob_get_contents();
		ob_end_clean();
		return $output;
	}
}
add_shortcode( 'litho_single_portfolio_share', 'litho_single_portfolio_share_shortcode' );

if ( ! function_exists( 'litho_single_product_share_shortcode' ) ) {
	/**
	 * Add [litho_single_product_share] Shortcode.
	 */
	function litho_single_product_share_shortcode() {

		global $post;

		if ( ! $post ) {
			return false;
		}

		$output              = '';
		$enable_social_share = get_theme_mod( 'litho_single_product_enable_social_share', '1' );
		$social_share_title  = get_theme_mod( 'litho_single_product_share_title', esc_html__( 'Share:', 'litho-addons' ) );
		$social_sharing      = get_theme_mod( 'litho_single_product_social_sharing', '' );
		$permalink           = get_permalink( $post->ID );
		$featured_image      = urlencode( wp_get_attachment_url( get_post_thumbnail_id( $post->ID ) ) );
		$post_title          = rawurlencode( get_the_title( $post->ID ) );

		ob_start();
		if ( is_woocommerce_activated() && 1 == $enable_social_share && ! empty( $social_sharing ) ) {
			?>
			<div class="default social-icons-wrapper">
				<?php
				if ( ! empty( $social_share_title ) ) {
					?>
					<span class="share-heading-text"><?php echo esc_html( $social_share_title ); ?></span>
					<?php
				}
				?>
				<ul class="default-icon">
					<?php
					$i = 0;

					$count = count( $social_sharing );
					foreach ( $social_sharing as $key => $value ) {
						if ( $i < $count ) {
							if ( '1' == $social_sharing[ $i+1 ] ) {
								switch ( $social_sharing[ $i ] ) {
									case 'facebook':
										?>
										<li><a class="social-sharing-icon facebook-f" href="//www.facebook.com/sharer.php?u=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" rel="nofollow" target="_blank" title="<?php echo esc_attr( $post_title ); ?>"><i class="fa-brands fa-facebook-f"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'twitter':
										?>
										<li><a class="social-sharing-icon twitter" href="//twitter.com/share?url=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;"  rel="nofollow" target="_blank" title="<?php echo esc_attr( $post_title ); ?>"><i class="fa-brands fa-x-twitter"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'linkedin':
										?>
										<li><a class="social-sharing-icon linkedin-in" href="//linkedin.com/shareArticle?mini=true&amp;url=<?php echo esc_url( $permalink ); ?>&amp;title=<?php echo esc_attr( $post_title ); ?>" target="_blank" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;"  rel="nofollow" title="<?php echo esc_attr( $post_title ); ?>"><i class="fa-brands fa-linkedin-in"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'pinterest':
										?>
										<li><a class="social-sharing-icon pinterest-p" href="//pinterest.com/pin/create/button/?url=<?php echo esc_url( $permalink ); ?>&amp;media=<?php echo $featured_image; ?>&amp;description=<?php echo esc_attr ( $post_title ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" rel="nofollow" target="_blank" title="<?php echo esc_attr( $post_title ); ?>"><i class="fa-brands fa-pinterest-p"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'reddit':
										?>
										<li><a class="social-sharing-icon reddit" href="//reddit.com/submit?url=<?php echo esc_url( $permalink ); ?>&amp;title=<?php echo esc_attr( $post_title ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-reddit"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'stumbleupon':
										?>
										<li><a class="social-sharing-icon stumbleupon" href="http://www.stumbleupon.com/submit?url=<?php echo esc_url( $permalink ); ?>&amp;title=<?php echo esc_attr( $post_title ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-stumbleupon"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'digg':
										?>
										<li><a class="social-sharing-icon digg" href="//www.digg.com/submit?url=<?php echo esc_url( $permalink ); ?>&amp;title=<?php echo esc_attr( $post_title ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-digg"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'vk':
										?>
										<li><a class="social-sharing-icon vk" href="//vk.com/share.php?url=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-vk"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'xing':
										?>
										<li><a class="social-sharing-icon xing" href="//www.xing.com/app/user?op=share&url=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-xing"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'telegram':
										?>
										<li><a class="social-sharing-icon telegram-plane" href="//t.me/share/url?url=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-telegram-plane"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'ok':
										?>
										<li><a class="social-sharing-icon odnoklassniki" href="//connect.ok.ru/offer?url=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-odnoklassniki"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'viber':
										?>
										<li><a class="social-sharing-icon viber" href="//viber://forward?text=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-viber"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'whatsapp':
										?>
										<li><a class="social-sharing-icon whatsapp" href="//api.whatsapp.com/send?text=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-whatsapp"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
									case 'skype':
										?>
										<li><a class="social-sharing-icon skype" href="//web.skype.com/share?source=button&url=<?php echo esc_url( $permalink ); ?>" onclick="window.open(this.href,this.title,'width=500,height=500,top=300px,left=300px'); return false;" data-pin-custom="true"><i class="fa-brands fa-skype"></i><span></span><span class="screen-reader-text"><?php echo esc_html__( 'Social Share', 'litho-addons' ); ?></span></a></li>
										<?php
										break;
								}
							}
							$i = $i + 3;
						}
					}
					?>
				</ul>
			</div>
		<?php } ?>
		<?php
		$output = ob_get_contents();
		ob_end_clean();
		return $output;
	}
}
add_shortcode( 'litho_single_product_share', 'litho_single_product_share_shortcode' );

if ( ! function_exists( 'litho_add_additional_hover_animations' ) ) {
	/**
	 * Add additional hover animations effect.
	 */
	function litho_add_additional_hover_animations() {
		$animations = [
			'forward'                => __( 'Fordward', 'litho-addons' ),
			'backword'               => __( 'Backward', 'litho-addons' ),
			'fade'                   => __( 'Fade', 'litho-addons' ),
			'back-pulse'             => __( 'Back Pulse', 'litho-addons' ),
			'sweep-to-right'         => __( 'Sweep To Right', 'litho-addons' ),
			'sweep-to-left'          => __( 'Sweep To Left', 'litho-addons' ),
			'sweep-to-bottom'        => __( 'Sweep To Bottom', 'litho-addons' ),
			'sweep-to-top'           => __( 'Sweep To Top', 'litho-addons' ),
			'bounce-to-right'        => __( 'Bounce To Right', 'litho-addons' ),
			'bounce-to-left'         => __( 'Bounce To Left', 'litho-addons' ),
			'bounce-to-bottom'       => __( 'Bounce To Bottom', 'litho-addons' ),
			'bounce-to-top'          => __( 'Bounce To Top', 'litho-addons' ),
			'radial-out'             => __( 'Radial Out', 'litho-addons' ),
			'radial-in'              => __( 'Radial In', 'litho-addons' ),
			'rectangle-in'           => __( 'Rectangle In', 'litho-addons' ),
			'rectangle-out'          => __( 'Rectangle Out', 'litho-addons' ),
			'shutter-in-horizontal'  => __( 'Shutter In Horizontal', 'litho-addons' ),
			'shutter-out-horizontal' => __( 'Shutter Out Horizontal', 'litho-addons' ),
			'shutter-in-vertical'    => __( 'Shutter In Vertical', 'litho-addons' ),
			'shutter-out-vertical'   => __( 'Shutter Out Vertical', 'litho-addons' ),
			'hollow'                 => __( 'Hollow', 'litho-addons' ),
			'trim'                   => __( 'Trim', 'litho-addons' ),
			'ripple-out'             => __( 'Ripple Out', 'litho-addons' ),
			'ripple-in'              => __( 'Ripple In', 'litho-addons' ),
			'outline-out'            => __( 'Outline Out', 'litho-addons' ),
			'outline-in'             => __( 'Outline In', 'litho-addons' ),
			'round-corners'          => __( 'Round Corners', 'litho-addons' ),
			'underline-from-left'    => __( 'Underline From Left', 'litho-addons' ),
			'underline-from-center'  => __( 'Underline From Center', 'litho-addons' ),
			'underline-from-right'   => __( 'Underline From Right', 'litho-addons' ),
			'reveal'                 => __( 'Reveal', 'litho-addons' ),
			'underline-reveal'       => __( 'Underline Reveal', 'litho-addons' ),
			'overline-reveal'        => __( 'Overline Reveal', 'litho-addons' ),
			'overline-from-left'     => __( 'Overline From Left', 'litho-addons' ),
			'overline-from-center'   => __( 'Overline From Center', 'litho-addons' ),
			'overline-from-right'    => __( 'Overline From Right', 'litho-addons' ),
			'shadow'                 => __( 'Shadow', 'litho-addons' ),
			'grow-shadow'            => __( 'Grow Shadow', 'litho-addons' ),
			'float-shadow'           => __( 'Float Shadow', 'litho-addons' ),
			'glow'                   => __( 'Glow', 'litho-addons' ),
			'shadow-radial'          => __( 'Shadow Radial', 'litho-addons' ),
			'box-shadow-outset'      => __( 'Box Shadow Outset', 'litho-addons' ),
			'box-shadow-inset'       => __( 'Box Shadow Inset', 'litho-addons' ),
			'bubble-top'             => __( 'Bubble Top', 'litho-addons' ),
			'bubble-right'           => __( 'Bubble Right', 'litho-addons' ),
			'bubble-bottom'          => __( 'Bubble Bottom', 'litho-addons' ),
			'bubble-left'            => __( 'Bubble Left', 'litho-addons' ),
			'bubble-float-top'       => __( 'Bubble Float Top', 'litho-addons' ),
			'bubble-float-right'     => __( 'Bubble Float Right', 'litho-addons' ),
			'bubble-float-bottom'    => __( 'Bubble Float Bottom', 'litho-addons' ),
			'bubble-float-left'      => __( 'Bubble Float Left', 'litho-addons' ),
			'curl-top-left'          => __( 'Curl Top Left', 'litho-addons' ),
			'curl-top-right'         => __( 'Curl Top Right', 'litho-addons' ),
			'curl-bottom-right'      => __( 'Curl Bottom Right', 'litho-addons' ),
			'curl-bottom-left'       => __( 'Curl Bottom Left', 'litho-addons' ),
			'float-3px'              => __( 'Litho Float Three Up', 'litho-addons' ),
			'float-5px'              => __( 'Litho Float Five Up', 'litho-addons' ),
			'float-10px'             => __( 'Litho Float Ten Up', 'litho-addons' ),
			'scale-effect'           => __( 'Litho Scale', 'litho-addons' ),
			'scale-3d-effect'        => __( 'Litho Scale (3d)', 'litho-addons' ),
			'scale-9-effect'         => __( 'Litho Scale (9)', 'litho-addons' ),
			'zoom-effect'            => __( 'Litho Zoom', 'litho-addons' ),
			'btn-slide-up-bg'        => __( 'Litho Button Slide Up', 'litho-addons' ),
			'btn-slide-down-bg'      => __( 'Litho Button Slide Down', 'litho-addons' ),
			'btn-slide-left-bg'      => __( 'Litho Button Slide Left', 'litho-addons' ),
			'btn-slide-right-bg'     => __( 'Litho Button Slide Right', 'litho-addons' ),
			'btn-expand-ltr'         => __( 'Litho Button Expand Width', 'litho-addons' ),
			'icon-sweep-bottom'      => __( 'Litho Icon Sweep To Bottom', 'litho-addons' ),
		];
		return $animations;
	}
}
add_filter( 'elementor/controls/hover_animations/additional_animations', 'litho_add_additional_hover_animations' );

if ( ! function_exists( 'litho_before_register_style_js_callback' ) ) {
	/**
	 * Callback to deregister JavaScript files before they are registered.
	 *
	 */
	function litho_before_register_style_js_callback() {
		if ( litho_load_javascript_by_key( 'swiper' ) ) {
			wp_deregister_script( 'swiper' );
		}
	}
}
add_action( 'litho_before_register_style_js', 'litho_before_register_style_js_callback' );

if ( ! function_exists( 'litho_remove_e_swiper_style' ) ) {
	/**
	 * Filter the <link> tag for styles and remove Elementor's Swiper stylesheet from output.
	 *
	 * This is useful when you want to suppress loading without deregistering the style (e.g., to prevent dependency issues).
	 *
	 * @param string $tag    The HTML <link> tag for the enqueued style.
	 * @param string $handle The style handle.
	 * @return string Modified (or original) tag.
	 */
	function litho_remove_e_swiper_style( $tag, $handle ) {
		if ( is_elementor_activated() && 'e-swiper' === $handle ) {
			return ''; // remove from output
		}
		return $tag;
	}
}
add_filter( 'style_loader_tag', 'litho_remove_e_swiper_style', 10, 2 );

if ( ! function_exists( 'litho_addons_additional_js_output' ) ) {
	/**
	 * Output custom JavaScript on the frontend if available.
	 *
	 * This function retrieves custom JS from the 'litho_custom_js' option and outputs it
	 * inline after ensuring the target script ('litho-main') is enqueued.
	 *
	 * Hooked to 'wp_enqueue_scripts' with a high priority to ensure scripts are already enqueued.
	 */
	function litho_addons_additional_js_output() {

		$custom_js = get_option( 'litho_custom_js', '' );

		// Bail early if there's no custom JS.
		if ( '' === $custom_js ) {
			return;
		}

		// Ensure the target script is enqueued before adding inline JS.
		if ( wp_script_is( 'litho-main', 'enqueued' ) ) {
			wp_add_inline_script( 'litho-main', $custom_js );
		}
	}
}
add_action( 'wp_enqueue_scripts', 'litho_addons_additional_js_output', 999 );

if ( ! function_exists( 'litho_customizer_settings' ) ) {
	/**
	 * Handles the import and export of Customizer settings.
	 *
	 * @param WP_Customize_Manager $wp_customize The Customizer object.
	 */
	function litho_customizer_settings( $wp_customize ) {
		if ( current_user_can( 'edit_theme_options' ) ) {
			if ( isset( $_REQUEST['litho-export'] ) ) {
				litho_customizer_export( $wp_customize );
			}
			if ( isset( $_REQUEST['litho-import'] ) && isset( $_FILES['litho-import-file'] ) ) {
				litho_customizer_import( $wp_customize );
			}
		}
	}
}
add_action( 'customize_register', 'litho_customizer_settings', 100 );

if ( ! function_exists( 'litho_customizer_export' ) ) {
	/**
	 * Export Customizer settings to a downloadable JSON file.
	 *
	 * @param WP_Customize_Manager $wp_customize The Customizer object.
	 */
	function litho_customizer_export( $wp_customize ) {
		if ( ! wp_verify_nonce( $_REQUEST['litho-export'], 'litho-exporting' ) ) {
			return;
		}

		$core_options = array(
			'page_for_posts',
			'blogname',
			'show_on_front',
			'blogdescription',
			'page_on_front',
		);

		$theme_name = get_stylesheet();
		$template   = get_template();
		$charset    = get_option( 'blog_charset' );

		$settings_data  = array(
			'template' => $template,
			'mods'     => get_theme_mods() ? get_theme_mods() : [],
			'options'  => [],
		);

		// Get options from the Customizer API.
		$settings = $wp_customize->settings();

		foreach ( $settings as $key => $setting ) {

			if ( 'option' == $setting->type ) {

				// Don't save widget data.
				if ( 'widget_' === substr( strtolower( $key ), 0, 7 ) ) {
					continue;
				}

				// Don't save sidebar data.
				if ( 'sidebars_' === substr( strtolower( $key ), 0, 9 ) ) {
					continue;
				}

				// Don't save core options.
				if ( in_array( $key, $core_options, true ) ) {
					continue;
				}

				$settings_data['options'][ $key ] = $setting->value();
			}
		}

		if ( function_exists( 'wp_get_custom_css_post' ) ) {
			$settings_data['wp_css'] = wp_get_custom_css();
		}

		// Set the download headers.
		header( 'Content-disposition: attachment; filename=' . sanitize_file_name( $theme_name ) . '-export.json' );
		header( 'Content-Type: application/octet-stream; charset=' . esc_attr( $charset ) );

		// Serialize the export data.
		echo wp_json_encode( $settings_data );

		// Start the download.
		die();
	}
}

// Customizer settings import.
if ( ! function_exists( 'litho_customizer_import' ) ) {
	function litho_customizer_import( $wp_customize ) {
		// Make sure import form.
		if ( ! wp_verify_nonce( $_REQUEST['litho-import'], 'litho-importing' ) ) {
			return;
		}

		// Make sure WordPress upload support is loaded.
		if ( ! function_exists( 'wp_handle_upload' ) ) {
			require_once ABSPATH . 'wp-admin/includes/file.php';
		}

		// Setup global vars.
		global $wp_customize;

		// Setup internal vars.
		$template  = get_template();
		$overrides = array( 'test_form' => false, 'test_type' => false, 'mimes' => array( 'json' => 'text/plain' ) );
		$file      = wp_handle_upload( $_FILES['litho-import-file'], $overrides );

		// Make sure we have an uploaded file.
		if ( isset( $file['error'] ) && ! file_exists( $file['file'] ) ) {
			return;
		}

		// Get the upload data.
		$file_content = file_get_contents( $file['file'] );
		$file_data    = @unserialize( $file_content );

		// Remove the uploaded file.
		unlink( $file['file'] );

		// Data checks.
		if ( 'array' != gettype( $file_data ) ) {
			$error_message = __( 'Error importing settings! Please check that you uploaded a customizer export file.', 'litho-addons' );
			echo "<script type='text/javascript'>alert('$error_message');</script>";
			return;
		}
		if ( ! isset( $file_data['template'] ) || ! isset( $file_data['mods'] ) ) {
			$error_message = __( 'Error importing settings! Please check that you uploaded a customizer export file.', 'litho-addons' );
			echo "<script type='text/javascript'>alert('$error_message');</script>";
			return;
		}
		if ( $file_data['template'] != $template ) {
			$error_message = __( 'Error importing settings! The settings you uploaded are not for the current theme.', 'litho-addons' );
			echo "<script type='text/javascript'>alert('$error_message');</script>";
			return;
		}

		// If wp_css is set then import it.
		if ( function_exists( 'wp_update_custom_css_post' ) && isset( $file_data['wp_css'] ) && '' !== $file_data['wp_css'] ) {
			wp_update_custom_css_post( $file_data['wp_css'] );
		}

		// Call the customize_save action.
		do_action( 'customize_save', $wp_customize );

		// Loop through the mods.
		foreach ( $file_data['mods'] as $key => $val ) {

			// Call the customize_save_ dynamic action.
			do_action( 'customize_save_' . $key, $wp_customize );

			// Save the mod.
			set_theme_mod( $key, $val );
		}

		// Call the customize_save_after action.
		do_action( 'customize_save_after', $wp_customize );
	}
}

// Remove revslider conflict in widget.php.
if ( ! function_exists( 'litho_revslider_gutenberg_cgb_editor_assets' ) ) {
	function litho_revslider_gutenberg_cgb_editor_assets() {
		global $pagenow;
		if ( 'widgets.php' == $pagenow ) {
			wp_dequeue_script( 'revslider_gutenberg-cgb-block-js' );
		}
	}
}
add_action( 'enqueue_block_editor_assets', 'litho_revslider_gutenberg_cgb_editor_assets' );

if ( ! function_exists( 'litho_load_stylesheet_by_key' ) ) {
	/**
	 * Determine whether a stylesheet identified by a key should be loaded.
	 *
	 * Checks the 'litho_disable_style_details' option to see if a given stylesheet key
	 * has been marked for exclusion. Returns true if the stylesheet should be loaded.
	 *
	 * @param string $value The key identifying the stylesheet.
	 * @return bool Whether the stylesheet should be loaded.
	 */
	function litho_load_stylesheet_by_key( $value ) {

		$flag = true;

		// Get comma-separated list of disabled styles from theme options.
		$disabled_styles = litho_option( 'litho_disable_style_details', '' );

		if ( ! empty( $disabled_styles ) ) {
			$disabled_styles_array = explode( ',', $disabled_styles );

			if ( in_array( $value, $disabled_styles_array, true ) ) {
				$flag = false;
			}
		}

		/**
		 * Filter whether a specific stylesheet should be loaded.
		 *
		 * @param bool   $flag  Whether the stylesheet should load.
		 * @param string $value The key identifying the stylesheet.
		 */
		return apply_filters( 'litho_load_stylesheet_by_key', $flag, $value );
	}
}

if ( ! function_exists( 'litho_load_javascript_by_key' ) ) {
	/**
	 * Determine whether a JavaScript file identified by a key should be loaded.
	 *
	 * Checks the 'litho_disable_script_details' option to see if a given script key
	 * has been marked for exclusion. Returns true if the script should be loaded.
	 *
	 * @param string $value The key identifying the JavaScript file.
	 * @return bool Whether the JavaScript file should be loaded.
	 */
	function litho_load_javascript_by_key( $value ) {
		$flag = true;

		// Get comma-separated list of disabled scripts from theme options.
		$disabled_scripts = litho_option( 'litho_disable_script_details', '' );
		if ( ! empty( $disabled_scripts ) ) {
			$disabled_scripts_array = explode( ',', $disabled_scripts );

			if ( in_array( $value, $disabled_scripts_array, true ) ) {
				$flag = false;
			}
		}

		/**
		 * Filter whether a specific JavaScript file should be loaded.
		 *
		 * @param bool   $flag  Whether the script should load.
		 * @param string $value The key identifying the script.
		 */
		return apply_filters( 'litho_load_javascript_by_key', $flag, $value );
	}
}

if ( ! function_exists( 'litho_admin_dequeue_scripts' ) ) {
	/**
	 * Fix icon conflict with SG Optimizer by dequeuing conflicting icon styles on specific admin pages.
	 *
	 * SG Optimizer and Litho Theme both use the same `data-icon` attribute, causing icon conflicts.
	 * This function dequeues/deregisters icon styles (Font Awesome, ET Line, Themify) only on SG Optimizer pages.
	 */
	function litho_admin_dequeue_scripts() {
		// Get current admin page slug.
		$page = isset( $_GET['page'] ) ? $_GET['page'] : '';

		// List of SG Optimizer-related admin pages.
		$sg_pages = array(
			'sg-security',
			'site-security',
			'login-settings',
			'activity-log',
			'post-hack-actions',
			'sgo_frontend',
			'sg-cachepress',
			'sgo_caching',
			'sgo_environment',
			'sgo_media',
			'sgo_analysis',
		);

		// Dequeue Litho icon styles only on SG Optimizer pages.
		if ( is_admin() && in_array( $page, $sg_pages, true ) ) {
			$styles = array(
				'font-awesome',
				'et-line-icons',
				'themify-icons',
			);

			foreach ( $styles as $style ) {
				wp_deregister_style( $style );
				wp_dequeue_style( $style );
			}
		}
	}
}
add_action( 'admin_enqueue_scripts', 'litho_admin_dequeue_scripts', 999 );

/**
 * Dequeue Contact Form 7 plugin styles and scripts.
 *
 * This function removes the Contact Form 7 styles and scripts from.
 * being loaded on the front end, which can help optimize performance.
 * if the forms are not used on certain pages.
 */
function litho_dequeue_script_styles_contact_form7() {
	// Only proceed if Contact Form 7 is active.
	if ( ! class_exists( 'WPCF7' ) ) {
		return;
	}
	
	// Dequeue Contact Form 7 stylesheet if enqueued.
	if ( wp_style_is( 'contact-form-7', 'enqueued' ) ) {
		wp_dequeue_style( 'contact-form-7' );
	}

	// Dequeue Contact Form 7 script if enqueued.
	if ( wp_script_is( 'contact-form-7', 'enqueued' ) ) {
		wp_dequeue_script( 'contact-form-7' );
	}
}
add_action( 'wp_enqueue_scripts', 'litho_dequeue_script_styles_contact_form7', 20 );

/**
 * Load Contact Form 7 scripts and styles in the Contact Form 7 Widget.
 *
 * This function checks if the widget is of type 'litho-contact-form'.
 * and enqueues the necessary scripts and styles for Contact Form 7.
 *
 * @param mixed $widget The widget instance containing settings.
 */
function litho_load_script_styles_contact_form7( $widget ) {
	// Only proceed if Contact Form 7 plugin is active.
	if ( ! class_exists( 'WPCF7' ) ) {
		return;
	}

	// Check if the widget is a Contact Form 7 widget.
	if ( 'litho-contact-form' !== $widget->get_name() ) {
		return; // Early return for better readability.
	}

	// Enqueue Contact Form 7 scripts and styles if the functions exist.
	if ( function_exists( 'wpcf7_enqueue_scripts' ) ) {
		wpcf7_enqueue_scripts();
	}

	if ( function_exists( 'wpcf7_enqueue_styles' ) ) {
		wpcf7_enqueue_styles();
	}
}
add_action( 'elementor/frontend/widget/before_render', 'litho_load_script_styles_contact_form7' );

/**
 * Disable crawling and indexing for specific post types.
 *
 * Prevents search engines from indexing or following links
 * on 'litho-mega-menu' and 'sectionbuilder' single pages.
 *
 * @param array $robots Array of robots meta directives.
 * @return array Modified robots directives.
 */
add_filter( 'wp_robots', function( $robots ) {
	if ( is_singular( 'litho-mega-menu' ) || is_singular( 'sectionbuilder' ) ) {
		$robots['noindex']  = true;
		$robots['nofollow'] = true;
	}

	return $robots;
} );

add_filter( 'wpml_pb_elementor_register_string_name_litho-team-memeber-carousel', function( $name, $args ) {
	$widgetType = $args['element']['widgetType'];
	return $widgetType . '-' . $args['key'] . '-' . $args['field'] . '-' . $args['nodeId'] . '-' . $args['item']['_id'];
}, 10, 2 );

add_filter( 'wpml_pb_elementor_register_string_name_litho-slider', function( $name, $args ) {
	$widgetType = $args['element']['widgetType'];
	return $widgetType . '-' . $args['key'] . '-' . $args['field'] . '-' . $args['nodeId'] . '-' . $args['item']['_id'];
}, 10, 2 );

if ( ! function_exists( 'litho_get_attachment_html' ) ) {
	/**
	 * Generate HTML for an attachment image.
	 *
	 * @param mixed $attachment_id Attachment ID.
	 * @param mixed $attachment_url Attachment URL.
	 * @param mixed $attachment_size Attachment size.
	 */
	function litho_get_attachment_html( $attachment_id, $attachment_url, $attachment_size = 'full', $classes = array() ) {
		$image_html = '';

		// If attachment ID and URL are provided.
		if ( ! empty( $attachment_id ) && ! empty( $attachment_url ) ) {
			// Get image alt and title from metadata.
			$alt_data   = litho_option_image_alt( $attachment_id );
			$title_data = litho_option_image_title( $attachment_id );

			$image_data = [];

			if ( is_array( $classes ) && ! empty( $classes ) ) {
				$classes_list        = implode( ' ', $classes );
				$image_data['class'] = esc_attr( $classes_list );
			}

			if ( isset( $alt_data['alt'] ) && ! empty( $alt_data['alt'] ) ) {
				$image_data['alt'] = esc_attr( $alt_data['alt'] );
			}

			if ( isset( $alt_data['title'] ) && ! empty( $alt_data['title'] ) ) {
				$image_data['title'] = esc_attr( $alt_data['title'] );
			}

			// Build image HTML with attributes.
			$image_html = wp_get_attachment_image(
				$attachment_id,
				$attachment_size,
				false,
				$image_data
			);
		} elseif ( ! empty( $attachment_url ) ) {
			// Output fallback image using provided URL.
			$image_html = sprintf(
				'<img src="%1$s" alt="%2$s" />',
				esc_url( $attachment_url ),
				esc_attr__( 'Placeholder Image', 'litho-addons' )
			);
		}

		// Output the result safely.
		if ( $image_html ) {
			echo $image_html; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		}
	}
}

if ( ! function_exists( 'litho_load_admin_footer_css' ) ) {
	/**
	 * Outputs custom CSS to the WordPress admin footer.
	 */
	function litho_load_admin_footer_css() {
		// Define the custom CSS to be output in the admin area.
		$output_css = '
			#adminmenu .wp-menu-image img {
				padding: 5px 0 0;
				opacity: 1;
				width: 24px;
			}
		';

		// Print the CSS directly in the footer wrapped in <style> tags.
		// Using wp_kses() here is optional but helps guard against accidental unescaped output.
		echo '<style type="text/css">' . wp_strip_all_tags( $output_css ) . '</style>';
	}
}
add_action( 'admin_footer', 'litho_load_admin_footer_css' );

if ( ! function_exists( 'litho_admin_bar_menu' ) ) {
	/**
	 * Add menu item in Admin bar
	 *
	 * @param object $admin_bar Array of menus.
	 */
	function litho_admin_bar_menu( $admin_bar ) {
		// Early return if not administrator.
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$admin_bar->add_menu(
			array(
				'id'    => 'litho-adminbar-item',
				'title' => esc_html__( 'Litho Settings', 'litho-addons' ),
				'href'  => admin_url( 'admin.php?page=litho-theme-setup' ),
			)
		);

		$admin_bar->add_menu(
			array(
				'parent' => 'litho-adminbar-item',
				'id'     => 'litho-header-and-footer',
				'title'  => esc_html__( 'Header & Footer', 'litho-addons' ),
				'href'   => admin_url( 'edit.php?post_type=sectionbuilder&template_type=header' ),
			)
		);

		$admin_bar->add_menu(
			array(
				'parent' => 'litho-adminbar-item',
				'id'     => 'litho-page-title',
				'title'  => esc_html__( 'Page Title', 'litho-addons' ),
				'href'   => admin_url( 'edit.php?post_type=sectionbuilder&template_type=custom-title' ),
			)
		);

		$admin_bar->add_menu(
			array(
				'parent' => 'litho-adminbar-item',
				'id'     => 'litho-archive-all',
				'title'  => esc_html__( 'Archive', 'litho-addons' ),
			)
		);

		$admin_bar->add_menu(
			array(
				'parent' => 'litho-archive-all',
				'id'     => 'litho-archive',
				'title'  => esc_html__( 'Post Archive ( Blog, Author, Category, Search, etc... )', 'litho-addons' ),
				'href'   => admin_url( 'edit.php?post_type=sectionbuilder&template_type=archive' ),
			)
		);

		$admin_bar->add_menu(
			array(
				'parent' => 'litho-archive-all',
				'id'     => 'litho-archive-portfolio',
				'title'  => esc_html__( 'Portfolio Archive', 'litho-addons' ),
				'href'   => admin_url( 'edit.php?post_type=sectionbuilder&template_type=archive-portfolio' ),
			)
		);


		$admin_bar->add_menu(
			array(
				'parent' => 'litho-adminbar-item',
				'id'     => 'litho-maintenance-mode',
				'title'  => esc_html__( 'Maintenance Mode', 'litho-addons' ),
				'href'   => admin_url( 'admin.php?page=elementor-tools#tab-maintenance_mode' ),
			)
		);

		$admin_bar->add_menu(
			array(
				'parent' => 'litho-adminbar-item',
				'id'     => 'litho-system-info',
				'title'  => esc_html__( 'System Info', 'litho-addons' ),
				'href'   => admin_url( 'site-health.php?tab=debug' ),
				'meta'   => array(
					'target' => '_blank',
				),
			)
		);

		if ( ! is_admin() ) {
			$admin_bar->add_menu(
				array(
					'parent' => 'litho-adminbar-item',
					'id'     => 'litho-reset-temporary-data',
					'title'  => esc_html__( 'Clear Temporary Data', 'litho-addons' ),
					'href'   => '?litho_reset_temporary_data=true',
				)
			);
		}
	}
}
add_action( 'admin_bar_menu', 'litho_admin_bar_menu', 98 );

if ( ! function_exists( 'litho_clear_cache_data' ) ) {
	/**
	 * Add litho clear cache data
	 */
	function litho_clear_cache_data() {
		// Early return if not administrator.
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		if ( ! is_admin() ) {
			if ( isset( $_GET['litho_reset_temporary_data'] ) ) { // phpcs:ignore
				if ( is_multisite() ) {
					// Deletes the temporary library cache option from the database.
					delete_site_option( 'litho_library_cache' );
				}

				// Deletes the temporary library cache option from the database.
				delete_option( 'litho_library_cache' );

				// Deletes the stored transient containing the local_google_fonts and fonts css files.
				if ( is_elementor_activated() ) {
					delete_option( '_elementor_local_google_fonts' );
					\Elementor\Core\Files\Fonts\Google_Font::clear_cache();
				}
			}
		}
	}
}
add_action( 'init', 'litho_clear_cache_data', 20 );
