<?php
namespace KeyStone\Compatibility;

use KeyStone\Utils;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class KeyStone_Elementor {

	private static $instance;

	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	public function __construct() {
		// Register elementor locations
		add_action( 'elementor/theme/register_locations', [ $this, 'register_locations' ] );

		// Get body classes
		add_filter( 'body_class', [ $this, 'elementor_body_classes' ] );

		add_action( 'wp_enqueue_scripts', [ $this, 'load_elementor_styles' ] );

		require_once KEYSTONE_COMPATIBILITY_PATH . '/elementor/widgets.php';
		require_once KEYSTONE_COMPATIBILITY_PATH . '/elementor/site-settings.php';
		require_once KEYSTONE_COMPATIBILITY_PATH . '/elementor/widget-controls.php';
		require_once KEYSTONE_COMPATIBILITY_PATH . '/elementor/library-template-layout.php';

		if ( '' != \KeyStone\Utils::get_option( 'typekit_id' ) ) {
			require_once KEYSTONE_COMPATIBILITY_PATH . '/elementor/typekit.php';
		}

		add_action( 'init', [ $this, 'update_default_elementor_kit' ] );

		add_action( 'elementor/init', [ $this, 'load_keystone_library' ], 15 );
		add_action( 'init', [ $this, 'clear_library_cache' ] );

		add_filter( 'elementor_page_title_switch', [ $this, 'check_hide_title' ] );

		add_action( 'after_switch_theme', [ $this, 'enable_cpt_support' ] );
		add_action( 'after_switch_theme', [ $this, 'dismiss_plugin_notices' ] );

		add_action( 'wp_dashboard_setup', [ $this, 'remove_dashboard_widgets' ] );
		add_action( 'admin_menu', [ $this, 'elementor_free_menu' ], 801 );

		add_action( 'admin_enqueue_scripts', [ $this, 'elementor_free_styles' ] );
		add_action( 'elementor/editor/before_enqueue_scripts', [ $this, 'elementor_free_styles' ] );
		add_action( 'elementor/preview/enqueue_styles', [ $this, 'elementor_editor_styles' ] );

		add_filter( 'elementor/init', [ $this, 'update_experiments_data' ] );
	}

	public static function elementor_body_classes( $classes ) {
		$classes[] = '';

		if ( 'keystone-library' == \KeyStone\Utils::get_option( 'elementor_default_library' ) ) {
			$classes[] = 'keystone-elementor-library';
		}

		return $classes;
	}

	public static function load_keystone_library() {
		$license_key = \KeyStone\License\Admin::get_license_key();
		if ( $license_key && 'keystone-library' == \KeyStone\Utils::get_option( 'elementor_default_library' ) ) {
			include KEYSTONE_COMPATIBILITY_PATH . '/elementor/library.php';
			$unregister_source = function($id) {
				unset( $this->_registered_sources[ $id ] );
			};
			$unregister_source->call( \Elementor\Plugin::instance()->templates_manager, 'remote');
			\Elementor\Plugin::instance()->templates_manager->register_source( 'Elementor\TemplateLibrary\KeyStone_Source' );
		}
	}

	public static function clear_library_cache() {
		if ( get_option( 'library_force_update' ) != \KeyStone\Utils::get_option( 'elementor_default_library' ) ) {
			\KeyStone\Utils::clear_transient_cache();
		}
		update_option( 'library_force_update', \KeyStone\Utils::get_option( 'elementor_default_library' ) );
	}

	public function register_locations( $manager ) {
		$manager->register_all_core_location();
	}

	public function load_elementor_styles() {
		\Elementor\Plugin::$instance->frontend->enqueue_styles();
	}

	public function update_default_elementor_kit() {

		if ( ! \KeyStone\Utils::is_keystone_theme() ) {
			return;
		}

		add_option( 'default_keystone_kit', 1 );
		if ( get_option( 'default_keystone_kit' ) == 1 ) {

			$kit = \Elementor\Plugin::$instance->kits_manager->get_active_kit();

			if ( ! $kit->get_id() ) {
				return;
			}

			$kit->update_settings( [
				'container_width' => array(
					'size' => "1240",
				),
				'system_colors' => array(
					 0 => array(
						'_id' => 'primary',
						'title' => 'Primary',
						'color' => defined( 'KEYSTONE_ELEMENTOR_PRIMARY_COLOR' ) ? KEYSTONE_ELEMENTOR_PRIMARY_COLOR : '#0073aa',
					 ),
					 1 => array(
						'_id' => 'secondary',
						'title' => 'Secondary',
						'color' => defined( 'KEYSTONE_ELEMENTOR_SECONDARY_COLOR' ) ? KEYSTONE_ELEMENTOR_SECONDARY_COLOR : '#6c757d',
					 ),
					 2 => array(
						'_id' => 'headline',
						'title' => 'Headline',
						'color' => defined( 'KEYSTONE_ELEMENTOR_HEADLINE_COLOR' ) ? KEYSTONE_ELEMENTOR_HEADLINE_COLOR : '#212529',
					 ),
					 3 => array(
						'_id' => 'text',
						'title' => 'Text',
						'color' => defined( 'KEYSTONE_ELEMENTOR_TEXT_COLOR' ) ? KEYSTONE_ELEMENTOR_TEXT_COLOR : '#495057',
					 ),
					 4 => array(
						'_id' => 'accent',
						'title' => 'Accent',
						'color' => defined( 'KEYSTONE_ELEMENTOR_ACCENT_COLOR' ) ? KEYSTONE_ELEMENTOR_ACCENT_COLOR : '#28a745',
					 ),
					 5 => array(
						'_id' => 'light',
						'title' => 'Light Background',
						'color' => defined( 'KEYSTONE_ELEMENTOR_LIGHT_COLOR' ) ? KEYSTONE_ELEMENTOR_LIGHT_COLOR : '#f8f9fa',
					 ), 
					 6 => array(
						'_id' => 'light_dark',
						'title' => 'Dark Light Background',
						'color' => defined( 'KEYSTONE_ELEMENTOR_DARK_LIGHT_COLOR' ) ? KEYSTONE_ELEMENTOR_DARK_LIGHT_COLOR : '#e9ecef',
					 ),
					 7 => array(
						'_id' => 'border',
						'title' => 'Border',
						'color' => defined( 'KEYSTONE_ELEMENTOR_BORDER_COLOR' ) ? KEYSTONE_ELEMENTOR_BORDER_COLOR : '#dee2e6',
					),

					8 => array(
						'_id' => 'black',
						'title' => 'Black',
						'color' => defined( 'KEYSTONE_ELEMENTOR_BLACK_COLOR' ) ? KEYSTONE_ELEMENTOR_BLACK_COLOR : '#000000',
					),
					9 => array(
						'_id' => 'white',
						'title' => 'White',
						'color' => defined( 'KEYSTONE_ELEMENTOR_WHITE_COLOR' ) ? KEYSTONE_ELEMENTOR_WHITE_COLOR : '#ffffff',
					),
				),
				'system_typography' => array(
					0 => array(
					   '_id' => 'primary',
					   'title' => 'Primary',
					   'typography_typography' => 'custom',
					   'typography_font_family' => KEYSTONE_ELEMENTOR_PRIMARY_FONT_FAMILY,
					   'typography_font_weight' => KEYSTONE_ELEMENTOR_PRIMARY_FONT_WEIGHT,
					),
					1 => array(
						'_id' => 'secondary',
						'title' => 'Secondary',
						'typography_typography' => 'custom',
						'typography_font_family' => KEYSTONE_ELEMENTOR_SECONDARY_FONT_FAMILY,
						'typography_font_weight' => KEYSTONE_ELEMENTOR_SECONDARY_FONT_WEIGHT,
					 ),
					 2 => array(
						'_id' => 'text',
						'title' => 'Text',
						'typography_typography' => 'custom',
						'typography_font_family' => KEYSTONE_ELEMENTOR_TEXT_FONT_FAMILY,
						'typography_font_weight' => KEYSTONE_ELEMENTOR_TEXT_FONT_WEIGHT,
					 ),
					 3 => array(
						'_id' => 'accent',
						'title' => 'Accent',
						'typography_typography' => 'custom',
						'typography_font_family' => KEYSTONE_ELEMENTOR_ACCENT_FONT_FAMILY,
						'typography_font_weight' => KEYSTONE_ELEMENTOR_ACCENT_FONT_WEIGHT,
					 ),
			   ),
			] );

			\Elementor\Plugin::instance()->files_manager->clear_cache();
			update_option( 'default_keystone_kit', 0 );
		}
	}

	public static function get_site_settings_link() {
		$kit_id = get_option( 'elementor_active_kit' );
		$site_settings_url = '';

		if ( $kit_id ) {
			$kit = \Elementor\Plugin::$instance->documents->get( $kit_id );
			
			if ( $kit && method_exists( $kit, 'get_edit_url' ) ) {
				$site_settings_url = $kit->get_edit_url();
			}
		}

		return $site_settings_url;
	}

	public static function elementor_free_styles() {
		if ( defined( 'ELEMENTOR_PRO_VERSION' ) ) {
			return;
		}

		wp_enqueue_style(
			'keystone-elementor-free',
			KEYSTONE_ASSETS_URL . 'css/keystone-elementor-free.css',
			[],
			KEYSTONE_VERSION
		);
	}

	public static function elementor_editor_styles() {
		wp_enqueue_style(
			'keystone-elementor-styles',
			KEYSTONE_ASSETS_URL . 'css/keystone-elementor-editor.css',
			[],
			KEYSTONE_VERSION
		);
	}

	public function check_hide_title( $val ) {
		$current_doc = \Elementor\Plugin::instance()->documents->get( get_the_ID() );
		if ( $current_doc && 'yes' === $current_doc->get_settings( 'hide_title' ) ) {
			$val = false;
		}

		return $val;
	}

	public function enable_cpt_support() {
		$cpt_support = get_option( 'elementor_cpt_support' );
		if ( ! $cpt_support ) {
			$cpt_support = [ 'page', 'post', 'keystone-portfolio' ];
			update_option( 'elementor_cpt_support', $cpt_support );
		} else if ( ! in_array( 'keystone-portfolio', $cpt_support ) ) {
			$cpt_support[] = 'keystone-portfolio';
			update_option( 'elementor_cpt_support', $cpt_support );
		}
	}

	public function remove_dashboard_widgets() {
		remove_meta_box( 'e-dashboard-overview', 'dashboard', 'normal' );
	}

	public static function elementor_free_menu() {
		if ( defined( 'ELEMENTOR_PRO_VERSION' ) ) {
			return;
		}
		remove_submenu_page( 'elementor', 'e-form-submissions' );
		remove_submenu_page( 'elementor', 'elementor_custom_fonts' );
		remove_submenu_page( 'elementor', 'elementor_custom_icons' );
		remove_submenu_page( 'elementor', 'elementor_custom_code' );
		remove_submenu_page( 'elementor', 'elementor-apps' );
		remove_submenu_page( 'elementor', 'go_elementor_pro' );
	}

	public static function update_experiments_data() {
		// Enable the Container experiment by default
		if ( ! \Elementor\Plugin::$instance->experiments->is_feature_active( 'container' ) ) {
			update_option( 'elementor_experiment-container', 'active' );
		}

		// Disable the Inline Font Icons experiment
		if ( \Elementor\Plugin::$instance->experiments->is_feature_active( 'e_font_icon_svg' ) ) {
			update_option( 'elementor_experiment-e_font_icon_svg', 'inactive' );
		}

		// Disable the Optimized Control Loading feature
		if ( \Elementor\Plugin::$instance->experiments->is_feature_active( 'e_optimized_control_loading' ) ) {
			update_option( 'elementor_experiment-e_optimized_control_loading', 'inactive' );
		}

		// Disable Lazy Load Background Images feature
		if ( \Elementor\Plugin::$instance->experiments->is_feature_active( 'e_lazyload' ) ) {
			update_option( 'elementor_experiment-e_lazyload', 'inactive' );
		}

		// Disable Element Caching feature
		if ( \Elementor\Plugin::$instance->experiments->is_feature_active( 'e_element_cache' ) ) {
			update_option( 'elementor_experiment-e_element_cache', 'inactive' );
		}
	}

	public static function dismiss_plugin_notices() {
		update_option( 'elementor_tracker_notice', '1' );
	}

}
KeyStone_Elementor::instance();
