class VamtamImage extends elementorModules.frontend.handlers.Base {
	getDefaultSettings() {
		return {
			selectors: {
				container: '.elementor-widget-container',
				image: '.elementor-image img',
				imageWrap: '.vamtam-image-wrapper',
				widget: '.elementor-widget-container',
			},
		};
	}

	getDefaultElements() {
		const selectors = this.getSettings( 'selectors' );
		return {
			$container: this.$element.find( selectors.container ),
			$image: this.$element.find( selectors.image ),
			$imageWrap: this.$element.find( selectors.imageWrap ),
			$widget: this.$element.find( selectors.widget ),
		};
	}

	onInit( ...args ) {
		super.onInit( ...args );
		this.initImageGrowAnims();
	}

	initImageGrowAnims() {
		const $swiperEl = this.$element.closest( '.swiper' );
		if ( ! $swiperEl.length ) {
			this.handleImageGrowAnims();
		} else {
			// If the image is inside a swiper loop, run handleImageGrowAnims() after swiper has been initialized.
			const swiperInstance = $swiperEl.data( 'swiper' );
			if ( swiperInstance ) {
				if ( swiperInstance.initialized ) {
					this.handleImageGrowAnims();
				} else {
					swiperInstance.on( 'init', () => {
						this.handleImageGrowAnims();
					} );
				}
			} else {
				// Observe and wait for the swiper to be initialized.
				const swiperObserver = new MutationObserver( ( mutationsList ) => {
					for ( const mutation of mutationsList ) {
						if ( mutation.attributeName === 'class' && $swiperEl.hasClass( 'swiper-initialized' ) ) {
							this.handleImageGrowAnims();
							swiperObserver.disconnect();
							break;
						}
					}
				} );
				swiperObserver.observe( $swiperEl[0], { attributes: true, attributeFilter: ['class'] } );
			}
		}
	}

	handleImageGrowAnims() {
		if ( this.$element.hasClass( 'vamtam-animated' ) ) {
			return;
		}

		var elementSettings    = this.getElementSettings(),
			growAnims          = [ 'growFromLeft', 'growFromRight' ],
			growWithScaleAnims = [ 'imageGrowWithScaleLeft', 'imageGrowWithScaleRight', 'imageGrowWithScaleTop', 'imageGrowWithScaleBottom' ];

			if ( growAnims.includes( elementSettings._animation ) || growAnims.includes( elementSettings.animation ) ) {
				// Apply the anim to the img element.
				this.elements.$image.addClass( 'elementor-invisible' );
				window.VAMTAM_FRONT.WidgetsObserver.observe( this.$element[0], this.animateScale.bind(this) );
			} else if ( growWithScaleAnims.includes( elementSettings._animation ) || growWithScaleAnims.includes( elementSettings.animation ) ) {
				// The grow part of the animation is being animated by the global handler
				// since it is on the main $element. The image part (scale) we animate manually.
				this.elements.$imageWrap.addClass( 'elementor-invisible' );
				this.elements.$image.addClass( 'elementor-invisible' );
				this.animDuration = this.calcAnimDuration();
				window.VAMTAM_FRONT.WidgetsObserver.observe( this.$element[0], this.animate.bind(this) );
			}
	}

	animate() {
		if ( this.$element.hasClass( 'vamtam-animated' ) ) {
			return;
		}

		this.animateGrow();
		this.animateScale();
	}

	calcAnimDuration() {
		const speed  = this.$element.hasClass( 'animated-slow' ) ? 250 : this.$element.hasClass( 'animated-fast' ) ? 850 : 550, // px per second
			duration = this.elements.$imageWrap[0].offsetWidth / speed;
		return duration;
	}

	animateGrow() {
		const $imageWrap  = this.elements.$imageWrap,
			animation = this.getAnimation();

		if ( 'none' === animation ) {
			$imageWrap.removeClass( 'elementor-invisible' );
			return;
		}

		const elementSettings = this.getElementSettings(),
			animationDelay    = elementSettings._animation_delay || elementSettings.animation_delay || 0,
			animationDuration = this.animDuration || elementSettings.animation_duration || '';

		$imageWrap.removeClass( animation );

		if ( this.currentAnimation ) {
			$imageWrap.removeClass( this.currentAnimation );
		}

		this.currentAnimation = animation;

		if ( animationDuration ) {
			// Calculated animation duration for constant speed, regardless of distance traveled.
			$imageWrap[ 0 ].style[ '-webkit-animation-duration' ] = this.animDuration + 's';
			$imageWrap[ 0 ].style[ '-moz-animation-duration' ] = this.animDuration + 's';
			$imageWrap[ 0 ].style[ '-o-animation-duration' ] = this.animDuration + 's';
			$imageWrap[ 0 ].style[ 'animation-duration' ] = this.animDuration + 's';
		} else {
			// Fallback to Elementor's default values. Variable speed, based on distance traveled.
			if ( this.$element.hasClass( 'animated-slow' ) ) {
				$imageWrap.addClass( 'animated-slow' );
			}

			if ( this.$element.hasClass( 'animated-fast' ) ) {
				$imageWrap.addClass( 'animated-fast' );
			}
		}

		setTimeout( () => {
			window.requestAnimationFrame( () => {
				$imageWrap.removeClass( 'elementor-invisible' ).addClass( 'animated ' + animation );
			} );
		}, animationDelay );
	}

	animateScale() {
		const $image  = this.elements.$image,
			animation = this.getAnimation();

		if ( 'none' === animation ) {
			$image.removeClass( 'elementor-invisible' );
			return;
		}

		const elementSettings = this.getElementSettings(),
			animationDelay    = elementSettings._animation_delay || elementSettings.animation_delay || 0,
			animationDuration = this.animDuration || elementSettings.animation_duration || '';

		$image.removeClass( animation );

		if ( this.currentAnimation ) {
			$image.removeClass( this.currentAnimation );
		}

		this.currentAnimation = animation;

		if ( animationDuration ) {
			// Calculated animation duration for constant speed, regardless of distance traveled.
			$image[ 0 ].style[ '-webkit-animation-duration' ] = animationDuration + 's';
			$image[ 0 ].style[ '-moz-animation-duration' ] = animationDuration + 's';
			$image[ 0 ].style[ '-o-animation-duration' ] = animationDuration + 's';
			$image[ 0 ].style[ 'animation-duration' ] = animationDuration + 's';
		} else {
			// Fallback to Elementor's default values. Variable speed, based on distance traveled.
			if ( this.$element.hasClass( 'animated-slow' ) ) {
				$image.addClass( 'animated-slow' );
			}

			if ( this.$element.hasClass( 'animated-fast' ) ) {
				$image.addClass( 'animated-fast' );
			}
		}

		setTimeout( () => {
			window.requestAnimationFrame( () => {
				$image.removeClass( 'elementor-invisible' ).addClass( 'animated ' + animation );
				this.animationCleanup( animation );
			} );
		}, animationDelay );
	}

	animationCleanup( animation ) {
		// Animation cleanup so other anims (like hover) can work ok after the entrance anim is done.
		const $image  = this.elements.$image,
			_this = this;

		$image.one('animationend webkitAnimationEnd oAnimationEnd MSAnimationEnd', function () {
			// add vamtam-animated class to the main element
			_this.$element.addClass( 'vamtam-animated' );

			// also remove the animation-duration styles from the image and wrapper
			jQuery( this, _this.elements.$imageWrap ).css({
				'-webkit-animation-duration': '',
				'-moz-animation-duration': '',
				'-o-animation-duration': '',
				'animation-duration': ''
			});
		});
	}

	getAnimation() {
		return this.getCurrentDeviceSetting( 'animation' ) || this.getCurrentDeviceSetting( '_animation' );
	}

	onElementChange( propertyName ) {
		if ( /^_?animation/.test( propertyName ) ) {
			this.animate();
		}
	}
}


jQuery( window ).on( 'elementor/frontend/init', () => {
	if ( ! elementorFrontend.elementsHandler || ! elementorFrontend.elementsHandler.attachHandler ) {
		const addHandler = ( $element ) => {
			elementorFrontend.elementsHandler.addHandler( VamtamImage, {
				$element,
			} );
		};

		elementorFrontend.hooks.addAction( 'frontend/element_ready/image.default', addHandler, 100 );
	} else {
		elementorFrontend.elementsHandler.attachHandler( 'image', VamtamImage );
	}
} );
