<?php

declare(strict_types=1);

/*
 * Copyright (c) Ne-Lexa
 *
 * For the full copyright and license information, please view
 * the LICENSE file that was distributed with this source code.
 *
 * @see https://github.com/Ne-Lexa/google-play-scraper
 */

namespace Nelexa\GPlay\Scraper;

use GuzzleHttp\Psr7\Request;
use GuzzleHttp\Psr7\Utils;
use Nelexa\GPlay\Enum\SortEnum;
use Nelexa\GPlay\GPlayApps;
use Nelexa\GPlay\Model\AppId;
use Psr\Http\Message\RequestInterface;

/**
 * @internal
 */
class PlayStoreUiRequest
{
    public const LIMIT_REVIEW_ON_PAGE = 199;

    public const LIMIT_APPS_ON_PAGE = 100;

    private const RPC_ID_REVIEWS = 'UsvDTd';

    private const RPC_ID_APPS = 'qnKhOb';

    private const RPC_ID_PERMISSIONS = 'xdSrCf';

    private const RPC_ID_SUGGEST = 'IJ4APc';

    private const RPC_ID_PAGE = 'w3QCWb';

    private const RPC_ID_TOP = 'vyAe2';

    /**
     * @param AppId       $requestApp
     * @param int         $count
     * @param SortEnum    $sort
     * @param string|null $token
     *
     * @return RequestInterface
     */
    public static function getReviewsRequest(
        AppId $requestApp,
        int $count,
        SortEnum $sort,
        ?string $token = null
    ): RequestInterface {
        $limit = min(self::LIMIT_REVIEW_ON_PAGE, max(1, $count));
        $queryParams = [
            'rpcids' => self::RPC_ID_REVIEWS,
            GPlayApps::REQ_PARAM_LOCALE => $requestApp->getLocale(),
            GPlayApps::REQ_PARAM_COUNTRY => $requestApp->getCountry(),
            'authuser' => null,
            'soc-app' => 121,
            'soc-platform' => 1,
            'soc-device' => 1,
        ];
        $url = GPlayApps::GOOGLE_PLAY_URL . '/_/PlayStoreUi/data/batchexecute?' . http_build_query($queryParams);
        $formParams = [
            'f.req' => '[[["' . self::RPC_ID_REVIEWS . '","[null,null,[2,' . $sort->value(
                ) . ',[' . $limit . ',null,' . ($token === null ? 'null' : '\\"' . $token . '\\"')
                . ']],[\\"' . $requestApp->getId() . '\\",7]]",null,"generic"]]]',
        ];
        $headers = [
            'Content-Type' => 'application/x-www-form-urlencoded;charset=utf-8',
        ];
        $body = Utils::streamFor(http_build_query($formParams));

        return new Request('POST', $url, $headers, $body);
    }

    /**
     * @param string $locale
     * @param string $country
     * @param int    $count
     * @param string $token
     *
     * @return RequestInterface
     */
    public static function getAppsRequest(string $locale, string $country, int $count, string $token): RequestInterface
    {
        $limit = min(self::LIMIT_APPS_ON_PAGE, max(1, $count));
        $queryParams = [
            'rpcids' => self::RPC_ID_APPS,
            GPlayApps::REQ_PARAM_LOCALE => $locale,
            GPlayApps::REQ_PARAM_COUNTRY => $country,
            'soc-app' => 121,
            'soc-platform' => 1,
            'soc-device' => 1,
        ];
        $url = GPlayApps::GOOGLE_PLAY_URL . '/_/PlayStoreUi/data/batchexecute?' . http_build_query($queryParams);

        $formParams = [
            'f.req' => '[[["' . self::RPC_ID_APPS . '","[[null,[[8,[20,' . $limit . ']],true,null,[64,1,195,71,8,72,9,10,11,139,12,16,145,148,150,151,152,27,30,31,96,32,34,163,100,165,104,169,108,110,113,55,56,57,122],[null,null,[[[true],null,[[[]]],null,null,null,null,[null,2],null,null,null,null,null,null,[1],null,null,null,null,null,null,null,[1]],[null,[[[]]]],[null,[[[]]],null,[true]],[null,[[[]]]],null,null,null,null,[[[[]]]],[[[[]]]]],[[[[7,1],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,31],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,104],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,9],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,8],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,27],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,12],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,65],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,110],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,88],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,11],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,56],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,55],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,96],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,10],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,122],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,72],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,71],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,64],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,113],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,139],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,150],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,169],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,165],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,151],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,163],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,32],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,16],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,108],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,100],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[9,1],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,31],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,104],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,9],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,8],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,27],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,12],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,65],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,110],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,88],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,11],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,56],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,55],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,96],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,10],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,122],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,72],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,71],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,64],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,113],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,139],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,150],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,169],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,165],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,151],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,163],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,32],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,16],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,108],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,100],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[17,1],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,31],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,104],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,9],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,8],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,27],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,12],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,65],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,110],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,88],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,11],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,56],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,55],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,96],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,10],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,122],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,72],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,71],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,64],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,113],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,139],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,150],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,169],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,165],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,151],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,163],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,32],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,16],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,108],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,100],[[1,7,9,25,13,31,5,27,8,14,10]]],[[10,1],[[1,7,6,9]]],[[10,31],[[1,7,6,9]]],[[10,104],[[1,7,6,9]]],[[10,9],[[1,7,6,9]]],[[10,8],[[1,7,6,9]]],[[10,27],[[1,7,6,9]]],[[10,12],[[1,7,6,9]]],[[10,65],[[1,7,6,9]]],[[10,110],[[1,7,6,9]]],[[10,88],[[1,7,6,9]]],[[10,11],[[1,7,6,9]]],[[10,56],[[1,7,6,9]]],[[10,55],[[1,7,6,9]]],[[10,96],[[1,7,6,9]]],[[10,10],[[1,7,6,9]]],[[10,122],[[1,7,6,9]]],[[10,72],[[1,7,6,9]]],[[10,71],[[1,7,6,9]]],[[10,64],[[1,7,6,9]]],[[10,113],[[1,7,6,9]]],[[10,139],[[1,7,6,9]]],[[10,150],[[1,7,6,9]]],[[10,169],[[1,7,6,9]]],[[10,165],[[1,7,6,9]]],[[10,151],[[1,7,6,9]]],[[10,163],[[1,7,6,9]]],[[10,32],[[1,7,6,9]]],[[10,16],[[1,7,6,9]]],[[10,108],[[1,7,6,9]]],[[10,100],[[1,7,6,9]]],[[1,1],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,31],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,104],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,9],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,8],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,27],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,12],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,65],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,110],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,88],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,11],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,56],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,55],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,96],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,10],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,122],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,72],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,71],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,64],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,113],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,139],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,150],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,169],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,165],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,151],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,163],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,32],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,16],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,108],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,100],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[4,1],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,31],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,104],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,9],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,8],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,27],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,12],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,65],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,110],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,88],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,11],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,56],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,55],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,96],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,10],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,122],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,72],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,71],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,64],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,113],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,139],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,150],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,169],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,165],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,151],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,163],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,32],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,16],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,108],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,100],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[3,1],[[1,5,14,4,10,17]]],[[3,31],[[1,5,14,4,10,17]]],[[3,104],[[1,5,14,4,10,17]]],[[3,9],[[1,5,14,4,10,17]]],[[3,8],[[1,5,14,4,10,17]]],[[3,27],[[1,5,14,4,10,17]]],[[3,12],[[1,5,14,4,10,17]]],[[3,65],[[1,5,14,4,10,17]]],[[3,110],[[1,5,14,4,10,17]]],[[3,88],[[1,5,14,4,10,17]]],[[3,11],[[1,5,14,4,10,17]]],[[3,56],[[1,5,14,4,10,17]]],[[3,55],[[1,5,14,4,10,17]]],[[3,96],[[1,5,14,4,10,17]]],[[3,10],[[1,5,14,4,10,17]]],[[3,122],[[1,5,14,4,10,17]]],[[3,72],[[1,5,14,4,10,17]]],[[3,71],[[1,5,14,4,10,17]]],[[3,64],[[1,5,14,4,10,17]]],[[3,113],[[1,5,14,4,10,17]]],[[3,139],[[1,5,14,4,10,17]]],[[3,150],[[1,5,14,4,10,17]]],[[3,169],[[1,5,14,4,10,17]]],[[3,165],[[1,5,14,4,10,17]]],[[3,151],[[1,5,14,4,10,17]]],[[3,163],[[1,5,14,4,10,17]]],[[3,32],[[1,5,14,4,10,17]]],[[3,16],[[1,5,14,4,10,17]]],[[3,108],[[1,5,14,4,10,17]]],[[3,100],[[1,5,14,4,10,17]]],[[2,1],[[1,5,7,4,13,16,12,18]]],[[2,31],[[1,5,7,4,13,16,12,18]]],[[2,104],[[1,5,7,4,13,16,12,18]]],[[2,9],[[1,5,7,4,13,16,12,18]]],[[2,8],[[1,5,7,4,13,16,12,18]]],[[2,27],[[1,5,7,4,13,16,12,18]]],[[2,12],[[1,5,7,4,13,16,12,18]]],[[2,65],[[1,5,7,4,13,16,12,18]]],[[2,110],[[1,5,7,4,13,16,12,18]]],[[2,88],[[1,5,7,4,13,16,12,18]]],[[2,11],[[1,5,7,4,13,16,12,18]]],[[2,56],[[1,5,7,4,13,16,12,18]]],[[2,55],[[1,5,7,4,13,16,12,18]]],[[2,96],[[1,5,7,4,13,16,12,18]]],[[2,10],[[1,5,7,4,13,16,12,18]]],[[2,122],[[1,5,7,4,13,16,12,18]]],[[2,72],[[1,5,7,4,13,16,12,18]]],[[2,71],[[1,5,7,4,13,16,12,18]]],[[2,64],[[1,5,7,4,13,16,12,18]]],[[2,113],[[1,5,7,4,13,16,12,18]]],[[2,139],[[1,5,7,4,13,16,12,18]]],[[2,150],[[1,5,7,4,13,16,12,18]]],[[2,169],[[1,5,7,4,13,16,12,18]]],[[2,165],[[1,5,7,4,13,16,12,18]]],[[2,151],[[1,5,7,4,13,16,12,18]]],[[2,163],[[1,5,7,4,13,16,12,18]]],[[2,32],[[1,5,7,4,13,16,12,18]]],[[2,16],[[1,5,7,4,13,16,12,18]]],[[2,108],[[1,5,7,4,13,16,12,18]]],[[2,100],[[1,5,7,4,13,16,12,18]]]]]],null,null,[[[1,2],[10,8,9],[],[]]]],null,\"' . $token . '\"],[true]]",null,"generic"]]]',
        ];

        $headers = [
            'Content-Type' => 'application/x-www-form-urlencoded;charset=utf-8',
        ];
        $body = Utils::streamFor(http_build_query($formParams));

        return new Request('POST', $url, $headers, $body);
    }

    /**
     * @param AppId $requestApp
     *
     * @throws \Exception
     *
     * @return RequestInterface
     */
    public static function getPermissionsRequest(AppId $requestApp): RequestInterface
    {
        $queryParams = [
            'rpcids' => self::RPC_ID_PERMISSIONS,
            GPlayApps::REQ_PARAM_LOCALE => $requestApp->getLocale(),
            GPlayApps::REQ_PARAM_COUNTRY => $requestApp->getCountry(),
            'soc-app' => 121,
            'soc-platform' => 1,
            'soc-device' => 1,
        ];
        $url = GPlayApps::GOOGLE_PLAY_URL . '/_/PlayStoreUi/data/batchexecute?' . http_build_query($queryParams);
        $formParams = [
            'f.req' => '[[["' . self::RPC_ID_PERMISSIONS . '","[[null,[\"'
                . $requestApp->getId() . '\",7],[]]]",null,"1"]]]',
        ];
        $headers = [
            'Content-Type' => 'application/x-www-form-urlencoded;charset=utf-8',
        ];
        $body = Utils::streamFor(http_build_query($formParams));

        return new Request('POST', $url, $headers, $body);
    }

    /**
     * @param string $query
     * @param string $locale
     * @param string $country
     *
     * @throws \Exception
     *
     * @return RequestInterface
     */
    public static function getSuggestRequest(
        string $query,
        string $locale = GPlayApps::DEFAULT_LOCALE,
        string $country = GPlayApps::DEFAULT_COUNTRY
    ): RequestInterface {
        $queryParams = [
            'rpcids' => self::RPC_ID_SUGGEST,
            GPlayApps::REQ_PARAM_LOCALE => $locale,
            GPlayApps::REQ_PARAM_COUNTRY => $country,
            'source-path' => '/store/apps',
            'authuser' => null,
            'soc-app' => 121,
            'soc-platform' => 1,
            'soc-device' => 1,
        ];
        $url = GPlayApps::GOOGLE_PLAY_URL . '/_/PlayStoreUi/data/batchexecute?' . http_build_query($queryParams);
        $formParams = [
            'f.req' => '[[["' . self::RPC_ID_SUGGEST . '","[[null,[\"'
                . str_replace('"', '\\\\\\"', $query)
                . '\"],[10],[2],4]]",null,"generic"]]]',
        ];
        $headers = [
            'Content-Type' => 'application/x-www-form-urlencoded;charset=utf-8',
        ];
        $body = Utils::streamFor(http_build_query($formParams));

        return new Request('POST', $url, $headers, $body);
    }

    /**
     * @param string $token
     * @param string $locale
     * @param string $country
     *
     * @return RequestInterface
     */
    public static function getClusterPagesRequest(string $token, string $locale, string $country): RequestInterface
    {
        $queryParams = [
            'rpcids' => self::RPC_ID_PAGE,
            GPlayApps::REQ_PARAM_LOCALE => $locale,
            GPlayApps::REQ_PARAM_COUNTRY => $country,
            'source-path' => '/store/apps',
            'authuser' => null,
            'soc-app' => 121,
            'soc-platform' => 1,
            'soc-device' => 1,
        ];
        $url = GPlayApps::GOOGLE_PLAY_URL . '/_/PlayStoreUi/data/batchexecute?' . http_build_query($queryParams);
        $formParams = [
            'f.req' => '[[["' . self::RPC_ID_PAGE . '","[[null,2,null,null,[[10,[10,50],null,\"' . $token . '\"],true,null,[96,27,8,57,30,110,11,16,1,9,31,12,104,55,56,145,51,34,10],[null,null,[[[true],null,[[[]]],null,null,null,null,[null,2],null,null,null,null,null,null,[1],null,null,null,null,null,null,null,[1]],[null,[[[]]]],[null,[[[]]],null,[true]],[null,[[[]]]],null,null,null,null,[[[[]]]],[[[[]]]]],[[[[7,1],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,31],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,104],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,9],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,8],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,27],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,12],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,65],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,110],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,88],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,11],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,56],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,55],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,96],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,10],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,122],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,72],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,71],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,64],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,113],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,139],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,150],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,169],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,165],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,151],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,163],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,32],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,16],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,108],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,100],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[9,1],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,31],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,104],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,9],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,8],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,27],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,12],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,65],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,110],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,88],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,11],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,56],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,55],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,96],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,10],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,122],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,72],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,71],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,64],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,113],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,139],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,150],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,169],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,165],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,151],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,163],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,32],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,16],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,108],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,100],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[17,1],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,31],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,104],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,9],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,8],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,27],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,12],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,65],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,110],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,88],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,11],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,56],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,55],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,96],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,10],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,122],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,72],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,71],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,64],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,113],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,139],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,150],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,169],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,165],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,151],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,163],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,32],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,16],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,108],[[1,7,9,25,13,31,5,27,8,14,10]]],[[17,100],[[1,7,9,25,13,31,5,27,8,14,10]]],[[10,1],[[1,7,6,9]]],[[10,31],[[1,7,6,9]]],[[10,104],[[1,7,6,9]]],[[10,9],[[1,7,6,9]]],[[10,8],[[1,7,6,9]]],[[10,27],[[1,7,6,9]]],[[10,12],[[1,7,6,9]]],[[10,65],[[1,7,6,9]]],[[10,110],[[1,7,6,9]]],[[10,88],[[1,7,6,9]]],[[10,11],[[1,7,6,9]]],[[10,56],[[1,7,6,9]]],[[10,55],[[1,7,6,9]]],[[10,96],[[1,7,6,9]]],[[10,10],[[1,7,6,9]]],[[10,122],[[1,7,6,9]]],[[10,72],[[1,7,6,9]]],[[10,71],[[1,7,6,9]]],[[10,64],[[1,7,6,9]]],[[10,113],[[1,7,6,9]]],[[10,139],[[1,7,6,9]]],[[10,150],[[1,7,6,9]]],[[10,169],[[1,7,6,9]]],[[10,165],[[1,7,6,9]]],[[10,151],[[1,7,6,9]]],[[10,163],[[1,7,6,9]]],[[10,32],[[1,7,6,9]]],[[10,16],[[1,7,6,9]]],[[10,108],[[1,7,6,9]]],[[10,100],[[1,7,6,9]]],[[1,1],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,31],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,104],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,9],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,8],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,27],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,12],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,65],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,110],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,88],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,11],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,56],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,55],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,96],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,10],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,122],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,72],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,71],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,64],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,113],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,139],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,150],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,169],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,165],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,151],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,163],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,32],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,16],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,108],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[1,100],[[1,5,14,38,19,29,7,34,4,12,11,6,30,43,40,42,16,10]]],[[4,1],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,31],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,104],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,9],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,8],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,27],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,12],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,65],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,110],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,88],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,11],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,56],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,55],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,96],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,10],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,122],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,72],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,71],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,64],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,113],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,139],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,150],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,169],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,165],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,151],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,163],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,32],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,16],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,108],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,100],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[3,1],[[1,5,14,4,10,17]]],[[3,31],[[1,5,14,4,10,17]]],[[3,104],[[1,5,14,4,10,17]]],[[3,9],[[1,5,14,4,10,17]]],[[3,8],[[1,5,14,4,10,17]]],[[3,27],[[1,5,14,4,10,17]]],[[3,12],[[1,5,14,4,10,17]]],[[3,65],[[1,5,14,4,10,17]]],[[3,110],[[1,5,14,4,10,17]]],[[3,88],[[1,5,14,4,10,17]]],[[3,11],[[1,5,14,4,10,17]]],[[3,56],[[1,5,14,4,10,17]]],[[3,55],[[1,5,14,4,10,17]]],[[3,96],[[1,5,14,4,10,17]]],[[3,10],[[1,5,14,4,10,17]]],[[3,122],[[1,5,14,4,10,17]]],[[3,72],[[1,5,14,4,10,17]]],[[3,71],[[1,5,14,4,10,17]]],[[3,64],[[1,5,14,4,10,17]]],[[3,113],[[1,5,14,4,10,17]]],[[3,139],[[1,5,14,4,10,17]]],[[3,150],[[1,5,14,4,10,17]]],[[3,169],[[1,5,14,4,10,17]]],[[3,165],[[1,5,14,4,10,17]]],[[3,151],[[1,5,14,4,10,17]]],[[3,163],[[1,5,14,4,10,17]]],[[3,32],[[1,5,14,4,10,17]]],[[3,16],[[1,5,14,4,10,17]]],[[3,108],[[1,5,14,4,10,17]]],[[3,100],[[1,5,14,4,10,17]]],[[2,1],[[1,5,7,18,4,13,16,12]]],[[2,31],[[1,5,7,18,4,13,16,12]]],[[2,104],[[1,5,7,18,4,13,16,12]]],[[2,9],[[1,5,7,18,4,13,16,12]]],[[2,8],[[1,5,7,18,4,13,16,12]]],[[2,27],[[1,5,7,18,4,13,16,12]]],[[2,12],[[1,5,7,18,4,13,16,12]]],[[2,65],[[1,5,7,18,4,13,16,12]]],[[2,110],[[1,5,7,18,4,13,16,12]]],[[2,88],[[1,5,7,18,4,13,16,12]]],[[2,11],[[1,5,7,18,4,13,16,12]]],[[2,56],[[1,5,7,18,4,13,16,12]]],[[2,55],[[1,5,7,18,4,13,16,12]]],[[2,96],[[1,5,7,18,4,13,16,12]]],[[2,10],[[1,5,7,18,4,13,16,12]]],[[2,122],[[1,5,7,18,4,13,16,12]]],[[2,72],[[1,5,7,18,4,13,16,12]]],[[2,71],[[1,5,7,18,4,13,16,12]]],[[2,64],[[1,5,7,18,4,13,16,12]]],[[2,113],[[1,5,7,18,4,13,16,12]]],[[2,139],[[1,5,7,18,4,13,16,12]]],[[2,150],[[1,5,7,18,4,13,16,12]]],[[2,169],[[1,5,7,18,4,13,16,12]]],[[2,165],[[1,5,7,18,4,13,16,12]]],[[2,151],[[1,5,7,18,4,13,16,12]]],[[2,163],[[1,5,7,18,4,13,16,12]]],[[2,32],[[1,5,7,18,4,13,16,12]]],[[2,16],[[1,5,7,18,4,13,16,12]]],[[2,108],[[1,5,7,18,4,13,16,12]]],[[2,100],[[1,5,7,18,4,13,16,12]]]]]]],null,2],[true]]",null,"generic"]]]',
        ];
        $headers = [
            'Content-Type' => 'application/x-www-form-urlencoded;charset=utf-8',
        ];
        $body = Utils::streamFor(http_build_query($formParams));

        return new Request('POST', $url, $headers, $body);
    }

    /**
     * @param string $topSlug
     * @param string $categoryName
     * @param int    $limit
     * @param string $locale
     * @param string $country
     *
     * @return RequestInterface
     */
    public static function getTopCategoryApps(string $topSlug, string $categoryName, string $locale, string $country, int $limit = 1000): RequestInterface
    {
        $queryParams = [
            'rpcids' => self::RPC_ID_TOP,
            GPlayApps::REQ_PARAM_LOCALE => $locale,
            GPlayApps::REQ_PARAM_COUNTRY => $country,
            'source-path' => '/store/apps',
            'authuser' => null,
            'soc-app' => 121,
            'soc-platform' => 1,
            'soc-device' => 1,
        ];
        $url = GPlayApps::GOOGLE_PLAY_URL . '/_/PlayStoreUi/data/batchexecute?' . http_build_query($queryParams);
        $formParams = [
            'f.req' => '[[["' . self::RPC_ID_TOP . '","[[null,[[8,[20,' . $limit . ']],true,null,[64,1,195,71,8,72,9,10,11,139,12,16,145,148,150,151,152,27,30,31,96,32,34,163,100,165,104,169,108,110,113,55,56,57,122],[null,null,[[[true],null,[[[]]],null,null,null,null,[null,2],null,null,null,null,null,null,[1],null,null,null,null,null,null,null,[1]],[null,[[[]]]],[null,[[[]]],null,[true]],[null,[[[]]]],null,null,null,null,[[[[]]]],[[[[]]]]],[[[[7,1],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,31],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,104],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,9],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,8],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,27],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,12],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,65],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,110],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,88],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,11],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,56],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,55],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,96],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,10],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,122],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,72],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,71],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,64],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,113],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,139],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,150],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,169],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,165],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,151],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,163],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,32],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,16],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,108],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[7,100],[[1,73,96,103,97,58,50,92,52,112,69,19,31,101,123,74,49,80,20,10,14,79,43,42,139]]],[[9,1],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,31],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,104],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,9],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,8],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,27],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,12],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,65],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,110],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,88],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,11],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,56],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,55],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,96],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,10],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,122],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,72],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,71],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,64],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,113],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,139],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,150],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,169],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,165],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,151],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,163],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,32],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,16],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,108],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[9,100],[[1,7,9,24,12,31,5,15,27,8,13,10]]],[[17,1],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,31],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,104],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,9],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,8],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,27],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,12],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,65],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,110],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,88],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,11],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,56],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,55],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,96],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,10],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,122],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,72],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,71],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,64],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,113],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,139],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,150],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,169],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,165],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,151],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,163],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,32],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,16],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,108],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[17,100],[[1,7,9,25,13,31,5,41,27,8,14,10]]],[[10,1],[[1,7,6,9]]],[[10,31],[[1,7,6,9]]],[[10,104],[[1,7,6,9]]],[[10,9],[[1,7,6,9]]],[[10,8],[[1,7,6,9]]],[[10,27],[[1,7,6,9]]],[[10,12],[[1,7,6,9]]],[[10,65],[[1,7,6,9]]],[[10,110],[[1,7,6,9]]],[[10,88],[[1,7,6,9]]],[[10,11],[[1,7,6,9]]],[[10,56],[[1,7,6,9]]],[[10,55],[[1,7,6,9]]],[[10,96],[[1,7,6,9]]],[[10,10],[[1,7,6,9]]],[[10,122],[[1,7,6,9]]],[[10,72],[[1,7,6,9]]],[[10,71],[[1,7,6,9]]],[[10,64],[[1,7,6,9]]],[[10,113],[[1,7,6,9]]],[[10,139],[[1,7,6,9]]],[[10,150],[[1,7,6,9]]],[[10,169],[[1,7,6,9]]],[[10,165],[[1,7,6,9]]],[[10,151],[[1,7,6,9]]],[[10,163],[[1,7,6,9]]],[[10,32],[[1,7,6,9]]],[[10,16],[[1,7,6,9]]],[[10,108],[[1,7,6,9]]],[[10,100],[[1,7,6,9]]],[[1,1],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,31],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,104],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,9],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,8],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,27],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,12],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,65],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,110],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,88],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,11],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,56],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,55],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,96],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,10],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,122],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,72],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,71],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,64],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,113],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,139],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,150],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,169],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,165],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,151],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,163],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,32],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,16],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,108],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[1,100],[[1,5,14,38,19,29,34,4,12,11,6,30,43,40,42,16,10,7]]],[[4,1],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,31],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,104],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,9],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,8],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,27],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,12],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,65],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,110],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,88],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,11],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,56],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,55],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,96],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,10],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,122],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,72],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,71],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,64],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,113],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,139],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,150],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,169],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,165],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,151],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,163],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,32],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,16],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,108],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[4,100],[[1,3,5,4,7,6,11,19,21,17,15,12,16,20]]],[[3,1],[[1,5,14,4,10,17]]],[[3,31],[[1,5,14,4,10,17]]],[[3,104],[[1,5,14,4,10,17]]],[[3,9],[[1,5,14,4,10,17]]],[[3,8],[[1,5,14,4,10,17]]],[[3,27],[[1,5,14,4,10,17]]],[[3,12],[[1,5,14,4,10,17]]],[[3,65],[[1,5,14,4,10,17]]],[[3,110],[[1,5,14,4,10,17]]],[[3,88],[[1,5,14,4,10,17]]],[[3,11],[[1,5,14,4,10,17]]],[[3,56],[[1,5,14,4,10,17]]],[[3,55],[[1,5,14,4,10,17]]],[[3,96],[[1,5,14,4,10,17]]],[[3,10],[[1,5,14,4,10,17]]],[[3,122],[[1,5,14,4,10,17]]],[[3,72],[[1,5,14,4,10,17]]],[[3,71],[[1,5,14,4,10,17]]],[[3,64],[[1,5,14,4,10,17]]],[[3,113],[[1,5,14,4,10,17]]],[[3,139],[[1,5,14,4,10,17]]],[[3,150],[[1,5,14,4,10,17]]],[[3,169],[[1,5,14,4,10,17]]],[[3,165],[[1,5,14,4,10,17]]],[[3,151],[[1,5,14,4,10,17]]],[[3,163],[[1,5,14,4,10,17]]],[[3,32],[[1,5,14,4,10,17]]],[[3,16],[[1,5,14,4,10,17]]],[[3,108],[[1,5,14,4,10,17]]],[[3,100],[[1,5,14,4,10,17]]],[[2,1],[[1,5,7,4,13,16,12,18]]],[[2,31],[[1,5,7,4,13,16,12,18]]],[[2,104],[[1,5,7,4,13,16,12,18]]],[[2,9],[[1,5,7,4,13,16,12,18]]],[[2,8],[[1,5,7,4,13,16,12,18]]],[[2,27],[[1,5,7,4,13,16,12,18]]],[[2,12],[[1,5,7,4,13,16,12,18]]],[[2,65],[[1,5,7,4,13,16,12,18]]],[[2,110],[[1,5,7,4,13,16,12,18]]],[[2,88],[[1,5,7,4,13,16,12,18]]],[[2,11],[[1,5,7,4,13,16,12,18]]],[[2,56],[[1,5,7,4,13,16,12,18]]],[[2,55],[[1,5,7,4,13,16,12,18]]],[[2,96],[[1,5,7,4,13,16,12,18]]],[[2,10],[[1,5,7,4,13,16,12,18]]],[[2,122],[[1,5,7,4,13,16,12,18]]],[[2,72],[[1,5,7,4,13,16,12,18]]],[[2,71],[[1,5,7,4,13,16,12,18]]],[[2,64],[[1,5,7,4,13,16,12,18]]],[[2,113],[[1,5,7,4,13,16,12,18]]],[[2,139],[[1,5,7,4,13,16,12,18]]],[[2,150],[[1,5,7,4,13,16,12,18]]],[[2,169],[[1,5,7,4,13,16,12,18]]],[[2,165],[[1,5,7,4,13,16,12,18]]],[[2,151],[[1,5,7,4,13,16,12,18]]],[[2,163],[[1,5,7,4,13,16,12,18]]],[[2,32],[[1,5,7,4,13,16,12,18]]],[[2,16],[[1,5,7,4,13,16,12,18]]],[[2,108],[[1,5,7,4,13,16,12,18]]],[[2,100],[[1,5,7,4,13,16,12,18]]]]]],null,null,[[[1,2],[10,8,9],[],[]]]],[2,\"' . $topSlug . '\",\"' . $categoryName . '\"]]]",null,"generic"]]]',
        ];
        $headers = [
            'Content-Type' => 'application/x-www-form-urlencoded;charset=utf-8',
        ];
        $body = Utils::streamFor(http_build_query($formParams));

        return new Request('POST', $url, $headers, $body);
    }
}
