<?php
/**
 * Plugin Name: Motors All Import Addon
 * Description: Easily import Listing posts into the Motors theme using Motors All Import Addon. Perfect for managing vehicle inventories and classified ads.
 * Author: StylemixThemes
 * Author URI: https://stylemixthemes.com/
 * Version: 1.0.2
 */

require_once 'rapid-addon.php';

class MotorsAllImportAddon extends RapidAddon {

	public function __construct() {
		parent::__construct( 'Motors All Import Addon', 'motors_all_import_addon' );

		$this->add_field( 'featured_image', 'Group of image URLs (first image will be assigned as featured)', 'images' );
		$this->add_field( 'stm_images_sep', 'Images URL delimiter (By default - ",")', 'text' );
		$this->add_field( 'price', 'Price', 'text' );
		$this->add_field( 'sale_price', 'Sale price', 'text' );
		$this->add_field( 'marketing_price', 'Marketing price', 'text' );
		$this->add_field( 'regular_price_label', 'Regular price label', 'text' );
		$this->add_field( 'regular_price_description', 'Regular price description', 'text' );
		$this->add_field( 'special_price_label', 'Special price label', 'text' );
		$this->add_field( 'special_car', 'Special offer', 'text' );
		$this->add_field( 'special_text', 'Special offer text', 'text' );
		$this->add_field( 'badge_text', 'Special badge text', 'text' );
		$this->add_field( 'badge_bg_color', 'Special badge color', 'text' );
		$this->add_field( 'car_mark_as_sold', 'Mark as sold', 'text' );
		$this->add_field( 'car_price_form_label', 'Listing price form label', 'text' );
		$this->add_field( 'car_price_form', 'Listing price form', 'text' );
		$this->add_field( 'instant_savings_label', 'Instant savings label', 'text' );
		$this->add_field( 'stock_number', 'Stock number', 'text' );
		$this->add_field( 'vin_number', 'VIN', 'text' );
		$this->add_field( 'registration_date', 'Registration date', 'text' );
		$this->add_field( 'history', 'History', 'text' );
		$this->add_field( 'history_link', 'History link', 'text' );
		$this->add_field( 'city_mpg', 'City MPG', 'text' );
		$this->add_field( 'highway_mpg', 'Highway MPG', 'text' );
		$this->add_field( 'stm_car_location', 'Car location (Address)', 'text' );
		$this->add_field( 'stm_lat_car_admin', 'Car location lat', 'text' );
		$this->add_field( 'stm_lng_car_admin', 'Car location lng', 'text' );
		$this->add_field( 'gallery_video', 'Gallery video', 'text' );
		$this->add_field( 'stm_car_user', 'User added (User ID)', 'text' );
		$this->add_field( 'automanager_id', 'Automanager ID', 'text' );
		$this->add_field( 'additional_features', 'Additional features', 'text' );

		$filter_options = get_option( 'stm_vehicle_listing_options' );

		// multilisting compatibility
		if ( function_exists( 'wp_all_import_get_import_post_type' ) && function_exists( 'stm_listings_post_type' ) ) {
			$current_type = wp_all_import_get_import_post_type();
			if ( $current_type && stm_listings_post_type() !== $current_type ) {
				$filter_options = get_option( "stm_{$current_type}_options", array() );
			}
		}

		if ( ! empty( $filter_options ) && is_array( $filter_options ) ) {
			foreach ( $filter_options as $filter_option ) {
				if ( ! empty( $filter_option['numeric'] ) ) {
					$this->add_field( $filter_option['slug'], $filter_option['single_name'], 'text' );
				}
			}
		}

		$this->set_import_function( array( $this, 'motors_aia_import_generate_content' ) );
		$this->set_post_saved_function( array( $this, 'saved_post' ) );

		$this->admin_notice();

		$run_array['post_types'] = array( 'listings' );

		if ( function_exists( 'stm_is_multilisting' ) && stm_is_multilisting() ) {
			$slugs = STMMultiListing::stm_get_listing_type_slugs();
			if ( ! empty( $slugs ) && is_array( $slugs ) ) {
				$run_array['post_types'] = array_merge( $run_array['post_types'], $slugs );
			}
		}

		$this->run( $run_array );
	}

	public function motors_aia_import_generate_content( $post_id, $data, $import_options, $postData, $logger ) {
		remove_action( 'save_post', 'stm_butterbean_save_post', 100 );
		remove_action( 'save_post', 'stm_save_metaboxes' );

		if ( class_exists( 'MotorsVehiclesListing\ListingMetaboxes' ) ) {
			MotorsVehiclesListing\ListingMetaboxes::instance()->remove_action( 'save_post' );
		}

		update_post_meta( $post_id, '_wpb_vc_js_status', 'false' );
		update_post_meta( $post_id, 'title', 'hide' );

		$logger && call_user_func( $logger, '<p><b>Post ID: </b>' . $post_id . '</p>' );
		$logger && call_user_func( $logger, '<p><b>Post Title: </b>' . get_the_title( $post_id ) . '</p>' );

		$keys = array(
			'stock_number',
			'price',
			'regular_price_label',
			'regular_price_description',
			'regular_price_description',
			'special_price_label',
			'instant_savings_label',
			'vin_number',
			'registration_date',
			'history',
			'history_link',
			'city_mpg',
			'highway_mpg',
			'stm_car_location',
			'stm_lat_car_admin',
			'stm_lng_car_admin',
			'gallery_video',
			'stm_car_user',
			'automanager_id',
			'car_price_form_label',
			'additional_features',
			'badge_text',
			'badge_bg_color',
			'special_text',
		);

		foreach ( $keys as $meta_key ) {
			if ( $this->can_update_meta( $meta_key, $import_options ) ) {
				update_post_meta( $post_id, $meta_key, $this->get_data_value( $data, $meta_key ) );
			}
		}

		if ( $this->can_update_meta( 'stm_car_user', $import_options ) ) {
			if ( ! empty( $data['stm_car_user'] ) ) {
				wp_update_post(
					array(
						'ID'          => $post_id,
						'post_author' => $data['stm_car_user'],
					)
				);
			}
		}

		if ( $this->can_update_meta( 'marketing_price', $import_options ) ) {
			if ( floatval( $this->get_data_value( $data, 'marketing_price' ) ) > 0 ) {
				update_post_meta( $post_id, 'sale_price', $this->get_data_value( $data, 'marketing_price' ) );
			} else {
				if ( $this->can_update_meta( 'sale_price', $import_options ) ) {

					if ( floatval( $this->get_data_value( $data, 'sale_price' ) ) > 0 ) {
						update_post_meta( $post_id, 'sale_price', $this->get_data_value( $data, 'sale_price' ) );
					} else {
						delete_post_meta( $post_id, 'sale_price' );
					}
				}
			}
		}

		if ( $this->can_update_meta( 'special_car', $import_options ) ) {
			if ( ! empty( $data['special_car'] ) ) {
				$special_car_value = $this->get_data_value( $data, 'special_car' );
				update_post_meta( $post_id, 'special_car', $special_car_value );
			}
		}

		if ( $this->can_update_meta( 'car_mark_as_sold', $import_options ) ) {
			if ( ! empty( $data['car_mark_as_sold'] ) ) {
				$car_mark_as_sold_value = $this->get_data_value( $data, 'car_mark_as_sold' );
				update_post_meta( $post_id, 'car_mark_as_sold', $car_mark_as_sold_value );
			}
		}

		if ( $this->can_update_meta( 'car_price_form', $import_options ) ) {
			if ( ! empty( $data['car_price_form'] ) && 'yes' === $this->get_data_value( $data, 'car_price_form' ) ) {
				update_post_meta( $post_id, 'car_price_form', 'on' );
			} else {
				delete_post_meta( $post_id, 'car_price_form' );
			}
		}

		// Update post meta values from attributes.
		$filter_options = get_option( 'stm_vehicle_listing_options' );

		if ( function_exists( 'wp_all_import_get_import_post_type' ) && function_exists( 'stm_listings_post_type' ) ) {
			$current_type = wp_all_import_get_import_post_type();
			if ( $current_type && stm_listings_post_type() !== $current_type ) {
				$filter_options = get_option( "stm_{$current_type}_options", array() );
			}
		}

		if ( ! empty( $filter_options ) ) {
			foreach ( $filter_options as $filter_option ) {
				$slug = $filter_option['slug'];
				if ( ! empty( $data[ $slug ] ) && update_post_meta( $post_id, $slug, $data[ $slug ] ) ) {
					$logger && call_user_func( $logger, "- Attribute saved: `{$slug}`: {$data[ $slug ]}" );
				}
			}
		}

		if ( ! empty( $data['featured_image'] ) ) {
			$gallery = array();

			$post_thumbnail_id = $data['featured_image'][0]['attachment_id'];

			$logger && call_user_func( $logger, '<p><b>Featured Image: </b>' . $post_thumbnail_id . '</p>' );
			update_post_meta( $post_id, 'featured_image_tmp', $post_thumbnail_id );

			foreach ( $data['featured_image'] as $image ) {
				if ( $image['attachment_id'] !== $post_thumbnail_id ) {
					$gallery[] = $image['attachment_id'];
				}
			}

			update_post_meta( $post_id, 'gallery', $gallery );

			$logger && call_user_func( $logger, '<p><b>Gallery: </b>' . wp_json_encode( $gallery ) . '</p>' );
		}
	}

	protected function get_data_value( $data, $key ) {
		$value = '';

		if ( isset( $data[ $key ] ) ) {
			$arr   = explode( '|', $data[ $key ] );
			$value = end( $arr );
		}

		return $value;
	}

	public function saved_post( $post_id, $import = null, $logger = null ) {
		// Additional features.
		if ( function_exists( 'stm_get_wpb_def_tmpl' ) && stm_get_wpb_def_tmpl() ) {
			$post_data['ID']           = $post_id;
			$post_data['post_content'] = stm_get_wpb_def_tmpl();
			update_post_meta( $post_id, '_wpb_vc_js_status', 'true' );

			wp_update_post( $post_data );
		}

		// Hidden price.
		$price      = get_post_meta( $post_id, 'price', true );
		$sale_price = get_post_meta( $post_id, 'sale_price', true );

		if ( ! empty( $sale_price ) ) {
			$price = $sale_price;
		}

		update_post_meta( $post_id, 'stm_genuine_price', $price );

		// Update post meta values from attributes.
		$filter_options = get_option( 'stm_vehicle_listing_options' );

		if ( function_exists( 'wp_all_import_get_import_post_type' ) && function_exists( 'stm_listings_post_type' ) ) {
			$current_type = wp_all_import_get_import_post_type();
			if ( $current_type && stm_listings_post_type() !== $current_type ) {
				$filter_options = get_option( "stm_{$current_type}_options", array() );
			}
		}

		if ( ! empty( $filter_options ) ) {
			foreach ( $filter_options as $filter_option ) {
				if ( $filter_option['numeric'] ) {
					continue;
				}

				$old_value  = get_post_meta( $post_id, $filter_option['slug'], true );
				$meta_value = null;
				$terms      = wp_get_post_terms( $post_id, $filter_option['slug'] );
				if ( ! empty( $terms ) ) {
					$meta_value = implode( ',', wp_list_pluck( $terms, 'slug' ) );
				}

				if ( ! empty( $meta_value ) ) {
					if ( $meta_value !== $old_value && update_post_meta( $post_id, $filter_option['slug'], $meta_value ) ) {
						$logger && call_user_func( $logger, "- Attribute saved: `{$filter_option['slug']}`: {$meta_value}" );
					}
				} elseif ( ! empty( $old_value ) && delete_post_meta( $post_id, $filter_option['slug'] ) ) {
					$logger && call_user_func( $logger, "- Attribute deleted: `{$filter_option['slug']}`" );
				}
			}
		}

		$gallery = get_post_meta( $post_id, 'gallery', true );

		$taxonomy = 'media_category';

		$listing_type = 'listings';

		if ( function_exists( 'stm_listings_post_type' ) ) {
			if ( $current_type && stm_listings_post_type() !== $current_type ) {
				$listing_type = $current_type;
			}
		}

		$term = get_term_by( 'slug', $listing_type, $taxonomy );

		if ( ! $term ) {
			$new = wp_insert_term( 'Listings', $taxonomy, array( 'slug' => $listing_type ) );
			if ( ! is_wp_error( $new ) ) {
				$term = get_term( $new['term_id'] );
			}
		}

		if ( $term ) {
			foreach ( $gallery as $id ) {
				wp_set_object_terms( $id, $term->term_id, $taxonomy );
			}
		}
	}

	public function getCurrentSalePrice( $salePrice, $marketingPrice ) {
		if ( floatval( $marketingPrice ) > 0 ) {
			return $marketingPrice;
		} else {
			return $salePrice;
		}
	}
}

add_action( 'plugins_loaded', 'initialize_motors_all_import_addon' );
function initialize_motors_all_import_addon() {
	new MotorsAllImportAddon();
}

// After post saved.
add_action( 'pmxi_saved_post', 'motors_aia_after_pmxi_post_import', 100, 1 );
function motors_aia_after_pmxi_post_import( $post_id ) {
	wp_cache_flush();

	$filter_options = get_option( 'stm_vehicle_listing_options' );

	if ( function_exists( 'wp_all_import_get_import_post_type' ) && function_exists( 'stm_listings_post_type' ) ) {
		$current_type = wp_all_import_get_import_post_type();
		if ( $current_type && stm_listings_post_type() !== $current_type ) {
			$filter_options = get_option( "stm_{$current_type}_options", array() );
		}
	}

	if ( ! empty( $filter_options ) ) {
		foreach ( $filter_options as $filter_option ) {

			if ( $filter_option['numeric'] ) {
				continue;
			}

			$old_value  = get_post_meta( $post_id, $filter_option['slug'], true );
			$meta_value = null;

			$terms = wp_get_post_terms( $post_id, $filter_option['slug'] );

			if ( ! empty( $terms ) ) {
				$meta_value = implode( ',', wp_list_pluck( $terms, 'slug' ) );
			}

			if ( ! empty( $meta_value ) ) {
				if ( $meta_value !== $old_value ) {
					update_post_meta( $post_id, $filter_option['slug'], $meta_value );
				}
			} elseif ( ! empty( $old_value ) ) {
				delete_post_meta( $post_id, $filter_option['slug'] );
			}
		}
	}

	$old_value  = get_post_meta( $post_id, 'additional_features', true );
	$meta_value = null;

	if ( empty( $old_value ) ) {
		$terms = wp_get_post_terms( $post_id, 'stm_additional_features' );

		if ( ! empty( $terms ) ) {
			$meta_value = implode( ',', wp_list_pluck( $terms, 'name' ) );
			if ( ! empty( $meta_value ) ) {
				if ( $meta_value !== $old_value ) {
					update_post_meta( $post_id, 'additional_features', $meta_value );
				}
			}
		}
	}

	$featuredImgTmp = get_post_meta( $post_id, 'featured_image_tmp', true );
	if ( ! empty( $featuredImgTmp ) ) {
		update_post_meta( $post_id, '_thumbnail_id', $featuredImgTmp );
		delete_post_meta( $post_id, 'featured_image_tmp' );
	}
}
