<?php
/**
 *    Oxygen WordPress Theme
 *
 *    Laborator.co
 *    www.laborator.co
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class Oxygen_Freemius {

	/**
	 * Instance.
	 *
	 * @var self
	 */
	private static $instance;

	/**
	 * Create the instance of the class.
	 *
	 * @return self
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Construct.
	 */
	private function __construct() {
		add_action( 'oxygen_fs_loaded', [ $this, 'init' ] );
		add_action( 'wp_ajax_theme_license_convert', [ $this, 'theme_license_convert' ] );
		add_action( 'wp_ajax_theme_license_activation_details', [ $this, 'theme_license_activation_details' ] );
	}

	/**
	 * Get Freemius instance.
	 *
	 * @return Freemius
	 */
	public function get_fs_instance() {
		return oxygen_fs();
	}

	/**
	 * Init
	 */
	public function init() {

		// License converter
		if ( $this->is_from_themeforest() ) {
			$this->get_fs_instance()->add_action( 'connect/before', [ $this, 'envato_license_converter' ] );
		}

		$return_hook_value = function ( $value ) {
			return function () use ( $value ) {
				return $value;
			};
		};

		// Activate theme sticky message text
		$activate_theme_text = sprintf( 'Thanks for installing %s! <b><a href="%s">Please enter your license key to unlock all features.</a></b>', wp_get_theme(), $this->get_fs_instance()->get_activation_url() );
		$this->get_fs_instance()->add_filter( 'sticky_message_connect_account', $return_hook_value( $activate_theme_text ) );

        // Pricing page
        $this->get_fs_instance()->add_filter( 'pricing/show_annual_in_monthly', '__return_false' );

		// Replace strings
		$this->replace_freemius_strings();
	}

	/**
	 * Is the theme installed from ThemeForest.
	 *
	 * @return bool
	 */
	public function is_from_themeforest() {
		return require_once __DIR__ . '/themeforest.php';
	}

	/**
	 * Envato to Freemius license converter.
	 */
	public function envato_license_converter() {
		wp_enqueue_style( 'oxygen-license-converter', oxygen_assets_url() . '/css/admin/license-converter.min.css' );
		wp_enqueue_script( 'oxygen-license-converter', oxygen_assets_url() . '/js/admin/license-converter.min.js' );

		$theme_name = (string) wp_get_theme();

		?>
		<div class="license-converter">
			<h1>Transfer Your Envato License &amp; Activate <?php echo $theme_name; ?></h1>
            <p>We have moved <?php echo $theme_name; ?> licensing and updates to Freemius, replacing the previous system with Envato. To continue using and receiving updates directly through your WordPress dashboard, you will need to transfer your existing license to Freemius.</p>
			<p>This will create an account with Freemius, our new license provider, and grant you theme updates.
				Please provide the email address you want to use for your <?php echo $theme_name; ?> account. If you
				already have a Freemius account, use that email.</p>

			<form action="<?php echo esc_attr( $_SERVER['REQUEST_URI'] ); ?>" method="post">
				<div class="form-field">
					<label for="email">Email:</label>
					<input type="email" class="regular-text" id="email" name="email" autocomplete="email" required="">
				</div>

				<div class="form-field">
					<label for="purchase-code">Purchase Code (<a href="https://documentation.laborator.co/kb/oxygen/how-to-find-purchase-code-oxygen/" target="_blank" rel="noopener noreferrer">How to find purchase code?</a>):</label>
					<input type="text" class="regular-text" id="purchase-code" name="purchase_code" required="">
				</div>

				<button type="submit" class="button button-primary">Activate <?php echo $theme_name; ?></button>
			</form>

			<span class="form-consent">By completing this form, you agree to let Freemius process your data for future updates.</span>
		</div>

		<div class="license-information">
			<h2>FAQ</h2>

			<details>
				<summary>What does this mean for me?</summary>
				<div>
					<p>
						If you purchased <strong><?php echo esc_html( $theme_name ); ?></strong> before <strong>November 1, 2024</strong> through <strong>ThemeForest (Envato)</strong>, you are entitled to a lifetime single-site license.
						This means no additional payments are required.
					</p>
					<p>
						Additionally, you will receive a <strong><?php echo esc_html( $theme_name ); ?> Legacy</strong> license, which grants you lifetime access to automatic updates directly in your WordPress Dashboard.
					</p>
					<p>
						For purchases made on or after <strong>November 1, 2024</strong>, the <strong><?php echo esc_html( $theme_name ); ?> Legacy</strong> license will include automatic updates for <strong>6 months</strong> from the date of purchase.
					</p>
					<p>
						Regardless of when you purchased, you can always download the latest version of the theme from the Envato Marketplace.
					</p>
				</div>
			</details>

			<details>
				<summary>Why has the price increased on ThemeForest?</summary>
				<div>
					<p>We've opted out of exclusivity with Envato, leading to a decrease in our author share and a price
						increase on ThemeForest. However, by purchasing directly from our website, you can take
						advantage of
						better pricing options, including exclusive deals, flexible plans, and additional benefits that
                        aren’t available on ThemeForest. <a href="https://laborator.co/themes/oxygen/" target="_blank" rel="noopener">Visit our site</a> to see how you can get more value and
						savings!</p>
				</div>
			</details>
		</div>
		<?php
	}

	/**
	 * Upgrade validate data.
	 */
	public function theme_license_convert() {
		if ( ! current_user_can( 'update_themes' ) ) {
			return;
		}

		$email         = trim( $_POST['email'] ?? '' );
		$purchase_code = trim( $_POST['purchaseCode'] ?? '' );

		// Transfer license
		$transfer_request = wp_remote_post(
			'https://api.laborator.co/?transfer-license=oxygen',
			[
				'timeout' => 30,
				'body'    => [
					'email'         => $email,
					'purchase_code' => $purchase_code,
				],
			]
		);

		// Errors
		if ( is_wp_error( $transfer_request ) ) {
			wp_send_json(
				[
					'success' => false,
					'message' => $transfer_request->get_error_message(),
				]
			);
		}

		$transfer_response = json_decode( wp_remote_retrieve_body( $transfer_request ), true );

		wp_send_json( $transfer_response );
	}

	/**
	 * Get license activation details to activate the theme.
	 *
	 * @since 4.0
	 */
	public function theme_license_activation_details() {
		if ( ! current_user_can( 'update_themes' ) ) {
			return;
		}

		$action = $this->get_fs_instance()->get_ajax_action( 'activate_license' );

		// Response
		wp_send_json(
			[
				'success'     => true,
				'ajax_url'    => Freemius::ajax_url( '' ),
				'ajax_action' => $action,
				'security'    => wp_create_nonce( $action ),
				'module_id'   => $this->get_fs_instance()->get_id(),
			]
		);
	}

	/**
	 * Get license object.
	 *
	 * @return FS_Plugin_License|bool
	 */
	public function get_license() {
		return $this->get_fs_instance()->_get_license();
	}

	/**
	 * Get license key.
	 *
	 * @return string|null
	 */
	public function get_license_key() {
		if ( $license_data = $this->get_data() ) {
			return $license_data['secret_key'];
		}

		return null;
	}

	/**
	 * Check if theme is registered.
	 *
	 * @return boolean
	 */
	public function is_theme_registered() {
		return $this->get_fs_instance()->is_registered();
	}

	/**
	 * Check if license is expired.
	 *
	 * @return bool
	 */
	public function is_expired() {
		$license = $this->get_license();

		return ! $license || $license->is_expired();
	}

	/**
	 * Checks if license is lifetime deal.
	 *
	 * @return bool
	 */
	public function is_lifetime() {
		$license = $this->get_license();

		if ( $license ) {
			return $license->is_lifetime();
		}

		return false;
	}

	/**
	 * Checks if theme has active subscription.
	 *
	 * @return bool
	 */
	public function is_active() {
		return $this->is_lifetime() || ! $this->is_expired();
	}

	/**
	 * Checks if subscription auto-renews.
	 *
	 * @return bool
	 */
	public function is_auto_renew() {
		$license = $this->get_license();

		if ( $license && ! $this->is_lifetime() ) {
			$subscription = $this->get_fs_instance()->_get_subscription( $license->id );

			if ( $subscription && $subscription->is_active() ) {
				return ! $subscription->is_first_payment_pending();
			}
		}

		return false;
	}

	/**
	 * Checks is license white labeled.
	 *
	 * @return bool
	 */
	public function is_white_labeled() {
		if ( $license = $this->get_license() ) {
			return $license->is_whitelabeled;
		}

		return false;
	}

	/**
	 * Wrap URL with license key.
	 *
	 * @param string $url
	 *
	 * @return string
	 */
	public function wrap_license_key( $url ) {
		if ( $url && ( $license_key = $this->get_license_key() ) ) {
			return add_query_arg( 'license_key', urlencode( $license_key ), $url );
		}

		return $url;
	}

	/**
	 * Get account page URL.
	 *
	 * @return string
	 */
	public function get_account_url() {
		return $this->get_fs_instance()->get_account_url();
	}

	/**
	 * Get plan.
	 *
	 * @return string
	 */
	public function get_plan_name() {
		if ( $plan = $this->get_fs_instance()->get_plan() ) {
			return $plan->name;
		}

		return null;
	}

	/**
	 * Get current plan title.
	 *
	 * @return string
	 */
	public function get_plan_title() {
		if ( $plan = $this->get_fs_instance()->get_plan() ) {
			return $plan->title;
		}

		return 'Unknown';
	}

	/**
	 * Get license data.
	 *
	 * @return array|null
	 */
	public function get_data() {
		if ( $license = $this->get_license() ) {
			$user = $this->get_fs_instance()->get_user();
			$site = $this->get_fs_instance()->get_site();

			return [
				'plan'              => $this->get_plan_title(),
				'subscription_term' => $this->is_lifetime() ? 'Lifetime' : 'Annual',
				'expiration'        => $this->is_lifetime() ? false : strtotime( $license->expiration ),
				'secret_key'        => $license->secret_key,
				'secret_key_masked' => $license->get_html_escaped_masked_secret_key(),
				'name'              => $user ? $user->get_name() : null,
				'email'             => $user ? $user->email : null,
				'site_id'           => $site ? $site->id : null,
			];
		}

		return null;
	}

	/**
	 * Replace Freemius SDK strings.
	 */
	private function replace_freemius_strings() {
		$this->get_fs_instance()->override_i18n(
			[
				'yee-haw'                => 'Awesome',
				'plan-activated-message' => wp_get_theme() . ' was successfully activated.',
				'plan-upgraded-message'  => wp_get_theme() . ' was successfully upgraded.',
			]
		);
	}
}

// Initialize the class.
Oxygen_Freemius::instance();
