<?php
/**
 *    Oxygen WordPress Theme
 *
 *    Laborator.co
 *    www.laborator.co
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * Checks if WPBakery is used as theme-plugin mode.
 *
 * @return bool
 */
function oxygen_wpb_is_vc_theme_plugin_mode() {
	if ( function_exists( 'vc_license' ) && vc_license()->isActivated() ) {
		return false;
	}

	return get_option( 'oxygen_wpb_theme_plugin_mode', true );
}

/**
 * Disable updater, our theme provides updates for this plugin.
 */
function oxygen_wpb_disable_vc_updater() {
	vc_manager()->disableUpdater();
}

/**
 * Allow automatic update for WPBakery.
 */
function oxygen_wpb_allow_automatic_update() {
	remove_filter( 'upgrader_pre_download', [ vc_updater(), 'preUpgradeFilter' ], 10, 4 );
}

/**
 * Hide promo popups.
 */
function oxygen_wpb_hide_promo_popups() {
	global $pagenow;

	if ( in_array( $pagenow, [ 'post.php', 'post-new.php' ], true ) ) {
		echo '<style>#vc_ui-helper-promo-popup {display: none !important;}</style>';
	}
}

/**
 * Hide AI icon from param type.
 *
 * @param string $output
 *
 * @return string
 */
function oxygen_wpb_hide_vc_ai_icon( $output ) {
	return str_replace( 'class="vc_ui-icon-ai"', 'class="vc_ui-icon-ai hidden" style="display:none"', $output );
}

/**
 * Hide several VC notices.
 *
 * @return array
 */
function oxygen_wpb_hide_vc_notices( $notices ) {
	return [ 'empty_api_response' => true ];
}

/**
 * Hide user notices for WPBakery.
 * 
 * @param mixed $value
 * @param int $object_id
 * @param string $meta_key
 * 
 * @return mixed
 */
function oxygen_wpb_hide_user_notices( $value, $object_id, $meta_key ) {
	if ( 'wpb_expire_close_list' !== $meta_key ) {
		return $value;
	}

	// Get current user metadata
	remove_filter( 'get_user_metadata', 'oxygen_wpb_hide_user_notices', 10, 3 );
	$metadata = get_user_meta( $object_id, $meta_key, true );
	add_filter( 'get_user_metadata', 'oxygen_wpb_hide_user_notices', 10, 3 );

	return json_encode(
		array_merge(
			(array) json_decode( $metadata, true ),
			[
				'not_active_license' => [
					defined( 'WPB_VC_VERSION' ) ? WPB_VC_VERSION : false,
				],
			]
		)
	);
}

/**
 * Theme-plugin mode option.
 */
function oxygen_wpb_vc_theme_plugin_mode_option() {
	if ( ! defined( 'WPB_VC_VERSION' ) ) {
		return;
	}

	$theme_plugin_mode_field_name = 'oxygen_wpb_theme_plugin_mode';

	$field_callback = function () use ( $theme_plugin_mode_field_name ) {
		$checked = get_option( $theme_plugin_mode_field_name, true );

		if ( empty( $checked ) ) {
			$checked = false;
		}
		?>
		<label>
			<input type="checkbox"<?php checked( $checked ); ?> value="1"
			       id="<?php echo esc_attr( $theme_plugin_mode_field_name ); ?>"
			       name="<?php echo esc_attr( $theme_plugin_mode_field_name ); ?>">
			<?php esc_html_e( 'Enable', 'js_composer' ); ?>
		</label><br/>
		<p class="description indicator-hint">The theme automatically provides WPBakery plugin updates. If you hold a
			WPBakery license and intend to apply it to this site, please turn off this feature.</p>
		<?php
	};

	register_setting( 'wpb_js_composer_settings_general', $theme_plugin_mode_field_name, [
		'sanitize_callback' => function ( $state ) {
			return ! ! $state;
		}
	] );
	add_settings_field( $theme_plugin_mode_field_name, 'Theme plugin mode', $field_callback, 'vc_settings_general', 'wpb_js_composer_settings_general' );

	// Create option with default value
	add_option( 'oxygen_wpb_theme_plugin_mode', true );
}

add_action( 'admin_init', 'oxygen_wpb_vc_theme_plugin_mode_option', 0 );

if ( oxygen_wpb_is_vc_theme_plugin_mode() ) {
	add_action( 'vc_before_init', 'vc_set_as_theme' );
	add_action( 'vc_before_init', 'oxygen_wpb_disable_vc_updater' );
	add_action( 'vc_after_init', 'oxygen_wpb_allow_automatic_update' );
	add_action( 'admin_footer', 'oxygen_wpb_hide_promo_popups' );
	add_filter( 'vc_single_param_edit_holder_output', 'oxygen_wpb_hide_vc_ai_icon' );
	add_filter( 'transient_wpb_notice_list', 'oxygen_wpb_hide_vc_notices' );
	add_filter( 'get_user_metadata', 'oxygen_wpb_hide_user_notices', 10, 3 );
}
