<?php
namespace Phox\core\admin;
/**
 * Create sub menu on admin dashboard
 * @since 2.3.7
 */
class Dashboard_Menu {

    /**
     * The slug name for the parent menu.
     * @var string
     */
    private string $parent_slug;

    /**
     * The text to be displayed in the title tags of the page when the menu is selected.
     * @var string
     */
    private string $page_title;

    /**
     * The slug name to refer to this menu.
     * @var string
     */
    private string $menu_slug;

    /**
     * The text to be used for the menu.
     * @var string
     */
    private string $menu_title;

    /**
     * The capability required for this menu to be displayed to the user.
     * @var string
     */
    private string $capability;

    /**
     * Output the content for this page
     * @var string
     */
    private string $template_path;

    /**
     * The URL to the icon to be used for this menu
     * @var string
     */
    private string $menu_icon;

    /**
     * The position in the menu order this item should appear.
     * @var int|float|null
     */
    private int|float|null $position;

    /**
     * The text to be used for the menu.
     *
     * @var string
     */
    private string $menu_name;

    /**
     * Render Vars
     * @var array Pass data to render template
     */
    private array $render_vars = [];

    /**
     * @param string  $parent_slug The slug name for the parent menu or if your create new menu this will menu id
     * @param string  $page_title The text to be displayed in the title tags of the page when the menu
     *                                is selected.
     * @param string $menu_slug The slug name to refer to this menu by. Should be unique for this menu
     *                          and only include lowercase alphanumeric, dashes, and underscores characters
     *                          to be compatible with sanitize_key().
     * @param string $menu_title The text to be used for the menu.
     * @param string $capability The capability required for this menu to be displayed to the user.
     */
    public function __construct(
        string $parent_slug,
        string $page_title,
        string $menu_slug,
        string $menu_title,
        string $capability
    )
    {
        $this->parent_slug = $parent_slug;
        $this->page_title  = esc_html($page_title);
        $this->menu_slug   = $menu_slug;
        $this->menu_title  = esc_html($menu_title);
        $this->capability  = $capability;

        $this->position = null;

    }

    /**
     * Adds a submenu page.
     *
     * This function takes a capability which will be used to determine whether or not a page is included in the menu.
     *
     * @link https://developer.wordpress.org/reference/functions/add_submenu_page/
     * @return void
     */
    public function create_sub_menu():void
    {

        add_submenu_page(
            $this->parent_slug,
            $this->page_title,
            $this->menu_title,
            $this->capability,
            $this->menu_slug,
            [$this, 'render'],
            $this->position
        );

    }

    public function create_menu():void
    {
        add_menu_page(
            $this->page_title,
            $this->menu_title,
            $this->capability,
            $this->menu_slug,
            [$this, 'render'],
            $this->menu_icon,
            $this->position
        );
    }

    /**
     * Set Template
     *
     * @param string $template path to template that will have the output content
     * @return void
     */
    public function setTemplatePath(string $path): void
    {
        $this->template_path = $path;

    }

    /**
     * Set Menu Icon
     *
     * @param string $icon Pass a base64-encoded SVG using a data URI,
     * Pass the name of a Dashicons helper class
     * Pass 'none' to leave div.wp-menu-image empty
     * @return void
     */
    public function set_menu_icon(string $icon): void
    {
        $this->menu_icon = $icon;
    }

    /**
     * Set position
     *
     * @param int|float $position The position in the menu order this item should appear.

     * @return void
     */
    public function set_position(int|float $position): void
    {
        $this->position = $position;
    }

    /**
     * Set menu name
     *
     * @param string $menu_name
     * @return void
     */
    public function set_menu_name(string $menu_name): void
    {
        $this->menu_name = $menu_name;
    }

    public function set_render_vars(array $render_vars): void
    {
        $this->render_vars = $render_vars;
    }

    /**
     * Render Template
     *
     * @return void
     */
    public function render()
    {
        ob_start();
        
        $vars = $this->render_vars;

        require_once wp_normalize_path($this->template_path);

        ob_end_flush();
    }


    /**
     * This action is used to add extra submenus and menu options to the admin panel’s menu structure.
     *
     * @link https://developer.wordpress.org/reference/hooks/admin_menu/
     * @return void
     */
    public function register_menu($relation):void
    {
        if($relation == 'children')
            add_action('admin_menu', [$this,'create_sub_menu']);

        if($relation == 'parent')
            add_action('admin_menu', [$this,'create_menu']);

    }

    /**
     * All process to register sub menu
     *
     * @param string $path setTemplatPath method prama
     * @return void
     */
    public function register_sub_menu(string $path):void
    {

        $this->setTemplatePath($path);
        $this->register_menu('children');

    }

    public function change_menu_name():void
    {
        global $submenu;

        if ( current_user_can( 'edit_theme_options' ) ) {
            $submenu[$this->menu_slug][0][0] = esc_html__( $this->menu_name, 'phox' );
        }
    }

    public function register_main_menu(string $path, string $icon, int|float $position = null):void
    {
        $this->setTemplatePath($path);

        $this->set_menu_icon($icon);

        if( ! is_null($position) )
            $this->set_position($position);

        $this->register_menu('parent');
    }

    public function apply_new_menu_name(string $menu_name):void
    {
        $this->set_menu_name($menu_name);

        add_action('admin_menu', [$this,'change_menu_name'], 999);
    }

    /**
     * Remove TGMPA menu item and notice message
     * @see
     * @return void
     */
    public function remove_TGMPA_OCID_menu()
    {
        add_action('admin_menu', function (){
            remove_submenu_page( 'phox', 'wdes-plugins' );
            remove_submenu_page( 'themes.php', 'wdes-demo-import' );
            remove_action( 'admin_notices', [ $GLOBALS['tgmpa'], 'notices' ] );
        }, 999);

    }


}
