<?php
namespace Phox\core\admin;
/**
 * Register style and script
 * @since 2.3.7
 */
class Enqueue {
    /**
     * base folder directory
     *
     * @var string
     */
    private string $base_url;

    /**
     * Collection of style files that will use to add on enqueue style function
     * @var array
     */
    private array $style_collection = [];

    /**
     * Collection of script files that will use to add on enqueue script function
     * @var array
     */
    private array $script_collection = [];

    /**
     * Collection of inline scripts data that will use to add on enqueue script function
     * @var array
     */
    private array $inline_script_collection = [];

    /**
     * Collection of font data that will use to add on enqueue script function
     * @var array
     */
    private array $font_collection = [];

    /**
     * Hook Suffix
     * @var string hook suffix
     */
    private string $hook_suffix = '';

    /**
     * Set Hook Suffix
     *
     * @param string $hook_suffix
     * @return void
     */
    public function set_hook_suffix(string $hook_suffix): void
    {
        $this->hook_suffix = $hook_suffix;
    }

    /**
     * When use registers methods you need to set url fo css to js file
     *
     * @param string $base_dir dir path
     * @return void
     */
    public function set_base_url(string $base_url): void
    {
        $this->base_url = $base_url;
    }

    /**
     * Add style to collection.
     *
     * @param string $handle Name of the stylesheet. Should be unique. By default, add to all handle 'wdes-' and then parameter
     * @param string $filename only filename you need to add dir from setBaseDir
     * @param array $deps
     *
     * @return void
     */
    public function add_style(string $handle, string $filename, array $deps = []): void
    {

        $this->style_collection[$handle] = [
            'handle' => $handle,
            'filename' => $filename,
            'deps' => $deps,
        ];

    }

    /**
     * Add script to collection.
     *
     * @param string $handle Name of the stylesheet. Should be unique. By default, add to all handle 'wdes-' and then parameter
     * @param string $filename only filename you need to add dir from setBaseDir
     * @param array $deps
     *
     * @return void
     */
    public function add_script(string $handle, string $filename, array $deps = []): void
    {
        $this->script_collection[$handle] = [
            'handle' => $handle,
            'filename' => $filename,
            'deps' => $deps,
        ];
    }

    /**
     * Registers the style and enqueues.
     *
     * Looping from collection and use wp function to add style to enqueue
     *
     * @link https://developer.wordpress.org/reference/functions/wp_enqueue_style/
     * @param $hook_suffix
     * @return void
     */
    public function enqueue_style($hook_suffix):void
    {
        if($hook_suffix === $this->hook_suffix) {
            foreach ($this->style_collection as $style) {
                wp_enqueue_style(
                    'wdes-' . $style['handle'],
                    $this->base_url . '/css/' . $style['filename'] . '.css',
                    $style['deps'],
                    $this->timestamp(),
                    'all'
                );
            }
        }
    }

    /**
     * Registers the script and enqueues.
     *
     * Looping from collection and use wp function to add script to enqueue
     *
     * @link https://developer.wordpress.org/reference/functions/wp_enqueue_script/
     * @param $hook_suffix
     * @return void
     */
    public function enqueue_script($hook_suffix):void
    {
        if($hook_suffix === $this->hook_suffix) {
            foreach ($this->script_collection as $script) {
                wp_enqueue_script(
                    'wdes-' . $script['handle'],
                    $this->base_url . '/js/' . $script['filename'] . '.js',
                    $script['deps'],
                    $this->timestamp(),
                    true
                );
            }

            $this->enqueue_inline_script();

            $this->enqueue_google_font();
        }
    }

    public function media():void
    {
        wp_enqueue_media();

    }

    public function add_inline_script(string $handle, string $data, string $position = 'after'): void
    {
        $this->inline_script_collection[$handle] = [
            'handle' => $handle,
            'data' => $data,
            'position' => $position,
        ];
    }

    public function enqueue_inline_script():void
    {
        if( ! empty($this->inline_script_collection) ){
            foreach ($this->inline_script_collection as $script) {
                wp_add_inline_script(
                    $script['handle'],
                    $script['data'],
                    $script['position'],
                );
            }
        }

    }

    public function add_google_font(string $slug, array $sizes):void
    {
        $this->font_collection[$slug] = [
            'slug' => $slug,
            'size' =>implode(',', $sizes),
        ];
    }

    public function enqueue_google_font():void
    {
        if( ! empty($this->font_collection) ){

            $font_families = [];

            foreach ($this->font_collection as $font) {
                $font_families[] = $font['slug'] . ':' . $font['size'];
            }
            $query_args = array(
                'family' => urlencode( implode( '|', $font_families ) ),
                'subset' => urlencode( 'latin,latin-ext' ),
            );

            $fonts_url = add_query_arg( $query_args, 'https://fonts.googleapis.com/css' );

            wp_enqueue_style( 'admin-google-fonts', esc_url_raw( $fonts_url ), array(), null);
        }
    }

    public function enqueue_media():void
    {
        add_action( 'admin_enqueue_scripts', [$this, 'media'] );

    }

    /**
     * Add script and style to admin page to enqueue it there
     *
     * @link https://developer.wordpress.org/reference/hooks/admin_enqueue_scripts/
     * @return void
     */
    public function enqueue_admin()
    {
        add_action( 'admin_enqueue_scripts', [$this, 'enqueue_style'] );

        add_action( 'admin_enqueue_scripts', [$this, 'enqueue_script'] );


    }


    public function timestamp()
    {
        return WDES_THEME_VERSION;
    }



}