<?php

namespace Phox\core\admin\panel\wdes_options\pages\elements;

class Element_Validator
{
    /**
     * The Singleton's instance is stored in a static field.
     *
     * @since 2.3.6
     * @var array $instances
     */
    private static array $instances = [];

    private array  $validation_functions = [
        'bool' => 'is_bool',
        'int' => 'is_int',
        'string' => 'is_string',
        'url' => 'is_url',
        'image' => 'is_image',
        'font' => 'is_font',
        'color' => 'is_color',
        'options' => 'is_options',
        'multiple' => 'is_multiple',
    ];

    private bool $pass_validation = false;

    /**
     * URL register expiration
     *
     * @see https://github.com/symfony/symfony/blob/7.0/src/Symfony/Component/Validator/Constraints/UrlValidator.php
     * @since 2.3.6
     */
    private const PATTERN = '~^
            (http|https)://                                 # protocol
            (((?:[\_\.\pL\pN-]|%%[0-9A-Fa-f]{2})+:)?((?:[\_\.\pL\pN-]|%%[0-9A-Fa-f]{2})+)@)?  # basic auth
            (
                (?:
                    (?:xn--[a-z0-9-]++\.)*+xn--[a-z0-9-]++            # a domain name using punycode
                        |
                    (?:[\pL\pN\pS\pM\-\_]++\.)+[\pL\pN\pM]++          # a multi-level domain name
                        |
                    [a-z0-9\-\_]++                                    # a single-level domain name
                )\.?
                    |                                                 # or
                \d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}                    # an IP address
                    |                                                 # or
                \[
                    (?:(?:(?:(?:(?:(?:(?:[0-9a-f]{1,4})):){6})(?:(?:(?:(?:(?:[0-9a-f]{1,4})):(?:(?:[0-9a-f]{1,4})))|(?:(?:(?:(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9]))\.){3}(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9])))))))|(?:(?:::(?:(?:(?:[0-9a-f]{1,4})):){5})(?:(?:(?:(?:(?:[0-9a-f]{1,4})):(?:(?:[0-9a-f]{1,4})))|(?:(?:(?:(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9]))\.){3}(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9])))))))|(?:(?:(?:(?:(?:[0-9a-f]{1,4})))?::(?:(?:(?:[0-9a-f]{1,4})):){4})(?:(?:(?:(?:(?:[0-9a-f]{1,4})):(?:(?:[0-9a-f]{1,4})))|(?:(?:(?:(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9]))\.){3}(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9])))))))|(?:(?:(?:(?:(?:(?:[0-9a-f]{1,4})):){0,1}(?:(?:[0-9a-f]{1,4})))?::(?:(?:(?:[0-9a-f]{1,4})):){3})(?:(?:(?:(?:(?:[0-9a-f]{1,4})):(?:(?:[0-9a-f]{1,4})))|(?:(?:(?:(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9]))\.){3}(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9])))))))|(?:(?:(?:(?:(?:(?:[0-9a-f]{1,4})):){0,2}(?:(?:[0-9a-f]{1,4})))?::(?:(?:(?:[0-9a-f]{1,4})):){2})(?:(?:(?:(?:(?:[0-9a-f]{1,4})):(?:(?:[0-9a-f]{1,4})))|(?:(?:(?:(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9]))\.){3}(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9])))))))|(?:(?:(?:(?:(?:(?:[0-9a-f]{1,4})):){0,3}(?:(?:[0-9a-f]{1,4})))?::(?:(?:[0-9a-f]{1,4})):)(?:(?:(?:(?:(?:[0-9a-f]{1,4})):(?:(?:[0-9a-f]{1,4})))|(?:(?:(?:(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9]))\.){3}(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9])))))))|(?:(?:(?:(?:(?:(?:[0-9a-f]{1,4})):){0,4}(?:(?:[0-9a-f]{1,4})))?::)(?:(?:(?:(?:(?:[0-9a-f]{1,4})):(?:(?:[0-9a-f]{1,4})))|(?:(?:(?:(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9]))\.){3}(?:(?:25[0-5]|(?:[1-9]|1[0-9]|2[0-4])?[0-9])))))))|(?:(?:(?:(?:(?:(?:[0-9a-f]{1,4})):){0,5}(?:(?:[0-9a-f]{1,4})))?::)(?:(?:[0-9a-f]{1,4})))|(?:(?:(?:(?:(?:(?:[0-9a-f]{1,4})):){0,6}(?:(?:[0-9a-f]{1,4})))?::))))
                \]  # an IPv6 address
            )
            (:[0-9]+)?                              # a port (optional)
            (?:/ (?:[\pL\pN\-._\~!$&\'()*+,;=:@]|%%[0-9A-Fa-f]{2})* )*          # a path
            (?:\? (?:[\pL\pN\-._\~!$&\'\[\]()*+,;=:@/?]|%%[0-9A-Fa-f]{2})* )?   # a query (optional)
            (?:\# (?:[\pL\pN\-._\~!$&\'()*+,;=:@/?]|%%[0-9A-Fa-f]{2})* )?       # a fragment (optional)
        $~ixu';

    /**
     * Color register expiration for long hax
     *
     * @see https://github.com/symfony/symfony/blob/7.0/src/Symfony/Component/Validator/Constraints/CssColorValidator.php
     * @since 2.3.6
     */
    private const COLOR_PATTERNS = [
        'hex_long' => '/^#[0-9a-f]{6}$/i',
        'hex_short' => '/^#[0-9a-f]{3}$/i',
        'hex_long_with_alpha' => '/^#[0-9a-f]{8}$/i',
        'hex_short_with_alpha' => '/^#[0-9a-f]{4}$/i',
        'rgb' => '/^rgb\(\s*(0|255|25[0-4]|2[0-4]\d|1\d\d|0?\d?\d),\s*(0|255|25[0-4]|2[0-4]\d|1\d\d|0?\d?\d),\s*(0|255|25[0-4]|2[0-4]\d|1\d\d|0?\d?\d)\s*\)$/i',
        'rgba' => '/^rgba\(\s*(0|255|25[0-4]|2[0-4]\d|1\d\d|0?\d?\d),\s*(0|255|25[0-4]|2[0-4]\d|1\d\d|0?\d?\d),\s*(0|255|25[0-4]|2[0-4]\d|1\d\d|0?\d?\d),\s*(0|0?\.\d+|1(\.0)?)\s*\)$/i'
    ];


    /**
     * Validators
     *
     * Collect all items that fail on validator
     *
     * @since 2.3.6
     * @var array $validator it will have target with error message
     */
    private array $validator = [];

    /**
     * This is the static method that controls the access to the singleton
     * instance.
     *
     * @since 2.3.6
     */
    public static function getInstance(): Element_Validator
    {
        $cls = static::class;
        if (!isset(self::$instances[$cls])) {
            self::$instances[$cls] = new static();
        }

        return self::$instances[$cls];
    }

    private function add_validator($target ,$message): void
    {
        $this->validator = [];
        $this->validator[$target] = $message;
    }

    public function validate($type, $value, $target, $message):void
    {
        if( null === $value )
        {
            return;
        }

        if( ! in_array($type, array_keys($this->validation_functions)) ){

            return;

        }

        $is_valid = call_user_func([$this, $this->validation_functions[$type]], $value);

        if( $is_valid ) {

            $this->pass_validation = true;

            return;

        }

        $this->pass_validation = false;

        $this->add_validator($target ,$message);

    }

    public function get_validator(): array
    {
        return $this->validator;
    }

    public function is_string($string):bool
    {
        return is_string($string);
    }
    public function is_bool($bool):bool
    {
        return is_bool($bool);
    }

    public function is_int($int):bool
    {
        return is_int($int);
    }

    public function is_url($url):bool
    {
        return preg_match(self::PATTERN, $url) || empty($url) || $url === '#';
    }

    private function check_mime_type($mime_type, $url):bool
    {

        $ch = curl_init($url);

        // Set cURL options
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HEADER, true);
        curl_setopt($ch, CURLOPT_NOBODY, true);
        // Add User-Agent header
        curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36');


        // Execute cURL request
        $headers = curl_exec($ch);

        // Check for errors and initialize contentType to null
        if (curl_errno($ch) !== 0) {
            curl_close($ch);
            return false;
        }

        // Parse headers to find the Content-Type
        $mime_content = '';
        $headerLines = explode("\r\n", $headers);
        foreach ($headerLines as $header) {
            if (stripos($header, 'Content-Type:') === 0) {
                $mime_content = trim(substr($header, 13));
                break;
            }
        }

        // Close the cURL session
        curl_close($ch);

        return strpos($mime_content, $mime_type) !== false;

    }

    public function is_image($url):bool
    {
        if ( !$this->is_url($url) ) return false;

        if( empty($url) ) return true;

        return $this->check_mime_type('image', $url);
    }

    public function is_font($url):bool
    {
        if ( !$this->is_url($url) ) return false;

        if( empty($url) ) return true;

        return $this->check_mime_type('font', $url);
    }

    public function is_color($color):bool
    {
        foreach (self::COLOR_PATTERNS as $regex) {
            if (preg_match($regex, (string) $color)) {
                return true;
            }
        }
        return false;
    }

    public function is_options($selection_properties)
    {
        if(empty($selection_properties['chosen'])) return true;

        foreach ($selection_properties['choices'] as $choice) {

            if($choice['value'] === $selection_properties['chosen']){
                return true;
            }

        }

        return false;
    }

    public function is_multiple($selection_properties)
    {
        $values = $selection_properties['values'];
        if (empty($selection_properties['values'][0])) return true;
        $choices = array_column($selection_properties['choices'], 'value');

        return count(array_intersect($values, $choices)) === count($values);
    }

    public function check_status():bool
    {
        return $this->pass_validation;
    }

}