<?php

namespace PenciPaywall\Woocommerce;

use DateInterval;
use DateTime;
use DateTimeZone;
use Exception;
use WC_Order;

/**
 * Class Order - Optimized for all WooCommerce payment gateways
 *
 * @package PenciPaywall\Woocommerce
 */
class Order {
	/**
	 * @var Order
	 */
	private static $instance;

	/**
	 * Supported subscription payment methods
	 * @var array
	 */
	private $subscription_gateways = array();

	/**
	 * Order constructor.
	 */
	private function __construct() {
		$this->init_subscription_gateways();
		$this->init_hooks();
	}

	/**
	 * Initialize subscription payment gateways
	 */
	private function init_subscription_gateways() {
		$this->subscription_gateways = apply_filters( 'pencipw_subscription_gateways', array(
			'paypal',
			'stripe',
			'stripe_cc',
			'stripe_sepa',
			'square',
			'authorize_net',
			'braintree',
			'paypalsubscribe'
		) );
	}

	/**
	 * Initialize WordPress hooks
	 */
	private function init_hooks() {
		// Admin notices
		add_action( 'admin_notices', array( $this, 'paywall_notice' ) );
		add_action( 'wp_ajax_dismiss_paywall_notice', array( $this, 'dismiss_paywall_notice' ) );
		add_action( 'wp_ajax_nopriv_dismiss_paywall_notice', array( $this, 'dismiss_paywall_notice' ) );

		// Order processing
		add_filter( 'woocommerce_payment_complete_order_status', array( $this, 'payment_complete_order_status' ), 99, 3 );
		add_action( 'woocommerce_order_status_changed', array( $this, 'order_paid' ), 99, 3 );
		add_action( 'woocommerce_add_to_cart', array( $this, 'product_added' ), 10, 2 );
		add_action( 'woocommerce_before_thankyou', array( $this, 'auto_complete_order' ) );
		add_action( 'template_redirect', array( $this, 'redirect_after_login' ) );

		// WooCommerce Subscriptions Integration
		if ( class_exists( 'WC_Subscriptions' ) ) {
			add_action( 'woocommerce_add_to_cart', array( $this, 'subscription_added' ), 10, 2 );
			add_filter( 'woocommerce_payment_complete_order_status', array( $this, 'auto_complete_order_wcs' ), 10, 3 );
			add_action( 'woocommerce_subscription_status_updated', array( $this, 'on_subscription_status_changed' ) );
			add_action( 'woocommerce_subscription_status_active', array( $this, 'process_checkout' ) );
		}

		// Payment gateway specific hooks
		add_action( 'woocommerce_payment_complete', array( $this, 'payment_completed' ) );
		add_action( 'woocommerce_thankyou', array( $this, 'process_payment_success' ) );
	}

	/**
	 * Process payment completion for all gateways
	 *
	 * @param int $order_id
	 */
	public function payment_completed( $order_id ) {
		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			return;
		}

		$this->process_paywall_order( $order, 'completed' );
	}

	/**
	 * Process payment success on thank you page
	 *
	 * @param int $order_id
	 */
	public function process_payment_success( $order_id ) {
		$order = wc_get_order( $order_id );
		if ( ! $order || ! $this->is_paywall_order( $order ) ) {
			return;
		}

		$payment_method = $order->get_payment_method();
		
		// Handle subscription orders
		if ( $this->is_subscription_gateway( $payment_method ) ) {
			$this->process_subscription_order( $order );
		}
	}

	/**
	 * Check if order contains paywall products
	 *
	 * @param WC_Order $order
	 * @return bool
	 */
	private function is_paywall_order( $order ) {
		foreach ( $order->get_items() as $item ) {
			$product = $item->get_product();
			if ( $product && ( $product->is_type( 'paywall_subscribe' ) || $product->is_type( 'paywall_unlock' ) ) ) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Check if payment method supports subscriptions
	 *
	 * @param string $payment_method
	 * @return bool
	 */
	private function is_subscription_gateway( $payment_method ) {
		return in_array( $payment_method, $this->subscription_gateways, true );
	}

	/**
	 * Process subscription order for any gateway
	 *
	 * @param WC_Order $order
	 */
	private function process_subscription_order( $order ) {
		$user_id = $order->get_customer_id();
		if ( ! $user_id ) {
			return;
		}

		$payment_method = $order->get_payment_method();
		$subscription_data = $this->get_subscription_data( $order );

		if ( $subscription_data ) {
			$this->update_subscription_status( $user_id, $subscription_data, $payment_method );
		}
	}

	/**
	 * Get subscription data from order
	 *
	 * @param WC_Order $order
	 * @return array|false
	 */
	private function get_subscription_data( $order ) {
		$payment_method = $order->get_payment_method();
		$order_id = $order->get_id();

		// Check for WooCommerce Subscriptions
		if ( function_exists( 'wcs_get_subscriptions_for_order' ) ) {
			$subscriptions = wcs_get_subscriptions_for_order( $order_id );
			if ( ! empty( $subscriptions ) ) {
				$subscription = array_shift( $subscriptions );
				return $this->extract_wcs_subscription_data( $subscription );
			}
		}

		// Fallback to order meta for custom implementations
		return $this->get_custom_subscription_data( $order );
	}

	/**
	 * Extract subscription data from WooCommerce Subscriptions
	 *
	 * @param object $subscription
	 * @return array
	 */
	private function extract_wcs_subscription_data( $subscription ) {
		$data = $subscription->get_data();
		$expired = new DateTime();

		if ( isset( $data['schedule_next_payment'] ) && is_object( $data['schedule_next_payment'] ) ) {
			$expired->setTimestamp( $data['schedule_next_payment']->getTimestamp() );
		} elseif ( isset( $data['schedule_end'] ) && is_object( $data['schedule_end'] ) ) {
			$expired->setTimestamp( $data['schedule_end']->getTimestamp() );
		} else {
			// Default to 1 month from now
			$expired->add( new DateInterval( 'P1M' ) );
		}

		$expired->setTimezone( new DateTimeZone( 'UTC' ) );

		return array(
			'subscription_id' => $data['id'],
			'status' => 'ACTIVE',
			'expired_date' => $expired->format( 'Y-m-d H:i:s' )
		);
	}

	/**
	 * Get custom subscription data from order meta
	 *
	 * @param WC_Order $order
	 * @return array|false
	 */
	private function get_custom_subscription_data( $order ) {
		$order_id = $order->get_id();
		$payment_method = $order->get_payment_method();
		
		// Check for subscription ID in order meta
		$subscription_id = '';
		$meta_keys = array(
			'subscription_id',
			'_subscription_id',
			$payment_method . '_subscription_id',
			'pencipw_' . $payment_method . '_subs_id'
		);

		foreach ( $meta_keys as $meta_key ) {
			$value = get_post_meta( $order_id, $meta_key, true );
			if ( ! empty( $value ) ) {
				$subscription_id = $value;
				break;
			}
		}

		if ( empty( $subscription_id ) ) {
			return false;
		}

		// Default subscription period
		$expired = new DateTime();
		$expired->add( new DateInterval( 'P1M' ) );
		$expired->setTimezone( new DateTimeZone( 'UTC' ) );

		return array(
			'subscription_id' => $subscription_id,
			'status' => 'ACTIVE',
			'expired_date' => $expired->format( 'Y-m-d H:i:s' )
		);
	}

	/**
	 * Update user subscription status
	 *
	 * @param int $user_id
	 * @param array $subscription_data
	 * @param string $payment_method
	 */
	private function update_subscription_status( $user_id, $subscription_data, $payment_method ) {
		update_user_option( $user_id, 'pencipw_subscribe_id', $subscription_data['subscription_id'] );
		update_user_option( $user_id, 'pencipw_subscribe_status', $subscription_data['status'] );
		update_user_option( $user_id, 'pencipw_expired_date', $subscription_data['expired_date'] );
		update_user_option( $user_id, 'pencipw_payment_method', $payment_method );

		// Store gateway-specific subscription ID
		update_user_option( $user_id, 'pencipw_' . $payment_method . '_subs_id', $subscription_data['subscription_id'] );
	}

	/**
	 * Process paywall order with universal logic
	 *
	 * @param WC_Order $order
	 * @param string $status
	 */
	private function process_paywall_order( $order, $status ) {
		$user_id = $order->get_customer_id();
		if ( ! $user_id ) {
			return;
		}

		$order_id = $order->get_id();
		$payment_method = $order->get_payment_method();
		$is_completed = in_array( $status, array( 'completed', 'processing' ), true );

		foreach ( $order->get_items() as $item ) {
			$product = $item->get_product();
			if ( ! $product ) {
				continue;
			}

			if ( $product->is_type( 'paywall_subscribe' ) ) {
				$this->handle_subscription_product( $order, $product, $item, $is_completed );
			} elseif ( $product->is_type( 'paywall_unlock' ) ) {
				$this->handle_unlock_product( $order, $product, $item, $is_completed );
			}
		}

		// Update order completion status
		update_post_meta( $order_id, 'pencipw_paywall_completed', $is_completed ? 'yes' : 'no' );
	}

	/**
	 * Handle subscription product processing
	 *
	 * @param WC_Order $order
	 * @param object $product
	 * @param object $item
	 * @param bool $is_completed
	 */
	private function handle_subscription_product( $order, $product, $item, $is_completed ) {
		$user_id = $order->get_customer_id();
		$payment_method = $order->get_payment_method();

		if ( $is_completed ) {
			$subscription_data = $this->get_subscription_data( $order );
			if ( $subscription_data ) {
				$this->update_subscription_status( $user_id, $subscription_data, $payment_method );
			}
		} else {
			// Cancel/refund - clear subscription data
			$this->clear_subscription_data( $user_id );
		}
	}

	/**
	 * Handle unlock product processing
	 *
	 * @param WC_Order $order
	 * @param object $product
	 * @param object $item
	 * @param bool $is_completed
	 */
	private function handle_unlock_product( $order, $product, $item, $is_completed ) {
		$user_id = $order->get_customer_id();
		$unlock_remaining = get_user_option( 'pencipw_unlock_remaining', $user_id ) ?: 0;
		$unlocked_posts = get_user_option( 'pencipw_unlocked_post_list', $user_id ) ?: array();

		if ( $unlock_remaining < 0 ) {
			$unlock_remaining = 0;
		}

		$unlock_amount = $product->get_total_unlock() * $item->get_quantity();

		if ( $is_completed ) {
			$unlock_remaining += $unlock_amount;
		} else {
			// Handle refund/cancellation
			if ( get_post_meta( $order->get_id(), 'pencipw_paywall_completed', true ) === 'yes' ) {
				if ( $unlock_remaining >= $unlock_amount ) {
					$unlock_remaining -= $unlock_amount;
				} else {
					$leftover = $unlock_amount - $unlock_remaining;
					$unlock_remaining = 0;
					// Remove unlocked posts
					for ( $i = 0; $i < $leftover; $i++ ) {
						array_pop( $unlocked_posts );
					}
				}
			}
		}

		update_user_option( $user_id, 'pencipw_unlock_remaining', $unlock_remaining );
		update_user_option( $user_id, 'pencipw_unlocked_post_list', $unlocked_posts );
	}

	/**
	 * Clear subscription data
	 *
	 * @param int $user_id
	 */
	private function clear_subscription_data( $user_id ) {
		update_user_option( $user_id, 'pencipw_subscribe_id', false );
		update_user_option( $user_id, 'pencipw_subscribe_status', false );
		update_user_option( $user_id, 'pencipw_expired_date', false );
		update_user_option( $user_id, 'pencipw_payment_method', false );

		// Clear gateway-specific subscription IDs
		foreach ( $this->subscription_gateways as $gateway ) {
			update_user_option( $user_id, 'pencipw_' . $gateway . '_subs_id', false );
		}
	}

	/**
	 * Update subscribe status after the subscription specified with $subscription has had its status changed.
	 *
	 * @param object $subscription An instance of a WC_Subscription object
	 */
	public function process_checkout( $subscription ) {
		if ( function_exists( 'wcs_get_subscription' ) ) {
			if ( ! self::should_connect_paywall( $subscription ) ) {
				return;
			}
			$this->subscribe_status( $subscription );
		}
	}

	/**
	 * Update user subscription status
	 *
	 * @param object $subscription An instance of a WC_Subscription object
	 */
	public function subscribe_status( $subscription ) {
		if ( function_exists( 'wcs_get_subscription' ) ) {
			$subscription_data = $this->extract_wcs_subscription_data( $subscription );
			$user_id = $subscription->get_user_id();
			$payment_method = $subscription->get_payment_method();
			
			$this->update_subscription_status( $user_id, $subscription_data, $payment_method );
		}
	}

	/**
	 * Check if subscription product is set for Penci Paywall
	 *
	 * @param mixed $subscription
	 * @return bool
	 */
	public static function should_connect_paywall( $subscription ) {
		if ( function_exists( 'wcs_get_subscription' ) ) {
			$order_items = $subscription->get_items();
			$order_products = array_filter(
				array_map(
					function ( $item ) {
						return $item->get_product();
					},
					$order_items
				),
				function ( $product ) {
					return ! ! $product;
				}
			);
			
			if ( count( $order_products ) > 0 ) {
				$is_wcs_pencipw = array_reduce(
					$order_products,
					function ( $virtual_order_so_far, $product ) {
						return $virtual_order_so_far && $product->is_virtual() && 'subscription' === $product->get_type() && 'yes' === $product->get_meta( '_pencipw_subscription_paywall' );
					},
					true
				);
				if ( $is_wcs_pencipw ) {
					return true;
				}
			}
		}

		return false;
	}

	/**
	 * Fire whenever a subscription's status is changed.
	 *
	 * @param object $subscription An instance of a WC_Subscription object
	 */
	public function on_subscription_status_changed( $subscription ) {
		if ( function_exists( 'wcs_get_subscription' ) ) {
			$is_cancelled = $subscription->has_status( 'cancelled' );
			$is_expired = $subscription->has_status( 'expired' );

			if ( $is_cancelled || $is_expired ) {
				$user_id = $subscription->get_user_id();
				$pencipw_subscribe_id = get_user_option( 'pencipw_subscribe_id', $user_id );
				
				if ( $subscription->get_id() == $pencipw_subscribe_id ) {
					$this->clear_subscription_data( $user_id );
				}
			}
		}
	}

	/**
	 * Triggers when a subscription switch is added to the cart.
	 *
	 * @param string $cart_item_key The new cart item's key.
	 * @param int $product_id The product added to the cart.
	 */
	public function subscription_added( $cart_item_key, $product_id ) {
		if ( function_exists( 'wcs_get_subscription' ) ) {
			foreach ( WC()->cart->get_cart() as $key => $cart_item ) {
				$item_data = $cart_item['data'];
				if ( 'subscription' === $item_data->get_type() && 'yes' === $item_data->get_meta( '_pencipw_subscription_paywall' ) ) {
					if ( $product_id === $cart_item['product_id'] ) {
						WC()->cart->set_quantity( $key, 1 );
					} else {
						WC()->cart->set_quantity( $key, 0 );
					}
				} else {
					$product = wc_get_product( $product_id );
					if ( $product && 'subscription' === $product->get_type() && 'yes' === $product->get_meta( '_pencipw_subscription_paywall' ) ) {
						WC()->cart->set_quantity( $key, 0 );
					}
				}
			}
		}

		return $cart_item_key;
	}

	/**
	 * Automatically set the order's status to complete for WCS.
	 *
	 * @param string $new_order_status
	 * @param int $order_id
	 * @param WC_Order $order
	 * @return string $new_order_status
	 */
	public function auto_complete_order_wcs( $status, $order_id, $order ) {
		if ( function_exists( 'wcs_get_subscription' ) ) {
			$current_status = $order->get_status();
			$allowed_current_statuses = array( 'on-hold', 'pending', 'failed' );
			
			if ( 'processing' === $status && in_array( $current_status, $allowed_current_statuses, true ) ) {
				$order_items = $order->get_items();
				$order_products = array_filter(
					array_map(
						function ( $item ) {
							return $item->get_product();
						},
						$order_items
					),
					function ( $product ) {
						return ! ! $product;
					}
				);
				
				if ( count( $order_products ) > 0 ) {
					$is_wcs_pencipw = array_reduce(
						$order_products,
						function ( $virtual_order_so_far, $product ) {
							return $virtual_order_so_far && $product->is_virtual() && 'subscription' === $product->get_type() && 'yes' === $product->get_meta( '_pencipw_subscription_paywall' );
						},
						true
					);
					if ( $is_wcs_pencipw ) {
						$status = 'completed';
						$order->update_status( 'completed' );
					}
				}
			}
		}

		return $status;
	}

	/**
	 * Dismiss paywall notice
	 */
	public function dismiss_paywall_notice() {
		update_option( 'penci_dismiss_paywall_notice', true );
		wp_die();
	}

	/**
	 * Paywall notice
	 */
	public function paywall_notice() {
		if ( ! class_exists( 'WooCommerce' ) && ! get_option( 'penci_dismiss_paywall_notice', false ) ) {
			$this->print_paywall_notice();
		}
	}

	/**
	 * Print paywall notice
	 */
	public function print_paywall_notice() {
		?>
		<div class="notice notice-error is-dismissible" data-dismissible="paywall-notice">
			<p>
				<?php
				printf(
					wp_kses(
						__(
							'<strong>Penci Paywall:</strong> Please install and activate the WooCommerce plugin. <a href="%s" class="button-primary">Manage Plugins</a>',
							'penci-paywall'
						),
						array(
							'strong' => array(),
							'a' => array(
								'href' => true,
								'class' => true,
							),
						)
					),
					esc_url( admin_url( 'plugins.php' ) )
				);
				?>
			</p>
		</div>
		<script>
		jQuery(document).ready(function($) {
			$(document).on('click', '.notice-dismiss', function() {
				$.post(ajaxurl, {
					action: 'dismiss_paywall_notice'
				});
			});
		});
		</script>
		<?php
	}

	/**
	 * Add product to cart after login
	 */
	public function redirect_after_login() {
		if ( ! function_exists( 'is_checkout' ) || ! is_checkout() ) {
			return;
		}

		$product_id = isset( $_COOKIE['paywall_product'] ) ? absint( $_COOKIE['paywall_product'] ) : 0;
		if ( ! $product_id ) {
			return;
		}

		$product = wc_get_product( $product_id );
		if ( ! $product ) {
			return;
		}

		try {
			WC()->cart->add_to_cart( $product_id );
			// Clear the cookie after adding to cart
			setcookie( 'paywall_product', '', time() - 3600, '/' );
		} catch ( Exception $e ) {
			wc_add_notice( $e->getMessage(), 'error' );
		}
	}

	/**
	 * Check if product already added in cart
	 *
	 * @param string $cart_item_key
	 * @param int $product_id
	 * @return string
	 */
	public function product_added( $cart_item_key, $product_id ) {
		foreach ( WC()->cart->get_cart() as $key => $cart_item ) {
			$item_data = $cart_item['data'];
			if ( 'paywall_subscribe' === $item_data->get_type() ) {
				if ( $product_id === $cart_item['product_id'] ) {
					WC()->cart->set_quantity( $key, 1 );
				} else {
					WC()->cart->set_quantity( $key, 0 );
				}
			} else {
				$product = wc_get_product( $product_id );
				if ( $product && 'paywall_subscribe' === $product->get_type() ) {
					WC()->cart->set_quantity( $key, 0 );
				}
			}
		}

		return $cart_item_key;
	}

	/**
	 * Auto Complete Order - Universal for all payment gateways
	 *
	 * @param int $order_id
	 */
	public function auto_complete_order( $order_id ) {
		if ( ! $order_id ) {
			return;
		}

		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			return;
		}

		// Remove order again button for paywall products
		if ( $this->is_paywall_order( $order ) ) {
			remove_action( 'woocommerce_order_details_after_order_table', 'woocommerce_order_again_button' );
		}

		// Process the order
		$this->process_paywall_order( $order, $order->get_status() );
	}

	/**
	 * Product paywall unlock doesn't need processing. Make it completed!
	 *
	 * @param string $status order status.
	 * @param int $id unique ID for this object.
	 * @param WC_Order $order Order class.
	 * @return string
	 */
	public function payment_complete_order_status( $status, $id, $order ) {
		foreach ( $order->get_items() as $item ) {
			if ( $item->is_type( 'line_item' ) ) {
				$product = $item->get_product();
				if ( $product && $product->is_type( 'paywall_unlock' ) ) {
					return 'completed';
				}
			}
		}

		return $status;
	}

	/**
	 * Order paid - Universal handler for all payment gateways
	 *
	 * @param int $order_id
	 * @param string $old_status
	 * @param string $new_status
	 * @throws Exception
	 */
	public function order_paid( $order_id, $old_status, $new_status ) {
		$order = wc_get_order( $order_id );
		if ( ! $order || ! $this->is_paywall_order( $order ) ) {
			return;
		}

		$this->process_paywall_order( $order, $new_status );
	}

	/**
	 * Check if user is allowed to purchase
	 *
	 * @param int $product_id
	 * @return bool
	 */
	protected function allow_purchase( $product_id ) {
		// Add codes here if need to restrict user from purchasing.
		return apply_filters( 'pencipw_allow_purchase', true, $product_id );
	}

	/**
	 * Get instance
	 *
	 * @return Order
	 */
	public static function instance() {
		if ( null === static::$instance ) {
			static::$instance = new static();
		}

		return static::$instance;
	}
}