<?php

/**
 * Elementor Widget
 * @package SWIFTCART
 * @since 1.0.0
 */

namespace Elementor;

class Swiftcart_Property_Widget extends Widget_Base
{

	/**
	 * Get widget name.
	 *
	 * Retrieve Elementor widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name()
	{
		return 'swiftcart-property-widget';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Elementor widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title()
	{
		return esc_html__('Property', 'swiftcart-core');
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve Elementor widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-product-info';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the Elementor widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories()
	{
		return ['swiftcart_widgets'];
	}

	/**
	 * Register Elementor widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function _register_controls()
	{

		$this->start_controls_section(
			'layout_section',
			[
				'label' => __('Layout', 'swiftcart-core'),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'layout_type',
			[
				'label' => __('Select Layout', 'swiftcart-core'),
				'type' => \Elementor\Controls_Manager::SELECT2,
				'default' => 'layout_one',
				'options' => [
					'layout_one' => __('Layout One', 'swiftcart-core'),
					'layout_two' => __('Layout Two', 'swiftcart-core'),
					'layout_three' => __('Layout Three', 'swiftcart-core'),
					'layout_four' => __('Layout Four', 'swiftcart-core'),
					'layout_five' => __('Layout Five', 'swiftcart-core'),
					'layout_six' => __('Layout Six', 'swiftcart-core'),
					'layout_seven' => __('Layout Seven', 'swiftcart-core'),
				]
			]
		);

		$this->end_controls_section();

		/*
		* content
		*/
		$this->start_controls_section(
			'content',
			[
				'label' => esc_html__('Content', 'swiftcart-core'),
				'tab'   => Controls_Manager::TAB_CONTENT,
			]
		);
		$this->add_control(
			'heading_status',
			[
				'label' => __('Enable Pagination?', 'swiftcart-core'),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => __('Yes', 'swiftcart-core'),
				'label_off' => __('No', 'swiftcart-core'),
				'return_value' => 'yes',
				'default' => 'yes',

			]
		);
		$this->add_control(
			'title',
			[
				'label' => __('Title', 'swiftcart-core'),
				'type' => \Elementor\Controls_Manager::TEXTAREA,
				'placeholder' => __('Add Title', 'swiftcart-core'),
				'default' => __('Add Title', 'swiftcart-core'),
			]
		);

		$this->add_control(
			'sub_title',
			[
				'label' => __('Sub Title', 'swiftcart-core'),
				'type' => \Elementor\Controls_Manager::TEXTAREA,
				'placeholder' => __('Add Sub Title', 'swiftcart-core'),
				'default' => __('Add Sub Title', 'swiftcart-core'),
			]
		);

		$this->add_control(
			'button_label',
			[
				'label' => esc_html__('Button Text', 'swiftcart-core'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__('Descover', 'swiftcart-core'),
				'label_block' => true,
				'condition' => [
					'layout_type' => ['layout_two']
				]
			]
		);

		$this->add_control(
			'button_url',
			[
				'label' => esc_html__('Button Url', 'swiftcart-core'),
				'type' => \Elementor\Controls_Manager::URL,
				'placeholder' => esc_html__('#', 'swiftcart-core'),
				'show_external' => true,
				'default' => [
					'url' => '#',
					'is_external' => false,
					'nofollow' => false,
				],
				'show_label' => false,
				'condition' => [
					'layout_type' => ['layout_two']
				]
			]
		);



		$this->end_controls_section();
		//post options

		$this->start_controls_section(
			'post_options',
			[
				'label' => esc_html__('Post Options', 'swiftcart-core'),
				'tab'   => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'post_count',
			[
				'label' => __('Number Of Posts', 'swiftcart-core'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['count'],
				'range' => [
					'count' => [
						'min' => 0,
						'max' => 15,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'count',
					'size' => 6,
				],
			]
		);
		$this->add_control(
			'post_title_count',
			[
				'label' => __('Title Character Limit', 'swiftcart-core'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['count'],
				'range' => [
					'count' => [
						'min' => 0,
						'max' => 400,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'count',
					'size' => 15,
				],
			]
		);
		$this->add_control(
			'word_limit',
			[
				'label' => __('Word Limit', 'swiftcart-core'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['count'],
				'range' => [
					'count' => [
						'min' => 0,
						'max' => 100,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'count',
					'size' => 15,
				],
			]
		);


		$this->add_control(
			'select_cat',
			[
				'label'    => esc_html__('Select Status', 'swiftcart-core'),
				'type'     => Controls_Manager::SELECT2,
				'multiple' => true,
				'options'  => swiftcart_core_post_tag('property_status'),

			]
		);


		$this->add_control(
			'exclude_cat',
			[
				'label'    => esc_html__('Exclude Status', 'swiftcart-core'),
				'type'     => Controls_Manager::SELECT2,
				'multiple' => true,
				'options'  => swiftcart_core_taxonomy_list('property_status'),
			]
		);

		$this->add_control(
			'select_tag',
			[
				'label'    => esc_html__('Select Type', 'swiftcart-core'),
				'type'     => Controls_Manager::SELECT2,
				'multiple' => true,
				'options'  => swiftcart_core_post_tag('property_cat'),
			]
		);
		$this->add_control(
			'select_tags',
			[
				'label'    => esc_html__('Select Type', 'swiftcart-core'),
				'type'     => Controls_Manager::SELECT2,
				'multiple' => true,
				'options'  => swiftcart_core_post_tag('property_tag'),
			]
		);

		$this->add_control(
			'orderby',
			[
				'label'   => esc_html__('Order by', 'swiftcart-core'),
				'type'    => Controls_Manager::SELECT2,
				'options' => array(
					'author' => esc_html__('Author', 'swiftcart-core'),
					'title'  => esc_html__('Title', 'swiftcart-core'),
					'date'   => esc_html__('Date', 'swiftcart-core'),
					'rand'   => esc_html__('Random', 'swiftcart-core'),
				),
				'default' => 'date'

			]
		);

		$this->add_control(
			'order',
			[
				'label'   => esc_html__('Order', 'swiftcart-core'),
				'type'    => Controls_Manager::SELECT2,
				'options' => array(
					'desc' => esc_html__('DESC', 'swiftcart-core'),
					'asc'  => esc_html__('ASC', 'swiftcart-core'),
				),
				'default' => 'desc'

			]
		);

		$this->add_control(
			'pagination_status',
			[
				'label' => __('Enable Pagination?', 'swiftcart-core'),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => __('Yes', 'swiftcart-core'),
				'label_off' => __('No', 'swiftcart-core'),
				'return_value' => 'yes',
				'default' => 'no',
				'condition' => [
					'layout_type' => ['layout_seven']
				]

			]
		);

		$this->end_controls_section();



		//General style
		$this->start_controls_section(
			'general_style',
			[
				'label' => esc_html__('Section Header Style', 'swiftcart-core'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'layout_type' => ['layout_one', 'layout_two']
				]
			]
		);

		swiftcart_core_elementor_general_style_options($this, 'Title', '{{WRAPPER}} .section-title h2', ['layout_one', 'layout_two', 'layout_three', 'layout_four', 'layout_five']);
		swiftcart_core_elementor_general_style_options($this, 'Sub Title', '{{WRAPPER}} .section-title h6', ['layout_one', 'layout_two', 'layout_three', 'layout_four', 'layout_five']);

		$this->end_controls_section();

		//post style
		$this->start_controls_section(
			'post_style',
			[
				'label' => esc_html__('Post Style', 'swiftcart-core'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		swiftcart_core_elementor_link_style_options($this, 'Post Title', '{{WRAPPER}} .single-product-wrap .product-details-inner h4 a', '{{WRAPPER}} .single-product-wrap .product-details-inner h4 a:hover', ['layout_one', 'layout_three', 'layout_four']);
		swiftcart_core_elementor_general_style_options($this, 'Post Meta', '{{WRAPPER}} .single-product-wrap .product-details-inner .meta-inner li,{{WRAPPER}} ul li .media .media-body .post-info span', ['layout_one', 'layout_three', 'layout_four']);
		swiftcart_core_elementor_general_style_options($this, 'Post Summary', '{{WRAPPER}} .single-product-wrap .product-details-inner p', ['layout_one', 'layout_two', 'layout_three']);

		swiftcart_core_elementor_button_style_options($this, 'Read More', '{{WRAPPER}} .single-product-wrap.style-bottom-radius .product-details-inner .read-more,{{WRAPPER}} .btn-base', '{{WRAPPER}} .single-product-wrap.style-bottom-radius .product-details-inner .read-more:hover, {{WRAPPER}} .btn-base:hover', ['layout_one', 'layout_two']);



		$this->end_controls_section();
	}

	/**
	 * Render Elementor widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();


		include swiftcart_get_template('property-one.php');
		include swiftcart_get_template('property-two.php');
		include swiftcart_get_template('property-three.php');
		include swiftcart_get_template('property-four.php');
		include swiftcart_get_template('property-five.php');
		include swiftcart_get_template('property-six.php');
		include swiftcart_get_template('property-seven.php');
	}
}

Plugin::instance()->widgets_manager->register(new Swiftcart_Property_Widget());
