<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Repeater;
use Elementor\Group_Control_Image_Size;

class Nutreko_Elementor_ImgBoxs extends Nutreko_Base_Widgets_Swiper {
    /**
     * Get widget name.
     *
     * Retrieve imgbox widget name.
     *
     * @return string Widget name.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_name() {
        return 'nutreko-imgbox';
    }

    /**
     * Get widget title.
     *
     * Retrieve imgbox widget title.
     *
     * @return string Widget title.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_title() {
        return esc_html__('Nutreko Image Box Carousel', 'nutreko');
    }

    /**
     * Get widget icon.
     *
     * Retrieve imgbox widget icon.
     *
     * @return string Widget icon.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_icon() {
        return 'eicon-carousel';
    }

    public function get_script_depends() {
        return ['nutreko-elementor-imgbox', 'nutreko-elementor-swiper'];
    }

    public function get_categories() {
        return array('nutreko-addons');
    }

    /**
     * Register imgbox widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since  1.0.0
     * @access protected
     */
    protected function register_controls() {
        $this->start_controls_section(
            'section_imgbox',
            [
                'label' => esc_html__('ImgBoxs', 'nutreko'),
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'imgbox_title',
            [
                'label'   => esc_html__('Title', 'nutreko'),
                'type'    => Controls_Manager::TEXT,
                'default'     => '',
            ]
        );

        $repeater->add_control(
            'imgbox_button',
            [
                'label'     => esc_html__('Button Text', 'nutreko'),
                'type'      => Controls_Manager::TEXT,
                'dynamic'   => [
                    'active' => true,
                ],
                'default'   => esc_html__('Click Here', 'nutreko'),
                'separator' => 'before',
            ]
        );

        $repeater->add_control(
            'imgbox_link',
            [
                'label'       => esc_html__('Link to', 'nutreko'),
                'placeholder' => esc_html__('https://your-link.com', 'nutreko'),
                'type'        => Controls_Manager::URL,
            ]
        );

        $repeater->add_control(
            'imgbox_image',
            [
                'label'      => esc_html__('Choose Image', 'nutreko'),
                'type'       => Controls_Manager::MEDIA,
                'show_label' => false,
            ]
        );

        $this->add_control(
            'imgboxs',
            [
                'label'       => esc_html__('Items', 'nutreko'),
                'type'        => Controls_Manager::REPEATER,
                'fields'      => $repeater->get_controls(),
                'title_field' => '{{{ imgbox_title }}}',
            ]
        );

        $this->add_group_control(
            Elementor\Group_Control_Image_Size::get_type(),
            [
                'name'      => 'imgbox_image',
                'default'   => 'full',
                'separator' => 'none',
            ]
        );

        $this->add_responsive_control(
            'column',
            [
                'label'        => esc_html__('Columns', 'nutreko'),
                'type'         => \Elementor\Controls_Manager::SELECT,
                'default'      => 1,
                'options'      => [1 => 1, 2 => 2, 3 => 3, 4 => 4, 5 => 5, 6 => 6],
                'selectors' => [
                    '{{WRAPPER}} .d-grid' => 'grid-template-columns: repeat({{VALUE}}, 1fr)',
                ],
                'condition' => ['enable_carousel!' => 'yes']
            ]
        );

        $this->add_responsive_control(
            'gutter',
            [
                'label'      => esc_html__('Gutter', 'nutreko'),
                'type'       => Controls_Manager::SLIDER,
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                    ],
                ],
                'size_units' => ['px'],
                'selectors'  => [
                    '{{WRAPPER}} .d-grid' => 'grid-gap:{{SIZE}}{{UNIT}}',
                ],
                'condition'  => ['enable_carousel!' => 'yes']
            ]
        );

        $this->add_responsive_control(
            'imgbox_vertical',
            [
                'label'     => esc_html__('Vertical Align', 'nutreko'),
                'type'      => Controls_Manager::CHOOSE,
                'options'   => [
                    'flex-start' => [
                        'title' => esc_html__('flex-start', 'nutreko'),
                        'icon'  => 'eicon-v-align-top',
                    ],
                    'center'     => [
                        'title' => esc_html__('center', 'nutreko'),
                        'icon'  => 'eicon-v-align-middle',
                    ],
                    'flex-end'   => [
                        'title' => esc_html__('flex-end', 'nutreko'),
                        'icon'  => 'eicon-v-align-bottom',
                    ],
                ],
                'default'   => 'flex-end',
                'selectors'   => [
                    '{{WRAPPER}} .imgbox-content' => 'justify-content: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'imgbox_horizontal',
            [
                'label'       => esc_html__('Horizontal Position', 'nutreko'),
                'type'        => Controls_Manager::CHOOSE,
                'options'     => [
                    'left'   => [
                        'title' => esc_html__('left', 'nutreko'),
                        'icon'  => 'eicon-h-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__('center', 'nutreko'),
                        'icon'  => 'eicon-h-align-center',
                    ],
                    'right'  => [
                        'title' => esc_html__('right', 'nutreko'),
                        'icon'  => 'eicon-h-align-right',
                    ],
                ],
                'label_block' => false,
                'default'      => 'left',
                'selectors'   => [
                    '{{WRAPPER}} .imgbox-title' => 'text-align: {{VALUE}};',
                    '{{WRAPPER}} .imgbox-button' => 'text-align: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'enable_carousel',
            [
                'label' => esc_html__('Enable Carousel', 'nutreko'),
                'type'  => Controls_Manager::SWITCHER,
            ]
        );

        $this->add_control(
            'view',
            [
                'label'   => esc_html__('View', 'nutreko'),
                'type'    => Controls_Manager::HIDDEN,
                'default' => 'traditional',
            ]
        );
        $this->end_controls_section();


        // WRAPPER STYLE
        $this->start_controls_section(
            'section_style_imgbox_wrapper',
            [
                'label' => esc_html__('Wrapper', 'nutreko'),
                'tab'   => Controls_Manager::TAB_STYLE,

            ]
        );

        $this->add_responsive_control(
            'height_imgbox_wrapper',
            [
                'label'          => esc_html__('Height', 'nutreko'),
                'type'           => Controls_Manager::SLIDER,
                'default'        => [
                    'unit' => 'px',
                ],
                'tablet_default' => [
                    'unit' => 'px',
                ],
                'mobile_default' => [
                    'unit' => 'px',
                ],
                'size_units'     => ['px', 'vh'],
                'range'          => [
                    'px' => [
                        'min' => 1,
                        'max' => 500,
                    ],
                    'vh' => [
                        'min' => 1,
                        'max' => 100,
                    ],
                ],
                'selectors'      => [
                    '{{WRAPPER}} .imgbox-wrapper .imgbox-content' => 'height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'padding_imgbox_wrapper',
            [
                'label'      => esc_html__('Padding', 'nutreko'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .imgbox-wrapper .imgbox-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->start_controls_tabs('tabs_wrapper');

        $this->start_controls_tab(
            'tab_wrapper_normal',
            [
                'label' => esc_html__('Normal', 'nutreko'),
            ]
        );

        $this->add_control(
            'imgbox_wrapper',
            [
                'label'     => esc_html__('Background Color', 'nutreko'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .imgbox-wrapper' => 'background: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'label'     => esc_html__('Boxshadow', 'nutreko'),
                'name'     => 'wrapper_box_shadow',
                'selector' => '{{WRAPPER}} .imgbox-wrapper',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_wrapper_hover',
            [
                'label' => esc_html__('Hover', 'nutreko'),
            ]
        );

        $this->add_control(
            'imgbox_wrapper_hover',
            [
                'label'     => esc_html__('Background Color', 'nutreko'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .imgbox-wrapper:hover' => 'background: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'label'     => esc_html__('Boxshadow', 'nutreko'),
                'name'     => 'wrapper_box_shadow_hover',
                'selector' => '{{WRAPPER}} .imgbox-wrapper:hover',
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();

        $this->end_controls_section();

        // Image style
        $this->start_controls_section(
            'section_style_imgbox_image',
            [
                'label' => esc_html__('Image', 'nutreko'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );


        $this->add_responsive_control(
            'image_width',
            [
                'label'          => esc_html__('Width', 'nutreko'),
                'type'           => Controls_Manager::SLIDER,
                'default'        => [
                    'unit' => '%',
                ],
                'tablet_default' => [
                    'unit' => '%',
                ],
                'mobile_default' => [
                    'unit' => '%',
                ],
                'size_units'     => ['%', 'px', 'vw'],
                'range'          => [
                    '%'  => [
                        'min' => 1,
                        'max' => 100,
                    ],
                    'px' => [
                        'min' => 1,
                        'max' => 200,
                    ],
                    'vw' => [
                        'min' => 1,
                        'max' => 100,
                    ],
                ],
                'selectors'      => [
                    '{{WRAPPER}} .imgbox-image img' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->add_responsive_control(
            'image_height',
            [
                'label'          => esc_html__('Height', 'nutreko'),
                'type'           => Controls_Manager::SLIDER,
                'default'        => [
                    'unit' => 'px',
                ],
                'tablet_default' => [
                    'unit' => 'px',
                ],
                'mobile_default' => [
                    'unit' => 'px',
                ],
                'size_units'     => ['px', 'vh'],
                'range'          => [
                    'px' => [
                        'min' => 1,
                        'max' => 500,
                    ],
                    'vh' => [
                        'min' => 1,
                        'max' => 100,
                    ],
                ],
                'selectors'      => [
                    '{{WRAPPER}} .imgbox-image img' => 'height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'background_size_image',
            [
                'label'          => esc_html__('Background Size', 'nutreko'),
                'type'           => Controls_Manager::SLIDER,
                'default'        => [
                    'unit' => 'px',
                ],
                'tablet_default' => [
                    'unit' => 'px',
                ],
                'mobile_default' => [
                    'unit' => 'px',
                ],
                'size_units'     => ['px', 'vh'],
                'range'          => [
                    'px' => [
                        'min' => 1,
                        'max' => 500,
                    ],
                    'vh' => [
                        'min' => 1,
                        'max' => 100,
                    ],
                ],
                'selectors'      => [
                    '{{WRAPPER}} .imgbox-image:before' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'image_border_radius',
            [
                'label'      => esc_html__('Border Radius', 'nutreko'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .imgbox-image img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'image_margin',
            [
                'label'      => esc_html__('Margin', 'nutreko'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .imgbox-image' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Title.
        $this->start_controls_section(
            'section_style_imgbox_title',
            [
                'label' => esc_html__('Title', 'nutreko'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'title_text_color',
            [
                'label'     => esc_html__('Color', 'nutreko'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '#fff',
                'selectors' => [
                    '{{WRAPPER}} .imgbox-title a' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .imgbox-title' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'title_text_color_hover',
            [
                'label'     => esc_html__('Color Hover', 'nutreko'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .imgbox-title a:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'title_typography',
                'selector' => '{{WRAPPER}} .imgbox-title',
            ]
        );

        $this->add_responsive_control(
            'title_spacing',
            [
                'size_units' => ['px', 'em', '%'],
                'label'      => esc_html__('Spacing', 'nutreko'),
                'type'       => Controls_Manager::SLIDER,
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 300,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .imgbox-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        //Description
        $this->start_controls_section(
            'section_style_imgbox_button',
            [
                'label' => esc_html__('Button', 'nutreko'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'button_color',
            [
                'label'     => esc_html__('Color', 'nutreko'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '#fff',
                'selectors' => [
                    '{{WRAPPER}} .imgbox-button' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .imgbox-button a' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .imgbox-button svg' => 'fill: {{VALUE}};',
                    '{{WRAPPER}} .imgbox-button i' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'button_typography',
                'selector' => '{{WRAPPER}} .imgbox-button',
            ]
        );

        $this->end_controls_section();

        // Carousel options
        $this->add_control_carousel(['enable_carousel' => 'yes']);

    }

    /**
     * Render imgbox widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since  1.0.0
     * @access protected
     */
    protected function render() {
        $settings = $this->get_settings_for_display();
        if (!empty($settings['imgboxs']) && is_array($settings['imgboxs'])) {
            $this->add_render_attribute('wrapper', 'class', 'elementor-imgbox-item-wrapper');
            // Carousel
            $this->get_data_elementor_columns();
            // Item
            $this->add_render_attribute('item', 'class', 'elementor-imgbox-item');

            ?>
            <div <?php $this->print_render_attribute_string('wrapper'); // WPCS: XSS ok. ?>>
                <div <?php $this->print_render_attribute_string('row'); // WPCS: XSS ok. ?>>
                    <?php foreach ($settings['imgboxs'] as $index => $imgbox): ?>
                        <?php
                        $class_item = 'elementor-repeater-item-' . $imgbox['_id'];
                        $tab_title_setting_key = $this->get_repeater_setting_key('item', 'items', $index);
                        $this->add_render_attribute($tab_title_setting_key, ['class' => ['imgbox-wrapper', $class_item],]); ?>

                        <div <?php $this->print_render_attribute_string('item'); // WPCS: XSS ok. ?>>
                            <div <?php $this->print_render_attribute_string($tab_title_setting_key); ?>>
                                <?php $this->render_image($settings, $imgbox); ?>
                                <div class="imgbox-content">
                                    <?php $imgbox_title = $imgbox['imgbox_title'];
                                    if (!empty($imgbox['imgbox_link']['url'])) {
                                        $imgbox_title = '<a href="' . esc_url($imgbox['imgbox_link']['url']) . '">' . esc_html($imgbox_title) . '</a>';
                                    }
                                    printf('<h2 class="imgbox-title">%s</h2>', $imgbox_title);
                                    ?>
                                    <?php $imgbox_button = $imgbox['imgbox_button'];
                                    if (!empty($imgbox['imgbox_link']['url'])) {
                                        $imgbox_button = '<a href="' . esc_url($imgbox['imgbox_link']['url']) . '">' . esc_html($imgbox_button) . ' <span class="elementor-button-icon"><svg xmlns="http://www.w3.org/2000/svg" width="20" height="32" viewBox="0 0 20 32"><path d="M18.614 14.841l-13.091-13.091c-0.773-0.636-1.545-0.636-2.318 0-0.636 0.773-0.636 1.545 0 2.318l11.932 11.932-11.932 11.932c-0.636 0.773-0.636 1.545 0 2.318 0.773 0.637 1.545 0.637 2.318 0l13.091-13.091c0.636-0.773 0.636-1.545 0-2.318z" class="btn-icon__icon"></path><path d="M18.614 14.841l-13.091-13.091c-0.773-0.636-1.545-0.636-2.318 0-0.636 0.773-0.636 1.545 0 2.318l11.932 11.932-11.932 11.932c-0.636 0.773-0.636 1.545 0 2.318 0.773 0.637 1.545 0.637 2.318 0l13.091-13.091c0.636-0.773 0.636-1.545 0-2.318z" class="btn-icon__icon"></path></svg></span></a>';
                                    }
                                    printf('<div class="imgbox-button">%s</div>', $imgbox_button);
                                    ?>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php $this->render_swiper_pagination_navigation();?>
            <?php
        }
    }

    private function render_image($settings, $imgbox) {
        if (!empty($imgbox['imgbox_image']['url'])) :
            ?>
            <div class="imgbox-image">
                <?php
                $imgbox['imgbox_image_size']             = $settings['imgbox_image_size'];
                $imgbox['imgbox_image_custom_dimension'] = $settings['imgbox_image_custom_dimension'];
                echo Group_Control_Image_Size::get_attachment_image_html($imgbox, 'imgbox_image');
                ?>
            </div>
        <?php
        endif;
    }
}

$widgets_manager->register(new Nutreko_Elementor_ImgBoxs());

