<?php

/**
 * AJAX Handlers for Plugin Updater
 *
 * @package WP_Rig
 */

namespace Kivicare\Utility\PluginUpdater;

use Kivicare\Utility\Component_Interface;
use function add_action;
use function wp_send_json_success;
use function wp_send_json_error;

/**
 * AJAX Component for Plugin Updater
 */
class Ajax_Component implements Component_Interface
{

	/**
	 * Gets the unique identifier for the theme component.
	 *
	 * @return string Component slug.
	 */
	public function get_slug(): string
	{
		return 'plugin_updater_ajax';
	}

	/**
	 * Adds the action and filter hooks to integrate with WordPress.
	 */
	public function initialize()
	{
		// Dismiss update notice
		add_action('wp_ajax_kivicare_dismiss_premium_update_notice', array($this, 'dismiss_update_notice'));
		
		// Force check for updates
		add_action('wp_ajax_kivicare_force_update_check', array($this, 'force_update_check'));
	}

	/**
	 * Dismiss premium plugin update notice
	 */
	public function dismiss_update_notice()
	{
		// Verify user permissions
		if (!current_user_can('update_plugins')) {
			wp_send_json_error(array(
				'message' => __('You do not have permission to dismiss this notice.', 'kivicare'),
			));
			return;
		}

		// Set transient to remember dismissal for 7 days
		$user_id = get_current_user_id();
		set_transient('kivicare_premium_update_notice_dismissed_' . $user_id, true, 7 * DAY_IN_SECONDS);

		wp_send_json_success(array(
			'message' => __('Notice dismissed. You will be reminded in 7 days.', 'kivicare'),
		));
	}

	/**
	 * Force check for plugin updates
	 */
	public function force_update_check()
	{
		// Verify user permissions
		if (!current_user_can('update_plugins')) {
			wp_send_json_error(array(
				'message' => __('You do not have permission to update plugins.', 'kivicare'),
			));
			return;
		}

		// Clear all caches
		delete_transient('kivicare_premium_plugin_updates');
		delete_site_transient('update_plugins');

		// Trigger WordPress update check
		wp_update_plugins();

		wp_send_json_success(array(
			'message' => __('Update check completed. Please refresh the page.', 'kivicare'),
		));
	}
}
