<?php
/**
 * Kivicare\Utility\Import\Component class
 *
 * @package kivicare
 */

namespace Kivicare\Utility\Import;

use Kivicare\Utility\Component_Interface;
use function add_action;


class Component implements Component_Interface
{
	/**
	 * Gets the unique identifier for the theme component.
	 *
	 * @return string Component slug.
	 */
	public function get_slug(): string
	{
		return 'import';
	}

	/**
	 * Adds the action and filter hooks to integrate with WordPress.
	 */
	public function initialize()
	{

		add_filter('merlin_generate_child_functions_php', array($this, 'kivicare_generate_child_functions_php'), 10, 2);
		add_filter('merlin_import_files', array($this, 'kivicare_import_files'));
		add_action('merlin_after_all_import', array($this, 'kivicare_after_import_setup'));
		add_filter('merlin_generate_child_screenshot', function () {
			return trailingslashit(get_template_directory()) . 'screenshot.png';
		});
	}

	function kivicare_generate_child_functions_php($output, $slug)
	{

		$slug_no_hyphens = strtolower(preg_replace('#[^a-zA-Z]#', '', $slug));

		$output = "
		<?php
		/**
		 * Theme functions and definitions.
		 */
		add_action( 'wp_enqueue_scripts', '{$slug_no_hyphens}_enqueue_styles' ,99);

		function {$slug_no_hyphens}_enqueue_styles() {
				
			wp_enqueue_style( 'parent-style', get_stylesheet_directory_uri() . '/style.css'); 
			wp_enqueue_style( 'child-style',get_stylesheet_directory_uri() . '/style.css');
		}

		/**
		 * Set up My Child Theme's textdomain.
		*
		* Declare textdomain for this child theme.
		* Translations can be added to the /languages/ directory.
		*/
		function {$slug_no_hyphens}_child_theme_setup() {
			load_child_theme_textdomain( 'streamit', get_stylesheet_directory() . '/languages' );
		}
		add_action( 'after_setup_theme', '{$slug_no_hyphens}_child_theme_setup' );
	";

		// Let's remove the tabs so that it displays nicely.
		$output = trim(preg_replace('/\t+/', '', $output));

		// Filterable return.
		return $output;
	}

	public function generate_child_style_css($slug, $parent, $author, $version)
	{

		$output = "
			/**
			* Theme Name: {$parent} Child
			* Description: This is a child theme of {$parent}, generated by iQonic Design.
			* Author: {$author}
			* Template: {$slug}
			* Version: {$version}
			*/\n
		";

		// Let's remove the tabs so that it displays nicely.
		$output = trim(preg_replace('/\t+/', '', $output));

		return  $output;
	}

	public function kivicare_import_files(): array
	{
		return array(
			array(
				'import_file_name' => esc_html__('All Content', 'kivicare'),
				'local_import_file' => trailingslashit(get_template_directory()) . 'inc/Import/Demo/kivicare-content.xml',
				'local_import_widget_file' => trailingslashit(get_template_directory()) . 'inc/Import/Demo/kivicare-widget.wie',
				'local_import_customizer_file' => trailingslashit(get_template_directory()) . 'inc/Import/Demo/kivicare-export.dat',
				'local_import_redux' => array(
					array(
						'file_path' => trailingslashit(get_template_directory()) . 'inc/Import/Demo/kivicare_redux.json',
						'option_name' => 'kivi_options',
					),
				),
				'import_preview_image_url' => get_template_directory_uri() . '/inc/Import/Demo/preview_import_image.png',
				'import_notice' => esc_html__('DEMO IMPORT REQUIREMENTS: Memory Limit of 128 MB and max execution time (php time limit) of 300 seconds. ', 'kivicare') . '
									</br></br>' . esc_html__('Based on your INTERNET SPEED it could take 5 to 25 minutes. ', 'kivicare'),
				'preview_url' => 'https://wordpress.iqonic.design/product/wp/kivicare/',
			),
		);
	}

	public function kivicare_after_import_setup()
	{
		//get file
		global $wp_filesystem;
		if(empty($wp_filesystem)) {
			require_once(ABSPATH . '/wp-admin/includes/file.php');
			WP_Filesystem();
		}
		$content    =   '';
		$import_file =  trailingslashit(get_template_directory()) . 'inc/Import/Demo/elementor-site-settings.json';
		
		// Assign menus to their locations.
		$locations = get_theme_mod('nav_menu_locations'); // registered menu locations in theme
		$menus = wp_get_nav_menus(); // registered menus

		if ($menus) {
			foreach ($menus as $menu) { // assign menus to theme locations

				if ($menu->name == 'Main Menu') {
					$locations['top'] = $menu->term_id;
				}

				if( $menu->name == 'Departments menu' ) {
					$locations['kivi-departments-menu'] = $menu->term_id;
				}

			}
		}
		set_theme_mod('nav_menu_locations', $locations); // set menus to locations

		//Save menus
		$this->save_menus();

		$front_page_id = get_page_by_title('General Doctor');
		$blog_page_id = get_page_by_title('Blog');

		update_option('show_on_front', 'page');
		update_option('page_on_front', $front_page_id->ID);
		update_option('page_for_posts', $blog_page_id->ID);
		update_option('woosq_button_type', 'link');
		update_option('woosq_button_position', '0');

		// yith wishlist
		update_option('yith_wcwl_button_position','shortcode');
		update_option('yith_wcwl_add_to_wishlist_text', '');
		update_option('yith_wcwl_add_to_cart_text', '');
		update_option('yith_wcwl_browse_wishlist_text', '');

		//post-types selection for edit with elementor option
		$enable_edit_with_elementor = [
			"post",
			"page",
			"team",
			"iqonic_hf_layout",
			"testimonial",
			"service"
		];
		update_option('elementor_cpt_support', $enable_edit_with_elementor);

		// save woof setting
		$woof_setting_file =  trailingslashit(get_template_directory()) . 'inc/Import/Demo/kivicare-woof-setting.json';
		if (file_exists($woof_setting_file)) {
			$content =  $wp_filesystem->get_contents($woof_setting_file);
			if (!empty($content)) {
				$woof_options = json_decode($content, true);
				foreach ($woof_options as $option_name => $option_data) {
					if (is_serialized($option_data)) {
						$option_data = unserialize($option_data);
					}
					update_option($option_name, $option_data);
				}
			}
		}

		// icon 
		add_option( 'elementor_experiment-e_font_icon_svg', 'inactive' );
		update_option( 'elementor_experiment-e_optimized_markup', 'inactive' );

		//elementor global settings
		if (file_exists($import_file)) {
			$content = $wp_filesystem->get_contents($import_file);
			if (!empty($content)) {
				$default_post_id = get_option('elementor_active_kit');
				update_post_meta($default_post_id, '_elementor_page_settings', json_decode($content, true));
			}
		}

		$menu_item = get_posts([
            'post_type' => 'nav_menu_item',
            'post_status' => 'publish',
            'numberposts' => -1,
        ]);
        foreach ($menu_item as $key => $value) {
            wp_update_post(
                array(
                    'ID' => $value->ID,
                    'post_content' => $value->post_content,
                )
            );
        }

		// remove default post
		wp_delete_post(1, true);
	}

	public function save_menus()
	{
		global $wpdb;

		$menus = wp_get_nav_menus();
		if (empty($menus)) {
			error_log("No navigation menus found.");
			return;
		}

		foreach ($menus as $menu) {

			$menu_id      = (int) $menu->term_id;
			$taxonomy_id  = (int) $menu->term_taxonomy_id;

			error_log("Processing menu '{$menu->name}' (ID: {$menu_id})");

			/**
			 * Step 1 — Get items directly from DB
			 */
			$db_items = $wpdb->get_results($wpdb->prepare("
				SELECT p.ID, p.menu_order
				FROM {$wpdb->posts} p
				INNER JOIN {$wpdb->term_relationships} tr ON p.ID = tr.object_id
				WHERE tr.term_taxonomy_id = %d
				AND p.post_type = 'nav_menu_item'
				AND p.post_status = 'publish'
				ORDER BY p.menu_order ASC
			", $taxonomy_id));

			if (empty($db_items)) {
				error_log("Menu '{$menu->name}' has NO items.");
				continue;
			}

			/**
			 * Step 2 — Delete duplicate menu relationships
			 */
			$wpdb->query(
				$wpdb->prepare("
					DELETE tr 
					FROM {$wpdb->term_relationships} tr
					LEFT JOIN (
						SELECT 
							object_id,
							term_taxonomy_id,
							MIN(term_order) AS keep_order
						FROM {$wpdb->term_relationships}
						WHERE term_taxonomy_id = %d
						GROUP BY object_id, term_taxonomy_id
					) AS keep_rows
					ON tr.object_id = keep_rows.object_id
					AND tr.term_taxonomy_id = keep_rows.term_taxonomy_id
					AND tr.term_order = keep_rows.keep_order
					WHERE tr.term_taxonomy_id = %d
					AND keep_rows.keep_order IS NULL
				", $taxonomy_id, $taxonomy_id)
			);


			/**
			 * Step 3 — Reorder menu items
			 */
			$order = 0;
			foreach ($db_items as $item) {
				wp_update_post([
					'ID'         => (int) $item->ID,
					'menu_order' => $order++,
				]);
			}

			/**
			 * Step 4 — Fix taxonomy count
			 */
			$real_count = count($db_items);

			$wpdb->update(
				$wpdb->term_taxonomy,
				['count' => $real_count],
				['term_taxonomy_id' => $taxonomy_id]
			);

			/**
			 * Step 5 — Refresh internal WP cache
			 */
			clean_term_cache($menu_id, 'nav_menu');

			error_log("Menu '{$menu->name}' repaired. Item count: {$real_count}");
		}
	}

}
