<?php

/**
 * Premium Plugin Update Handler
 *
 * Handles automatic updates for premium WP_Rig plugins bundled with the theme
 *
 * @package WP_Rig
 */

namespace Kivicare\Utility\PluginUpdater;

use Kivicare\Utility\Component_Interface;
use function add_filter;
use function add_action;

/**
 * Class for handling premium plugin updates
 *
 * This class checks for and enables updates for premium plugins
 * that are bundled with the WP_Rig theme
 */
class Component implements Component_Interface
{

	/**
	 * Update server endpoint
	 *
	 * @var string
	 */
	private $update_server = 'https://assets.iqonic.design/wp/plugins/kivicare-new/plugins-update.json';

	/**
	 * Premium plugins to track
	 *
	 * @var array
	 */
	private $premium_plugins = array(
		'kivicare-pro/kivicare-clinic-management-system-pro.php',
		'kivicare-telemed-addon/kivicare-telemed-addon.php',
	);

	/**
	 * Gets the unique identifier for the theme component.
	 *
	 * @return string Component slug.
	 */
	public function get_slug(): string
	{
		return 'plugin_updater';
	}

	/**
	 * Adds the action and filter hooks to integrate with WordPress.
	 */
	public function initialize()
	{
		// Check for plugin updates
		add_filter('pre_set_site_transient_update_plugins', array($this, 'check_for_updates'));
		
		// Add plugin information for the update details
		add_filter('plugins_api', array($this, 'plugin_info'), 20, 3);
		
		// Show update notice after theme update
		add_action('admin_notices', array($this, 'show_update_notice'));
		
		// Auto-update premium plugins when theme is updated (optional)
		add_action('upgrader_process_complete', array($this, 'auto_update_premium_plugins'), 10, 2);
		
		// Force check for updates on theme switch
		add_action('after_switch_theme', array($this, 'force_update_check'));
	}

	/**
	 * Check for plugin updates from remote server
	 *
	 * @param object $transient Update transient object
	 * @return object Modified transient
	 */
	public function check_for_updates($transient)
	{
		if (empty($transient->checked)) {
			return $transient;
		}

		// Get update information from server
		$update_info = $this->get_remote_update_info();

		if (!$update_info) {
			return $transient;
		}

		// Check each premium plugin
		foreach ($this->premium_plugins as $plugin_file) {
			$plugin_slug = dirname($plugin_file);

			// Skip if plugin is not installed
			if (!isset($transient->checked[$plugin_file])) {
				continue;
			}

			$current_version = $transient->checked[$plugin_file];

			// Check if update is available
			if (isset($update_info[$plugin_slug]) && version_compare($current_version, $update_info[$plugin_slug]['version'], '<')) {
				$plugin_data = array(
					'slug'        => $plugin_slug,
					'plugin'      => $plugin_file,
					'new_version' => $update_info[$plugin_slug]['version'],
					'url'         => $update_info[$plugin_slug]['homepage'],
					'package'     => $update_info[$plugin_slug]['download_url'],
					'tested'      => $update_info[$plugin_slug]['tested'],
					'requires'    => $update_info[$plugin_slug]['requires'],
				);

				$transient->response[$plugin_file] = (object) $plugin_data;
			}
		}

		return $transient;
	}

	/**
	 * Get update information from remote server
	 *
	 * @return array|false Update information or false on failure
	 */
	private function get_remote_update_info()
	{
		// Check cache first
		$cache_key = 'kivicare_premium_plugin_updates';
		$cached_info = get_transient($cache_key);

		if ($cached_info !== false) {
			return $cached_info;
		}

		// Fetch from server
		$response = wp_remote_get(add_query_arg(['ver' => rand(1, 1000)], $this->update_server), array(
			'timeout' => 10,
			'headers' => array(
				'Accept' => 'application/json',
			),
		));

		if (is_wp_error($response)) {
			return false;
		}

		$body = wp_remote_retrieve_body($response);
		$data = json_decode($body, true);

		if (!$data) {
			return false;
		}

		// Cache for 12 hours
		set_transient($cache_key, $data, 12 * HOUR_IN_SECONDS);

		return $data;
	}

	/**
	 * Provide plugin information for the update details
	 *
	 * @param false|object|array $result The result object or array
	 * @param string             $action The type of information being requested
	 * @param object             $args   Plugin API arguments
	 * @return false|object Modified result
	 */
	public function plugin_info($result, $action, $args)
	{
		if ($action !== 'plugin_information') {
			return $result;
		}

		// Check if this is one of our premium plugins
		$plugin_slug = $args->slug;
		$premium_slugs = array_map('dirname', $this->premium_plugins);

		if (!in_array($plugin_slug, $premium_slugs, true)) {
			return $result;
		}

		// Get update info
		$update_info = $this->get_remote_update_info();

		if (!isset($update_info[$plugin_slug])) {
			return $result;
		}

		$info = $update_info[$plugin_slug];

		// Build plugin info object
		$plugin_info = new \stdClass();
		$plugin_info->name = $info['name'];
		$plugin_info->slug = $plugin_slug;
		$plugin_info->version = $info['version'];
		$plugin_info->author = $info['author'];
		$plugin_info->homepage = $info['homepage'];
		$plugin_info->requires = $info['requires'];
		$plugin_info->tested = $info['tested'];
		$plugin_info->downloaded = 0;
		$plugin_info->last_updated = $info['last_updated'];
		$plugin_info->sections = array(
			'description' => $info['description'],
			'changelog'   => $info['changelog'],
		);
		$plugin_info->download_link = $info['download_url'];

		return $plugin_info;
	}

	/**
	 * Show admin notice if premium plugin updates are available
	 */
	public function show_update_notice()
	{
		// Only show on admin pages
		if (!current_user_can('update_plugins')) {
			return;
		}

		// Check if user dismissed the notice
		$user_id = get_current_user_id();
		$dismissed = get_transient('kivicare_premium_update_notice_dismissed_' . $user_id);
		if ($dismissed) {
			return;
		}

		// Get update info
		$update_info = $this->get_remote_update_info();
		if (!$update_info) {
			return;
		}

		$updates_available = array();

		// Check each premium plugin
		foreach ($this->premium_plugins as $plugin_file) {
			$plugin_slug = dirname($plugin_file);
			
			// Skip if plugin is not active
			if (!is_plugin_active($plugin_file)) {
				continue;
			}

			// Get current version
			$plugin_data = get_plugin_data(WP_PLUGIN_DIR . '/' . $plugin_file);
			$current_version = $plugin_data['Version'];

			// Check if update is available
			if (isset($update_info[$plugin_slug]) && version_compare($current_version, $update_info[$plugin_slug]['version'], '<')) {
				$updates_available[] = array(
					'name'            => $plugin_data['Name'],
					'current_version' => $current_version,
					'new_version'     => $update_info[$plugin_slug]['version'],
				);
			}
		}

		// Show notice if updates are available
		if (!empty($updates_available)) {
			?>
			<div class="notice notice-warning is-dismissible kivicare-premium-update-notice">
				<h3><?php esc_html_e('KiviCare Premium Plugin Updates Available', 'kivicare'); ?></h3>
				<p><?php esc_html_e('The following premium plugins bundled with KiviCare theme have updates available:', 'kivicare'); ?></p>
				<ul>
					<?php foreach ($updates_available as $plugin) : ?>
						<li>
							<strong><?php echo esc_html($plugin['name']); ?></strong>: 
							<?php 
							printf(
								/* translators: 1: current version, 2: new version */
								esc_html__('Version %1$s → %2$s', 'kivicare'),
								esc_html($plugin['current_version']),
								esc_html($plugin['new_version'])
							);
							?>
						</li>
					<?php endforeach; ?>
				</ul>
				<p>
					<a href="<?php echo esc_url(admin_url('plugins.php')); ?>" class="button button-primary">
						<?php esc_html_e('Update Plugins Now', 'kivicare'); ?>
					</a>
					<button type="button" class="button kivicare-dismiss-update-notice">
						<?php esc_html_e('Remind Me Later', 'kivicare'); ?>
					</button>
				</p>
			</div>
			<script>
			jQuery(document).ready(function($) {
				$('.kivicare-dismiss-update-notice').on('click', function() {
					$('.kivicare-premium-update-notice').fadeOut();
					$.post(ajaxurl, {
						action: 'kivicare_dismiss_premium_update_notice'
					});
				});
			});
			</script>
			<?php
		}
	}

	/**
	 * Auto-update premium plugins when theme is updated
	 *
	 * @param object $upgrader_object WP_Upgrader instance
	 * @param array  $options         Array of bulk item update data
	 */
	public function auto_update_premium_plugins($upgrader_object, $options)
	{
		// Only proceed if this is a theme update
		if ($options['type'] !== 'theme') {
			return;
		}

		// Only proceed if KiviCare theme was updated
		if (!isset($options['themes']) || !in_array('kivicare', $options['themes'], true)) {
			return;
		}

		// Clear update cache to force fresh check
		delete_transient('kivicare_premium_plugin_updates');

		// Trigger WordPress to check for plugin updates
		wp_clean_plugins_cache();
		wp_update_plugins();

		// Optional: Schedule auto-update of premium plugins
		// Uncomment the following to enable automatic updates
		/*
		require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
		require_once ABSPATH . 'wp-admin/includes/plugin.php';

		$skin = new \Automatic_Upgrader_Skin();
		$upgrader = new \Plugin_Upgrader($skin);

		foreach ($this->premium_plugins as $plugin_file) {
			if (is_plugin_active($plugin_file)) {
				$upgrader->upgrade($plugin_file);
			}
		}
		*/
	}

	/**
	 * Force update check when theme is switched
	 */
	public function force_update_check()
	{
		// Clear cached update info
		delete_transient('kivicare_premium_plugin_updates');
		
		// Clear WordPress plugin update cache
		delete_site_transient('update_plugins');
		
		// Trigger update check
		wp_update_plugins();
	}
}
