/**
 * Listeo iCal Admin JavaScript
 */

(function($) {
    'use strict';

    var ListeoIcalAdmin = {

        init: function() {
            this.bindEvents();
            this.initTooltips();
        },

        bindEvents: function() {
            console.log('Binding events...'); // Debug log

            // Single import actions
            $(document).on('click', '.ical-force-update', this.handleForceUpdate);
            $(document).on('click', '.ical-preview', this.handlePreview);
            $(document).on('click', '.ical-remove', this.handleRemove);

            // Single listing actions
            $(document).on('click', '.ical-force-update-single', this.handleForceUpdateSingle);
            $(document).on('click', '.ical-preview-single', this.handlePreviewSingle);

            // Bulk actions
            $(document).on('click', '#bulk-force-update', this.handleBulkForceUpdate);
            $(document).on('click', '#bulk-preview', this.handleBulkPreview);

            // Inline form submissions - Handle div-based forms due to WordPress table restrictions
            $(document).on('submit', 'form.inline-import-form-enhanced', this.handleInlineAddImport);
            $(document).on('submit', 'form.inline-import-form', this.handleInlineAddImport);

            // Handle button click for div-based forms (WordPress strips form tags from table cells)
            $(document).on('click', '.ical-add-import-btn', this.handleInlineAddImportButton);

            // Also bind to click on submit button as backup for actual forms
            $(document).on('click', '.inline-import-form-enhanced button[type="submit"]', function(e) {
                e.preventDefault();
                $(this).closest('form').trigger('submit');
            });

            // Copy export URL
            $(document).on('click', '.copy-export-url', this.handleCopyExportUrl);

            // Preview modal
            $(document).on('click', '.ical-preview-close', this.closePreviewModal);
            $(document).on('click', '.ical-preview-modal', function(e) {
                if (e.target === this) {
                    ListeoIcalAdmin.closePreviewModal();
                }
            });

            // Escape key to close modal
            $(document).on('keydown', function(e) {
                if (e.keyCode === 27) { // ESC key
                    ListeoIcalAdmin.closePreviewModal();
                }
            });
        },

        initTooltips: function() {
            // Initialize tooltips for elements with title attributes (if jQuery UI is available)
            if ($.fn.tooltip) {
                $('[title]').tooltip();
            }
        },

        handleForceUpdate: function(e) {
            e.preventDefault();

            var $button = $(this);
            var listingId = $button.data('listing-id');
            var importIndex = $button.data('import-index');

            if (!confirm(listeoIcalAdmin.strings.confirmBulkUpdate)) {
                return;
            }

            ListeoIcalAdmin.setButtonLoading($button, true);
            ListeoIcalAdmin.updateStatus(listingId, importIndex, 'loading', listeoIcalAdmin.strings.processing);

            $.ajax({
                url: listeoIcalAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'listeo_ical_admin_force_update',
                    nonce: listeoIcalAdmin.nonce,
                    listing_id: listingId,
                    import_index: importIndex
                },
                success: function(response) {
                    if (response.success) {
                        var result = response.data.results[0];
                        var message = ListeoIcalAdmin.formatImportResult(result);
                        ListeoIcalAdmin.updateStatus(listingId, importIndex, 'success', message);
                        ListeoIcalAdmin.showNotice('success', response.data.message);
                    } else {
                        ListeoIcalAdmin.updateStatus(listingId, importIndex, 'error', response.data);
                        ListeoIcalAdmin.showNotice('error', response.data);
                    }
                },
                error: function(xhr, status, error) {
                    ListeoIcalAdmin.updateStatus(listingId, importIndex, 'error', error);
                    ListeoIcalAdmin.showNotice('error', listeoIcalAdmin.strings.error);
                },
                complete: function() {
                    ListeoIcalAdmin.setButtonLoading($button, false);
                }
            });
        },

        handlePreview: function(e) {
            e.preventDefault();

            var $button = $(this);
            var listingId = $button.data('listing-id');
            var importIndex = $button.data('import-index');

            ListeoIcalAdmin.setButtonLoading($button, true);

            $.ajax({
                url: listeoIcalAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'listeo_ical_admin_preview_import',
                    nonce: listeoIcalAdmin.nonce,
                    listing_id: listingId,
                    import_index: importIndex
                },
                success: function(response) {
                    if (response.success) {
                        ListeoIcalAdmin.showPreviewModal(response.data.previews);
                    } else {
                        ListeoIcalAdmin.showNotice('error', response.data);
                    }
                },
                error: function(xhr, status, error) {
                    ListeoIcalAdmin.showNotice('error', listeoIcalAdmin.strings.error);
                },
                complete: function() {
                    ListeoIcalAdmin.setButtonLoading($button, false);
                }
            });
        },

        handleRemove: function(e) {
            e.preventDefault();

            var $button = $(this);
            var listingId = $button.data('listing-id');
            var importIndex = $button.data('import-index');

            if (!confirm(listeoIcalAdmin.strings.confirmBulkDelete)) {
                return;
            }

            ListeoIcalAdmin.setButtonLoading($button, true);

            $.ajax({
                url: listeoIcalAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'listeo_ical_admin_remove_import',
                    nonce: listeoIcalAdmin.nonce,
                    listing_id: listingId,
                    import_index: importIndex
                },
                success: function(response) {
                    if (response.success) {
                        $button.closest('.ical-import-item').fadeOut(400, function() {
                            $(this).remove();
                        });
                        ListeoIcalAdmin.showNotice('success', response.data.message);
                    } else {
                        ListeoIcalAdmin.showNotice('error', response.data);
                    }
                },
                error: function(xhr, status, error) {
                    ListeoIcalAdmin.showNotice('error', listeoIcalAdmin.strings.error);
                },
                complete: function() {
                    ListeoIcalAdmin.setButtonLoading($button, false);
                }
            });
        },

        handleForceUpdateSingle: function(e) {
            e.preventDefault();

            var $button = $(this);
            var listingId = $button.data('listing-id');

            if (!confirm(listeoIcalAdmin.strings.confirmBulkUpdate)) {
                return;
            }

            ListeoIcalAdmin.setButtonLoading($button, true);

            $.ajax({
                url: listeoIcalAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'listeo_ical_admin_force_update',
                    nonce: listeoIcalAdmin.nonce,
                    listing_id: listingId
                },
                success: function(response) {
                    if (response.success) {
                        var message = response.data.results.length + ' imports processed';
                        ListeoIcalAdmin.showNotice('success', message);
                    } else {
                        ListeoIcalAdmin.showNotice('error', response.data);
                    }
                },
                error: function(xhr, status, error) {
                    ListeoIcalAdmin.showNotice('error', listeoIcalAdmin.strings.error);
                },
                complete: function() {
                    ListeoIcalAdmin.setButtonLoading($button, false);
                }
            });
        },

        handlePreviewSingle: function(e) {
            e.preventDefault();

            var $button = $(this);
            var listingId = $button.data('listing-id');

            ListeoIcalAdmin.setButtonLoading($button, true);

            $.ajax({
                url: listeoIcalAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'listeo_ical_admin_preview_import',
                    nonce: listeoIcalAdmin.nonce,
                    listing_id: listingId
                },
                success: function(response) {
                    if (response.success) {
                        ListeoIcalAdmin.showPreviewModal(response.data.previews);
                    } else {
                        ListeoIcalAdmin.showNotice('error', response.data);
                    }
                },
                error: function(xhr, status, error) {
                    ListeoIcalAdmin.showNotice('error', listeoIcalAdmin.strings.error);
                },
                complete: function() {
                    ListeoIcalAdmin.setButtonLoading($button, false);
                }
            });
        },

        handleBulkForceUpdate: function(e) {
            e.preventDefault();

            var $button = $(this);
            var listingId = $button.data('listing-id');

            if (!confirm(listeoIcalAdmin.strings.confirmBulkUpdate)) {
                return;
            }

            ListeoIcalAdmin.setButtonLoading($button, true);

            $.ajax({
                url: listeoIcalAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'listeo_ical_admin_bulk_operation',
                    nonce: listeoIcalAdmin.nonce,
                    operation: 'force_update_all',
                    listing_ids: [listingId]
                },
                success: function(response) {
                    if (response.success) {
                        ListeoIcalAdmin.showNotice('success', response.data.message);
                        // Update all import statuses
                        $('.ical-import-item').each(function() {
                            var $item = $(this);
                            var importIndex = $item.data('import-index');
                            ListeoIcalAdmin.updateStatus(listingId, importIndex, 'success', listeoIcalAdmin.strings.completed);
                        });
                    } else {
                        ListeoIcalAdmin.showNotice('error', response.data);
                    }
                },
                error: function(xhr, status, error) {
                    ListeoIcalAdmin.showNotice('error', listeoIcalAdmin.strings.error);
                },
                complete: function() {
                    ListeoIcalAdmin.setButtonLoading($button, false);
                }
            });
        },

        handleBulkPreview: function(e) {
            e.preventDefault();

            var $button = $(this);
            var listingId = $button.data('listing-id');

            ListeoIcalAdmin.setButtonLoading($button, true);

            $.ajax({
                url: listeoIcalAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'listeo_ical_admin_bulk_operation',
                    nonce: listeoIcalAdmin.nonce,
                    operation: 'preview_all',
                    listing_ids: [listingId]
                },
                success: function(response) {
                    if (response.success) {
                        var previews = response.data.results[listingId].previews;
                        ListeoIcalAdmin.showPreviewModal(previews);
                    } else {
                        ListeoIcalAdmin.showNotice('error', response.data);
                    }
                },
                error: function(xhr, status, error) {
                    ListeoIcalAdmin.showNotice('error', listeoIcalAdmin.strings.error);
                },
                complete: function() {
                    ListeoIcalAdmin.setButtonLoading($button, false);
                }
            });
        },

        setButtonLoading: function($button, loading) {
            if (loading) {
                $button.addClass('loading').prop('disabled', true);
                $button.data('original-text', $button.text());
                $button.text(listeoIcalAdmin.strings.processing);
            } else {
                $button.removeClass('loading').prop('disabled', false);
                var originalText = $button.data('original-text');
                if (originalText) {
                    $button.text(originalText);
                }
            }
        },

        updateStatus: function(listingId, importIndex, status, message) {
            var $statusDiv = $('#ical-status-' + listingId + '-' + importIndex);

            if ($statusDiv.length === 0) {
                return;
            }

            $statusDiv.removeClass('loading success error').addClass(status + ' active');
            $statusDiv.html(message);

            // Auto-hide success messages after 5 seconds
            if (status === 'success') {
                setTimeout(function() {
                    $statusDiv.fadeOut();
                }, 5000);
            }
        },

        showPreviewModal: function(previews) {
            var modalHtml = '<div class="ical-preview-modal active">' +
                '<div class="ical-preview-content">' +
                '<div class="ical-preview-header">' +
                '<h3><span class="dashicons dashicons-calendar-alt"></span> iCal Preview</h3>' +
                '<div class="ical-preview-view-toggle">' +
                '<button type="button" class="view-toggle-btn active" data-view="list">' +
                '<span class="dashicons dashicons-list-view"></span> List View' +
                '</button>' +
                '<button type="button" class="view-toggle-btn" data-view="calendar">' +
                '<span class="dashicons dashicons-calendar"></span> Calendar View' +
                '</button>' +
                '</div>' +
                '<button type="button" class="ical-preview-close">&times;</button>' +
                '</div>' +
                '<div class="ical-preview-body">' +
                '<div class="preview-view-container list-view active" data-view="list">' +
                ListeoIcalAdmin.generatePreviewContent(previews) +
                '</div>' +
                '<div class="preview-view-container calendar-view" data-view="calendar">' +
                ListeoIcalAdmin.generateCalendarView(previews) +
                '</div>' +
                '</div>' +
                '</div>' +
                '</div>';

            $('body').append(modalHtml);

            // Bind view toggle
            $('.view-toggle-btn').on('click', ListeoIcalAdmin.handleViewToggle);
        },

        generatePreviewContent: function(previews) {
            var content = '';

            if (!previews || previews.length === 0) {
                return '<p>No preview data available.</p>';
            }

            previews.forEach(function(preview) {
                content += '<div class="ical-preview-calendar">';
                content += '<div class="preview-header">';
                content += '<span class="dashicons dashicons-calendar-alt"></span>';
                content += '<h4>' + ListeoIcalAdmin.escapeHtml(preview.import_name) + '</h4>';
                content += '</div>';

                if (preview.error) {
                    content += '<div class="notice notice-error"><p><span class="dashicons dashicons-warning"></span> ' + ListeoIcalAdmin.escapeHtml(preview.error) + '</p></div>';
                } else if (preview.preview) {
                    content += '<div class="preview-info">';
                    content += '<p><strong>📄 Source:</strong> <a href="' + ListeoIcalAdmin.escapeHtml(preview.url) + '" target="_blank" class="preview-url">' + ListeoIcalAdmin.truncateUrl(preview.url) + '</a></p>';
                    content += '<p><strong>📊 Summary:</strong> ' + ListeoIcalAdmin.escapeHtml(preview.preview.message) + '</p>';
                    content += '</div>';

                    if (preview.preview.events && preview.preview.events.length > 0) {
                        content += '<div class="ical-preview-events">';
                        content += '<h5><span class="dashicons dashicons-calendar"></span> Upcoming Events/Bookings:</h5>';

                        preview.preview.events.forEach(function(event, index) {
                            var startDate = ListeoIcalAdmin.formatICalDate(event.start);
                            var endDate = ListeoIcalAdmin.formatICalDate(event.end);
                            var isAllDay = event.all_day || false;
                            var duration = ListeoIcalAdmin.calculateDuration(event.start, event.end);

                            content += '<div class="ical-preview-event enhanced">';

                            // Event title with status indicator
                            content += '<div class="event-header">';
                            content += '<span class="event-number">#' + (index + 1) + '</span>';
                            content += '<div class="event-title">' + ListeoIcalAdmin.escapeHtml(event.summary) + '</div>';
                            if (isAllDay) {
                                content += '<span class="event-type all-day">All Day</span>';
                            } else {
                                content += '<span class="event-type timed">Timed</span>';
                            }
                            content += '</div>';

                            // Enhanced date display
                            content += '<div class="event-dates">';
                            content += '<div class="date-range">';
                            content += '<span class="start-date"><span class="dashicons dashicons-calendar"></span> <strong>From:</strong> ' + startDate.formatted + '</span>';
                            content += '<span class="end-date"><span class="dashicons dashicons-calendar"></span> <strong>To:</strong> ' + endDate.formatted + '</span>';
                            content += '</div>';

                            if (duration.days > 0) {
                                content += '<div class="duration">';
                                content += '<span class="dashicons dashicons-clock"></span> ';
                                content += '<strong>Duration:</strong> ' + duration.text;
                                content += '</div>';
                            }
                            content += '</div>';

                            // Description if available
                            if (event.description && event.description.trim() !== '') {
                                content += '<div class="event-description">';
                                content += '<span class="dashicons dashicons-text-page"></span> ';
                                content += '<strong>Notes:</strong> ' + ListeoIcalAdmin.escapeHtml(event.description.substring(0, 100));
                                if (event.description.length > 100) content += '...';
                                content += '</div>';
                            }

                            content += '</div>';
                        });
                        content += '</div>';
                    } else {
                        content += '<div class="no-events">';
                        content += '<span class="dashicons dashicons-calendar"></span>';
                        content += '<p>No upcoming events found in this calendar.</p>';
                        content += '</div>';
                    }
                }

                content += '</div>';
                if (previews.length > 1) content += '<hr>';
            });

            return content;
        },

        formatICalDate: function(dateString) {
            if (!dateString) return {formatted: 'Invalid Date', date: null};

            // Handle different iCal date formats
            var date;
            var isAllDay = false;

            // Check for different formats: YYYYMMDD, YYYYMMDDTHHMMSS, etc.
            if (dateString.length === 8) {
                // All-day format: YYYYMMDD
                isAllDay = true;
                date = new Date(
                    dateString.substring(0, 4),
                    parseInt(dateString.substring(4, 6)) - 1,
                    dateString.substring(6, 8)
                );
            } else if (dateString.includes('T')) {
                // Date-time format: YYYYMMDDTHHMMSS or ISO format
                if (dateString.includes('-')) {
                    date = new Date(dateString);
                } else {
                    // YYYYMMDDTHHMMSS format
                    date = new Date(
                        dateString.substring(0, 4),
                        parseInt(dateString.substring(4, 6)) - 1,
                        dateString.substring(6, 8),
                        dateString.substring(9, 11) || 0,
                        dateString.substring(11, 13) || 0,
                        dateString.substring(13, 15) || 0
                    );
                }
            } else {
                date = new Date(dateString);
            }

            // Format the date nicely
            var options = {
                year: 'numeric',
                month: 'long',
                day: 'numeric'
            };

            if (!isAllDay && date.getHours() !== 0) {
                options.hour = '2-digit';
                options.minute = '2-digit';
            }

            var formatted = date.toLocaleDateString('en-US', options);

            return {
                formatted: formatted,
                date: date,
                isAllDay: isAllDay
            };
        },

        calculateDuration: function(startDate, endDate) {
            var start = ListeoIcalAdmin.formatICalDate(startDate).date;
            var end = ListeoIcalAdmin.formatICalDate(endDate).date;

            if (!start || !end) return {days: 0, text: ''};

            var diffTime = Math.abs(end - start);
            var diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));

            var text = '';
            if (diffDays === 1) {
                text = '1 day';
            } else if (diffDays < 7) {
                text = diffDays + ' days';
            } else if (diffDays < 30) {
                var weeks = Math.floor(diffDays / 7);
                var remainingDays = diffDays % 7;
                text = weeks + ' week' + (weeks > 1 ? 's' : '');
                if (remainingDays > 0) text += ' ' + remainingDays + ' day' + (remainingDays > 1 ? 's' : '');
            } else {
                var months = Math.floor(diffDays / 30);
                text = months + ' month' + (months > 1 ? 's' : '') + ' (' + diffDays + ' days)';
            }

            return {days: diffDays, text: text};
        },

        truncateUrl: function(url) {
            if (url.length <= 50) return url;
            return url.substring(0, 25) + '...' + url.substring(url.length - 20);
        },

        closePreviewModal: function() {
            $('.ical-preview-modal').remove();
        },

        formatImportResult: function(result) {
            if (!result || !result.result) {
                return 'Unknown result';
            }

            var res = result.result;
            var parts = [];

            if (res.imported) {
                parts.push(res.imported + ' imported');
            }
            if (res.skipped_already_booked) {
                parts.push(res.skipped_already_booked + ' already booked');
            }
            if (res.skipped_missing_slot) {
                parts.push(res.skipped_missing_slot + ' missing slot');
            }
            if (res.skipped_past) {
                parts.push(res.skipped_past + ' past events');
            }
            if (res.error) {
                return 'Error: ' + res.error;
            }

            return parts.length > 0 ? parts.join(', ') : 'No changes';
        },

        showNotice: function(type, message) {
            var noticeClass = type === 'success' ? 'notice-success' : 'notice-error';
            var notice = '<div class="notice ' + noticeClass + ' is-dismissible"><p>' +
                ListeoIcalAdmin.escapeHtml(message) + '</p></div>';

            $('.wrap h1').after(notice);

            // Auto-dismiss after 5 seconds
            setTimeout(function() {
                $('.notice').fadeOut();
            }, 5000);
        },

        handleInlineAddImport: function(e) {
            e.preventDefault();
            e.stopPropagation();

            console.log('Form submission intercepted'); // Debug log

            var $form = $(this);
            var $submitBtn = $form.find('button[type="submit"]');
            var listingId = $form.data('listing-id');
            var importName = $form.find('input[name="import_name"]').val();
            var importUrl = $form.find('input[name="import_url"]').val();
            var forceUpdate = $form.find('input[name="force_update"]').is(':checked');

            console.log('Form data:', {listingId, importName, importUrl, forceUpdate}); // Debug log

            if (!listeoIcalAdmin) {
                console.error('listeoIcalAdmin object not found');
                alert('Error: AJAX configuration not loaded');
                return false;
            }

            if (!importName || !importUrl) {
                ListeoIcalAdmin.showNotice('error', listeoIcalAdmin.strings.fillAllFields || 'Please fill all fields');
                return false;
            }

            ListeoIcalAdmin.setButtonLoading($submitBtn, true);

            $.ajax({
                url: listeoIcalAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'listeo_ical_admin_add_import',
                    nonce: listeoIcalAdmin.nonce,
                    listing_id: listingId,
                    import_name: importName,
                    import_url: importUrl,
                    force_update: forceUpdate
                },
                success: function(response) {
                    if (response.success) {
                        ListeoIcalAdmin.showNotice('success', response.data.message);
                        // Reset form
                        $form[0].reset();
                        // Reload the page to show the new import
                        setTimeout(function() {
                            location.reload();
                        }, 1500);
                    } else {
                        ListeoIcalAdmin.showNotice('error', response.data);
                    }
                },
                error: function(xhr, status, error) {
                    ListeoIcalAdmin.showNotice('error', listeoIcalAdmin.strings.error);
                },
                complete: function() {
                    ListeoIcalAdmin.setButtonLoading($submitBtn, false);
                }
            });
        },

        handleInlineAddImportButton: function(e) {
            e.preventDefault();
            e.stopPropagation();

            console.log('Button click intercepted for div-based form'); // Debug log

            var $button = $(this);
            var $formDiv = $button.closest('.inline-import-form-enhanced');
            var listingId = $formDiv.data('listing-id');
            var importName = $formDiv.find('input[name="import_name"]').val();
            var importUrl = $formDiv.find('input[name="import_url"]').val();
            var forceUpdate = $formDiv.find('input[name="force_update"]').is(':checked');

            console.log('Button form data:', {listingId, importName, importUrl, forceUpdate}); // Debug log

            if (!listeoIcalAdmin) {
                console.error('listeoIcalAdmin object not found');
                alert('Error: AJAX configuration not loaded');
                return false;
            }

            if (!importName || !importUrl) {
                ListeoIcalAdmin.showNotice('error', listeoIcalAdmin.strings.fillAllFields || 'Please fill all fields');
                return false;
            }

            ListeoIcalAdmin.setButtonLoading($button, true);

            $.ajax({
                url: listeoIcalAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'listeo_ical_admin_add_import',
                    nonce: listeoIcalAdmin.nonce,
                    listing_id: listingId,
                    import_name: importName,
                    import_url: importUrl,
                    force_update: forceUpdate
                },
                success: function(response) {
                    if (response.success) {
                        ListeoIcalAdmin.showNotice('success', response.data.message);
                        // Reset form fields
                        $formDiv.find('input[type="text"], input[type="url"]').val('');
                        $formDiv.find('input[type="checkbox"]').prop('checked', false);
                        // Reload the page to show the new import
                        setTimeout(function() {
                            location.reload();
                        }, 1500);
                    } else {
                        ListeoIcalAdmin.showNotice('error', response.data);
                    }
                },
                error: function(xhr, status, error) {
                    ListeoIcalAdmin.showNotice('error', listeoIcalAdmin.strings.error);
                },
                complete: function() {
                    ListeoIcalAdmin.setButtonLoading($button, false);
                }
            });
        },

        handleCopyExportUrl: function(e) {
            e.preventDefault();

            var $button = $(this);
            var exportUrl = $button.data('url');
            var listingId = $button.closest('.ical-export-wrapper').closest('tr').find('input[name="listing[]"]').val();

            // Copy to clipboard
            if (navigator.clipboard && window.isSecureContext) {
                navigator.clipboard.writeText(exportUrl).then(function() {
                    ListeoIcalAdmin.showNotice('success', listeoIcalAdmin.strings.urlCopied || 'URL copied to clipboard');

                    // Log the copy action
                    $.ajax({
                        url: listeoIcalAdmin.ajaxUrl,
                        type: 'POST',
                        data: {
                            action: 'listeo_ical_admin_copy_export_url',
                            nonce: listeoIcalAdmin.nonce,
                            listing_id: listingId,
                            export_url: exportUrl
                        }
                    });
                }).catch(function(err) {
                    ListeoIcalAdmin.fallbackCopyTextToClipboard(exportUrl, $button);
                });
            } else {
                ListeoIcalAdmin.fallbackCopyTextToClipboard(exportUrl, $button);
            }
        },

        fallbackCopyTextToClipboard: function(text, $button) {
            var $temp = $('<textarea>');
            $('body').append($temp);
            $temp.val(text).select();
            document.execCommand('copy');
            $temp.remove();

            ListeoIcalAdmin.showNotice('success', listeoIcalAdmin.strings.urlCopied || 'URL copied to clipboard');

            // Log the copy action
            var listingId = $button.closest('.ical-export-wrapper').closest('tr').find('input[name="listing[]"]').val();
            $.ajax({
                url: listeoIcalAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'listeo_ical_admin_copy_export_url',
                    nonce: listeoIcalAdmin.nonce,
                    listing_id: listingId,
                    export_url: text
                }
            });
        },

        generateCalendarView: function(previews) {
            if (!previews || previews.length === 0) {
                return '<p>No preview data available.</p>';
            }

            // Collect all events from all previews
            var allEvents = [];
            previews.forEach(function(preview) {
                if (preview.preview && preview.preview.events) {
                    preview.preview.events.forEach(function(event) {
                        var parsedStart = ListeoIcalAdmin.formatICalDate(event.start);
                        var parsedEnd = ListeoIcalAdmin.formatICalDate(event.end);

                        allEvents.push({
                            title: event.summary,
                            start: parsedStart.date,
                            end: parsedEnd.date,
                            startFormatted: parsedStart.formatted,
                            endFormatted: parsedEnd.formatted,
                            description: event.description,
                            allDay: event.all_day,
                            source: preview.import_name
                        });
                    });
                }
            });

            if (allEvents.length === 0) {
                return '<div class="no-events"><span class="dashicons dashicons-calendar"></span><p>No events to display in calendar view.</p></div>';
            }

            // Sort events by start date
            allEvents.sort(function(a, b) {
                return a.start - b.start;
            });

            // Group events by date for calendar display
            var eventsByDate = {};
            allEvents.forEach(function(event) {
                if (!event.start) return;

                // Create a date key for each day this event spans
                var currentDate = new Date(event.start);
                var endDate = event.end || event.start;

                while (currentDate <= endDate) {
                    var dateKey = currentDate.getFullYear() + '-' +
                                 String(currentDate.getMonth() + 1).padStart(2, '0') + '-' +
                                 String(currentDate.getDate()).padStart(2, '0');

                    if (!eventsByDate[dateKey]) {
                        eventsByDate[dateKey] = [];
                    }
                    eventsByDate[dateKey].push(event);

                    // Move to next day
                    currentDate.setDate(currentDate.getDate() + 1);
                }
            });

            // Generate calendar HTML
            var content = '<div class="ical-calendar-grid-view">';
            content += '<div class="calendar-summary"><strong>' + allEvents.length + '</strong> events from <strong>' + previews.length + '</strong> calendar(s)</div>';

            // Get unique months from events
            var months = {};
            allEvents.forEach(function(event) {
                if (!event.start) return;
                var monthKey = event.start.getFullYear() + '-' + (event.start.getMonth() + 1);
                if (!months[monthKey]) {
                    months[monthKey] = {
                        year: event.start.getFullYear(),
                        month: event.start.getMonth()
                    };
                }
            });

            // Render calendar grid for each month
            Object.keys(months).sort().forEach(function(monthKey) {
                var monthData = months[monthKey];
                content += ListeoIcalAdmin.generateMonthGrid(monthData.year, monthData.month, eventsByDate, previews);
            });

            content += '</div>';
            return content;
        },

        generateMonthGrid: function(year, month, eventsByDate, previews) {
            var monthNames = ['January', 'February', 'March', 'April', 'May', 'June',
                            'July', 'August', 'September', 'October', 'November', 'December'];
            var dayNames = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];

            // Get first and last day of month
            var firstDay = new Date(year, month, 1);
            var lastDay = new Date(year, month + 1, 0);
            var startingDayOfWeek = firstDay.getDay();
            var daysInMonth = lastDay.getDate();

            var html = '<div class="calendar-month-grid">';
            html += '<h4 class="calendar-month-header">';
            html += '<span class="dashicons dashicons-calendar-alt"></span> ';
            html += monthNames[month] + ' ' + year;
            html += '</h4>';

            // Calendar grid
            html += '<div class="calendar-grid">';

            // Day headers
            html += '<div class="calendar-week-header">';
            dayNames.forEach(function(day) {
                html += '<div class="calendar-day-name">' + day + '</div>';
            });
            html += '</div>';

            // Calendar days
            html += '<div class="calendar-days">';

            // Empty cells before month starts
            for (var i = 0; i < startingDayOfWeek; i++) {
                html += '<div class="calendar-day empty"></div>';
            }

            // Days of the month
            for (var day = 1; day <= daysInMonth; day++) {
                var dateKey = year + '-' +
                            String(month + 1).padStart(2, '0') + '-' +
                            String(day).padStart(2, '0');
                var hasEvents = eventsByDate[dateKey] && eventsByDate[dateKey].length > 0;
                var isToday = ListeoIcalAdmin.isToday(year, month, day);

                html += '<div class="calendar-day' + (hasEvents ? ' has-events' : '') + (isToday ? ' today' : '') + '">';
                html += '<div class="day-number">' + day + '</div>';

                if (hasEvents) {
                    var events = eventsByDate[dateKey];
                    var displayCount = Math.min(events.length, 3);

                    for (var e = 0; e < displayCount; e++) {
                        var event = events[e];
                        var eventColor = ListeoIcalAdmin.getColorForSource(event.source, previews);
                        var timeStr = '';

                        if (!event.allDay && event.start) {
                            timeStr = event.start.toLocaleTimeString('en-US', {
                                hour: 'numeric',
                                minute: '2-digit',
                                hour12: true
                            });
                        }

                        html += '<div class="calendar-event" style="border-left-color: ' + eventColor + '" title="' +
                                ListeoIcalAdmin.escapeHtml(event.title) +
                                (timeStr ? ' at ' + timeStr : '') +
                                ' (' + event.source + ')">';
                        html += '<span class="event-time">' + (timeStr || 'All Day') + '</span> ';
                        html += '<span class="event-title">' + ListeoIcalAdmin.escapeHtml(event.title.substring(0, 20));
                        if (event.title.length > 20) html += '...';
                        html += '</span>';
                        html += '</div>';
                    }

                    if (events.length > 3) {
                        html += '<div class="more-events">+' + (events.length - 3) + ' more</div>';
                    }
                }

                html += '</div>';
            }

            html += '</div>'; // calendar-days
            html += '</div>'; // calendar-grid
            html += '</div>'; // calendar-month-grid

            return html;
        },

        isToday: function(year, month, day) {
            var today = new Date();
            return today.getFullYear() === year &&
                   today.getMonth() === month &&
                   today.getDate() === day;
        },

        getColorForSource: function(source, previews) {
            var colors = ['#0073aa', '#46b450', '#f56e28', '#826eb4', '#dd3333', '#00a0d2', '#e1a948'];
            var index = 0;

            for (var i = 0; i < previews.length; i++) {
                if (previews[i].import_name === source) {
                    index = i;
                    break;
                }
            }

            return colors[index % colors.length];
        },

        handleViewToggle: function(e) {
            e.preventDefault();
            var $btn = $(this);
            var view = $btn.data('view');

            // Toggle active state on buttons
            $('.view-toggle-btn').removeClass('active');
            $btn.addClass('active');

            // Toggle view containers
            $('.preview-view-container').removeClass('active');
            $('.preview-view-container[data-view="' + view + '"]').addClass('active');
        },

        escapeHtml: function(text) {
            if (!text) return '';
            var map = {
                '&': '&amp;',
                '<': '&lt;',
                '>': '&gt;',
                '"': '&quot;',
                "'": '&#039;'
            };
            return text.replace(/[&<>"']/g, function(m) { return map[m]; });
        }
    };

    // Initialize when document is ready
    $(document).ready(function() {
        ListeoIcalAdmin.init();
    });

})(jQuery);