<?php
/**
 * Admin Menu Handler
 */

if (!defined('ABSPATH')) {
    exit;
}

class Listeo_iCal_Admin_Menu {

    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
    }

    /**
     * Add admin menu pages
     */
    public function add_admin_menu() {
        // Main menu page
        $hook = add_menu_page(
            __('iCal Manager', 'listeo-ical-admin'),
            __('iCal Manager', 'listeo-ical-admin'),
            'manage_options',
            'listeo-ical-manager',
            array($this, 'main_page'),
            'dashicons-calendar-alt',
            58
        );

        // Add screen options for the main page
        add_action("load-{$hook}", array($this, 'add_screen_options'));

        // Listings overview submenu
        add_submenu_page(
            'listeo-ical-manager',
            __('Listings Overview', 'listeo-ical-admin'),
            __('Listings Overview', 'listeo-ical-admin'),
            'manage_options',
            'listeo-ical-manager',
            array($this, 'main_page')
        );

        // Import history submenu
        // add_submenu_page(
        //     'listeo-ical-manager',
        //     __('Import History', 'listeo-ical-admin'),
        //     __('Import History', 'listeo-ical-admin'),
        //     'manage_options',
        //     'listeo-ical-history',
        //     array($this, 'history_page')
        // );

        // Settings submenu
        add_submenu_page(
            'listeo-ical-manager',
            __('Settings', 'listeo-ical-admin'),
            __('Settings', 'listeo-ical-admin'),
            'manage_options',
            'listeo-ical-settings',
            array($this, 'settings_page')
        );
    }

    /**
     * Add screen options
     */
    public function add_screen_options() {
        add_screen_option('per_page', array(
            'label'   => __('Listings per page', 'listeo-ical-admin'),
            'default' => 20,
            'option'  => 'listings_per_page'
        ));
    }

    /**
     * Main listings overview page
     */
    public function main_page() {
        if (isset($_GET['action']) && $_GET['action'] === 'view' && isset($_GET['listing_id'])) {
            $this->single_listing_page();
            return;
        }

        require_once plugin_dir_path(__FILE__) . 'class-listings-table-simple.php';
        $listings_table = new Listeo_iCal_Admin_Simple_Table();
        $listings_table->prepare_items();
        ?>
        <div class="wrap">
            <h1 class="wp-heading-inline"><?php _e('iCal Manager - Listings Overview', 'listeo-ical-admin'); ?></h1>

            <?php
            // Display bulk action success messages
            if (isset($_GET['bulk_action']) && $_GET['bulk_action'] === 'force_update_done') {
                echo '<div class="notice notice-success is-dismissible"><p>' . __('Bulk force update completed successfully!', 'listeo-ical-admin') . '</p></div>';
            }
            settings_errors('listeo_ical_messages');
            ?>

            <div class="notice notice-info">
                <h3><?php _e('📅 How iCal Management Works', 'listeo-ical-admin'); ?></h3>
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-top: 10px;">
                    <div>
                        <h4><?php _e('📥 Import (Sync External Calendars)', 'listeo-ical-admin'); ?></h4>
                        <p><?php _e('Add iCal URLs from external platforms (Airbnb, Booking.com, VRBO, etc.) to automatically block dates when your property is booked elsewhere. This prevents double bookings.', 'listeo-ical-admin'); ?></p>
                        <ul style="margin-left: 20px;">
                            <li><?php _e('• Add multiple calendar sources per listing', 'listeo-ical-admin'); ?></li>
                            <li><?php _e('• Automatic daily synchronization', 'listeo-ical-admin'); ?></li>
                            <li><?php _e('• Manual sync with "Update" button', 'listeo-ical-admin'); ?></li>
                        </ul>
                    </div>
                    <div>
                        <h4><?php _e('📤 Export (Share Your Calendar)', 'listeo-ical-admin'); ?></h4>
                        <p><?php _e('Share your listing\'s calendar with external platforms so they know when your property is already booked on your website.', 'listeo-ical-admin'); ?></p>
                        <ul style="margin-left: 20px;">
                            <li><?php _e('• Copy export URL and add to external platforms', 'listeo-ical-admin'); ?></li>
                            <li><?php _e('• Works with Google Calendar, Outlook, etc.', 'listeo-ical-admin'); ?></li>
                            <li><?php _e('• Real-time updates when bookings change', 'listeo-ical-admin'); ?></li>
                        </ul>
                    </div>
                </div>
            </div>

            <div class="notice notice-warning">
                <p><strong><?php _e('📋 Quick Guide:', 'listeo-ical-admin'); ?></strong></p>
                <p><?php _e('1️⃣ <strong>Import History</strong> - View all sync activities, errors, and success logs', 'listeo-ical-admin'); ?></p>
                <p><?php _e('2️⃣ <strong>Settings</strong> - Configure sync frequency, security, and performance options', 'listeo-ical-admin'); ?></p>
                <p><?php _e('3️⃣ <strong>Below Table</strong> - Manage imports/exports directly for each listing', 'listeo-ical-admin'); ?></p>
            </div>

            <form method="post">
                <?php
                $listings_table->search_box(__('Search listings', 'listeo-ical-admin'), 'listing');
                $listings_table->display();
                ?>
            </form>
        </div>
        <?php
    }

    /**
     * Single listing detail page
     */
    private function single_listing_page() {
        $listing_id = absint($_GET['listing_id']);
        $listing = get_post($listing_id);

        if (!$listing || $listing->post_type !== 'listing') {
            wp_die(__('Listing not found.', 'listeo-ical-admin'));
        }

        $ical_imports = get_post_meta($listing_id, 'listeo_ical_imports', true);
        $ical_imports = is_array($ical_imports) ? $ical_imports : array();
        ?>
        <div class="wrap">
            <h1 class="wp-heading-inline">
                <?php echo sprintf(__('iCal Management: %s', 'listeo-ical-admin'), esc_html($listing->post_title)); ?>
                <a href="<?php echo admin_url('admin.php?page=listeo-ical-manager'); ?>" class="page-title-action"><?php _e('← Back to Overview', 'listeo-ical-admin'); ?></a>
            </h1>

            <div class="listeo-ical-single-listing">
                <div class="postbox">
                    <h2 class="hndle"><?php _e('Listing Information', 'listeo-ical-admin'); ?></h2>
                    <div class="inside">
                        <table class="form-table">
                            <tr>
                                <th scope="row"><?php _e('Listing ID', 'listeo-ical-admin'); ?></th>
                                <td><?php echo $listing_id; ?></td>
                            </tr>
                            <tr>
                                <th scope="row"><?php _e('Title', 'listeo-ical-admin'); ?></th>
                                <td><?php echo esc_html($listing->post_title); ?></td>
                            </tr>
                            <tr>
                                <th scope="row"><?php _e('Status', 'listeo-ical-admin'); ?></th>
                                <td><span class="status-<?php echo $listing->post_status; ?>"><?php echo ucfirst($listing->post_status); ?></span></td>
                            </tr>
                            <tr>
                                <th scope="row"><?php _e('Booking Type', 'listeo-ical-admin'); ?></th>
                                <td><?php echo esc_html(listeo_get_booking_type($listing_id)); ?></td>
                            </tr>
                            <tr>
                                <th scope="row"><?php _e('Actions', 'listeo-ical-admin'); ?></th>
                                <td>
                                    <a href="<?php echo get_edit_post_link($listing_id); ?>" class="button"><?php _e('Edit Listing', 'listeo-ical-admin'); ?></a>
                                    <a href="<?php echo get_permalink($listing_id); ?>" class="button" target="_blank"><?php _e('View Listing', 'listeo-ical-admin'); ?></a>
                                </td>
                            </tr>
                        </table>
                    </div>
                </div>

                <div class="postbox">
                    <h2 class="hndle">
                        <?php _e('iCal Imports', 'listeo-ical-admin'); ?>
                        <span class="badge"><?php echo count($ical_imports); ?></span>
                    </h2>
                    <div class="inside">
                        <?php if (empty($ical_imports)) : ?>
                            <p><?php _e('No iCal imports configured for this listing.', 'listeo-ical-admin'); ?></p>
                        <?php else : ?>
                            <div class="ical-imports-list">
                                <?php foreach ($ical_imports as $index => $import) : ?>
                                    <div class="ical-import-item" data-listing-id="<?php echo $listing_id; ?>" data-import-index="<?php echo $index; ?>">
                                        <div class="ical-import-header">
                                            <h4><?php echo esc_html($import['name']); ?></h4>
                                            <div class="ical-import-actions">
                                                <button type="button" class="button ical-force-update" data-listing-id="<?php echo $listing_id; ?>" data-import-index="<?php echo $index; ?>">
                                                    <?php _e('Force Update', 'listeo-ical-admin'); ?>
                                                </button>
                                                <button type="button" class="button ical-preview" data-listing-id="<?php echo $listing_id; ?>" data-import-index="<?php echo $index; ?>">
                                                    <?php _e('Preview', 'listeo-ical-admin'); ?>
                                                </button>
                                                <button type="button" class="button button-link-delete ical-remove" data-listing-id="<?php echo $listing_id; ?>" data-import-index="<?php echo $index; ?>">
                                                    <?php _e('Remove', 'listeo-ical-admin'); ?>
                                                </button>
                                            </div>
                                        </div>
                                        <div class="ical-import-details">
                                            <p><strong><?php _e('URL:', 'listeo-ical-admin'); ?></strong>
                                                <a href="<?php echo esc_url($import['url']); ?>" target="_blank">
                                                    <?php echo esc_html($this->truncate_url($import['url'], 60)); ?>
                                                </a>
                                            </p>
                                            <p><strong><?php _e('Force Update:', 'listeo-ical-admin'); ?></strong> <?php echo isset($import['force_update']) && $import['force_update'] ? __('Yes', 'listeo-ical-admin') : __('No', 'listeo-ical-admin'); ?></p>
                                            <p><strong><?php _e('Author:', 'listeo-ical-admin'); ?></strong>
                                                <?php
                                                if (isset($import['bookings_author']) && $import['bookings_author']) {
                                                    $author = get_user_by('id', $import['bookings_author']);
                                                    echo $author ? esc_html($author->display_name) : __('Unknown', 'listeo-ical-admin');
                                                } else {
                                                    _e('System', 'listeo-ical-admin');
                                                }
                                                ?>
                                            </p>
                                        </div>
                                        <div class="ical-import-status" id="ical-status-<?php echo $listing_id; ?>-<?php echo $index; ?>"></div>
                                    </div>
                                <?php endforeach; ?>
                            </div>

                            <div class="ical-bulk-actions">
                                <h4><?php _e('Bulk Actions', 'listeo-ical-admin'); ?></h4>
                                <button type="button" class="button button-primary" id="bulk-force-update" data-listing-id="<?php echo $listing_id; ?>">
                                    <?php _e('Force Update All', 'listeo-ical-admin'); ?>
                                </button>
                                <button type="button" class="button" id="bulk-preview" data-listing-id="<?php echo $listing_id; ?>">
                                    <?php _e('Preview All', 'listeo-ical-admin'); ?>
                                </button>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <div class="postbox">
                    <h2 class="hndle"><?php _e('Recent Import History', 'listeo-ical-admin'); ?></h2>
                    <div class="inside">
                        <?php $this->show_recent_history($listing_id); ?>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Import history page
     */
    public function history_page() {
        $history_table = new Listeo_iCal_Admin_History_Table();
        $history_table->prepare_items();
        ?>
        <div class="wrap">
            <h1><?php _e('iCal Import History', 'listeo-ical-admin'); ?></h1>

            <form method="get">
                <input type="hidden" name="page" value="listeo-ical-history">
                <?php $history_table->search_box(__('Search history', 'listeo-ical-admin'), 'history'); ?>
            </form>

            <form method="post">
                <?php $history_table->display(); ?>
            </form>
        </div>
        <?php
    }

    /**
     * Settings page
     */
    public function settings_page() {
        $settings = new Listeo_iCal_Admin_Settings();
        $settings->render_page();
    }

    /**
     * Show recent import history for a listing
     */
    private function show_recent_history($listing_id) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'listeo_ical_import_history';
        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table_name WHERE listing_id = %d ORDER BY created_at DESC LIMIT 10",
            $listing_id
        ));

        if (empty($results)) {
            echo '<p>' . __('No import history found.', 'listeo-ical-admin') . '</p>';
            return;
        }

        echo '<table class="wp-list-table widefat fixed striped">';
        echo '<thead><tr>';
        echo '<th>' . __('Date', 'listeo-ical-admin') . '</th>';
        echo '<th>' . __('Calendar', 'listeo-ical-admin') . '</th>';
        echo '<th>' . __('Status', 'listeo-ical-admin') . '</th>';
        echo '<th>' . __('Events', 'listeo-ical-admin') . '</th>';
        echo '<th>' . __('Type', 'listeo-ical-admin') . '</th>';
        echo '</tr></thead><tbody>';

        foreach ($results as $row) {
            echo '<tr>';
            echo '<td>' . date('Y-m-d H:i', strtotime($row->created_at)) . '</td>';
            echo '<td>' . esc_html($row->ical_name) . '</td>';
            echo '<td><span class="status-' . esc_attr($row->status) . '">' . ucfirst($row->status) . '</span></td>';
            echo '<td>' . $row->events_imported . ' / ' . $row->events_skipped . '</td>';
            echo '<td>' . ucfirst($row->import_type) . '</td>';
            echo '</tr>';
        }

        echo '</tbody></table>';
        echo '<p><a href="' . admin_url('admin.php?page=listeo-ical-history&listing_id=' . $listing_id) . '">' . __('View full history', 'listeo-ical-admin') . '</a></p>';
    }

    /**
     * Truncate URL for display
     */
    private function truncate_url($url, $length = 50) {
        if (strlen($url) <= $length) {
            return $url;
        }
        return substr($url, 0, $length) . '...';
    }
}