<?php
/**
 * Bulk Operations Handler
 */

if (!defined('ABSPATH')) {
    exit;
}

class Listeo_iCal_Admin_Bulk_Operations {

    public function __construct() {
        add_action('wp_ajax_listeo_ical_bulk_operations', array($this, 'handle_bulk_request'));
    }

    /**
     * Handle bulk operation requests
     */
    public function handle_bulk_request() {
        // Verify nonce and permissions
        if (!wp_verify_nonce($_POST['nonce'] ?? '', 'listeo_ical_admin_nonce')) {
            wp_die(__('Security check failed', 'listeo-ical-admin'));
        }

        if (!current_user_can('manage_options')) {
            wp_die(__('Insufficient permissions', 'listeo-ical-admin'));
        }

        $operation = sanitize_text_field($_POST['operation'] ?? '');
        $listing_ids = isset($_POST['listing_ids']) ? array_map('absint', $_POST['listing_ids']) : array();

        if (empty($listing_ids)) {
            wp_send_json_error(__('No listings selected', 'listeo-ical-admin'));
        }

        switch ($operation) {
            case 'force_update_all':
                $this->bulk_force_update_all($listing_ids);
                break;
            case 'preview_all':
                $this->bulk_preview_all($listing_ids);
                break;
            case 'export_settings':
                $this->bulk_export_settings($listing_ids);
                break;
            case 'import_settings':
                $this->bulk_import_settings($listing_ids);
                break;
            case 'health_check':
                $this->bulk_health_check($listing_ids);
                break;
            default:
                wp_send_json_error(__('Invalid operation', 'listeo-ical-admin'));
        }
    }

    /**
     * Force update all iCal imports for multiple listings
     */
    private function bulk_force_update_all($listing_ids) {
        $results = array();
        $total_imports = 0;
        $successful_imports = 0;
        $failed_imports = 0;

        // Set time limit for bulk operations
        set_time_limit(300);

        foreach ($listing_ids as $listing_id) {
            $ical_imports = get_post_meta($listing_id, 'listeo_ical_imports', true);
            if (!is_array($ical_imports) || empty($ical_imports)) {
                $results[$listing_id] = array(
                    'status' => 'skipped',
                    'message' => __('No iCal imports found', 'listeo-ical-admin'),
                    'imports' => array()
                );
                continue;
            }

            $listing_results = array();

            foreach ($ical_imports as $import) {
                $total_imports++;
                $import['force_update'] = 'true';

                try {
                    if (class_exists('Listeo_Core_iCal')) {
                        $ical_core = Listeo_Core_iCal::instance();
                        $result = $ical_core->import_bookings_from_ical($import, $listing_id);

                        $status = isset($result['error']) ? 'error' : 'success';
                        if ($status === 'success') {
                            $successful_imports++;
                        } else {
                            $failed_imports++;
                        }

                        // Log the operation
                        $this->log_bulk_operation($listing_id, $import['name'], $import['url'], 'bulk_force', $status, $result);

                        $listing_results[] = array(
                            'import_name' => $import['name'],
                            'status' => $status,
                            'result' => $result
                        );

                    } else {
                        $failed_imports++;
                        $listing_results[] = array(
                            'import_name' => $import['name'],
                            'status' => 'error',
                            'result' => array('error' => __('Listeo Core iCal class not found', 'listeo-ical-admin'))
                        );
                    }
                } catch (Exception $e) {
                    $failed_imports++;
                    $error_result = array('error' => $e->getMessage());
                    $this->log_bulk_operation($listing_id, $import['name'], $import['url'], 'bulk_force', 'error', $error_result);

                    $listing_results[] = array(
                        'import_name' => $import['name'],
                        'status' => 'error',
                        'result' => $error_result
                    );
                }

                // Add delay between imports to prevent server overload
                usleep(250000); // 0.25 seconds
            }

            $results[$listing_id] = array(
                'status' => 'completed',
                'imports' => $listing_results,
                'listing_title' => get_the_title($listing_id)
            );
        }

        wp_send_json_success(array(
            'message' => sprintf(
                __('Bulk update completed. %d/%d imports successful, %d failed.', 'listeo-ical-admin'),
                $successful_imports,
                $total_imports,
                $failed_imports
            ),
            'total_listings' => count($listing_ids),
            'total_imports' => $total_imports,
            'successful_imports' => $successful_imports,
            'failed_imports' => $failed_imports,
            'results' => $results
        ));
    }

    /**
     * Generate preview for all imports across multiple listings
     */
    private function bulk_preview_all($listing_ids) {
        $results = array();

        foreach ($listing_ids as $listing_id) {
            $listing = get_post($listing_id);
            $ical_imports = get_post_meta($listing_id, 'listeo_ical_imports', true);

            if (!is_array($ical_imports) || empty($ical_imports)) {
                $results[$listing_id] = array(
                    'listing_title' => $listing->post_title,
                    'status' => 'no_imports',
                    'message' => __('No iCal imports found', 'listeo-ical-admin'),
                    'previews' => array()
                );
                continue;
            }

            $listing_previews = array();

            foreach ($ical_imports as $import) {
                try {
                    $preview_data = $this->preview_ical_content($import['url']);
                    $listing_previews[] = array(
                        'import_name' => $import['name'],
                        'url' => $import['url'],
                        'status' => 'success',
                        'preview' => $preview_data
                    );
                } catch (Exception $e) {
                    $listing_previews[] = array(
                        'import_name' => $import['name'],
                        'url' => $import['url'],
                        'status' => 'error',
                        'error' => $e->getMessage()
                    );
                }

                // Small delay between requests
                usleep(100000); // 0.1 second
            }

            $results[$listing_id] = array(
                'listing_title' => $listing->post_title,
                'status' => 'completed',
                'previews' => $listing_previews
            );
        }

        wp_send_json_success(array(
            'message' => sprintf(__('Preview generated for %d listings', 'listeo-ical-admin'), count($listing_ids)),
            'results' => $results
        ));
    }

    /**
     * Export iCal settings for listings
     */
    private function bulk_export_settings($listing_ids) {
        $export_data = array();

        foreach ($listing_ids as $listing_id) {
            $listing = get_post($listing_id);
            $ical_imports = get_post_meta($listing_id, 'listeo_ical_imports', true);
            $booking_type = listeo_get_booking_type($listing_id);

            $export_data[] = array(
                'listing_id' => $listing_id,
                'listing_title' => $listing->post_title,
                'listing_slug' => $listing->post_name,
                'booking_type' => $booking_type,
                'ical_imports' => is_array($ical_imports) ? $ical_imports : array(),
                'export_date' => current_time('c'),
                'export_version' => '1.0'
            );
        }

        $filename = 'listeo-ical-settings-export-' . date('Y-m-d-H-i-s') . '.json';
        $json_data = wp_json_encode($export_data, JSON_PRETTY_PRINT);

        wp_send_json_success(array(
            'message' => __('Export data generated successfully', 'listeo-ical-admin'),
            'filename' => $filename,
            'data' => $json_data,
            'download_url' => $this->create_download_file($filename, $json_data)
        ));
    }

    /**
     * Import iCal settings to listings
     */
    private function bulk_import_settings($data) {
        // This would handle importing settings from a JSON file
        // Implementation would depend on how the import data is provided
        wp_send_json_error(__('Import functionality not yet implemented', 'listeo-ical-admin'));
    }

    /**
     * Perform health check on iCal imports
     */
    private function bulk_health_check($listing_ids) {
        $health_results = array();

        foreach ($listing_ids as $listing_id) {
            $listing = get_post($listing_id);
            $ical_imports = get_post_meta($listing_id, 'listeo_ical_imports', true);

            $listing_health = array(
                'listing_title' => $listing->post_title,
                'status' => 'healthy',
                'issues' => array(),
                'recommendations' => array(),
                'imports_count' => 0,
                'working_imports' => 0,
                'broken_imports' => 0
            );

            if (!is_array($ical_imports) || empty($ical_imports)) {
                $listing_health['status'] = 'no_imports';
                $listing_health['issues'][] = __('No iCal imports configured', 'listeo-ical-admin');
            } else {
                $listing_health['imports_count'] = count($ical_imports);

                foreach ($ical_imports as $import) {
                    $import_health = $this->check_import_health($import);

                    if ($import_health['status'] === 'working') {
                        $listing_health['working_imports']++;
                    } else {
                        $listing_health['broken_imports']++;
                        $listing_health['issues'][] = sprintf(
                            __('Import "%s": %s', 'listeo-ical-admin'),
                            $import['name'],
                            $import_health['message']
                        );
                    }
                }

                if ($listing_health['broken_imports'] > 0) {
                    $listing_health['status'] = 'issues';
                }

                // Add recommendations based on findings
                if ($listing_health['imports_count'] > 5) {
                    $listing_health['recommendations'][] = __('Consider reducing the number of imports for better performance', 'listeo-ical-admin');
                }

                if ($listing_health['working_imports'] === 0 && $listing_health['imports_count'] > 0) {
                    $listing_health['status'] = 'critical';
                    $listing_health['recommendations'][] = __('All imports are broken - check URLs and credentials', 'listeo-ical-admin');
                }
            }

            $health_results[$listing_id] = $listing_health;
        }

        wp_send_json_success(array(
            'message' => sprintf(__('Health check completed for %d listings', 'listeo-ical-admin'), count($listing_ids)),
            'results' => $health_results
        ));
    }

    /**
     * Check individual import health
     */
    private function check_import_health($import) {
        try {
            // Basic URL validation
            if (!filter_var($import['url'], FILTER_VALIDATE_URL)) {
                return array(
                    'status' => 'broken',
                    'message' => __('Invalid URL format', 'listeo-ical-admin')
                );
            }

            // Try to make a HEAD request to check if URL is accessible
            $response = wp_remote_head($import['url'], array(
                'timeout' => 10,
                'user-agent' => 'Listeo iCal Health Check/1.0'
            ));

            if (is_wp_error($response)) {
                return array(
                    'status' => 'broken',
                    'message' => sprintf(__('Network error: %s', 'listeo-ical-admin'), $response->get_error_message())
                );
            }

            $response_code = wp_remote_retrieve_response_code($response);

            if ($response_code !== 200) {
                return array(
                    'status' => 'broken',
                    'message' => sprintf(__('HTTP error: %d', 'listeo-ical-admin'), $response_code)
                );
            }

            // Check content type if available
            $content_type = wp_remote_retrieve_header($response, 'content-type');
            if ($content_type && !preg_match('/text\/(calendar|plain)|application\/calendar/i', $content_type)) {
                return array(
                    'status' => 'warning',
                    'message' => sprintf(__('Unexpected content type: %s', 'listeo-ical-admin'), $content_type)
                );
            }

            return array(
                'status' => 'working',
                'message' => __('Import appears to be working correctly', 'listeo-ical-admin')
            );

        } catch (Exception $e) {
            return array(
                'status' => 'broken',
                'message' => $e->getMessage()
            );
        }
    }

    /**
     * Preview iCal content without importing
     */
    private function preview_ical_content($url) {
        if (!class_exists('Listeo_Core_iCal_Reader')) {
            throw new Exception(__('iCal Reader class not found', 'listeo-ical-admin'));
        }

        $ical = new Listeo_Core_iCal_Reader(array($url));

        if (!$ical->has_events()) {
            return array(
                'events_count' => 0,
                'events' => array(),
                'message' => __('No events found in calendar', 'listeo-ical-admin')
            );
        }

        $events = array();
        $event_count = 0;
        $max_preview_events = 5; // Limit for bulk preview

        foreach ($ical->events() as $event) {
            if ($event_count >= $max_preview_events) {
                break;
            }

            $events[] = array(
                'summary' => isset($event->summary) ? $event->summary : __('No title', 'listeo-ical-admin'),
                'start' => isset($event->dtstart) ? $event->dtstart : '',
                'end' => isset($event->dtend) ? $event->dtend : '',
                'all_day' => isset($event->all_day_event) ? $event->all_day_event : false
            );

            $event_count++;
        }

        return array(
            'events_count' => $ical->event_count,
            'events' => $events,
            'showing_count' => count($events),
            'message' => sprintf(__('Found %d events (showing first %d)', 'listeo-ical-admin'), $ical->event_count, count($events))
        );
    }

    /**
     * Log bulk operation
     */
    private function log_bulk_operation($listing_id, $ical_name, $ical_url, $type, $status, $result = array()) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'listeo_ical_import_history';

        $wpdb->insert(
            $table_name,
            array(
                'listing_id'      => $listing_id,
                'ical_name'       => $ical_name,
                'ical_url'        => $ical_url,
                'import_type'     => $type,
                'status'          => $status,
                'events_imported' => isset($result['imported']) ? $result['imported'] : 0,
                'events_skipped'  => isset($result['skipped_already_booked']) ? $result['skipped_already_booked'] + $result['skipped_missing_slot'] + $result['skipped_past'] : 0,
                'error_message'   => isset($result['error']) ? $result['error'] : null,
                'created_at'      => current_time('mysql')
            ),
            array('%d', '%s', '%s', '%s', '%s', '%d', '%d', '%s', '%s')
        );
    }

    /**
     * Create downloadable file
     */
    private function create_download_file($filename, $content) {
        $upload_dir = wp_upload_dir();
        $file_path = $upload_dir['basedir'] . '/' . $filename;

        if (file_put_contents($file_path, $content)) {
            return $upload_dir['baseurl'] . '/' . $filename;
        }

        return false;
    }
}