<?php
/**
 * Simple Working Listings Table
 */

if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('WP_List_Table')) {
    require_once(ABSPATH . 'wp-admin/includes/class-wp-list-table.php');
}

class Listeo_iCal_Admin_Simple_Table extends WP_List_Table {

    public function __construct() {
        parent::__construct(array(
            'singular' => 'listing',
            'plural'   => 'listings',
            'ajax'     => false
        ));
    }

    public function get_columns() {
        return array(
            'cb'              => '<input type="checkbox" />',
            'title'           => __('Title', 'listeo-ical-admin'),
            'status'          => __('Status', 'listeo-ical-admin'),
            'ical_management' => __('iCal Management', 'listeo-ical-admin'),
            'ical_export'     => __('iCal Export', 'listeo-ical-admin')
        );
    }

    public function get_sortable_columns() {
        return array(
            'title'  => array('title', false),
            'status' => array('post_status', false)
        );
    }

    public function get_bulk_actions() {
        return array(
            'force_update' => __('Force Update All iCal Imports', 'listeo-ical-admin'),
            'export_data'  => __('Export Data', 'listeo-ical-admin')
        );
    }

    /**
     * Process bulk actions
     */
    public function process_bulk_action() {
        // Security check
        if (!current_user_can('manage_options')) {
            return;
        }

        $action = $this->current_action();
        if (!$action) {
            return;
        }

        // Get selected listings
        $listing_ids = isset($_POST['listing']) ? array_map('absint', $_POST['listing']) : array();
        if (empty($listing_ids)) {
            return;
        }

        // Verify nonce
        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'bulk-listings')) {
            wp_die(__('Security check failed', 'listeo-ical-admin'));
        }

        switch ($action) {
            case 'force_update':
                $this->bulk_force_update($listing_ids);
                break;
            case 'export_data':
                $this->bulk_export_data($listing_ids);
                break;
        }
    }

    /**
     * Bulk force update iCal imports
     */
    private function bulk_force_update($listing_ids) {
        $updated_count = 0;
        $ical_instance = Listeo_Core_iCal::instance();

        foreach ($listing_ids as $listing_id) {
            $ical_imports = get_post_meta($listing_id, 'listeo_ical_imports', true);
            if (empty($ical_imports) || !is_array($ical_imports)) {
                continue;
            }

            foreach ($ical_imports as $import) {
                if (!isset($import['url']) || !isset($import['name'])) {
                    continue;
                }

                // Force update this import
                $import['force_update'] = 'true';
                $import['bookings_author'] = get_current_user_id();

                try {
                    $ical_instance->import_bookings_from_ical($import, $listing_id);
                    $updated_count++;
                } catch (Exception $e) {
                    // Log error but continue
                    if (defined('WP_DEBUG') && WP_DEBUG) {
                        error_log("Bulk force update error for listing {$listing_id}: " . $e->getMessage());
                    }
                }
            }
        }

        // Show success message
        add_settings_error(
            'listeo_ical_messages',
            'bulk_force_update',
            sprintf(__('Successfully triggered force update for %d iCal import(s).', 'listeo-ical-admin'), $updated_count),
            'success'
        );

        // Redirect to avoid form resubmission
        wp_safe_redirect(add_query_arg('bulk_action', 'force_update_done', wp_get_referer()));
        exit;
    }

    /**
     * Bulk export data to CSV
     */
    private function bulk_export_data($listing_ids) {
        $filename = 'listeo-ical-export-' . date('Y-m-d-His') . '.csv';

        // Set headers for download
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=' . $filename);

        // Create output stream
        $output = fopen('php://output', 'w');

        // Add BOM for UTF-8
        fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));

        // CSV headers
        fputcsv($output, array(
            'Listing ID',
            'Title',
            'Status',
            'Booking Enabled',
            'iCal Import Name',
            'iCal URL',
            'Force Update',
            'Export URL'
        ));

        // Export data
        foreach ($listing_ids as $listing_id) {
            $listing = get_post($listing_id);
            if (!$listing) {
                continue;
            }

            $booking_enabled = get_post_meta($listing_id, '_booking_status', true);
            $ical_imports = get_post_meta($listing_id, 'listeo_ical_imports', true);
            $export_url = '';

            if (function_exists('listeo_ical_export_url')) {
                $export_url = listeo_ical_export_url($listing_id);
            }

            if (empty($ical_imports) || !is_array($ical_imports)) {
                // Listing with no imports
                fputcsv($output, array(
                    $listing_id,
                    $listing->post_title,
                    $listing->post_status,
                    $booking_enabled === 'on' ? 'Yes' : 'No',
                    'No imports',
                    '',
                    '',
                    $export_url
                ));
            } else {
                // Listing with imports
                foreach ($ical_imports as $import) {
                    fputcsv($output, array(
                        $listing_id,
                        $listing->post_title,
                        $listing->post_status,
                        $booking_enabled === 'on' ? 'Yes' : 'No',
                        isset($import['name']) ? $import['name'] : '',
                        isset($import['url']) ? $import['url'] : '',
                        isset($import['force_update']) && $import['force_update'] ? 'Yes' : 'No',
                        $export_url
                    ));
                }
            }
        }

        fclose($output);
        exit;
    }

    public function prepare_items() {
        // Process bulk actions first
        $this->process_bulk_action();

        $this->_column_headers = array($this->get_columns(), array(), $this->get_sortable_columns());

        // Get per page from user preference or default to 20
        $per_page = $this->get_items_per_page('listings_per_page', 20);
        $current_page = $this->get_pagenum();

        // Get search term
        $search = isset($_REQUEST['s']) ? sanitize_text_field($_REQUEST['s']) : '';

        // Get sort parameters
        $orderby = isset($_REQUEST['orderby']) ? sanitize_text_field($_REQUEST['orderby']) : 'date';
        $order = isset($_REQUEST['order']) && $_REQUEST['order'] === 'asc' ? 'ASC' : 'DESC';

        // Get status filter
        $filter_status = isset($_REQUEST['filter_status']) ? sanitize_text_field($_REQUEST['filter_status']) : '';

        // Build query with proper pagination
        $args = array(
            'post_type'      => 'listing',
            'post_status'    => !empty($filter_status) ? $filter_status : array('publish', 'pending', 'draft', 'private'),
            'posts_per_page' => $per_page,
            'paged'          => $current_page,
            'orderby'        => $orderby,
            'order'          => $order
        );

        // Add search parameter
        if (!empty($search)) {
            $args['s'] = $search;
        }

        // Execute query
        $query = new WP_Query($args);
        $this->items = $query->posts;

        // Set pagination args
        $this->set_pagination_args(array(
            'total_items' => $query->found_posts,
            'per_page'    => $per_page,
            'total_pages' => ceil($query->found_posts / $per_page)
        ));
    }

    public function column_cb($item) {
        return sprintf('<input type="checkbox" name="listing[]" value="%s" />', $item->ID);
    }

    public function column_title($item) {
        // make tittle link to view listing
        return '<strong><a href="' . esc_url(get_permalink($item->ID)) . '">' . esc_html($item->post_title) . '</a></strong>';
    }

    public function column_id($item) {
        return $item->ID;
    }

    public function column_status($item) {
        $status_colors = array(
            'publish' => 'green',
            'pending' => 'orange',
            'draft'   => 'gray',
            'private' => 'blue',
            'expired' => 'red'
        );

        $status_labels = array(
            'publish' => __('Published', 'listeo-ical-admin'),
            'pending' => __('Pending', 'listeo-ical-admin'),
            'draft'   => __('Draft', 'listeo-ical-admin'),
            'private' => __('Private', 'listeo-ical-admin'),
            'expired' => __('Expired', 'listeo-ical-admin')
        );

        $color = isset($status_colors[$item->post_status]) ? $status_colors[$item->post_status] : 'gray';
        $label = isset($status_labels[$item->post_status]) ? $status_labels[$item->post_status] : ucfirst($item->post_status);

        return sprintf(
            '<span class="status-badge status-%s" style="display: inline-block; padding: 3px 8px; border-radius: 3px; font-size: 11px; font-weight: 600; color: #fff; background-color: %s;">%s</span>',
            esc_attr($color),
            $this->get_status_color($color),
            esc_html($label)
        );
    }

    /**
     * Get status color hex code
     */
    private function get_status_color($color) {
        $colors = array(
            'green'  => '#46b450',
            'orange' => '#f56e28',
            'gray'   => '#8c8f94',
            'blue'   => '#2271b1',
            'red'    => '#dc3232'
        );
        return isset($colors[$color]) ? $colors[$color] : $colors['gray'];
    }

    public function column_ical_management($item) {
        $ical_imports = get_post_meta($item->ID, 'listeo_ical_imports', true);
        $ical_imports = is_array($ical_imports) ? $ical_imports : array();
        $booking_enabled = get_post_meta($item->ID, '_booking_status', true);

        $output = '<div class="ical-management-wrapper" data-listing-id="' . esc_attr($item->ID) . '">';

        // Booking status indicator
        if ($booking_enabled !== 'on') {
            $output .= '<div class="booking-disabled-notice">';
            $output .= '<span class="dashicons dashicons-warning"></span>';
            $output .= '<p><strong>Booking is disabled</strong><br><small>Enable booking to use iCal imports</small></p>';
            $output .= '</div>';
            $output .= '</div>';
            return $output;
        }

        // Show existing imports in a nicer format
        if (!empty($ical_imports)) {
            $output .= '<div class="existing-imports">';
            $output .= '<div class="imports-header">';
            $output .= '<span class="dashicons dashicons-calendar-alt"></span>';
            $output .= '<strong>' . sprintf('%d Active Import(s)', count($ical_imports)) . '</strong>';
            $output .= '</div>';

            foreach ($ical_imports as $index => $import) {
                if (!isset($import['name']) || !isset($import['url'])) continue;

                $output .= '<div class="import-item-enhanced" data-import-index="' . esc_attr($index) . '">';
                $output .= '<div class="import-header">';
                $output .= '<span class="import-name">' . esc_html($import['name']) . '</span>';
                $output .= '<div class="import-actions-inline">';
                $output .= '<button type="button" class="button-link ical-force-update" data-listing-id="' . esc_attr($item->ID) . '" data-import-index="' . esc_attr($index) . '" title="Sync now"><span class="dashicons dashicons-update"></span></button>';
                $output .= '<button type="button" class="button-link ical-preview" data-listing-id="' . esc_attr($item->ID) . '" data-import-index="' . esc_attr($index) . '" title="Preview events"><span class="dashicons dashicons-visibility"></span></button>';
                $output .= '<button type="button" class="button-link ical-remove" data-listing-id="' . esc_attr($item->ID) . '" data-import-index="' . esc_attr($index) . '" title="Remove import"><span class="dashicons dashicons-trash"></span></button>';
                $output .= '</div>';
                $output .= '</div>';
                $output .= '<div class="import-url">';
                $output .= '<a href="' . esc_url($import['url']) . '" target="_blank" title="' . esc_attr($import['url']) . '">';
                $output .= esc_html(strlen($import['url']) > 45 ? substr($import['url'], 0, 45) . '...' : $import['url']);
                $output .= '</a>';
                $output .= '</div>';
                $output .= '<div id="ical-status-' . esc_attr($item->ID) . '-' . esc_attr($index) . '" class="ical-import-status"></div>';
                $output .= '</div>';
            }
            $output .= '</div>';
        }

        // Add new import form with better design
        $output .= '<div class="add-import-form-enhanced" style="' . (!empty($ical_imports) ? 'margin-top: 15px; border-top: 1px solid #ddd; padding-top: 15px;' : '') . '">';

        // Show status if no imports exist
        if (empty($ical_imports)) {
            $output .= '<div class="no-imports-notice">';
            $output .= '<span class="dashicons dashicons-calendar-alt"></span>';
            $output .= '<small style="color: #666; margin-left: 5px;">No iCal imports configured</small>';
            $output .= '</div>';
        }

        $output .= '<div class="form-header">';
        $output .= '<span class="dashicons dashicons-plus-alt"></span>';
        $output .= '<strong>Add iCal Import</strong>';
        $output .= '</div>';
        $output .= '<div class="inline-import-form-enhanced" data-listing-id="' . esc_attr($item->ID) . '" data-form-type="ical-import">';
        $output .= '<div class="form-row">';
        $output .= '<label for="import_name_' . $item->ID . '">Name:</label>';
        $output .= '<input type="text" id="import_name_' . $item->ID . '" placeholder="e.g. Airbnb Calendar" name="import_name" required>';
        $output .= '</div>';
        $output .= '<div class="form-row">';
        $output .= '<label for="import_url_' . $item->ID . '">iCal URL:</label>';
        $output .= '<input type="url" id="import_url_' . $item->ID . '" placeholder="https://..." name="import_url" required>';
        $output .= '</div>';
        $output .= '<div class="form-row checkbox-row">';
        $output .= '<label>';
        $output .= '<input type="checkbox" name="force_update">';
        $output .= 'Override existing bookings';
        $output .= '<small class="help-text">Check this if you want to overwrite conflicting dates</small>';
        $output .= '</label>';
        $output .= '</div>';
        $output .= '<div class="form-actions">';
        $output .= '<button type="button" class="button button-primary button-small ical-add-import-btn" data-listing-id="' . esc_attr($item->ID) . '">';
        $output .= '<span class="dashicons dashicons-plus"></span> Add Import';
        $output .= '</button>';
        $output .= '</div>';
        $output .= '</div>';
        $output .= '</div>';

        $output .= '</div>';
        return $output;
    }

    public function column_ical_export($item) {
        $booking_enabled = get_post_meta($item->ID, '_booking_status', true);

        if ($booking_enabled !== 'on') {
            return '<div class="export-disabled">'
                 . '<span class="dashicons dashicons-warning"></span>'
                 . '<span class="status-inactive">Booking disabled</span>'
                 . '</div>';
        }

        // Generate export URL (check if function exists, otherwise create basic URL)
        if (function_exists('listeo_ical_export_url')) {
            $export_url = listeo_ical_export_url($item->ID);
        } else {
            $export_url = home_url('/listeo-ical-export/' . $item->ID . '/');
        }

        $output = '<div class="ical-export-wrapper-enhanced">';
        $output .= '<div class="export-header">';
        $output .= '<span class="dashicons dashicons-download"></span>';
        $output .= '<strong>Export Calendar</strong>';
        $output .= '</div>';
        $output .= '<div class="export-content">';
        $output .= '<div class="export-url-wrapper">';
        $output .= '<input type="text" value="' . esc_attr($export_url) . '" class="export-url-input" readonly>';
        $output .= '<button type="button" class="button button-secondary button-small copy-export-url" data-url="' . esc_attr($export_url) . '" data-listing-id="' . esc_attr($item->ID) . '">';
        $output .= '<span class="dashicons dashicons-clipboard"></span> Copy';
        $output .= '</button>';
        $output .= '</div>';
        $output .= '<small class="export-help">Share this URL with external calendar apps (Google Calendar, Outlook, etc.)</small>';
        $output .= '</div>';
        $output .= '</div>';

        return $output;
    }

    /**
     * Display extra table navigation
     */
    protected function extra_tablenav($which) {
        if ('top' !== $which) {
            return;
        }

        ?>
        <div class="alignleft actions">
            <select name="filter_status" id="filter-by-status">
                <option value=""><?php _e('All Statuses', 'listeo-ical-admin'); ?></option>
                <option value="publish" <?php selected(isset($_REQUEST['filter_status']) ? $_REQUEST['filter_status'] : '', 'publish'); ?>><?php _e('Published', 'listeo-ical-admin'); ?></option>
                <option value="pending" <?php selected(isset($_REQUEST['filter_status']) ? $_REQUEST['filter_status'] : '', 'pending'); ?>><?php _e('Pending', 'listeo-ical-admin'); ?></option>
                <option value="draft" <?php selected(isset($_REQUEST['filter_status']) ? $_REQUEST['filter_status'] : '', 'draft'); ?>><?php _e('Draft', 'listeo-ical-admin'); ?></option>
                <option value="private" <?php selected(isset($_REQUEST['filter_status']) ? $_REQUEST['filter_status'] : '', 'private'); ?>><?php _e('Private', 'listeo-ical-admin'); ?></option>
            </select>
            <?php submit_button(__('Filter', 'listeo-ical-admin'), '', 'filter_action', false); ?>
        </div>
        <?php
    }

    /**
     * Display table summary information
     */
    public function display_table_info() {
        $total = $this->_pagination_args['total_items'];
        $per_page = $this->_pagination_args['per_page'];
        $current_page = $this->get_pagenum();

        $start = ($current_page - 1) * $per_page + 1;
        $end = min($current_page * $per_page, $total);

        ?>
        <div class="tablenav-pages-info" style="float: right; margin: 10px 0; color: #646970;">
            <?php
            printf(
                __('Displaying %s-%s of %s listings', 'listeo-ical-admin'),
                number_format_i18n($start),
                number_format_i18n($end),
                number_format_i18n($total)
            );
            ?>
        </div>
        <?php
    }
}