<?php
/**
 * Settings Page
 */

if (!defined('ABSPATH')) {
    exit;
}

class Listeo_iCal_Admin_Settings {

    private $options;

    public function __construct() {
        add_action('admin_init', array($this, 'init_settings'));
    }

    /**
     * Initialize settings
     */
    public function init_settings() {
        register_setting(
            'listeo_ical_admin_settings',
            'listeo_ical_admin_options',
            array($this, 'sanitize_options')
        );

        add_settings_section(
            'listeo_ical_admin_general',
            __('General Settings', 'listeo-ical-admin'),
            array($this, 'section_general_callback'),
            'listeo_ical_admin_settings'
        );

        add_settings_section(
            'listeo_ical_admin_performance',
            __('Performance Settings', 'listeo-ical-admin'),
            array($this, 'section_performance_callback'),
            'listeo_ical_admin_settings'
        );

        add_settings_section(
            'listeo_ical_admin_security',
            __('Security Settings', 'listeo-ical-admin'),
            array($this, 'section_security_callback'),
            'listeo_ical_admin_settings'
        );

        add_settings_section(
            'listeo_ical_admin_logging',
            __('Logging & History', 'listeo-ical-admin'),
            array($this, 'section_logging_callback'),
            'listeo_ical_admin_settings'
        );

        // General settings fields
        add_settings_field(
            'batch_size',
            __('Batch Size', 'listeo-ical-admin'),
            array($this, 'field_batch_size_callback'),
            'listeo_ical_admin_settings',
            'listeo_ical_admin_general'
        );

        add_settings_field(
            'cron_frequency',
            __('Cron Frequency', 'listeo-ical-admin'),
            array($this, 'field_cron_frequency_callback'),
            'listeo_ical_admin_settings',
            'listeo_ical_admin_general'
        );

        // Performance settings fields
        add_settings_field(
            'max_execution_time',
            __('Max Execution Time', 'listeo-ical-admin'),
            array($this, 'field_max_execution_time_callback'),
            'listeo_ical_admin_settings',
            'listeo_ical_admin_performance'
        );

        add_settings_field(
            'memory_limit_check',
            __('Memory Limit Check', 'listeo-ical-admin'),
            array($this, 'field_memory_limit_check_callback'),
            'listeo_ical_admin_settings',
            'listeo_ical_admin_performance'
        );

        add_settings_field(
            'rate_limit_delay',
            __('Rate Limit Delay', 'listeo-ical-admin'),
            array($this, 'field_rate_limit_delay_callback'),
            'listeo_ical_admin_settings',
            'listeo_ical_admin_performance'
        );

        // Security settings fields
        add_settings_field(
            'allowed_domains',
            __('Allowed Domains', 'listeo-ical-admin'),
            array($this, 'field_allowed_domains_callback'),
            'listeo_ical_admin_settings',
            'listeo_ical_admin_security'
        );

        add_settings_field(
            'timeout_seconds',
            __('Request Timeout', 'listeo-ical-admin'),
            array($this, 'field_timeout_seconds_callback'),
            'listeo_ical_admin_settings',
            'listeo_ical_admin_security'
        );

        // Logging settings fields
        add_settings_field(
            'log_retention_days',
            __('Log Retention Days', 'listeo-ical-admin'),
            array($this, 'field_log_retention_days_callback'),
            'listeo_ical_admin_settings',
            'listeo_ical_admin_logging'
        );

        add_settings_field(
            'detailed_logging',
            __('Detailed Logging', 'listeo-ical-admin'),
            array($this, 'field_detailed_logging_callback'),
            'listeo_ical_admin_settings',
            'listeo_ical_admin_logging'
        );
    }

    /**
     * Render settings page
     */
    public function render_page() {
        $this->options = get_option('listeo_ical_admin_options', $this->get_default_options());

        if (isset($_POST['cleanup_history'])) {
            $this->cleanup_old_history();
        }

        if (isset($_POST['test_import'])) {
            $this->test_import_functionality();
        }
        ?>
        <div class="wrap airs-admin-wrap">
            <div class="airs-header">
                <div class="airs-header-content">
                    <div class="airs-header-icon">
                        📅
                    </div>
                    <div class="airs-header-text">
                        <h1><?php _e('iCal Manager Settings', 'listeo-ical-admin'); ?></h1>
                        <p><?php _e('Configure iCal import/export settings, performance options, and security controls', 'listeo-ical-admin'); ?></p>
                    </div>
                </div>
            </div>

            <div class="listeo-ical-settings-wrapper">
                <div class="listeo-ical-settings-main">
                    <form method="post" action="options.php">
                        <?php
                        settings_fields('listeo_ical_admin_settings');
                        do_settings_sections('listeo_ical_admin_settings');
                        submit_button();
                        ?>
                    </form>
                </div>

                <div class="listeo-ical-settings-sidebar">
                    <div class="airs-card">
                        <div class="airs-card-header">
                            <h3><?php _e('System Status', 'listeo-ical-admin'); ?></h3>
                            <p><?php _e('Server environment and plugin status', 'listeo-ical-admin'); ?></p>
                        </div>
                        <div class="airs-card-body">
                            <?php $this->render_system_status(); ?>
                        </div>
                    </div>

                    <div class="airs-card">
                        <div class="airs-card-header">
                            <h3><?php _e('Statistics', 'listeo-ical-admin'); ?></h3>
                            <p><?php _e('Import activity and performance metrics', 'listeo-ical-admin'); ?></p>
                        </div>
                        <div class="airs-card-body">
                            <?php $this->render_statistics(); ?>
                        </div>
                    </div>

                    <div class="airs-card">
                        <div class="airs-card-header">
                            <h3><?php _e('Maintenance', 'listeo-ical-admin'); ?></h3>
                            <p><?php _e('Tools for testing and cleanup', 'listeo-ical-admin'); ?></p>
                        </div>
                        <div class="airs-card-body">
                            <?php $this->render_maintenance_tools(); ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Get default options
     */
    private function get_default_options() {
        return array(
            'batch_size'          => 10,
            'cron_frequency'      => '30min',
            'max_execution_time'  => 25,
            'memory_limit_check'  => 50,
            'rate_limit_delay'    => 500,
            'allowed_domains'     => implode("\n", array(
                'calendar.google.com',
                'outlook.live.com',
                'outlook.office365.com',
                'calendars.icloud.com',
                'airbnb.com',
                'booking.com',
                'airtable.com'
            )),
            'timeout_seconds'     => 30,
            'log_retention_days'  => 30,
            'detailed_logging'    => false
        );
    }

    /**
     * Sanitize options
     */
    public function sanitize_options($input) {
        $sanitized = array();

        $sanitized['batch_size'] = absint($input['batch_size']);
        if ($sanitized['batch_size'] < 1) $sanitized['batch_size'] = 10;
        if ($sanitized['batch_size'] > 100) $sanitized['batch_size'] = 100;

        $sanitized['cron_frequency'] = sanitize_text_field($input['cron_frequency']);
        $sanitized['max_execution_time'] = absint($input['max_execution_time']);
        $sanitized['memory_limit_check'] = absint($input['memory_limit_check']);
        $sanitized['rate_limit_delay'] = absint($input['rate_limit_delay']);
        $sanitized['allowed_domains'] = sanitize_textarea_field($input['allowed_domains']);
        $sanitized['timeout_seconds'] = absint($input['timeout_seconds']);
        $sanitized['log_retention_days'] = absint($input['log_retention_days']);
        $sanitized['detailed_logging'] = !empty($input['detailed_logging']);

        return $sanitized;
    }

    /**
     * Section callbacks
     */
    public function section_general_callback() {
        echo '<p>' . __('Configure general iCal import settings.', 'listeo-ical-admin') . '</p>';
    }

    public function section_performance_callback() {
        echo '<p>' . __('Optimize performance for large numbers of imports.', 'listeo-ical-admin') . '</p>';
    }

    public function section_security_callback() {
        echo '<p>' . __('Security settings for iCal URL validation and access control.', 'listeo-ical-admin') . '</p>';
    }

    public function section_logging_callback() {
        echo '<p>' . __('Configure logging and history retention settings.', 'listeo-ical-admin') . '</p>';
    }

    /**
     * Get options (with fallback to defaults)
     */
    private function get_options() {
        if (!$this->options) {
            $this->options = get_option('listeo_ical_admin_options', $this->get_default_options());
        }
        return $this->options;
    }

    /**
     * Field callbacks
     */
    public function field_batch_size_callback() {
        $options = $this->get_options();
        printf(
            '<input type="number" id="batch_size" name="listeo_ical_admin_options[batch_size]" value="%d" min="1" max="100" />',
            isset($options['batch_size']) ? $options['batch_size'] : 10
        );
        echo '<p class="description">' . __('Number of listings to process per cron job execution (1-100).', 'listeo-ical-admin') . '</p>';
    }

    public function field_cron_frequency_callback() {
        $options = $this->get_options();
        $frequencies = array(
            '15min' => __('Every 15 minutes', 'listeo-ical-admin'),
            '30min' => __('Every 30 minutes', 'listeo-ical-admin'),
            'hourly' => __('Hourly', 'listeo-ical-admin'),
            '2hourly' => __('Every 2 hours', 'listeo-ical-admin'),
            'daily' => __('Daily', 'listeo-ical-admin')
        );

        echo '<select id="cron_frequency" name="listeo_ical_admin_options[cron_frequency]">';
        foreach ($frequencies as $value => $label) {
            printf(
                '<option value="%s"%s>%s</option>',
                esc_attr($value),
                selected(isset($options['cron_frequency']) ? $options['cron_frequency'] : '30min', $value, false),
                esc_html($label)
            );
        }
        echo '</select>';
        echo '<p class="description">' . __('How often to automatically check for iCal updates.', 'listeo-ical-admin') . '</p>';
    }

    public function field_max_execution_time_callback() {
        $options = $this->get_options();
        printf(
            '<input type="number" id="max_execution_time" name="listeo_ical_admin_options[max_execution_time]" value="%d" min="10" max="300" />',
            isset($options['max_execution_time']) ? $options['max_execution_time'] : 25
        );
        echo '<p class="description">' . __('Maximum execution time per cron job in seconds (10-300).', 'listeo-ical-admin') . '</p>';
    }

    public function field_memory_limit_check_callback() {
        $options = $this->get_options();
        printf(
            '<input type="number" id="memory_limit_check" name="listeo_ical_admin_options[memory_limit_check]" value="%d" min="10" max="500" />',
            isset($options['memory_limit_check']) ? $options['memory_limit_check'] : 50
        );
        echo '<p class="description">' . __('Stop import if memory usage increases by this many MB (10-500).', 'listeo-ical-admin') . '</p>';
    }

    public function field_rate_limit_delay_callback() {
        $options = $this->get_options();
        printf(
            '<input type="number" id="rate_limit_delay" name="listeo_ical_admin_options[rate_limit_delay]" value="%d" min="100" max="5000" step="100" />',
            isset($options['rate_limit_delay']) ? $options['rate_limit_delay'] : 500
        );
        echo '<p class="description">' . __('Delay between imports in milliseconds (100-5000).', 'listeo-ical-admin') . '</p>';
    }

    public function field_allowed_domains_callback() {
        $options = $this->get_options();
        printf(
            '<textarea id="allowed_domains" name="listeo_ical_admin_options[allowed_domains]" rows="8" cols="50" class="large-text">%s</textarea>',
            isset($options['allowed_domains']) ? esc_textarea($options['allowed_domains']) : ''
        );
        echo '<p class="description">' . __('One domain per line. Used for iCal URL validation. Leave empty to allow all domains.', 'listeo-ical-admin') . '</p>';
    }

    public function field_timeout_seconds_callback() {
        $options = $this->get_options();
        printf(
            '<input type="number" id="timeout_seconds" name="listeo_ical_admin_options[timeout_seconds]" value="%d" min="5" max="120" />',
            isset($options['timeout_seconds']) ? $options['timeout_seconds'] : 30
        );
        echo '<p class="description">' . __('HTTP request timeout in seconds (5-120).', 'listeo-ical-admin') . '</p>';
    }

    public function field_log_retention_days_callback() {
        $options = $this->get_options();
        printf(
            '<input type="number" id="log_retention_days" name="listeo_ical_admin_options[log_retention_days]" value="%d" min="1" max="365" />',
            isset($options['log_retention_days']) ? $options['log_retention_days'] : 30
        );
        echo '<p class="description">' . __('How many days to keep import history records (1-365).', 'listeo-ical-admin') . '</p>';
    }

    public function field_detailed_logging_callback() {
        $options = $this->get_options();
        printf(
            '<input type="checkbox" id="detailed_logging" name="listeo_ical_admin_options[detailed_logging]" value="1" %s />',
            checked(isset($options['detailed_logging']) ? $options['detailed_logging'] : false, true, false)
        );
        echo '<label for="detailed_logging">' . __('Enable detailed logging for debugging', 'listeo-ical-admin') . '</label>';
        echo '<p class="description">' . __('Warning: May increase log file sizes significantly.', 'listeo-ical-admin') . '</p>';
    }

    /**
     * Render system status
     */
    private function render_system_status() {
        $status_items = array(
            array(
                'label' => __('PHP Version', 'listeo-ical-admin'),
                'value' => PHP_VERSION,
                'status' => version_compare(PHP_VERSION, '7.4', '>=') ? 'good' : 'warning'
            ),
            array(
                'label' => __('WordPress Version', 'listeo-ical-admin'),
                'value' => get_bloginfo('version'),
                'status' => version_compare(get_bloginfo('version'), '5.0', '>=') ? 'good' : 'warning'
            ),
            array(
                'label' => __('Listeo Core', 'listeo-ical-admin'),
                'value' => class_exists('Listeo_Core_iCal') ? __('Active', 'listeo-ical-admin') : __('Not Found', 'listeo-ical-admin'),
                'status' => class_exists('Listeo_Core_iCal') ? 'good' : 'error'
            ),
            array(
                'label' => __('Memory Limit', 'listeo-ical-admin'),
                'value' => ini_get('memory_limit'),
                'status' => 'good'
            ),
            array(
                'label' => __('Max Execution Time', 'listeo-ical-admin'),
                'value' => ini_get('max_execution_time') . 's',
                'status' => 'good'
            )
        );

        echo '<table class="listeo-status-table">';
        foreach ($status_items as $item) {
            printf(
                '<tr><td>%s:</td><td class="status-%s">%s</td></tr>',
                esc_html($item['label']),
                esc_attr($item['status']),
                esc_html($item['value'])
            );
        }
        echo '</table>';
    }

    /**
     * Render statistics
     */
    private function render_statistics() {
        global $wpdb;

        $history_table = $wpdb->prefix . 'listeo_ical_import_history';

        // Get statistics
        $total_imports = $wpdb->get_var("SELECT COUNT(*) FROM $history_table");
        $successful_imports = $wpdb->get_var("SELECT COUNT(*) FROM $history_table WHERE status = 'success'");
        $failed_imports = $wpdb->get_var("SELECT COUNT(*) FROM $history_table WHERE status = 'error'");
        $today_imports = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $history_table WHERE DATE(created_at) = %s",
            current_time('Y-m-d')
        ));

        $listings_with_icals = $wpdb->get_var("
            SELECT COUNT(*) FROM {$wpdb->postmeta}
            WHERE meta_key = 'listeo_ical_imports'
            AND meta_value != ''
            AND meta_value != 'a:0:{}'
        ");

        echo '<table class="listeo-stats-table">';
        printf('<tr><td>%s:</td><td><strong>%d</strong></td></tr>', __('Total Imports', 'listeo-ical-admin'), $total_imports);
        printf('<tr><td>%s:</td><td><strong class="status-good">%d</strong></td></tr>', __('Successful', 'listeo-ical-admin'), $successful_imports);
        printf('<tr><td>%s:</td><td><strong class="status-error">%d</strong></td></tr>', __('Failed', 'listeo-ical-admin'), $failed_imports);
        printf('<tr><td>%s:</td><td><strong>%d</strong></td></tr>', __('Today', 'listeo-ical-admin'), $today_imports);
        printf('<tr><td>%s:</td><td><strong>%d</strong></td></tr>', __('Listings with iCal', 'listeo-ical-admin'), $listings_with_icals);
        echo '</table>';
    }

    /**
     * Render maintenance tools
     */
    private function render_maintenance_tools() {
        echo '<form method="post">';
        echo '<p><button type="submit" name="cleanup_history" class="button">' . __('Cleanup Old History', 'listeo-ical-admin') . '</button></p>';
        echo '<p><button type="submit" name="test_import" class="button">' . __('Test Import Function', 'listeo-ical-admin') . '</button></p>';
        echo '</form>';
    }

    /**
     * Cleanup old history records
     */
    private function cleanup_old_history() {
        global $wpdb;

        $retention_days = $this->options['log_retention_days'];
        $table_name = $wpdb->prefix . 'listeo_ical_import_history';

        $deleted = $wpdb->query($wpdb->prepare(
            "DELETE FROM $table_name WHERE created_at < DATE_SUB(NOW(), INTERVAL %d DAY)",
            $retention_days
        ));

        echo sprintf(
            '<div class="notice notice-success"><p>%s</p></div>',
            sprintf(__('Cleaned up %d old history records.', 'listeo-ical-admin'), $deleted)
        );
    }

    /**
     * Test import functionality
     */
    private function test_import_functionality() {
        $test_url = 'https://calendar.google.com/calendar/ical/en.christian%23holiday%40group.v.calendar.google.com/public/basic.ics';

        try {
            if (class_exists('Listeo_Core_iCal_Reader')) {
                $ical = new Listeo_Core_iCal_Reader(array($test_url));

                if ($ical->has_events()) {
                    echo sprintf(
                        '<div class="notice notice-success"><p>%s</p></div>',
                        sprintf(__('Test successful: Found %d events in test calendar.', 'listeo-ical-admin'), $ical->event_count)
                    );
                } else {
                    echo '<div class="notice notice-warning"><p>' . __('Test completed but no events found.', 'listeo-ical-admin') . '</p></div>';
                }
            } else {
                echo '<div class="notice notice-error"><p>' . __('Test failed: iCal Reader class not found.', 'listeo-ical-admin') . '</p></div>';
            }
        } catch (Exception $e) {
            echo sprintf(
                '<div class="notice notice-error"><p>%s</p></div>',
                sprintf(__('Test failed: %s', 'listeo-ical-admin'), $e->getMessage())
            );
        }
    }
}