<?php
/**
 * Plugin Name: Listeo iCal Admin Manager
 * Plugin URI: https://purethemes.net/
 * Description: Advanced admin management for Listeo iCal imports/exports. Provides comprehensive oversight, bulk operations, and import history for all listings.
 * Version: 1.0.1
 * Author: PureThemes
 * Author URI: https://purethemes.net/
 * Requires at least: 5.0
 * Tested up to: 6.4
 * Requires PHP: 7.4
 * Text Domain: listeo-ical-admin
 * Domain Path: /languages
 * Network: false
 * License: GPL v2 or later
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('LISTEO_ICAL_ADMIN_VERSION', '1.0.0');
define('LISTEO_ICAL_ADMIN_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('LISTEO_ICAL_ADMIN_PLUGIN_URL', plugin_dir_url(__FILE__));
define('LISTEO_ICAL_ADMIN_BASENAME', plugin_basename(__FILE__));

/**
 * Main plugin class
 */
class Listeo_iCal_Admin {

    private static $instance = null;

    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action('plugins_loaded', array($this, 'init'));
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }

    /**
     * Initialize plugin
     */
    public function init() {
        // Check if Listeo Core is active
        if (!class_exists('Listeo_Core_iCal')) {
            add_action('admin_notices', array($this, 'missing_listeo_notice'));
            return;
        }

        // Load text domain
        load_plugin_textdomain('listeo-ical-admin', false, dirname(LISTEO_ICAL_ADMIN_BASENAME) . '/languages');

        // Initialize components
        $this->includes();
        $this->init_hooks();
    }

    /**
     * Include required files
     */
    private function includes() {
        require_once LISTEO_ICAL_ADMIN_PLUGIN_DIR . 'includes/class-admin-menu.php';
        require_once LISTEO_ICAL_ADMIN_PLUGIN_DIR . 'includes/class-listings-table.php';
        require_once LISTEO_ICAL_ADMIN_PLUGIN_DIR . 'includes/class-import-history.php';
        require_once LISTEO_ICAL_ADMIN_PLUGIN_DIR . 'includes/class-bulk-operations.php';
        require_once LISTEO_ICAL_ADMIN_PLUGIN_DIR . 'includes/class-settings.php';
        require_once LISTEO_ICAL_ADMIN_PLUGIN_DIR . 'includes/class-ajax-handlers.php';
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        add_action('admin_enqueue_scripts', array($this, 'admin_scripts'));
        add_filter('set-screen-option', array($this, 'save_screen_options'), 10, 3);

        // Initialize admin components
        new Listeo_iCal_Admin_Menu();
        new Listeo_iCal_Admin_Ajax_Handlers();
        new Listeo_iCal_Admin_Settings(); // Initialize settings to register hooks
    }

    /**
     * Save screen options
     */
    public function save_screen_options($status, $option, $value) {
        if ('listings_per_page' === $option) {
            return $value;
        }
        return $status;
    }

    /**
     * Enqueue admin scripts and styles
     */
    public function admin_scripts($hook) {
        // Only load on our admin pages
        if (strpos($hook, 'listeo-ical') === false) {
            return;
        }

        wp_enqueue_style(
            'listeo-ical-admin-style',
            LISTEO_ICAL_ADMIN_PLUGIN_URL . 'assets/admin.css',
            array(),
            LISTEO_ICAL_ADMIN_VERSION
        );

        wp_enqueue_script(
            'listeo-ical-admin-script',
            LISTEO_ICAL_ADMIN_PLUGIN_URL . 'assets/admin.js',
            array('jquery'),
            LISTEO_ICAL_ADMIN_VERSION,
            true
        );

        wp_localize_script('listeo-ical-admin-script', 'listeoIcalAdmin', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('listeo_ical_admin_nonce'),
            'strings' => array(
                'confirmBulkUpdate' => __('Are you sure you want to force update selected iCal imports?', 'listeo-ical-admin'),
                'confirmBulkDelete' => __('Are you sure you want to delete selected iCal imports?', 'listeo-ical-admin'),
                'processing' => __('Processing...', 'listeo-ical-admin'),
                'completed' => __('Completed', 'listeo-ical-admin'),
                'error' => __('Error occurred', 'listeo-ical-admin')
            )
        ));
    }

    /**
     * Plugin activation
     */
    public function activate() {
        // Create necessary database tables
        $this->create_tables();

        // Set default options
        add_option('listeo_ical_admin_version', LISTEO_ICAL_ADMIN_VERSION);
        add_option('listeo_ical_batch_size', 10);
        add_option('listeo_ical_log_retention_days', 30);
    }

    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Clean up cron jobs if any
        wp_clear_scheduled_hook('listeo_ical_admin_cleanup');
    }

    /**
     * Create database tables
     */
    private function create_tables() {
        global $wpdb;

        $charset_collate = $wpdb->get_charset_collate();

        // Import history table
        $table_name = $wpdb->prefix . 'listeo_ical_import_history';
        $sql = "CREATE TABLE $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            listing_id bigint(20) NOT NULL,
            ical_name varchar(255) NOT NULL,
            ical_url text NOT NULL,
            import_type varchar(50) NOT NULL DEFAULT 'manual',
            status varchar(50) NOT NULL,
            events_imported int(11) NOT NULL DEFAULT 0,
            events_skipped int(11) NOT NULL DEFAULT 0,
            error_message text DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY listing_id (listing_id),
            KEY status (status),
            KEY created_at (created_at)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    /**
     * Show notice if Listeo Core is missing
     */
    public function missing_listeo_notice() {
        echo '<div class="notice notice-error"><p>';
        echo __('Listeo iCal Admin Manager requires Listeo Core plugin to be active.', 'listeo-ical-admin');
        echo '</p></div>';
    }
}

// Initialize the plugin
Listeo_iCal_Admin::get_instance();