<?php
if ( ! defined( 'ABSPATH' ) ) {
exit;
}

/**
* Listeo_Core_Paid_Listings_Subscriptions
*/
class Listeo_Core_Paid_Listings_Subscriptions {

	/** @var object Class Instance */
	private static $instance;

	/** @var array Cache for subscription check results to prevent repeated queries */
	private static $subscription_check_cache = array();

	/**
	 * Get the class instance
	 *
	 * @return static
	 */
	public static function get_instance() {
		return null === self::$instance ? ( self::$instance = new self ) : self::$instance;
	}

	/**
	 * Constructor
	 */
	public function __construct() {
		
		if ( class_exists( 'WC_Subscriptions_Synchroniser' ) && method_exists( 'WC_Subscriptions_Synchroniser', 'save_subscription_meta' ) ) {
			add_action( 'woocommerce_process_product_meta_listing_package_subscription', 'WC_Subscriptions_Synchroniser::save_subscription_meta', 10 );
		}

		add_action( 'added_post_meta', array( $this, 'updated_post_meta' ), 20, 4 );
		add_action( 'updated_post_meta', array( $this, 'updated_post_meta' ), 20, 4 );
		add_filter( 'woocommerce_is_subscription', array( $this, 'woocommerce_is_subscription' ), 10, 2 );
		add_action( 'wp_trash_post', array( $this, 'wp_trash_post' ) );
		add_action( 'untrash_post', array( $this, 'untrash_post' ) );
		add_action( 'publish_to_expired', array( $this, 'check_expired_listing' ) );

		// Register our product type as switchable
		add_filter( 'wcs_is_product_switchable_type', array( $this, 'is_product_switchable_type' ), 10, 2 );

		// Subscription is paused
		add_action( 'woocommerce_subscription_status_on-hold', array( $this, 'subscription_paused' ) ); // When a subscription is put on hold

		// Subscription is ended
		add_action( 'woocommerce_subscription_status_expired', array( $this, 'subscription_ended' ) ); // When a subscription expires
		add_action( 'woocommerce_subscription_status_cancelled', array( $this, 'subscription_ended' ) ); // When the subscription status changes to cancelled

		// Subscription starts
		add_action( 'woocommerce_subscription_status_active', array( $this, 'subscription_activated' ) ); // When the subscription status changes to active

		// On renewal
		add_action( 'woocommerce_subscription_renewal_payment_complete', array( $this, 'subscription_renewed' ) ); // When the subscription is renewed

		// Subscription is switched
		add_action( 'woocommerce_subscriptions_switched_item', array( $this, 'subscription_switched' ), 10, 3 ); // When the subscription is switched and a new subscription is created
		add_action( 'woocommerce_subscription_item_switched', array( $this, 'subscription_item_switched' ), 10, 4 ); // When the subscription is switched and only the item is changed
	}

	/**
	 * Prevent listings linked to subscriptions from expiring.
	 *
	 * @param int         $meta_id
	 * @param int|WP_Post $object_id
	 * @param string      $meta_key
	 * @param mixed       $meta_value
	 */
	public function updated_post_meta( $meta_id, $object_id, $meta_key, $meta_value ) {
		// Only process _listing_expires meta key
		if ( '_listing_expires' !== $meta_key ) {
			return;
		}

		// CRITICAL: Only process actual listing posts, not orders/subscriptions/etc
		if ( 'listing' !== get_post_type( $object_id ) ) {
			return;
		}

		// Skip if already set to sentinel value or empty
		if ( '' === $meta_value || ' ' === $meta_value ) {
			return;
		}

		// CRITICAL FIX: Add static flag to prevent re-entry during our own update
		static $is_updating = array();
		if ( isset( $is_updating[ $object_id ] ) && $is_updating[ $object_id ] === true ) {
			return;
		}

		// Check cache first to avoid repeated queries
		if ( ! isset( self::$subscription_check_cache[ $object_id ] ) ) {
			self::$subscription_check_cache[ $object_id ] = $this->get_listing_subscription_order_id( $object_id );
		}

		if ( false !== self::$subscription_check_cache[ $object_id ] ) {
			// Set flag to prevent re-entry
			$is_updating[ $object_id ] = true;

			// Temporarily remove hooks to prevent infinite loop
			remove_action( 'added_post_meta', array( $this, 'updated_post_meta' ), 20 );
			remove_action( 'updated_post_meta', array( $this, 'updated_post_meta' ), 20 );

			update_post_meta( $object_id, '_listing_expires', ' ' ); // Never expire automatically

			// Re-add hooks
			add_action( 'added_post_meta', array( $this, 'updated_post_meta' ), 20, 4 );
			add_action( 'updated_post_meta', array( $this, 'updated_post_meta' ), 20, 4 );

			// Clear flag
			unset( $is_updating[ $object_id ] );
		}
	}


	/**
	 * Clear the subscription check cache for a specific listing or all listings
	 *
	 * @param int|null $listing_id Listing ID to clear cache for, or null to clear all
	 */
	private function clear_subscription_cache( $listing_id = null ) {
		if ( null === $listing_id ) {
			self::$subscription_check_cache = array();
		} else {
			unset( self::$subscription_check_cache[ $listing_id ] );
		}
	}

	/**
	 * If the job listing is tied to a subscription of type 'listing', return the order ID.
	 *
	 * @param int $job_id
	 *
	 * @return bool|int False if not found or is not the correct subscription type.
	 */
	private function get_listing_subscription_order_id( $listing_id ) {
		// Defensive check: ensure valid listing ID
		if ( empty( $listing_id ) || ! is_numeric( $listing_id ) ) {
			return false;
		}

		if ( 'listing' === get_post_type( $listing_id ) ) {
			$user_package_id = get_post_meta( $listing_id, '_user_package_id', true );

			// Early return if no package ID
			if ( empty( $user_package_id ) ) {
				return false;
			}

			$user_package    = listeo_core_get_user_package( $user_package_id );

			// Early return if package doesn't exist or has no package
			if ( ! $user_package || ! $user_package->has_package() ) {
				return false;
			}

			$package_id      = get_post_meta( $listing_id, '_package_id', true );

			// Early return if no product ID
			if ( empty( $package_id ) ) {
				return false;
			}

			$package         = wc_get_product( $package_id );

			// Check if it's a subscription package
			if ( $package && ( $package instanceof WC_Product_Listing_Package_Subscription) ) {
				return $user_package->get_order_id();
			}
		}
		return false;
	}

	/**
	 * get subscription type for package by ID
	 *
	 * @param  int $product_id
	 * @return string
	 */
	public function get_package_subscription_type( $product_id ) {
		$subscription_type = get_post_meta( $product_id, '_package_subscription_type', true );
		return empty( $subscription_type ) ? 'package' : $subscription_type;
	}

	/**
	 * Is this a subscription product?
	 *
	 * @param bool $is_subscription
	 * @param int  $product_id
	 * @return bool
	 */
	public function woocommerce_is_subscription( $is_subscription, $product_id ) {
		$product = wc_get_product( $product_id );
		if ( $product && $product->is_type( 'listing_package_subscription') ) {
			$is_subscription = true;
		}
		return $is_subscription;
	}

	/**
	 * If a listing is expired, the pack may need it's listing count changing
	 *
	 * @param WP_Post $post
	 */
	public function check_expired_listing( $post ) {
		global $wpdb;

		if ( 'listing' === $post->post_type ) {
			$package_product_id = get_post_meta( $post->ID, '_package_id', true );
			$package_id         = get_post_meta( $post->ID, '_user_package_id', true );
			$package_product    = get_post( $package_product_id );

			if ( $package_product_id ) {
				$subscription_type = $this->get_package_subscription_type( $package_product_id );
				
				//if($subscription_type == 'listing_package_subscription') {

					$new_count = $wpdb->get_var( $wpdb->prepare( "SELECT package_count FROM {$wpdb->prefix}listeo_core_user_packages WHERE id = %d;", $package_id ) );
					$new_count --;

					$wpdb->update(
						"{$wpdb->prefix}listeo_core_user_packages",
						array(
							'package_count'  => max( 0, $new_count ),
						),
						array(
							'id' => $package_id,
						)
					);

					// Remove package meta after adjustment
					delete_post_meta( $post->ID, '_package_id' );
					delete_post_meta( $post->ID, '_user_package_id' );
				//}
				
				
				
			}
		}
	}

	/**
	 * If a listing gets trashed/deleted, the pack may need it's listing count changing
	 *
	 * @param int $id
	 */
	public function wp_trash_post( $id ) {
		global $wpdb;

		if ( $id > 0 ) {
			$post_type = get_post_type( $id );

			if ( 'listing' === $post_type  ) {
				$package_product_id = get_post_meta( $id, '_package_id', true );
				$package_id         = get_post_meta( $id, '_user_package_id', true );
				$package_product    = get_post( $package_product_id );

				if ( $package_product_id ) {
					
					
					$new_count = $wpdb->get_var( $wpdb->prepare( "SELECT package_count FROM {$wpdb->prefix}listeo_core_user_packages WHERE id = %d;", $package_id ) );
					$new_count --;

					$wpdb->update(
						"{$wpdb->prefix}listeo_core_user_packages",
						array(
							'package_count'  => max( 0, $new_count ),
						),
						array(
							'id' => $package_id,
						)
					);
					
				}
			}
		}
	}

/**
 * If a listing gets restored, the pack may need it's listing count changing
 *
 * @param int $id
 */
public function untrash_post( $id ) {
	global $wpdb;

	if ( $id > 0 ) {
		$post_type = get_post_type( $id );

		if ( 'listing' === $post_type) {
			$package_product_id = get_post_meta( $id, '_package_id', true );
			$package_id         = get_post_meta( $id, '_user_package_id', true );
			$package_product    = get_post( $package_product_id );

			if ( $package_product_id ) {
				
				
				$package  = $wpdb->get_row( $wpdb->prepare( "SELECT package_count, package_limit FROM {$wpdb->prefix}listeo_core_user_packages WHERE id = %d;", $package_id ) );
				$new_count = $package->package_count + 1;

				$wpdb->update(
					"{$wpdb->prefix}listeo_core_user_packages",
					array(
						'package_count'  => min( $package->package_limit, $new_count ),
					),
					array(
						'id' => $package_id,
					)
				);
				
			}
		}
	}
}

/**
 * Subscription is on-hold for payment. Suspend package and listings.
 *
 * @param WC_Subscription $subscription
 */
public function subscription_paused( $subscription ) {
	$this->subscription_ended( $subscription, true );
}

/**
 * Subscription has expired - cancel job packs
 *
 * @param WC_Subscription $subscription
 * @param bool            $paused
 */
public function subscription_ended( $subscription, $paused = false ) {
	global $wpdb;

	// Clear cache since subscription status is changing
	$this->clear_subscription_cache();

	foreach ( $subscription->get_items() as $item ) {
		

		/**
		 * @var WC_Order $parent
		 */
		
		$parent            = $subscription->get_parent();
		$parent_id         = ! empty( $parent ) ?  $parent->get_id() : null;
		$legacy_id         = isset( $parent_id ) ? $parent_id : $subscription->get_id();
		$user_package      = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$wpdb->prefix}listeo_core_user_packages WHERE order_id IN ( %d, %d ) AND product_id = %d;", $subscription->get_id(), $legacy_id, $item['product_id'] ) );

		if ( $user_package ) {
			// Delete the package
			$wpdb->delete(
				"{$wpdb->prefix}listeo_core_user_packages",
				array(
					'id' => $user_package->id,
				)
			);

			// Expire listings posted with package
			
			$listing_ids = listeo_core_get_listings_for_package( $user_package->id );
				
			foreach ( $listing_ids as $listing_id ) {

					if ( $paused ) {
						// Record the current post status in case subscription is resumed
						update_post_meta( $listing_id, '_post_status_before_package_pause', get_post_status( $listing_id ) );
					} else {
						delete_post_meta( $listing_id, '_post_status_before_package_pause' );
					}
					$listing = array(
						'ID' => $listing_id,
						'post_status' => 'expired',
					);
					
					wp_update_post( $listing );

					// Make a record of the subscription ID in case of re-activation
					update_post_meta( $listing_id, '_expired_subscription_id', $subscription->get_id() );
				}
			
		}
	}// End foreach().

	delete_post_meta(  $subscription->get_id(), 'listeo_core_subscription_packages_processed' );
}

/**
 * Subscription activated
 *
 * @param WC_Subscription $subscription
 */
public function subscription_activated( $subscription ) {
	global $wpdb;

	if ( get_post_meta( $subscription->get_id(), 'listeo_core_subscription_packages_processed', true ) ) {
		return;
	}

	// Clear cache since we're activating subscriptions
	$this->clear_subscription_cache();

	// Remove any old packages for this subscription
	$parent            = $subscription->get_parent();
	$parent_id         = ! empty( $parent ) ?  $parent->get_id() : null;
	$legacy_id         = isset( $parent_id ) ? $parent_id : $subscription->get_id();
	$wpdb->delete( "{$wpdb->prefix}listeo_core_user_packages", array(
		'order_id' => $legacy_id,
	) );
	$wpdb->delete( "{$wpdb->prefix}listeo_core_user_packages", array(
		'order_id' =>  $subscription->get_id(),
	) );

	foreach ( $subscription->get_items() as $item ) {
		$product           = wc_get_product( $item['product_id'] );
		$subscription_type = $this->get_package_subscription_type( $item['product_id'] );


		// Give user packages for this subscription
		if ( $product->is_type( array( 'listing_package_subscription' ) ) && $subscription->get_user_id() && ! isset( $item['switched_subscription_item_id'] ) ) {
			
			// Give packages to user
			for ( $i = 0; $i < $item['qty']; $i ++ ) {
				$user_package_id = listeo_core_give_user_package( $subscription->get_user_id(), $product->get_id(), $subscription->get_id() );
				
			}

			/**
			 * If the subscription is associated with listings, see if any
			 * already match this ID and approve them (useful on
			 * re-activation of a sub).
			 */
			
			$listing_ids = (array) $wpdb->get_col( $wpdb->prepare( "SELECT post_id FROM $wpdb->postmeta WHERE meta_key=%s AND meta_value=%s", '_expired_subscription_id', $subscription->get_id() ) );
			

			$listing_ids[] = isset( $item['listing_id'] ) ? $item['listing_id'] : '';
			$listing_ids   = array_unique( array_filter( array_map( 'absint', $listing_ids ) ) );

			foreach ( $listing_ids as $listing_id ) {
				if ( in_array( get_post_status( $listing_id ), array( 'pending_payment', 'expired','publish' ) ) ) {
					listeo_core_approve_listing_with_package( $listing_id, $subscription->get_user_id(), $user_package_id );
					delete_post_meta( $listing_id, '_expired_subscription_id' );
				}
			}
		}
	}

	update_post_meta( $subscription->get_id(), 'listeo_core_subscription_packages_processed', true );
}

/**
 * Subscription renewed - renew the job pack
 *
 * @param WC_Subscription $subscription
 */
public function subscription_renewed( $subscription ) {
	global $wpdb;

	// Clear cache since subscription status is changing
	$this->clear_subscription_cache();

	foreach ( $subscription->get_items() as $item ) {
		$product           = wc_get_product( $item['product_id'] );
		$subscription_type = $this->get_package_subscription_type( $item['product_id'] );
		$parent            = $subscription->get_parent();
		$parent_id         = ! empty( $parent ) ? $parent->get_id() : null;
		$legacy_id         = isset( $parent_id ) ? $parent_id : $subscription->get_id();

		// Renew packages which refresh every term

		
			// Otherwise the listings stay active, but we can ensure they are synced in terms of featured status etc
		if ( $user_package_ids = $wpdb->get_col( $wpdb->prepare( "SELECT id FROM {$wpdb->prefix}listeo_core_user_packages WHERE order_id IN ( %d, %d ) AND product_id = %d;", $subscription->get_id(), $legacy_id, $item['product_id'] ) ) ) {
			foreach ( $user_package_ids as $user_package_id ) {
				$package = listeo_core_get_user_package( $user_package_id );

				if ( $listing_ids = listeo_core_get_listings_for_package( $user_package_id ) ) {
					foreach ( $listing_ids as $listing_id ) {

						// Featured or not
						update_post_meta( $listing_id, '_featured', $package->is_featured() ? 1 : 0 );
					}
				}
			}
		}
		
	}// End foreach().
}

/**
 * When switching a subscription we need to update old listings.
 *
 * No need to give the user a new package; that is still handled by the orders class.
 *
 * @param WC_Order        $order
 * @param WC_Subscription $subscription
 * @param int             $new_order_item_id
 * @param int             $old_order_item_id
 */
public function subscription_item_switched( $order, $subscription, $new_order_item_id, $old_order_item_id ) {
	global $wpdb;

	$new_order_item = WC_Subscriptions_Order::get_item_by_id( $new_order_item_id );
	$old_order_item = WC_Subscriptions_Order::get_item_by_id( $old_order_item_id );

	$new_subscription = (object) array(
		'id'           => $subscription->get_id(),
		'subscription' => $subscription,
		'product_id'   => $new_order_item['product_id'],
		'product'      => wc_get_product( $new_order_item['product_id'] ),
		'type'         => $this->get_package_subscription_type( $new_order_item['product_id'] ),
	);

	$old_subscription = (object) array(
		'id'           => $subscription->get_id(),
		'subscription' => $subscription,
		'product_id'   => $old_order_item['product_id'],
		'product'      => wc_get_product( $old_order_item['product_id'] ),
		'type'         => $this->get_package_subscription_type( $old_order_item['product_id'] ),
	);

	$this->switch_package( $subscription->get_user_id(), $new_subscription, $old_subscription );
}

/**
 * When switching a subscription we need to update old listings.
 *
 * No need to give the user a new package; that is still handled by the orders class.
 *
 * @param WC_Subscription $subscription
 * @param array           $new_order_item
 * @param array           $old_order_item
 */
public function subscription_switched( $subscription, $new_order_item, $old_order_item ) {
	global $wpdb;
		error_log('Subscription switching triggered: ' . json_encode(func_get_args()));
	$new_subscription = (object) array(
		'id'         => $subscription->get_id(),
		'product_id' => $new_order_item['product_id'],
		'product'    => wc_get_product( $new_order_item['product_id'] ),
		'type'       => $this->get_package_subscription_type( $new_order_item['product_id'] ),
	);

	$old_subscription = (object) array(
		'id'         => $wpdb->get_var( $wpdb->prepare( "SELECT order_id FROM {$wpdb->prefix}woocommerce_order_items WHERE order_item_id = %d ", $new_order_item['switched_subscription_item_id'] ) ),
		'product_id' => $old_order_item['product_id'],
		'product'    => wc_get_product( $old_order_item['product_id'] ),
		'type'       => $this->get_package_subscription_type( $old_order_item['product_id'] ),
	);

	$this->switch_package( $subscription->get_user_id(), $new_subscription, $old_subscription );
}

/**
 * Handle Switch Event
 *
 * @param int      $user_id
 * @param stdClass $new_subscription
 * @param stdClass $old_subscription
 */
public function switch_package( $user_id, $new_subscription, $old_subscription ) {
	global $wpdb;
	error_log('Switching package triggered: ' . json_encode(func_get_args()));
	
	// Get the user package
	/**
	 * @var null|WC_Subscription $parent
	 */
	$parent            = isset( $old_subscription->subscription ) ? $old_subscription->subscription->get_parent() : null;
	$parent_id         = ! empty( $parent ) ? $parent->get_id() : null;
	$legacy_id         = isset( $parent_id ) ? $parent_id : $old_subscription->id;
	$user_package      = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$wpdb->prefix}listeo_core_user_packages WHERE order_id IN ( %d, %d ) AND product_id = %d;", $old_subscription->id, $legacy_id, $old_subscription->product_id ) );

	if ( $user_package ) {
		error_log('User package found: ' . json_encode($user_package));
		// If invalid, abort
		if ( ! $new_subscription->product->is_type( array( 'listing_package_subscription') ) ) {
			return false;
		}

		// Give new package to user
		$switching_to_package_id = listeo_core_give_user_package( $user_id, $new_subscription->product_id, $new_subscription->id );

		// Upgrade?
		$is_upgrade = ( 0 === $new_subscription->product->get_limit() || $new_subscription->product->get_limit() >= $user_package->package_count );
error_log('Is upgrade: ' . $is_upgrade);
		// Delete the old package
		$wpdb->delete( "{$wpdb->prefix}listeo_core_user_packages", array(
			'id' => $user_package->id,
		) );

		$does_new_subscription_feature_listings = false;
		if ( $new_subscription->product instanceof WC_Product_Listing_Package_Subscription ) {
			$does_new_subscription_feature_listings = $new_subscription->product->is_listing_featured();
		}

		// Update old listings
		
		$listing_ids = listeo_core_get_listings_for_package( $user_package->id );

		foreach ( $listing_ids as $listing_id ) {
			// If we are not upgrading, expire the old listing
			if ( ! $is_upgrade ) {
				$listing = array(
					'ID' => $listing_id,
					'post_status' => 'expired',
				);
				wp_update_post( $listing );
			} else {
				/** This filter is documented in includes/package-functions.php */
				
				// Change the user package ID and package ID
				update_post_meta( $listing_id, '_user_package_id', $switching_to_package_id );
				update_post_meta( $listing_id, '_package_id', $new_subscription->product_id );
			}

			// Featured or not
			update_post_meta( $listing_id, '_featured', $does_new_subscription_feature_listings ? 1 : 0 );

			// Fire action
			do_action( 'wc_paid_listings_switched_subscription', $listing_id, $user_package );
		}

	}// End if().
}

	/**
	 * Mark listing package subscriptions as switchable product type
	 *
	 * @param bool $is_switchable
	 * @param string $product_type
	 * @return bool
	 */
	public function is_product_switchable_type( $is_switchable, $product_type ) {
		if ( 'listing_package_subscription' === $product_type ) {
			return true;
		}
		return $is_switchable;
	}
}

Listeo_Core_Paid_Listings_Subscriptions::get_instance();