<?php
/**
 * Admin Interface Class
 * 
 * Handles WordPress admin settings and management interface
 * 
 * @package Listeo_AI_Search
 * @since 1.0.5
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Listeo_AI_Search_Admin_Interface {
    
    /**
     * Constructor
     */
    public function __construct() {
        // Admin menu
        add_action('admin_menu', array($this, 'admin_menu'));
        
        // Settings
        add_action('admin_init', array($this, 'admin_init'));
        
        // Enqueue admin scripts
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        
        // AJAX handlers for settings
        add_action('wp_ajax_listeo_ai_save_settings', array($this, 'ajax_save_settings'));
        add_action('wp_ajax_listeo_ai_test_api_key', array($this, 'ajax_test_api_key'));
        add_action('wp_ajax_listeo_ai_clear_cache', array($this, 'ajax_clear_cache'));
        add_action('wp_ajax_listeo_ai_regenerate_embedding', array($this, 'ajax_regenerate_embedding'));
        
        // AJAX handler for manual batch processing
        add_action('wp_ajax_listeo_ai_manual_batch_action', array($this, 'ajax_manual_batch_action'));
        
        // Show debug mode notice if enabled
        if (get_option('listeo_ai_search_debug_mode', false)) {
            add_action('admin_notices', array($this, 'show_debug_mode_notice'));
        }
    }
    
    /**
     * Add admin menu
     */
    public function admin_menu() {
        add_menu_page(
            __('Listeo AI Search', 'listeo-ai-search'),           // Page title
            __('Listeo AI Search', 'listeo-ai-search'),                 // Menu title
            'manage_options',                                     // Capability
            'listeo-ai-search',                                   // Menu slug
            array($this, 'admin_page'),                          // Callback function
            'dashicons-search',                                   // Search icon
            30                                                    // Position (after Comments)
        );
    }

    /**
     * Initialize admin settings
     */
    public function admin_init() {
        register_setting('listeo_ai_search_settings', 'listeo_ai_search_api_key');
        register_setting('listeo_ai_search_settings', 'listeo_ai_search_ai_location_filtering_enabled');
        register_setting('listeo_ai_search_settings', 'listeo_ai_search_debug_mode');
        register_setting('listeo_ai_search_settings', 'listeo_ai_search_disable_cache');
        register_setting('listeo_ai_search_settings', 'listeo_ai_search_query_expansion');
        register_setting('listeo_ai_search_settings', 'listeo_ai_search_min_match_percentage');
        register_setting('listeo_ai_search_settings', 'listeo_ai_search_best_match_threshold');
        register_setting('listeo_ai_search_settings', 'listeo_ai_search_max_results');
        register_setting('listeo_ai_search_settings', 'listeo_ai_search_enable_analytics');
        register_setting('listeo_ai_search_settings', 'listeo_ai_search_suggestions_enabled');
        register_setting('listeo_ai_search_settings', 'listeo_ai_search_suggestions_source');
        register_setting('listeo_ai_search_settings', 'listeo_ai_search_custom_suggestions');
        register_setting('listeo_ai_search_settings', 'listeo_ai_search_batch_size');
        register_setting('listeo_ai_search_settings', 'listeo_ai_search_embedding_delay');
        register_setting('listeo_ai_search_settings', 'listeo_ai_search_rate_limit_per_hour');
    }

    /**
     * Add hidden fields to preserve other settings when submitting a form
     * 
     * @param array $exclude_fields Array of field names to exclude from hidden fields
     */
    private function add_hidden_fields_except($exclude_fields = array()) {
        $all_settings = array(
            'listeo_ai_search_api_key',
            'listeo_ai_search_ai_location_filtering_enabled',
            'listeo_ai_search_debug_mode',
            'listeo_ai_search_disable_cache',
            'listeo_ai_search_query_expansion',
            'listeo_ai_search_min_match_percentage',
            'listeo_ai_search_best_match_threshold',
            'listeo_ai_search_max_results',
            'listeo_ai_search_enable_analytics',
            'listeo_ai_search_suggestions_enabled',
            'listeo_ai_search_suggestions_source',
            'listeo_ai_search_custom_suggestions',
            'listeo_ai_search_batch_size',
            'listeo_ai_search_embedding_delay',
            'listeo_ai_search_rate_limit_per_hour'
        );

        foreach ($all_settings as $setting) {
            if (!in_array($setting, $exclude_fields)) {
                $value = get_option($setting);
                if ($value !== false && $value !== '') {
                    // Handle different input types
                    if (is_array($value)) {
                        foreach ($value as $key => $sub_value) {
                            echo '<input type="hidden" name="' . esc_attr($setting) . '[' . esc_attr($key) . ']" value="' . esc_attr($sub_value) . '">';
                        }
                    } else {
                        echo '<input type="hidden" name="' . esc_attr($setting) . '" value="' . esc_attr($value) . '">';
                    }
                }
            }
        }
    }

    /**
     * AJAX handler for saving settings
     */
    public function ajax_save_settings() {
        // Verify nonce
        if (!check_ajax_referer('listeo_ai_search_nonce', 'nonce', false)) {
            wp_send_json_error(array('message' => __('Security check failed.', 'listeo-ai-search')));
            return;
        }

        // Check user permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'listeo-ai-search')));
            return;
        }

        // Get all registered settings
        $all_settings = array(
            'listeo_ai_search_api_key',
            'listeo_ai_search_enabled_types',
            'listeo_ai_search_ai_location_filtering_enabled',
            'listeo_ai_search_debug_mode',
            'listeo_ai_search_disable_cache',
            'listeo_ai_search_query_expansion',
            'listeo_ai_search_min_match_percentage',
            'listeo_ai_search_best_match_threshold',
            'listeo_ai_search_max_results',
            'listeo_ai_search_enable_analytics',
            'listeo_ai_search_suggestions_enabled',
            'listeo_ai_search_suggestions_source',
            'listeo_ai_search_custom_suggestions',
            'listeo_ai_search_batch_size',
            'listeo_ai_search_embedding_delay',
            'listeo_ai_search_rate_limit_per_hour'
        );

        $updated_settings = array();
        
        // Process each setting
        foreach ($all_settings as $setting) {
            if (isset($_POST[$setting])) {
                $value = $_POST[$setting];
                
                // Sanitize different types of values
                if (is_array($value)) {
                    $value = array_map('sanitize_text_field', $value);
                } else {
                    // Handle checkboxes and text fields
                    switch ($setting) {
                        case 'listeo_ai_search_api_key':
                            $value = sanitize_text_field($value);
                            break;
                        case 'listeo_ai_search_custom_suggestions':
                            $value = sanitize_textarea_field($value);
                            break;
                        case 'listeo_ai_search_min_match_percentage':
                        case 'listeo_ai_search_best_match_threshold':
                        case 'listeo_ai_search_max_results':
                        case 'listeo_ai_search_batch_size':
                        case 'listeo_ai_search_embedding_delay':
                        case 'listeo_ai_search_rate_limit_per_hour':
                            $value = intval($value);
                            // Validate rate limit bounds
                            if ($setting === 'listeo_ai_search_rate_limit_per_hour') {
                                $value = max(10, min(10000, $value)); // Enforce min 10, max 10000
                            }
                            break;
                        default:
                            // For checkboxes and other fields
                            if ($value === '1' || $value === 'true') {
                                $value = 1;
                            } else {
                                $value = sanitize_text_field($value);
                            }
                            break;
                    }
                }
                
                // Update the option
                update_option($setting, $value);
                $updated_settings[$setting] = $value;
            } else {
                // Handle unchecked checkboxes (they won't be in $_POST)
                $checkbox_fields = array(
                    'listeo_ai_search_ai_location_filtering_enabled',
                    'listeo_ai_search_debug_mode',
                    'listeo_ai_search_disable_cache',
                    'listeo_ai_search_query_expansion',
                    'listeo_ai_search_enable_analytics',
                    'listeo_ai_search_suggestions_enabled'
                );
                
                if (in_array($setting, $checkbox_fields)) {
                    update_option($setting, 0);
                    $updated_settings[$setting] = 0;
                }
            }
        }

        wp_send_json_success(array(
            'message' => __('Settings saved successfully!', 'listeo-ai-search'),
            'updated_settings' => $updated_settings
        ));
    }
    
    /**
     * AJAX handler for testing API key
     */
    public function ajax_test_api_key() {
        // Verify nonce
        if (!check_ajax_referer('listeo_ai_search_nonce', 'nonce', false)) {
            wp_send_json_error(array('message' => __('Security check failed.', 'listeo-ai-search')));
            return;
        }

        // Check user permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'listeo-ai-search')));
            return;
        }

        // Get API key from POST data or current settings
        $api_key = sanitize_text_field($_POST['api_key'] ?? get_option('listeo_ai_search_api_key', ''));
        
        if (empty($api_key)) {
            wp_send_json_error(array('message' => __('Please enter an API key first.', 'listeo-ai-search')));
            return;
        }

        try {
            // Test the API key by making a simple request
            $response = wp_remote_get('https://api.openai.com/v1/models', array(
                'headers' => array(
                    'Authorization' => 'Bearer ' . $api_key,
                    'Content-Type' => 'application/json',
                ),
                'timeout' => 15,
            ));

            if (is_wp_error($response)) {
                wp_send_json_error(array('message' => __('Connection failed: ', 'listeo-ai-search') . $response->get_error_message()));
                return;
            }

            $response_code = wp_remote_retrieve_response_code($response);
            $response_body = wp_remote_retrieve_body($response);

            if ($response_code === 200) {
                wp_send_json_success(array(
                    'message' => __('✅ API key is valid!', 'listeo-ai-search')
                ));
            } elseif ($response_code === 401) {
                wp_send_json_error(array('message' => __('❌ Invalid API key. Please check your key and try again.', 'listeo-ai-search')));
            } elseif ($response_code === 429) {
                wp_send_json_error(array('message' => __('⚠️ API key valid but rate limit exceeded. Try again in a moment.', 'listeo-ai-search')));
            } else {
                $error_body = json_decode($response_body, true);
                $error_message = isset($error_body['error']['message']) ? $error_body['error']['message'] : __('Unknown error', 'listeo-ai-search');
                wp_send_json_error(array('message' => sprintf(__('❌ API Error (%d): %s', 'listeo-ai-search'), $response_code, $error_message)));
            }

        } catch (Exception $e) {
            wp_send_json_error(array('message' => __('❌ Test failed: ', 'listeo-ai-search') . $e->getMessage()));
        }
    }
    
    /**
     * AJAX handler for clearing cache
     */
    public function ajax_clear_cache() {
        // Verify nonce
        if (!check_ajax_referer('listeo_ai_search_nonce', 'nonce', false)) {
            wp_send_json_error(array('message' => __('Security check failed.', 'listeo-ai-search')));
            return;
        }

        // Check user permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'listeo-ai-search')));
            return;
        }

        try {
            $cleared_count = 0;
            $cleared_types = array();

            // Clear search result cache (pattern: listeo_ai_search_*)
            $cache_keys = $this->get_cache_keys();
            foreach ($cache_keys as $key) {
                if (delete_transient($key)) {
                    $cleared_count++;
                }
            }
            if (count($cache_keys) > 0) {
                $cleared_types[] = sprintf(__('search results (%d)', 'listeo-ai-search'), count($cache_keys));
            }

            // Clear API health cache
            if (delete_transient('listeo_ai_api_health')) {
                $cleared_count++;
                $cleared_types[] = __('API health status', 'listeo-ai-search');
            }

            // Clear rate limit cache (current hour)
            $rate_limit_key = 'listeo_ai_rate_limit_' . date('Y-m-d-H');
            if (delete_transient($rate_limit_key)) {
                $cleared_count++;
                $cleared_types[] = __('rate limiting', 'listeo-ai-search');
            }

            // Clear usage tracking cache (current hour)
            $usage_key = 'listeo_ai_usage_' . date('Y-m-d-H');
            if (delete_transient($usage_key)) {
                $cleared_count++;
                $cleared_types[] = __('usage tracking', 'listeo-ai-search');
            }

            // Clear Google Places cache (if any exist)
            global $wpdb;
            $google_transients = $wpdb->get_col(
                "SELECT option_name FROM {$wpdb->options} 
                 WHERE option_name LIKE '_transient_listeo_google_places_%'"
            );
            foreach ($google_transients as $transient_name) {
                $key = str_replace('_transient_', '', $transient_name);
                if (delete_transient($key)) {
                    $cleared_count++;
                }
            }
            if (count($google_transients) > 0) {
                $cleared_types[] = sprintf(__('Google Places data (%d)', 'listeo-ai-search'), count($google_transients));
            }

            // Clear processing delay cache
            $processing_keys = $wpdb->get_col(
                "SELECT option_name FROM {$wpdb->options} 
                 WHERE option_name LIKE '_transient_listeo_ai_processing_delay_%'"
            );
            foreach ($processing_keys as $transient_name) {
                $key = str_replace('_transient_', '', $transient_name);
                if (delete_transient($key)) {
                    $cleared_count++;
                }
            }
            if (count($processing_keys) > 0) {
                $cleared_types[] = sprintf(__('processing delays (%d)', 'listeo-ai-search'), count($processing_keys));
            }

            $message = $cleared_count > 0 
                ? sprintf(__('✅ Cleared %d cache entries: %s', 'listeo-ai-search'), $cleared_count, implode(', ', $cleared_types))
                : __('ℹ️ No cache entries found to clear.', 'listeo-ai-search');

            wp_send_json_success(array('message' => $message));

        } catch (Exception $e) {
            wp_send_json_error(array('message' => __('❌ Clear cache failed: ', 'listeo-ai-search') . $e->getMessage()));
        }
    }

    /**
     * Get all cache keys used by the plugin
     */
    private function get_cache_keys() {
        global $wpdb;
        
        // Get all transients that start with our plugin prefix
        $cache_keys = $wpdb->get_col(
            "SELECT REPLACE(option_name, '_transient_', '') as cache_key 
             FROM {$wpdb->options} 
             WHERE option_name LIKE '_transient_listeo_ai_search_%'"
        );
        
        return $cache_keys;
    }
    
    /**
     * AJAX handler for regenerating single embedding
     */
    public function ajax_regenerate_embedding() {
        // Verify nonce
        if (!check_ajax_referer('listeo_ai_search_nonce', 'nonce', false)) {
            wp_send_json_error(array('message' => __('Security check failed.', 'listeo-ai-search')));
            return;
        }

        // Check user permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'listeo-ai-search')));
            return;
        }

        // Get listing ID
        $listing_id = absint($_POST['listing_id'] ?? 0);
        if (!$listing_id) {
            wp_send_json_error(array('message' => __('Please enter a valid listing ID.', 'listeo-ai-search')));
            return;
        }

        // Check if listing exists
        $listing = get_post($listing_id);
        if (!$listing || $listing->post_type !== 'listing') {
            wp_send_json_error(array('message' => sprintf(__('Listing ID %d not found or not a listing post type.', 'listeo-ai-search'), $listing_id)));
            return;
        }

        try {
            // Check if API key is configured
            $api_key = get_option('listeo_ai_search_api_key', '');
            if (empty($api_key)) {
                wp_send_json_error(array('message' => __('OpenAI API key is not configured. Please configure it in Settings first.', 'listeo-ai-search')));
                return;
            }

            // Regenerate the embedding
            $result = Listeo_AI_Search_Database_Manager::generate_single_embedding($listing_id);
            
            if ($result['success']) {
                $message = sprintf(
                    __('✅ Embedding regenerated successfully for listing "%s" (ID: %d). Processed %d characters.', 'listeo-ai-search'), 
                    esc_html($listing->post_title), 
                    $listing_id,
                    $result['chars_processed'] ?? 0
                );
                
                wp_send_json_success(array(
                    'message' => $message,
                    'listing_title' => $listing->post_title,
                    'listing_id' => $listing_id,
                    'chars_processed' => $result['chars_processed'] ?? 0,
                    'embedding_dimensions' => $result['embedding_dimensions'] ?? 0
                ));
            } else {
                wp_send_json_error(array('message' => sprintf(
                    __('❌ Failed to regenerate embedding for listing "%s" (ID: %d): %s', 'listeo-ai-search'),
                    esc_html($listing->post_title),
                    $listing_id,
                    $result['error'] ?? __('Unknown error', 'listeo-ai-search')
                )));
            }

        } catch (Exception $e) {
            wp_send_json_error(array('message' => sprintf(
                __('❌ Error regenerating embedding for listing ID %d: %s', 'listeo-ai-search'),
                $listing_id,
                $e->getMessage()
            )));
        }
    }
    
    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts($hook) {
        // Only load on our settings page (updated hook for top-level menu)
        if ($hook !== 'toplevel_page_listeo-ai-search') {
            return;
        }
        
        // Enqueue CSS
        wp_enqueue_style(
            'listeo-ai-search-admin', 
            LISTEO_AI_SEARCH_PLUGIN_URL . 'assets/css/admin-styles.css', 
            [], 
            LISTEO_AI_SEARCH_VERSION
        );
        
        // Enqueue manual batch interface CSS if on database tab (needed for Safe Mode)
        $active_tab = isset($_GET['tab']) ? $_GET['tab'] : 'settings';
        if ($active_tab === 'database') {
            wp_enqueue_style(
                'listeo-ai-search-manual-batch', 
                LISTEO_AI_SEARCH_PLUGIN_URL . 'assets/css/manual-batch-interface.css', 
                [], 
                LISTEO_AI_SEARCH_VERSION
            );
            
            wp_enqueue_script(
                'listeo-ai-search-manual-batch', 
                LISTEO_AI_SEARCH_PLUGIN_URL . 'assets/js/manual-batch-interface.js', 
                array('jquery'), 
                LISTEO_AI_SEARCH_VERSION,
                true
            );
        }
        
        // Enqueue jQuery (it should already be loaded, but just to be sure)
        wp_enqueue_script('jquery');
        
        // Get the current URL to detect port
        $current_url = (is_ssl() ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'];
        $admin_ajax_url = $current_url . '/wp-admin/admin-ajax.php';
        
        // Add inline script with localized data
        wp_add_inline_script('jquery', '
            window.listeo_ai_search_ajax = {
                ajax_url: "' . esc_js($admin_ajax_url) . '",
                nonce: "' . wp_create_nonce('listeo_ai_search_nonce') . '"
            };
        ');
        
        // Add inline CSS for AJAX form styling
        wp_add_inline_style('listeo-ai-search-admin', '
            .airs-form-message {
                margin-top: 15px;
                padding: 10px 15px;
                border-radius: 5px;
                border-left: 4px solid;
                background: #fff;
            }
            .airs-form-message.airs-alert-success {
                border-left-color: #46b450;
                background: #ecf7ed;
                color: #1e4620;
            }
            .airs-form-message.airs-alert-error {
                border-left-color: #dc3232;
                background: #fbeaea;
                color: #761919;
            }
            .button-spinner {
                display: inline-flex;
                align-items: center;
                gap: 8px;
            }
            .airs-ajax-form button[type="submit"]:disabled {
                opacity: 0.7;
                cursor: not-allowed;
            }
            .airs-api-test-result {
                padding: 10px 15px;
                border-radius: 4px;
                border: 1px solid;
                font-size: 14px;
                line-height: 1.4;
            }
            .airs-api-test-result.airs-api-test-success {
                border-color: #46b450;
                background: #ecf7ed;
                color: #1e4620;
            }
            .airs-api-test-result.airs-api-test-error {
                border-color: #dc3232;
                background: #fbeaea;
                color: #761919;
            }
            .airs-button-secondary {
                background: #f7f7f7;
                color: #555;
                border: 1px solid #ccc;
            }
            .airs-button-secondary:hover {
                background: #e9e9e9;
                border-color: #999;
            }
            .airs-cache-actions {
                margin-top: 10px;
            }
            .airs-cache-actions button {
                display: inline-flex;
                align-items: center;
                gap: 5px;
            }
            @keyframes spin {
                0% { transform: rotate(0deg); }
                100% { transform: rotate(360deg); }
            }
            .spin {
                animation: spin 1s linear infinite;
            }
        ');
    }
    
    /**
     * Admin page content
     */
    public function admin_page() {
        $active_tab = isset($_GET['tab']) ? $_GET['tab'] : 'settings';
        ?>
        <div class="wrap airs-admin-wrap">
            <div class="airs-header">
                <div class="airs-header-content">
                    <div class="airs-header-icon">
                        🔍
                    </div>
                    <div class="airs-header-text">
                        <h1><?php _e('AI Search for Listeo', 'listeo-ai-search'); ?></h1>
                        <p><?php _e('Intelligent keyword search powered by OpenAI to help users find exactly what they\'re looking for', 'listeo-ai-search'); ?></p>
                    </div>
                </div>
            </div>

            <nav class="airs-nav-tab-wrapper nav-tab-wrapper">
                <a href="?page=listeo-ai-search&tab=settings" 
                   class="nav-tab <?php echo $active_tab == 'settings' ? 'nav-tab-active' : ''; ?>">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path d="M12 15.5A3.5 3.5 0 0 1 8.5 12A3.5 3.5 0 0 1 12 8.5a3.5 3.5 0 0 1 3.5 3.5a3.5 3.5 0 0 1-3.5 3.5m7.43-2.53c.04-.32.07-.64.07-.97c0-.33-.03-.66-.07-1l2.11-1.63c.19-.15.24-.42.12-.64l-2-3.46c-.12-.22-.39-.31-.61-.22l-2.49 1c-.52-.39-1.06-.73-1.69-.98l-.37-2.65A.506.506 0 0 0 14 2h-4c-.25 0-.46.18-.5.42l-.37 2.65c-.63.25-1.17.59-1.69.98l-2.49-1c-.22-.09-.49 0-.61.22l-2 3.46c-.13.22-.07.49.12.64L4.57 11c-.04.34-.07.67-.07 1c0 .33.03.65.07.97l-2.11 1.66c-.19.15-.25.42-.12.64l2 3.46c.12.22.39.3.61.22l2.49-1.01c.52.4 1.06.74 1.69.99l.37 2.65c.04.24.25.42.5.42h4c.25 0 .46-.18.5-.42l.37-2.65c.63-.26 1.17-.59 1.69-.99l2.49 1.01c.22.08.49 0 .61-.22l2-3.46c.12-.22.07-.49-.12-.64l-2.11-1.66Z" fill="currentColor"/>
                    </svg>
                    <?php _e('Settings', 'listeo-ai-search'); ?>
                </a>
                <a href="?page=listeo-ai-search&tab=database" 
                   class="nav-tab <?php echo $active_tab == 'database' ? 'nav-tab-active' : ''; ?>">
                   🗄️
                    <?php _e('Database Management', 'listeo-ai-search'); ?>
                </a>
                <a href="?page=listeo-ai-search&tab=stats" 
                   class="nav-tab <?php echo $active_tab == 'stats' ? 'nav-tab-active' : ''; ?>">
                   📊
                    <?php _e('Stats', 'listeo-ai-search'); ?>
                </a>
            </nav>

            <?php if ($active_tab == 'settings') : ?>
                <div class="airs-tab-content airs-settings-tab">
                    <?php $this->render_settings_tab(); ?>
                </div>
            <?php elseif ($active_tab == 'database') : ?>
                <div class="airs-tab-content airs-database-tab">
                    <?php $this->render_database_tab(); ?>
                </div>
            <?php else : ?>
                <div class="airs-tab-content airs-stats-tab">
                    <?php $this->render_stats_tab(); ?>
                </div>
            <?php endif; ?>
        </div>
        
        <?php $this->render_admin_scripts(); ?>
        <?php
    }
    
    /**
     * Render settings tab
     */
    private function render_settings_tab() {
        ?>
        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e('API Configuration', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Configure your OpenAI API settings and basic search preferences.', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">
                <form class="airs-ajax-form" data-section="api-config">
                    <div class="airs-form-group">
                        <label for="listeo_ai_search_api_key" class="airs-label">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path d="M12 1L3 5V11C3 16.55 6.84 21.74 12 23C17.16 21.74 21 16.55 21 11V5L12 1M12 7C13.4 7 14.8 8.6 14.8 10.5V11C15.4 11 16 11.4 16 12V16C16 16.6 15.6 17 15 17H9C8.4 17 8 16.6 8 16V12C8 11.4 8.4 11 9 11V10.5C9 8.6 10.6 7 12 7M12 8.2C11.2 8.2 10.2 9.2 10.2 10.5V11H13.8V10.5C13.8 9.2 12.8 8.2 12 8.2Z" fill="currentColor"/>
                            </svg>
                            <?php _e('OpenAI API Key', 'listeo-ai-search'); ?>
                        </label>
                        <input type="password" id="listeo_ai_search_api_key" name="listeo_ai_search_api_key" value="<?php echo esc_attr(get_option('listeo_ai_search_api_key')); ?>" class="airs-input" placeholder="sk-..." />
                        
                        <!-- Test API Key Button -->
                        <div class="airs-api-test-wrapper" style="margin-top: 10px;">
                            <button type="button" id="test-api-key" class="airs-button airs-button-secondary" style="font-size: 13px; padding: 8px 16px;">
                                <span class="button-text">
                                    <svg width="14" height="14" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" style="margin-right: 6px;">
                                        <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-2 15l-5-5 1.41-1.41L10 14.17l7.59-7.59L19 8l-9 9z" fill="currentColor"/>
                                    </svg>
                                    <?php _e('Test API Key', 'listeo-ai-search'); ?>
                                </span>
                                <span class="button-spinner" style="display: none;">
                                    <svg width="14" height="14" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                        <path d="M12,1A11,11,0,1,0,23,12,11,11,0,0,0,12,1Zm0,19a8,8,0,1,1,8-8A8,8,0,0,1,12,20Z" opacity="0.25"/>
                                        <path d="M12,4a8,8,0,0,1,7.89,6.7A1.53,1.53,0,0,0,21.38,12h0a1.5,1.5,0,0,0,1.48-1.75,11,11,0,0,0-21.72,0A1.5,1.5,0,0,0,2.62,12h0a1.53,1.53,0,0,0,1.49-1.3A8,8,0,0,1,12,4Z" fill="currentColor">
                                            <animateTransform attributeName="transform" dur="0.75s" repeatCount="indefinite" type="rotate" values="0 12 12;360 12 12"/>
                                        </path>
                                    </svg>
                                    <?php _e('Testing...', 'listeo-ai-search'); ?>
                                </span>
                            </button>
                            <div id="api-test-result" class="airs-api-test-result" style="margin-top: 8px; display: none;"></div>
                        </div>
                        
                        <div class="airs-help-text">
                            <?php _e('Enter your OpenAI API key from the OpenAI Dashboard.', 'listeo-ai-search'); ?>
                            <br><a href="https://docs.purethemes.net/listeo/knowledge-base/how-to-create-open-ai-api-key/" target="_blank"><?php _e('How to create Open AI API key →', 'listeo-ai-search'); ?></a>
                        </div>
                    </div>

                    <div class="airs-form-actions">
                        <button type="submit" class="airs-button airs-button-primary">
                            <span class="button-text"><?php _e('Save Settings', 'listeo-ai-search'); ?></span>
                            <span class="button-spinner" style="display: none;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M12,1A11,11,0,1,0,23,12,11,11,0,0,0,12,1Zm0,19a8,8,0,1,1,8-8A8,8,0,0,1,12,20Z" opacity="0.25"/>
                                    <path d="M12,4a8,8,0,0,1,7.89,6.7A1.53,1.53,0,0,0,21.38,12h0a1.5,1.5,0,0,0,1.48-1.75,11,11,0,0,0-21.72,0A1.5,1.5,0,0,0,2.62,12h0a1.53,1.53,0,0,0,1.49-1.3A8,8,0,0,1,12,4Z" fill="currentColor">
                                        <animateTransform attributeName="transform" dur="0.75s" repeatCount="indefinite" type="rotate" values="0 12 12;360 12 12"/>
                                    </path>
                                </svg>
                                <?php _e('Saving...', 'listeo-ai-search'); ?>
                            </span>
                        </button>
                    </div>
                    <div class="airs-form-message" style="display: none;"></div>
                </form>
            </div>
        </div>

        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e('Search Refining', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Extra AI-powered search features and optimizations.', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">
                <form class="airs-ajax-form" data-section="ai-features">

                    <div class="airs-form-group"  style="background: #fff;
    padding: 20px;
    border-radius: 5px;
    border: 1px solid #e0e0e0;">
                        <label class="airs-checkbox-label">
                            <input type="checkbox" name="listeo_ai_search_query_expansion" value="1" <?php checked(get_option('listeo_ai_search_query_expansion', false), 1); ?> />
                            <span class="airs-checkbox-custom"></span>
                            <span class="airs-checkbox-text">
                                <?php _e('AI Query Expansion', 'listeo-ai-search'); ?>
                            </span>
                        </label>
                        <div class="airs-help-text">
                            <?php _e('Expands queries with related keywords to find more relevant results, but may return broader matches.', 'listeo-ai-search'); ?>
<br>
                            <strong><?php _e('Examples:', 'listeo-ai-search'); ?></strong><br>
                            ➡️ "car broken down" → auto repair, mechanic, garage<br>
                            ➡️ "need somewhere to sleep" → hotels, hostels, apartments<br>
                            ➡️ "romantic evening" → restaurants, theaters, spas<br>
                           <small style="color: #666; display: block; margin-top: 10px; font-size: 13px;"><?php _e('⚠️ Adds ~500ms latency per search due to additional AI processing', 'listeo-ai-search'); ?></small>
                        </div>

                    </div>

                    <div class="airs-form-group" style="background: #fff;
    padding: 20px;
    border-radius: 5px;
    border: 1px solid #e0e0e0;">
                        <label class="airs-checkbox-label">
                            <input type="checkbox" name="listeo_ai_search_ai_location_filtering_enabled" value="1" <?php checked(get_option('listeo_ai_search_ai_location_filtering_enabled', false), 1); ?> />
                            <span class="airs-checkbox-custom"></span>
                            <span class="airs-checkbox-text">
                                <?php _e('AI Location Filtering', 'listeo-ai-search'); ?>
                            </span>
                        </label>
                        <div class="airs-help-text">
                            <span style="color: #f00;">It’s an experimental feature. Keep in mind that it works on a text basis, not coordinates - might be inaccurate.</span> <br>
                            <?php _e('Automatically detects location keywords in search queries and prioritizes geographically relevant results.', 'listeo-ai-search'); ?>
                            <br>
                            <strong><?php _e('Example:', 'listeo-ai-search'); ?></strong> <?php _e('"cozy restaurant in New York" will show New York restaurants', 'listeo-ai-search'); ?>
                            <br><small style="color: #666; display: block; margin-top: 10px; font-size: 13px;""><?php _e('⚠️ Adds ~500ms latency per search due to additional AI processing', 'listeo-ai-search'); ?></small>
                        </div>
                    </div>

                    <div class="airs-form-actions">
                        <button type="submit" class="airs-button airs-button-primary">
                            <span class="button-text"><?php _e('Save Settings', 'listeo-ai-search'); ?></span>
                            <span class="button-spinner" style="display: none;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M12,1A11,11,0,1,0,23,12,11,11,0,0,0,12,1Zm0,19a8,8,0,1,1,8-8A8,8,0,0,1,12,20Z" opacity="0.25"/>
                                    <path d="M12,4a8,8,0,0,1,7.89,6.7A1.53,1.53,0,0,0,21.38,12h0a1.5,1.5,0,0,0,1.48-1.75,11,11,0,0,0-21.72,0A1.5,1.5,0,0,0,2.62,12h0a1.53,1.53,0,0,0,1.49-1.3A8,8,0,0,1,12,4Z" fill="currentColor">
                                        <animateTransform attributeName="transform" dur="0.75s" repeatCount="indefinite" type="rotate" values="0 12 12;360 12 12"/>
                                    </path>
                                </svg>
                                <?php _e('Saving...', 'listeo-ai-search'); ?>
                            </span>
                        </button>
                    </div>
                    <div class="airs-form-message" style="display: none;"></div>
                </form>
            </div>
        </div>

        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e('Search Quality & Thresholds', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Fine-tune search quality and result filtering parameters.', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">
                <form class="airs-ajax-form" data-section="quality-thresholds">
                    <div class="airs-form-row">
                        <div class="airs-form-group">
                            <label for="listeo_ai_search_min_match_percentage" class="airs-label">
                                <?php _e('Minimum Match Percentage', 'listeo-ai-search'); ?>
                            </label>
                            <input type="number" id="listeo_ai_search_min_match_percentage" name="listeo_ai_search_min_match_percentage" value="<?php echo esc_attr(get_option('listeo_ai_search_min_match_percentage', 65)); ?>" min="0" max="100" step="5" class="airs-input airs-input-small" />
                            <span>%</span>
                            <div class="airs-help-text">
                                <?php _e('Only show AI search results with match percentage above this threshold. Lower values show more results but may be less relevant.', 'listeo-ai-search'); ?>
                                <br>
                                <span style="color: #059669;">60%</span> = Balanced, 
                                <span style="color: #2980b9;">75%</span> = Quality focused, 
                                <span style="color: #f39c12;">40%</span> = Many less relevant results
                            </div>
                        </div>

                        <div class="airs-form-group">
                            <label for="listeo_ai_search_best_match_threshold" class="airs-label">
                                <?php _e('Best Match Badge Threshold', 'listeo-ai-search'); ?>
                            </label>
                            <input type="number" id="listeo_ai_search_best_match_threshold" name="listeo_ai_search_best_match_threshold" value="<?php echo esc_attr(get_option('listeo_ai_search_best_match_threshold', 75)); ?>" min="50" max="95" step="1" class="airs-input airs-input-small" />
                            <span>%</span>
                            <div class="airs-help-text">
                                <?php _e('Show "Best Match" badge for search results above this similarity percentage. Higher values make the badge more exclusive.', 'listeo-ai-search'); ?>
                                <br>
                                <span style="color: #27ae60;">75%</span> = Balanced, 
                                <span style="color: #2980b9;">85%</span> = More exclusive, 
                                <span style="color: #f39c12;">65%</span> = More badges
                            </div>
                        </div>

                        <div class="airs-form-group">
                            <label for="listeo_ai_search_max_results" class="airs-label">
                                <?php _e('Maximum AI Top Picks Results', 'listeo-ai-search'); ?>
                            </label>
                            <input type="number" id="listeo_ai_search_max_results" name="listeo_ai_search_max_results" value="<?php echo esc_attr(get_option('listeo_ai_search_max_results', 10)); ?>" min="3" max="50" step="1" class="airs-input airs-input-small" />
                            <span><?php _e('results', 'listeo-ai-search'); ?></span>
                            <div class="airs-help-text">
                                <?php _e('Maximum number of AI Top Picks results to display in search results dropdown.', 'listeo-ai-search'); ?>
                                <br>
                                <span style="color: #27ae60;">5</span> = Balanced (recommended), 
                                <span style="color: #2980b9;">3</span> = Compact, 
                                <span style="color: #f39c12;">20</span> = Comprehensive
                            </div>
                        </div>
                    </div>

                    <div class="airs-form-actions">
                        <button type="submit" class="airs-button airs-button-primary">
                            <span class="button-text"><?php _e('Save Settings', 'listeo-ai-search'); ?></span>
                            <span class="button-spinner" style="display: none;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M12,1A11,11,0,1,0,23,12,11,11,0,0,0,12,1Zm0,19a8,8,0,1,1,8-8A8,8,0,0,1,12,20Z" opacity="0.25"/>
                                    <path d="M12,4a8,8,0,0,1,7.89,6.7A1.53,1.53,0,0,0,21.38,12h0a1.5,1.5,0,0,0,1.48-1.75,11,11,0,0,0-21.72,0A1.5,1.5,0,0,0,2.62,12h0a1.53,1.53,0,0,0,1.49-1.3A8,8,0,0,1,12,4Z" fill="currentColor">
                                        <animateTransform attributeName="transform" dur="0.75s" repeatCount="indefinite" type="rotate" values="0 12 12;360 12 12"/>
                                    </path>
                                </svg>
                                <?php _e('Saving...', 'listeo-ai-search'); ?>
                            </span>
                        </button>
                    </div>
                    <div class="airs-form-message" style="display: none;"></div>
                </form>
            </div>
        </div>

        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e('Search Suggestions', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Help users discover what they can search for with intelligent suggestions.', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">
                <form class="airs-ajax-form" data-section="search-suggestions">
                    <div class="airs-form-group">
                        <label class="airs-checkbox-label">
                            <input type="checkbox" name="listeo_ai_search_suggestions_enabled" value="1" <?php checked(get_option('listeo_ai_search_suggestions_enabled'), 1); ?> />
                            <span class="airs-checkbox-custom"></span>
                            <span class="airs-checkbox-text">
                                <?php _e('Enable search suggestions', 'listeo-ai-search'); ?>
                                <small><?php _e('Show helpful search suggestions below the search input to guide users.', 'listeo-ai-search'); ?></small>
                            </span>
                        </label>
                    </div>
                    
                    <div class="airs-form-group">
                        <label class="airs-label" style="font-weight: 600; margin-bottom: 8px; display: block;">
                            <?php _e('Suggestion Source:', 'listeo-ai-search'); ?>
                        </label>
                        <label class="airs-checkbox-label">
                            <input type="radio" name="listeo_ai_search_suggestions_source" value="top_searches" <?php checked(get_option('listeo_ai_search_suggestions_source', 'top_searches'), 'top_searches'); ?> />
                            <span class="airs-checkbox-custom"></span>
                            <span class="airs-checkbox-text"><?php _e('Show top 5 most popular searches', 'listeo-ai-search'); ?></span>
                        </label>
                        <label class="airs-checkbox-label">
                            <input type="radio" name="listeo_ai_search_suggestions_source" value="top_searches_10" <?php checked(get_option('listeo_ai_search_suggestions_source', 'top_searches'), 'top_searches_10'); ?> />
                            <span class="airs-checkbox-custom"></span>
                            <span class="airs-checkbox-text"><?php _e('Show top 10 most popular searches', 'listeo-ai-search'); ?></span>
                        </label>
                        <label class="airs-checkbox-label">
                            <input type="radio" name="listeo_ai_search_suggestions_source" value="custom" <?php checked(get_option('listeo_ai_search_suggestions_source', 'top_searches'), 'custom'); ?> />
                            <span class="airs-checkbox-custom"></span>
                            <span class="airs-checkbox-text"><?php _e('Use custom suggestions (comma-separated)', 'listeo-ai-search'); ?></span>
                        </label>
                    </div>
                    
                    <div class="airs-form-group">
                        <label for="listeo_ai_search_custom_suggestions" class="airs-label">
                            <?php _e('Custom Suggestions (comma-separated):', 'listeo-ai-search'); ?>
                        </label>
                        <textarea id="listeo_ai_search_custom_suggestions" name="listeo_ai_search_custom_suggestions" rows="3" class="airs-input" placeholder="pet-friendly apartments, cozy cafes"><?php echo esc_textarea(get_option('listeo_ai_search_custom_suggestions', '')); ?></textarea>
                        <div class="airs-help-text">
                            <?php _e('Enter search suggestions separated by commas. These will be displayed when "custom suggestions" is selected above.', 'listeo-ai-search'); ?>
                            <br><strong><?php _e('Examples:', 'listeo-ai-search'); ?></strong>
                            pet-friendly apartments, cozy cafes with wifi, outdoor wedding venues, 24/7 gyms downtown
                        </div>
                    </div>

                    <div class="airs-form-actions">
                        <button type="submit" class="airs-button airs-button-primary">
                            <span class="button-text"><?php _e('Save Settings', 'listeo-ai-search'); ?></span>
                            <span class="button-spinner" style="display: none;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M12,1A11,11,0,1,0,23,12,11,11,0,0,0,12,1Zm0,19a8,8,0,1,1,8-8A8,8,0,0,1,12,20Z" opacity="0.25"/>
                                    <path d="M12,4a8,8,0,0,1,7.89,6.7A1.53,1.53,0,0,0,21.38,12h0a1.5,1.5,0,0,0,1.48-1.75,11,11,0,0,0-21.72,0A1.5,1.5,0,0,0,2.62,12h0a1.53,1.53,0,0,0,1.49-1.3A8,8,0,0,1,12,4Z" fill="currentColor">
                                        <animateTransform attributeName="transform" dur="0.75s" repeatCount="indefinite" type="rotate" values="0 12 12;360 12 12"/>
                                    </path>
                                </svg>
                                <?php _e('Saving...', 'listeo-ai-search'); ?>
                            </span>
                        </button>
                    </div>
                    <div class="airs-form-message" style="display: none;"></div>
                </form>
            </div>
        </div>

        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e('Developer & Debug Options', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Advanced options for development and troubleshooting.', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">
                <form class="airs-ajax-form" data-section="developer-debug">
                    <div class="airs-form-group">
                        <label class="airs-checkbox-label">
                            <input type="checkbox" name="listeo_ai_search_debug_mode" value="1" <?php checked(get_option('listeo_ai_search_debug_mode'), 1); ?> />
                            <span class="airs-checkbox-custom"></span>
                            <span class="airs-checkbox-text">
                                <?php _e('Debug Mode', 'listeo-ai-search'); ?>
                                <small><?php _e('Enable debug logging to wp-content/debug.log', 'listeo-ai-search'); ?></small>
                            </span>
                        </label>
                        <div class="airs-help-text"><?php _e('When enabled, detailed search information will be logged to help troubleshoot issues. Make sure WP_DEBUG_LOG is enabled in wp-config.php.', 'listeo-ai-search'); ?></div>
                    </div>

                    <div class="airs-form-group">
                        <label class="airs-checkbox-label">
                            <input type="checkbox" name="listeo_ai_search_disable_cache" value="1" <?php checked(get_option('listeo_ai_search_disable_cache'), 1); ?> />
                            <span class="airs-checkbox-custom"></span>
                            <span class="airs-checkbox-text">
                                <?php _e('Disable Cache', 'listeo-ai-search'); ?>
                                <small><?php _e('Disable search result caching (for development)', 'listeo-ai-search'); ?></small>
                            </span>
                        </label>
                        <div class="airs-help-text"><?php _e('When enabled, search results will not be cached. Useful during development to see fresh results every time. Disable for production to improve performance.', 'listeo-ai-search'); ?></div>
                        
                        <!-- Clear Cache Button -->
                        <div class="airs-cache-actions" style="margin-top: 10px;">
                            <button type="button" id="listeo-clear-cache-btn" class="button button-secondary">
                                <span class="dashicons dashicons-trash" style="vertical-align: middle; margin-right: 5px;"></span>
                                <?php _e('Clear All Cache', 'listeo-ai-search'); ?>
                            </button>
                            <span id="listeo-clear-cache-status" style="margin-left: 10px; font-weight: bold;"></span>
                        </div>
                    </div>

                    <div class="airs-form-group">
                        <label class="airs-checkbox-label">
                            <input type="checkbox" name="listeo_ai_search_enable_analytics" value="1" <?php checked(get_option('listeo_ai_search_enable_analytics'), 1); ?> />
                            <span class="airs-checkbox-custom"></span>
                            <span class="airs-checkbox-text">
                                <?php _e('Search Analytics', 'listeo-ai-search'); ?>
                                <small><?php _e('Enable search analytics tracking', 'listeo-ai-search'); ?></small>
                            </span>
                        </label>
                        <div class="airs-help-text">
                            <?php _e('When enabled, tracks search queries, response times, and usage patterns to help improve search performance. Data is anonymized and stored locally.', 'listeo-ai-search'); ?>
                            <br><strong><?php _e('Privacy:', 'listeo-ai-search'); ?></strong> IP addresses are anonymized, only tracks search patterns for optimization.
                        </div>
                        <?php if (get_option('listeo_ai_search_enable_analytics')): ?>
                            <div style="margin-top: 10px; padding: 10px; background: #e8f5e8; border-left: 4px solid #46b450; border-radius: 5px;">
                                <?php
                                if (class_exists('Listeo_AI_Search_Analytics')) {
                                    $analytics = Listeo_AI_Search_Analytics::get_analytics(7);
                                    echo '<strong>Last 7 days:</strong> ' . $analytics['total_searches'] . ' searches, ';
                                    echo $analytics['avg_response_time'] . 'ms avg response ';
                                }
                                ?>
                            </div>
                        <?php endif; ?>
                    </div>

                    <div class="airs-form-group">
                        <label for="listeo_ai_search_rate_limit_per_hour" class="airs-label">
                            <?php _e('API Rate Limit (per hour)', 'listeo-ai-search'); ?>
                        </label>
                        <input type="number" id="listeo_ai_search_rate_limit_per_hour" name="listeo_ai_search_rate_limit_per_hour" value="<?php echo esc_attr(get_option('listeo_ai_search_rate_limit_per_hour', 1000)); ?>" min="10" max="10000" step="50" class="airs-input airs-input-small" />
                        <span><?php _e('API calls per hour', 'listeo-ai-search'); ?></span>
                        <div class="airs-help-text">
                            <?php _e('Maximum number of search requests allowed per hour to prevent excessive OpenAI API usage during search operations.', 'listeo-ai-search'); ?>
                            <?php
                            // Show current usage if available
                            $rate_limit_key = 'listeo_ai_rate_limit_' . date('Y-m-d-H');
                            $current_calls = get_transient($rate_limit_key) ?: 0;
                            if ($current_calls > 0) {
                                echo '<br><span style="background-color: rgba(0, 123, 255, 0.1); padding: 3px 7px; display: inline-block; margin-top: 5px; border-radius: 3px; color: rgba(0, 123, 255, 1);">Current hour usage: <strong style="color: rgba(0, 123, 255, 1)">' . $current_calls . '</strong> calls</span>';
                            }
                            ?>
                        </div>
                    </div>

                    <div class="airs-form-actions">
                        <button type="submit" class="airs-button airs-button-primary">
                            <span class="button-text"><?php _e('Save Settings', 'listeo-ai-search'); ?></span>
                            <span class="button-spinner" style="display: none;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M12,1A11,11,0,1,0,23,12,11,11,0,0,0,12,1Zm0,19a8,8,0,1,1,8-8A8,8,0,0,1,12,20Z" opacity="0.25"/>
                                    <path d="M12,4a8,8,0,0,1,7.89,6.7A1.53,1.53,0,0,0,21.38,12h0a1.5,1.5,0,0,0,1.48-1.75,11,11,0,0,0-21.72,0A1.5,1.5,0,0,0,2.62,12h0a1.53,1.53,0,0,0,1.49-1.3A8,8,0,0,1,12,4Z" fill="currentColor">
                                        <animateTransform attributeName="transform" dur="0.75s" repeatCount="indefinite" type="rotate" values="0 12 12;360 12 12"/>
                                    </path>
                                </svg>
                                <?php _e('Saving...', 'listeo-ai-search'); ?>
                            </span>
                        </button>
                    </div>
                    <div class="airs-form-message" style="display: none;"></div>
                </form>
            </div>
        </div>

        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e('🚀 Performance Optimization', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Memory optimization and batch processing settings for large datasets.', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">
                <form class="airs-ajax-form" data-section="performance-optimization">
                    <div class="airs-form-group">
                        <label for="listeo_ai_search_batch_size" class="airs-label">
                            <?php _e('Batch Size', 'listeo-ai-search'); ?>
                        </label>
                        <input type="number" id="listeo_ai_search_batch_size" name="listeo_ai_search_batch_size" value="<?php echo esc_attr(get_option('listeo_ai_search_batch_size', 500)); ?>" min="100" max="2000" step="100" class="airs-input airs-input-small" />
                        <span><?php _e('embeddings per batch', 'listeo-ai-search'); ?></span>
                        <div class="airs-help-text">
                            <?php _e('Number of embeddings to process in each batch. Lower values use less memory but may be slower. Higher values are faster but use more memory.', 'listeo-ai-search'); ?>
                            <br><strong><?php _e('Recommendations:', 'listeo-ai-search'); ?></strong>
                            <span style="color: #059669;">500</span> = Balanced (recommended), 
                            <span style="color: #2980b9;">300</span> = Low memory servers, 
                            <span style="color: #f39c12;">1000</span> = High performance servers
                        </div>
                    </div>

                    <div class="airs-form-group">
                        <label for="listeo_ai_search_embedding_delay" class="airs-label">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-2 15l-5-5 1.41-1.41L10 14.17l7.59-7.59L19 8l-9 9z" fill="currentColor"/>
                            </svg>
                            <?php _e('Embedding Generation Delay', 'listeo-ai-search'); ?>
                        </label>
                        <input type="number" id="listeo_ai_search_embedding_delay" name="listeo_ai_search_embedding_delay" value="<?php echo esc_attr(get_option('listeo_ai_search_embedding_delay', 5)); ?>" min="0" max="60" step="1" class="airs-input airs-input-small" />
                        <span><?php _e('minutes', 'listeo-ai-search'); ?></span>
                        <div class="airs-help-text">
                            <?php _e('Minimum delay between embedding regenerations for the same listing. Prevents unnecessary API costs from rapid successive edits.', 'listeo-ai-search'); ?>
                            <br><strong><?php _e('Examples:', 'listeo-ai-search'); ?></strong> 
                            <span style="color: #059669;">5</span> = Balanced (recommended), 
                            <span style="color: #2980b9;">10</span> = Conservative, 
                            <span style="color: #f39c12;">0</span> = No throttling
                            <br><small><?php _e('💡 Tip: User can make multiple quick edits within the delay period without triggering extra API calls', 'listeo-ai-search'); ?></small>
                        </div>
                    </div>

                    <div class="airs-form-actions">
                        <button type="submit" class="airs-button airs-button-primary">
                            <span class="button-text"><?php _e('Save Settings', 'listeo-ai-search'); ?></span>
                            <span class="button-spinner" style="display: none;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M12,1A11,11,0,1,0,23,12,11,11,0,0,0,12,1Zm0,19a8,8,0,1,1,8-8A8,8,0,0,1,12,20Z" opacity="0.25"/>
                                    <path d="M12,4a8,8,0,0,1,7.89,6.7A1.53,1.53,0,0,0,21.38,12h0a1.5,1.5,0,0,0,1.48-1.75,11,11,0,0,0-21.72,0A1.5,1.5,0,0,0,2.62,12h0a1.53,1.53,0,0,0,1.49-1.3A8,8,0,0,1,12,4Z" fill="currentColor">
                                        <animateTransform attributeName="transform" dur="0.75s" repeatCount="indefinite" type="rotate" values="0 12 12;360 12 12"/>
                                    </path>
                                </svg>
                                <?php _e('Saving...', 'listeo-ai-search'); ?>
                            </span>
                        </button>
                    </div>
                    <div class="airs-form-message" style="display: none;"></div>
                </form>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render database management tab
     */
    private function render_database_tab() {
        // Handle Safe Mode actions
        $this->handle_safe_mode_actions();
        
        // Ensure the batch table exists
        Listeo_AI_Search_Manual_Batch_Processor::create_batch_table();
        
        $is_safe_mode_active = $this->is_safe_mode_active();
        $is_safe_mode_mandatory = $this->is_safe_mode_mandatory();
        
        ?>
        <!-- Generate Embeddings Section -->
        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e('Generate Structured Embeddings', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Generate embeddings for all your listings using the structured format which improves search accuracy.', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">
                <?php if ($is_safe_mode_active): ?>
                    <!-- Safe Mode Interface -->
                    <?php if (!$is_safe_mode_mandatory): ?>
                        <div style="margin-bottom: 15px; padding: 10px; background: #fff7df; border-radius: 4px;">
                            <p style="margin: 0 0 10px 0; color: #725b14;"><strong><?php _e('Safe Mode is currently enabled.', 'listeo-ai-search'); ?></strong></p>
                            <form method="post" style="margin: 0;" onsubmit="return confirm('<?php _e('Are you sure you want to disable Safe Mode? This will switch to direct processing mode.', 'listeo-ai-search'); ?>');">
                                <?php wp_nonce_field('listeo_safe_mode_action'); ?>
                                <button type="submit" name="disable_safe_mode" class="airs-button airs-button-secondary" style="font-size: 13px; padding: 6px 12px;">
                                    <?php _e('Disable Safe Mode', 'listeo-ai-search'); ?>
                                </button>
                            </form>
                        </div>
                    <?php endif; ?>
                    <?php $this->render_manual_batch_interface(); ?>
                <?php else: ?>
                    <!-- Regular Interface -->
                    <?php $this->render_simple_batch_interface(); ?>
                <?php endif; ?>
            </div>
        </div>

        <!-- Database Actions Section -->
        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e('Database Actions', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Use these tools to manage your AI search database.', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">

                <div class="airs-form-group">
                    <label for="listing-id-input" class="airs-label">
                        <?php _e('Check Embedding', 'listeo-ai-search'); ?>
                    </label>
                    <div style="display: flex; gap: 10px; align-items: center;">
                        <input type="number" id="listing-id-input" placeholder="Enter Listing ID" class="airs-input airs-input-small" />
                        <button type="button" id="check-embedding" class="airs-button airs-button-secondary"><?php _e('Check Embedding', 'listeo-ai-search'); ?></button>
                    </div>
                    <div class="airs-help-text"><?php _e('Enter a listing ID to view its embedding data and processed content.', 'listeo-ai-search'); ?></div>
                </div>

                <div class="airs-form-group">
                    <label for="regenerate-listing-id-input" class="airs-label">
                        <?php _e('Regenerate Embedding', 'listeo-ai-search'); ?>
                    </label>
                    <div style="display: flex; gap: 10px; align-items: center;">
                        <input type="number" id="regenerate-listing-id-input" placeholder="Enter Listing ID" class="airs-input airs-input-small" />
                        <button type="button" id="regenerate-embedding" class="airs-button airs-button-primary">
                            <span class="button-text"><?php _e('Regenerate Embedding', 'listeo-ai-search'); ?></span>
                            <span class="button-spinner" style="display: none;">
                                <span class="dashicons dashicons-update spin" style="vertical-align: middle;"></span>
                                <?php _e('Processing...', 'listeo-ai-search'); ?>
                            </span>
                        </button>
                    </div>
                    <div class="airs-help-text"><?php _e('Enter a listing ID to regenerate its embedding data. This will fetch fresh content and create a new embedding.', 'listeo-ai-search'); ?></div>
                    <div id="regenerate-embedding-result" style="margin-top: 10px; display: none;"></div>
                </div>

                <div class="airs-form-group">
                    <button type="button" id="clear-database" class="airs-button airs-button-danger"><?php _e('Clear All Embeddings', 'listeo-ai-search'); ?></button>
                    <div class="airs-help-text"><?php _e('Delete all embeddings. You will need to regenerate them after clearing.', 'listeo-ai-search'); ?></div>
                </div>
            </div>
        </div>

        <!-- Embedding Viewer Section -->
        <div id="embedding-viewer" class="airs-card" style="display: none;">
            <div class="airs-card-header">
                <h3><?php _e('Embedding Data', 'listeo-ai-search'); ?></h3>
            </div>
            <div class="airs-card-body">
                <div id="embedding-content"></div>
                <div class="airs-form-actions">
                    <button type="button" id="close-embedding" class="airs-button airs-button-secondary"><?php _e('Close', 'listeo-ai-search'); ?></button>
                </div>
            </div>
        </div>

        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e('Database Status', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Monitor your AI search database status and view embedding statistics.', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">
                <div id="database-status" style="margin: 20px 0;">
                    <div id="status-content">
                        <p><em><?php _e('Loading database status...', 'listeo-ai-search'); ?></em></p>
                    </div>
                    <div class="airs-form-actions">
                        <button type="button" id="refresh-status" class="airs-button airs-button-secondary"><?php _e('Refresh Status', 'listeo-ai-search'); ?></button>
                        <button type="button" id="test-ajax" class="airs-button airs-button-secondary"><?php _e('Test AJAX', 'listeo-ai-search'); ?></button>
                    </div>
                </div>
            </div>
        </div>

        <div id="action-result" class="airs-card" style="display: none;">
            <div class="airs-card-body">
                <div id="result-message-content" class="airs-alert"></div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render simple batch interface for small catalogs
     */
    /**
     * Handle Safe Mode toggle actions
     */
    private function handle_safe_mode_actions() {
        if (isset($_POST['enable_safe_mode']) && wp_verify_nonce($_POST['_wpnonce'], 'listeo_safe_mode_action')) {
            update_option('listeo_ai_search_safe_mode_enabled', 1);
            add_action('admin_notices', function() {
                echo '<div class="notice notice-success is-dismissible"><p>' . __('Safe Mode has been enabled. You can now use advanced batch processing.', 'listeo-ai-search') . '</p></div>';
            });
        }
        
        if (isset($_POST['disable_safe_mode']) && wp_verify_nonce($_POST['_wpnonce'], 'listeo_safe_mode_action')) {
            $total_listings = Listeo_AI_Search_Manual_Batch_Processor::get_total_listings_count();
            if ($total_listings >= Listeo_AI_Search_Manual_Batch_Processor::MANUAL_MODE_THRESHOLD) {
                add_action('admin_notices', function() use ($total_listings) {
                    echo '<div class="notice notice-error is-dismissible"><p>' . sprintf(__('Safe Mode cannot be disabled for sites with %d+ listings. This is required for optimal performance and API rate limiting.', 'listeo-ai-search'), Listeo_AI_Search_Manual_Batch_Processor::MANUAL_MODE_THRESHOLD) . '</p></div>';
                });
            } else {
                update_option('listeo_ai_search_safe_mode_enabled', 0);
                add_action('admin_notices', function() {
                    echo '<div class="notice notice-success is-dismissible"><p>' . __('Safe Mode has been disabled. You can now use direct processing.', 'listeo-ai-search') . '</p></div>';
                });
            }
        }
    }

    /**
     * Check if Safe Mode is enabled or mandatory
     */
    private function is_safe_mode_active() {
        $total_listings = Listeo_AI_Search_Manual_Batch_Processor::get_total_listings_count();
        $safe_mode_enabled = get_option('listeo_ai_search_safe_mode_enabled', 0);
        
        // Safe Mode is mandatory for listings >= MANUAL_MODE_THRESHOLD
        if ($total_listings >= Listeo_AI_Search_Manual_Batch_Processor::MANUAL_MODE_THRESHOLD) {
            // Auto-enable Safe Mode option for sites above threshold
            if (!$safe_mode_enabled) {
                update_option('listeo_ai_search_safe_mode_enabled', 1);
            }
            return true;
        }
        
        return $safe_mode_enabled;
    }

    /**
     * Check if Safe Mode is mandatory (cannot be disabled)
     */
    private function is_safe_mode_mandatory() {
        $total_listings = Listeo_AI_Search_Manual_Batch_Processor::get_total_listings_count();
        return $total_listings >= Listeo_AI_Search_Manual_Batch_Processor::MANUAL_MODE_THRESHOLD;
    }

    private function render_simple_batch_interface() {
        ?>
        <p class="airs-help-text">
            <?php _e('This process will consume OpenAI API credits based on the number of listings processed.', 'listeo-ai-search'); ?>
        </p>
        
        <div id="listing-detection-info" style="background: #008c2810; color: #008c28; font-weight: 500; font-size: 14px; padding: 10px 15px; border-radius: 4px; margin: 0 0 15px 0; display: inline-block;">
                <span id="listing-count-text"><?php _e('Loading listing information...', 'listeo-ai-search'); ?></span>
        </div>
        
        <div id="regeneration-controls">
            <div class="airs-form-group">
                <label for="batch-size" style="margin:0;"class="airs-label"><?php _e('Batch Size:', 'listeo-ai-search'); ?></label>
                <span class="airs-help-text" style="margin: 0; margin-bottom: 9px; display: block;"><?php _e('Recommended between 20 and 100', 'listeo-ai-search'); ?></span>
                <input type="number" id="batch-size" value="20" min="1" max="50" class="airs-input airs-input-small" />
            </div>
            
            <div class="airs-form-actions">
                <button type="button" id="start-regeneration" style="zoom: 1.15" class="airs-button airs-button-primary"><?php _e('🚀 Start Generation', 'listeo-ai-search'); ?></button>
                <button type="button" id="stop-regeneration" class="airs-button airs-button-secondary" style="display: none;"><?php _e('Stop', 'listeo-ai-search'); ?></button>
            </div>
            
            <div class="airs-form-group" style="margin-top: 35px; margin-bottom: 0 !important;">
                <label style="display: flex; align-items: center; gap: 8px; margin: 0; font-weight: 500;">
                    <input type="checkbox" id="enable-safe-mode-checkbox" style="margin: 0;">
                    <?php _e('Enable Safe Mode', 'listeo-ai-search'); ?>
                </label>
                <div class="airs-help-text" style="margin: 5px 0 0 0;">
                    <?php _e('Check this option if you encounter issues like timeouts or "API rate limit exceeded" errors.', 'listeo-ai-search'); ?>
                </div>
            </div>
        </div>
        
        <div id="regeneration-progress" style="display: none; margin-top: 15px;">
            <div class="airs-progress-bar">
                <div id="progress-fill" class="airs-progress-bar-inner"></div>
            </div>
            <p id="progress-text" class="airs-progress-text">Starting...</p>
        </div>
        
        <div id="regeneration-log" style="display: none; margin-top: 15px;">
            <h4><?php _e('Progress Log:', 'listeo-ai-search'); ?></h4>
            <div id="log-content" class="airs-log"></div>
        </div>
        <?php
    }
    
    /**
     * Render manual batch interface for large catalogs
     */
    private function render_manual_batch_interface() {
        // Get current progress data
        $progress_data = Listeo_AI_Search_Manual_Batch_Processor::get_processing_progress();
        $batches = Listeo_AI_Search_Manual_Batch_Processor::get_all_batches();
        $batch_summary = $progress_data['batch_summary'];

        // Get current settings
        $batch_size = get_option('listeo_ai_search_manual_batch_size', Listeo_AI_Search_Manual_Batch_Processor::DEFAULT_BATCH_SIZE);
        $delay = get_option('listeo_ai_search_manual_batch_delay', Listeo_AI_Search_Manual_Batch_Processor::DEFAULT_DELAY);
        
        // Force enable background processing in Safe Mode for optimal performance
        $background_processing = 1;
        ?>
        
        <!-- Manual Batch Mode Notification -->
        <div class="batch-mode-info" style="background: #fff7df; border: none; border-radius: 4px; padding: 15px; margin-bottom: 15px; color: #725b14;">
            <p style="margin: 0; font-size: 15px; line-height: 1.5;">
                <strong style="font-size: 16px;"><?php _e('Safe Mode Enabled', 'listeo-ai-search'); ?></strong><br>
                <?php 
                $total_listings = Listeo_AI_Search_Manual_Batch_Processor::get_total_listings_count();
                if ($total_listings >= 200): ?>
                    <?php printf(
                        __('You have %d listings, which requires Safe Mode processing. Use the advanced controls below to process embeddings in manageable batches to avoid API rate limits.', 'listeo-ai-search'),
                        $total_listings
                    ); ?>
                <?php else: ?>
                    <?php _e('Safe Mode provides advanced batch processing with delays and background processing to avoid API rate limits.', 'listeo-ai-search'); ?>
                <?php endif; ?>
            </p>
        </div>
        
        <!-- Custom Status Messages (inside card) -->
        <div id="batch-status-messages" style="display: none; margin-bottom: 15px;"></div>

        <!-- Progress Overview -->
        <div class="manual-batch-overview">
            <h4><?php _e('Processing Overview', 'listeo-ai-search'); ?></h4>
            
            <div class="progress-stats">
                <div class="stat-box">
                    <div class="stat-number"><?php echo number_format($progress_data['total_listings']); ?></div>
                    <div class="stat-label"><?php _e('Total Listings', 'listeo-ai-search'); ?></div>
                </div>
                
                <div class="stat-box">
                    <div class="stat-number"><?php echo number_format($progress_data['processed_listings']); ?></div>
                    <div class="stat-label"><?php _e('Processed', 'listeo-ai-search'); ?></div>
                </div>
                
                <div class="stat-box">
                    <div class="stat-number"><?php echo number_format($progress_data['remaining_listings']); ?></div>
                    <div class="stat-label"><?php _e('Remaining', 'listeo-ai-search'); ?></div>
                </div>
                
                <div class="stat-box">
                    <div class="stat-number"><?php echo $progress_data['progress_percentage']; ?>%</div>
                    <div class="stat-label"><?php _e('Complete', 'listeo-ai-search'); ?></div>
                </div>
            </div>
            
            <div class="progress-bar-container">
                <div class="progress-bar">
                    <div class="progress-fill" style="width: <?php echo $progress_data['progress_percentage']; ?>%"></div>
                </div>
                <div class="progress-text">
                    <?php printf(
                        __('%d of %d listings processed (%s%%)', 'listeo-ai-search'),
                        $progress_data['processed_listings'],
                        $progress_data['total_listings'],
                        $progress_data['progress_percentage']
                    ); ?>
                </div>
            </div>
        </div>

        <!-- Batch Configuration -->
        <div class="batch-configuration">
            <h4><?php _e('Batch Configuration', 'listeo-ai-search'); ?></h4>
            
            <form id="batch-config-form">
                <table class="form-table">
                    <tr>
                        <th scope="row"><?php _e('Batch Size', 'listeo-ai-search'); ?></th>
                        <td>
                            <select id="batch-size" name="batch_size">
                                <option value="150" <?php selected($batch_size, 150); ?>>150 <?php _e('listings per batch', 'listeo-ai-search'); ?></option>
                                <option value="250" <?php selected($batch_size, 250); ?>>250 <?php _e('listings per batch', 'listeo-ai-search'); ?> (<?php _e('Default', 'listeo-ai-search'); ?>)</option>
                                <option value="500" <?php selected($batch_size, 500); ?>>500 <?php _e('listings per batch', 'listeo-ai-search'); ?></option>
                            </select>
                            <p class="description"><?php _e('Smaller batches are safer but take longer. Larger batches are faster but more likely to hit rate limits.', 'listeo-ai-search'); ?></p>
                        </td>
                    </tr>
                </table>
                
                <!-- Hidden input for delay (always 0 for maximum speed) -->
                <input type="hidden" id="batch-delay" name="batch_delay" value="0" />
                
                <div class="airs-form-actions">
                    <button type="button" id="initialize-batches" class="airs-button airs-button-success">
                        <span class="button-text"><?php _e('Initialize Batches', 'listeo-ai-search'); ?></span>
                        <span class="button-spinner" style="display: none;">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path d="M12,1A11,11,0,1,0,23,12,11,11,0,0,0,12,1Zm0,19a8,8,0,1,1,8-8A8,8,0,0,1,12,20Z" opacity="0.25"/>
                                <path d="M12,4a8,8,0,0,1,7.89,6.7A1.53,1.53,0,0,0,21.38,12h0a1.5,1.5,0,0,0,1.48-1.75,11,11,0,0,0-21.72,0A1.5,1.5,0,0,0,2.62,12h0a1.53,1.53,0,0,0,1.49-1.3A8,8,0,0,1,12,4Z" fill="currentColor">
                                    <animateTransform attributeName="transform" dur="0.75s" repeatCount="indefinite" type="rotate" values="0 12 12;360 12 12"/>
                                </path>
                            </svg>
                        </span>
                    </button>
                    <button type="button" id="save-batch-settings" class="airs-button airs-button-primary">
                        <span class="button-text"><?php _e('Save Settings', 'listeo-ai-search'); ?></span>
                        <span class="button-spinner" style="display: none;">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path d="M12,1A11,11,0,1,0,23,12,11,11,0,0,0,12,1Zm0,19a8,8,0,1,1,8-8A8,8,0,0,1,12,20Z" opacity="0.25"/>
                                <path d="M12,4a8,8,0,0,1,7.89,6.7A1.53,1.53,0,0,0,21.38,12h0a1.5,1.5,0,0,0,1.48-1.75,11,11,0,0,0-21.72,0A1.5,1.5,0,0,0,2.62,12h0a1.53,1.53,0,0,0,1.49-1.3A8,8,0,0,1,12,4Z" fill="currentColor">
                                    <animateTransform attributeName="transform" dur="0.75s" repeatCount="indefinite" type="rotate" values="0 12 12;360 12 12"/>
                                </path>
                            </svg>
                        </span>
                    </button>
                </div>
            </form>
        </div>

        <!-- Batch Processing Interface -->
        <?php if (!empty($batches)): ?>
        <div class="batch-processing">
            <h4><?php _e('Batch Processing', 'listeo-ai-search'); ?></h4>
            
                
                <!-- Batch Summary -->
                <div class="batch-summary">
                    <div class="batch-stat">
                        <span class="count"><?php echo $batch_summary['total_batches']; ?></span>
                        <span class="label"><?php _e('Total Batches', 'listeo-ai-search'); ?></span>
                    </div>
                    <div class="batch-stat ready">
                        <span class="count"><?php echo $batch_summary['ready']; ?></span>
                        <span class="label"><?php _e('Ready', 'listeo-ai-search'); ?></span>
                    </div>
                    <div class="batch-stat queued">
                        <span class="count"><?php echo $batch_summary['queued']; ?></span>
                        <span class="label"><?php _e('Queued', 'listeo-ai-search'); ?></span>
                    </div>
                    <div class="batch-stat processing">
                        <span class="count"><?php echo $batch_summary['processing']; ?></span>
                        <span class="label"><?php _e('Processing', 'listeo-ai-search'); ?></span>
                    </div>
                    <div class="batch-stat complete">
                        <span class="count"><?php echo $batch_summary['complete']; ?></span>
                        <span class="label"><?php _e('Complete', 'listeo-ai-search'); ?></span>
                    </div>
                    <div class="batch-stat failed">
                        <span class="count"><?php echo $batch_summary['failed']; ?></span>
                        <span class="label"><?php _e('Failed', 'listeo-ai-search'); ?></span>
                    </div>
                    <div class="batch-stat partial">
                        <span class="count"><?php echo $batch_summary['partial']; ?></span>
                        <span class="label"><?php _e('Partial', 'listeo-ai-search'); ?></span>
                    </div>
                </div>

                <!-- Batch Controls -->
                <div class="batch-controls">
                    <div class="bulk-actions">
                        <button type="button" id="select-all-batches" class="airs-button airs-button-secondary"><?php _e('Select All', 'listeo-ai-search'); ?></button>
                        <button type="button" id="select-none-batches" class="airs-button airs-button-secondary"><?php _e('Select None', 'listeo-ai-search'); ?></button>
                        <button type="button" id="select-ready-batches" class="airs-button airs-button-secondary"><?php _e('Select Ready', 'listeo-ai-search'); ?></button>
                        <button type="button" id="select-failed-batches" class="airs-button airs-button-secondary"><?php _e('Select Failed', 'listeo-ai-search'); ?></button>
                    </div>
                    
                    <div class="process-actions">
                        <button type="button" id="stop-processing" class="airs-button airs-button-warning" style="display: none;">
                            <?php _e('Stop Processing', 'listeo-ai-search'); ?>
                        </button>
                    </div>
                    
                    <div class="danger-actions">
                        <button type="button" id="reset-all-batches" class="airs-button airs-button-danger">
                            <span class="button-text"><?php _e('Reset All', 'listeo-ai-search'); ?></span>
                            <span class="button-spinner" style="display: none;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M12,1A11,11,0,1,0,23,12,11,11,0,0,0,12,1Zm0,19a8,8,0,1,1,8-8A8,8,0,0,1,12,20Z" opacity="0.25"/>
                                    <path d="M12,4a8,8,0,0,1,7.89,6.7A1.53,1.53,0,0,0,21.38,12h0a1.5,1.5,0,0,0,1.48-1.75,11,11,0,0,0-21.72,0A1.5,1.5,0,0,0,2.62,12h0a1.53,1.53,0,0,0,1.49-1.3A8,8,0,0,1,12,4Z" fill="currentColor">
                                        <animateTransform attributeName="transform" dur="0.75s" repeatCount="indefinite" type="rotate" values="0 12 12;360 12 12"/>
                                    </path>
                                </svg>
                            </span>
                        </button>
                    </div>
                </div>

                <!-- Batch List -->
                <div class="batch-list">
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <td class="manage-column column-cb check-column">
                                    <input type="checkbox" id="batch-select-all">
                                </td>
                                <th class="manage-column"><?php _e('Batch', 'listeo-ai-search'); ?></th>
                                <th class="manage-column"><?php _e('Range', 'listeo-ai-search'); ?></th>
                                <th class="manage-column"><?php _e('Progress', 'listeo-ai-search'); ?></th>
                                <th class="manage-column"><?php _e('Status', 'listeo-ai-search'); ?></th>
                                <th class="manage-column"><?php _e('Actions', 'listeo-ai-search'); ?></th>
                            </tr>
                        </thead>
                        <tbody id="batch-list-body">
                            <?php foreach ($batches as $batch): 
                                $progress_percent = $batch['total_count'] > 0 ? round(($batch['processed_count'] / $batch['total_count']) * 100, 1) : 0;
                                $status_class = $batch['status'];
                                $status_icon = '';
                                $status_text = '';
                                
                                switch ($batch['status']) {
                                    case 'ready':
                                        $status_icon = '⏳';
                                        $status_text = __('Ready', 'listeo-ai-search');
                                        break;
                                    case 'queued':
                                        $status_icon = '📋';
                                        $status_text = __('Queued', 'listeo-ai-search');
                                        break;
                                    case 'processing':
                                        $status_icon = '🔄';
                                        $status_text = __('Processing', 'listeo-ai-search');
                                        break;
                                    case 'complete':
                                        $status_icon = '✅';
                                        $status_text = __('Complete', 'listeo-ai-search');
                                        break;
                                    case 'failed':
                                        $status_icon = '⚠️';
                                        $status_text = __('Failed', 'listeo-ai-search');
                                        break;
                                    case 'partial':
                                        $status_icon = '⚙️';
                                        $status_text = __('Partial', 'listeo-ai-search');
                                        break;
                                }
                            ?>
                                <tr class="batch-row" data-batch-id="<?php echo esc_attr($batch['batch_id']); ?>" data-status="<?php echo esc_attr($batch['status']); ?>">
                                    <th scope="row" class="check-column">
                                        <input type="checkbox" 
                                               class="batch-checkbox" 
                                               value="<?php echo esc_attr($batch['batch_id']); ?>"
                                               <?php echo in_array($batch['status'], array('ready', 'failed', 'partial')) ? '' : 'disabled'; ?>>
                                    </th>
                                    <td class="batch-id">
                                        <strong><?php echo esc_html($batch['batch_id']); ?></strong>
                                    </td>
                                    <td class="batch-range">
                                        <?php printf(
                                            __('%d-%d', 'listeo-ai-search'),
                                            $batch['start_index'] + 1,
                                            $batch['end_index'] + 1
                                        ); ?>
                                        <span class="batch-count">(<?php echo $batch['total_count']; ?> <?php _e('listings', 'listeo-ai-search'); ?>)</span>
                                    </td>
                                    <td class="batch-progress">
                                        <div class="progress-bar-small">
                                            <div class="progress-fill-small" style="width: <?php echo $progress_percent; ?>%"></div>
                                        </div>
                                        <span class="progress-text-small">
                                            <?php echo $batch['processed_count']; ?>/<?php echo $batch['total_count']; ?> 
                                            (<?php echo $progress_percent; ?>%)
                                        </span>
                                    </td>
                                    <td class="batch-status">
                                        <div class="status-wrapper status-<?php echo $status_class; ?>">
                                            <span class="status-icon"><?php echo $status_icon; ?></span>
                                            <span class="status-text"><?php echo $status_text; ?></span>
                                        </div>
                                        <?php if ($batch['status'] === 'failed' && !empty($batch['error_message'])): ?>
                                            <div class="error-details" style="display: none;">
                                                <small><?php echo esc_html($batch['error_message']); ?></small>
                                            </div>
                                        <?php endif; ?>
                                        <?php if ($batch['retry_count'] > 0): ?>
                                            <span class="retry-count">(<?php printf(__('Retry %d', 'listeo-ai-search'), $batch['retry_count']); ?>)</span>
                                        <?php endif; ?>
                                    </td>
                                    <td class="batch-actions">
                                        <?php if (in_array($batch['status'], array('failed', 'partial'))): ?>
                                            <button type="button" 
                                                    class="button button-small retry-batch" 
                                                    data-batch-id="<?php echo esc_attr($batch['batch_id']); ?>">
                                                <?php _e('Retry', 'listeo-ai-search'); ?>
                                            </button>
                                        <?php endif; ?>
                                        
                                        <?php if ($batch['status'] === 'failed' && !empty($batch['error_message'])): ?>
                                            <button type="button" 
                                                    class="button button-small show-error" 
                                                    data-batch-id="<?php echo esc_attr($batch['batch_id']); ?>">
                                                <?php _e('Show Error', 'listeo-ai-search'); ?>
                                            </button>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Generate Embeddings Action -->
                <div class="generate-embeddings-action" style="text-align: center; margin-top: 20px; padding: 20px; background: #f8f9fa; border-radius: 4px;">
                    <button type="button" id="process-selected-batches" class="airs-button airs-button-primary" disabled style="font-size: 16px; padding: 12px 24px;">
                        <span class="button-text">🚀 <?php _e('Generate Embeddings', 'listeo-ai-search'); ?></span>
                        <span class="button-spinner" style="display: none;">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path d="M12,1A11,11,0,1,0,23,12,11,11,0,0,0,12,1Zm0,19a8,8,0,1,1,8-8A8,8,0,0,1,12,20Z" opacity="0.25"/>
                                <path d="M12,4a8,8,0,0,1,7.89,6.7A1.53,1.53,0,0,0,21.38,12h0a1.5,1.5,0,0,0,1.48-1.75,11,11,0,0,0-21.72,0A1.5,1.5,0,0,0,2.62,12h0a1.53,1.53,0,0,0,1.49-1.3A8,8,0,0,1,12,4Z" fill="currentColor">
                                    <animateTransform attributeName="transform" dur="0.75s" repeatCount="indefinite" type="rotate" values="0 12 12;360 12 12"/>
                                </path>
                            </svg>
                        </span>
                    </button>
                    <p style="margin: 10px 0 0 0; color: #6b7280; font-size: 14px;">
                        <?php _e('Select batches above to generate embeddings for your listings', 'listeo-ai-search'); ?><br>
                        <?php _e('Listings will be processed in the background. <strong>You can close this tab and return later to check progress.</strong>', 'listeo-ai-search'); ?>
                    </p>
                </div>

        </div>
        <?php endif; ?>

        <!-- Confirmation Modals -->
        <div id="reset-confirmation-modal" class="listeo-modal" style="display: none;">
            <div class="listeo-modal-content">
                <h3><?php _e('Reset All Batches and Embeddings?', 'listeo-ai-search'); ?></h3>
                <p><?php _e('This will permanently delete all existing embeddings and reset all batch progress. This action cannot be undone.', 'listeo-ai-search'); ?></p>
                <p><strong><?php _e('Are you sure you want to continue?', 'listeo-ai-search'); ?></strong></p>
                <div class="modal-actions">
                    <button type="button" id="confirm-reset" class="button button-primary"><?php _e('Yes, Reset Everything', 'listeo-ai-search'); ?></button>
                    <button type="button" id="cancel-reset" class="button button-secondary"><?php _e('Cancel', 'listeo-ai-search'); ?></button>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render stats tab
     */
    private function render_stats_tab() {
        ?>
        <!-- Search Analytics Section -->
        <?php if (get_option('listeo_ai_search_enable_analytics') && class_exists('Listeo_AI_Search_Analytics')): ?>
        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e('📊 Search Analytics', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Comprehensive analytics and insights for your AI search performance.', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">
                <div style="display: flex; gap: 20px; margin: 20px 0; flex-wrap: wrap;">
                    <!-- 7 Day Summary -->
                    <div style="background: #f8f9fa; padding: 20px; border-radius: 5px; flex: 1; min-width: 300px; border: 1px solid #e2e8f0;">
                        <h3 style="margin-top: 0;"><?php _e('Last 7 Days', 'listeo-ai-search'); ?></h3>
                        <?php
                        $analytics_7d = Listeo_AI_Search_Analytics::get_analytics(7);
                        ?>
                        <table style="width: 100%;">
                            <tr><td><strong><?php _e('Total Searches:', 'listeo-ai-search'); ?></strong></td><td><?php echo $analytics_7d['total_searches']; ?></td></tr>
                            <tr><td><strong><?php _e('Avg Response Time:', 'listeo-ai-search'); ?></strong></td><td><?php echo $analytics_7d['avg_response_time']; ?>ms</td></tr>
                        </table>
                    </div>
                    
                    <!-- 30 Day Summary -->
                    <div style="background: #e3f2fd; padding: 20px; border-radius: 5px; flex: 1; min-width: 300px; border: 1px solid #bbdefb;">
                        <h3 style="margin-top: 0;"><?php _e('Last 30 Days', 'listeo-ai-search'); ?></h3>
                        <?php
                        $analytics_30d = Listeo_AI_Search_Analytics::get_analytics(30);
                        ?>
                        <table style="width: 100%;">
                            <tr><td><strong><?php _e('Total Searches:', 'listeo-ai-search'); ?></strong></td><td><?php echo $analytics_30d['total_searches']; ?></td></tr>
                            <tr><td><strong><?php _e('Avg Response Time:', 'listeo-ai-search'); ?></strong></td><td><?php echo $analytics_30d['avg_response_time']; ?>ms</td></tr>
                        </table>
                    </div>
                </div>
                
                <!-- Popular Search Queries -->
                <div style="background: #fff3cd; padding: 20px; border-radius: 5px; margin: 20px 0; border: 1px solid #ffeaa7;">
                    <h3 style="margin-top: 0;"><?php _e('🔍 Popular Search Queries (Last 7 Days)', 'listeo-ai-search'); ?></h3>
                    <?php if (!empty($analytics_7d['popular_queries'])): ?>
                        <div style="display: flex; flex-wrap: wrap; gap: 10px;">
                            <?php foreach ($analytics_7d['popular_queries'] as $query => $count): ?>
                                <span style="background: #ffc107; color: #000; padding: 6px 12px; border-radius: 20px; font-size: 14px;">
                                    <strong><?php echo esc_html($query); ?></strong> (<?php echo $count; ?>x)
                                </span>
                            <?php endforeach; ?>
                        </div>
                    <?php else: ?>
                        <p><em><?php _e('No search queries recorded yet for the last 7 days.', 'listeo-ai-search'); ?></em></p>
                    <?php endif; ?>
                </div>

                <!-- Popular Search Queries - 30 Days -->
                <div style="background: #e3f2fd; padding: 20px; border-radius: 5px; margin: 20px 0; border: 1px solid #bbdefb;">
                    <h3 style="margin-top: 0;"><?php _e('🔍 Popular Search Queries (Last 30 Days)', 'listeo-ai-search'); ?></h3>
                    <?php if (!empty($analytics_30d['popular_queries'])): ?>
                        <div style="display: flex; flex-wrap: wrap; gap: 10px;">
                            <?php foreach ($analytics_30d['popular_queries'] as $query => $count): ?>
                                <span style="background: #2196f3; color: #fff; padding: 6px 12px; border-radius: 20px; font-size: 14px;">
                                    <strong><?php echo esc_html($query); ?></strong> (<?php echo $count; ?>x)
                                </span>
                            <?php endforeach; ?>
                        </div>
                    <?php else: ?>
                        <p><em><?php _e('No search queries recorded yet for the last 30 days.', 'listeo-ai-search'); ?></em></p>
                    <?php endif; ?>
                </div>

                <!-- Search Insights -->
                <div style="background: #f0f8ff; padding: 20px; border-radius: 5px; margin: 20px 0; border: 1px solid #cce7ff;">
                    <h3 style="margin-top: 0;"><?php _e('📈 Search Insights', 'listeo-ai-search'); ?></h3>
                    <div class="airs-help-text">
                        <?php _e('Use this popular query data to:', 'listeo-ai-search'); ?>
                        <br>• <?php _e('Optimize your listing content for popular search terms', 'listeo-ai-search'); ?>
                        <br>• <?php _e('Add more businesses in popular categories', 'listeo-ai-search'); ?>
                        <br>• <?php _e('Improve search suggestions based on actual user behavior', 'listeo-ai-search'); ?>
                        <br>• <?php _e('Identify trending topics and seasonal patterns', 'listeo-ai-search'); ?>
                    </div>
                </div>
                
                <!-- Analytics Actions -->
                <div>
                    <div class="airs-form-actions">
                        <button type="button" id="clear-analytics" class="airs-button airs-button-danger" onclick="return confirm('Are you sure? This will delete all analytics data.');">
                            <?php _e('Clear Analytics Data', 'listeo-ai-search'); ?>
                        </button>
                        <button type="button" id="refresh-analytics" class="airs-button airs-button-secondary" onclick="window.location.reload();">
                            <?php _e('Refresh Analytics', 'listeo-ai-search'); ?>
                        </button>
                    </div>
                    <div class="airs-help-text"><?php _e('Analytics data is automatically cleaned up after 1000 entries to prevent database bloat.', 'listeo-ai-search'); ?></div>
                </div>
            </div>
        </div>
        <?php else: ?>
        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e('📊 Search Analytics', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Analytics are currently disabled. Enable them in the Settings tab to start tracking search patterns.', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">
                <div style="background: #e2e3e5; padding: 40px 20px; border-radius: 5px; text-align: center;">
                    <h3><?php _e('📊 Search Analytics Disabled', 'listeo-ai-search'); ?></h3>
                    <p><?php _e('Enable search analytics in the Settings tab to track search patterns and performance.', 'listeo-ai-search'); ?></p>
                    <a href="?page=listeo-ai-search&tab=settings" class="airs-button airs-button-primary"><?php _e('Go to Settings', 'listeo-ai-search'); ?></a>
                </div>
            </div>
        </div>
        <?php endif; ?>
        <?php
    }
    
    /**
     * AJAX handler for manual batch processing actions
     */
    public function ajax_manual_batch_action() {
        // Verify nonce
        if (!check_ajax_referer('listeo_ai_search_nonce', 'nonce', false)) {
            wp_send_json_error(array('message' => __('Security check failed.', 'listeo-ai-search')));
            return;
        }

        // Check user permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'listeo-ai-search')));
            return;
        }
        
        $batch_action = sanitize_text_field($_POST['batch_action']);
        
        try {
            switch ($batch_action) {
                case 'initialize_batches':
                    $batch_size = isset($_POST['batch_size']) ? intval($_POST['batch_size']) : null;
                    $result = Listeo_AI_Search_Manual_Batch_Processor::initialize_batches($batch_size);
                    break;
                    
                case 'save_settings':
                    $batch_size = intval($_POST['batch_size']);
                    $batch_delay = intval($_POST['batch_delay']);
                    
                    update_option('listeo_ai_search_manual_batch_size', $batch_size);
                    update_option('listeo_ai_search_manual_batch_delay', $batch_delay);
                    // Background processing is always enabled in Safe Mode
                    update_option('listeo_ai_search_background_processing', 1);
                    
                    $result = array(
                        'success' => true,
                        'message' => __('Settings saved successfully', 'listeo-ai-search')
                    );
                    break;
                    
                case 'queue_batches':
                    $batch_ids = isset($_POST['batch_ids']) ? (array) $_POST['batch_ids'] : array();
                    $batch_ids = array_map('sanitize_text_field', $batch_ids);
                    $result = Listeo_AI_Search_Manual_Batch_Processor::queue_batches($batch_ids);
                    break;
                    
                case 'process_batch':
                    $batch_id = sanitize_text_field($_POST['batch_id']);
                    $delay = isset($_POST['batch_delay']) ? intval($_POST['batch_delay']) : null;
                    $result = Listeo_AI_Search_Manual_Batch_Processor::process_single_batch($batch_id, $delay);
                    break;
                    
                case 'process_selected_batches':
                    $batch_ids = array_map('sanitize_text_field', $_POST['batch_ids']);
                    $delay = isset($_POST['batch_delay']) ? intval($_POST['batch_delay']) : null;
                    $result = Listeo_AI_Search_Manual_Batch_Processor::process_selected_batches($batch_ids, $delay);
                    break;
                    
                case 'retry_batch':
                    $batch_id = sanitize_text_field($_POST['batch_id']);
                    $delay = isset($_POST['batch_delay']) ? intval($_POST['batch_delay']) : null;
                    $result = Listeo_AI_Search_Manual_Batch_Processor::retry_failed_batch($batch_id, $delay);
                    break;
                    
                case 'reset_all':
                    $result = Listeo_AI_Search_Manual_Batch_Processor::reset_all_batches();
                    break;
                    
                case 'get_progress':
                    $result = Listeo_AI_Search_Manual_Batch_Processor::get_processing_progress();
                    $result['success'] = true;
                    break;
                    
                case 'get_batch_status':
                    $batches = Listeo_AI_Search_Manual_Batch_Processor::get_all_batches();
                    $summary = Listeo_AI_Search_Manual_Batch_Processor::get_batch_summary();
                    $result = array(
                        'success' => true,
                        'batches' => $batches,
                        'summary' => $summary
                    );
                    break;
                    
                default:
                    wp_send_json_error(array('message' => __('Invalid batch action.', 'listeo-ai-search')));
                    return;
            }
            
            if (isset($result['success']) && $result['success']) {
                wp_send_json_success($result);
            } else {
                wp_send_json_error($result);
            }
            
        } catch (Exception $e) {
            wp_send_json_error(array('message' => $e->getMessage()));
        }
    }
    
    /**
     * Render admin JavaScript
     */
    private function render_admin_scripts() {
        ?>
        <script>
        jQuery(document).ready(function($) {
            console.log('Listeo AI Search admin script loaded');
            
            // Safe Mode Checkbox Handler
            $('#enable-safe-mode-checkbox').on('change', function() {
                if ($(this).is(':checked')) {
                    var confirmed = confirm('<?php _e('This will enable Safe Mode with advanced batch processing. Do you want to continue?', 'listeo-ai-search'); ?>');
                    if (confirmed) {
                        // Create and submit a form to enable Safe Mode
                        var form = $('<form method="post"></form>');
                        form.append('<?php echo wp_nonce_field('listeo_safe_mode_action', '_wpnonce', true, false); ?>');
                        form.append('<input type="hidden" name="enable_safe_mode" value="1">');
                        $('body').append(form);
                        form.submit();
                    } else {
                        // Uncheck the box if user cancels
                        $(this).prop('checked', false);
                    }
                }
            });
            
            // AJAX Settings Form Handler
            $('.airs-ajax-form').on('submit', function(e) {
                e.preventDefault();
                
                var $form = $(this);
                var $button = $form.find('button[type="submit"]');
                var $buttonText = $button.find('.button-text');
                var $buttonSpinner = $button.find('.button-spinner');
                var $message = $form.find('.airs-form-message');
                var section = $form.data('section');
                
                // Show loading state
                $button.prop('disabled', true);
                $buttonText.hide();
                $buttonSpinner.show();
                $message.hide();
                
                // Collect all current form values
                var formData = {};
                
                // Get all form inputs
                $form.find('input, textarea, select').each(function() {
                    var $input = $(this);
                    var name = $input.attr('name');
                    
                    if (name) {
                        if ($input.attr('type') === 'checkbox') {
                            formData[name] = $input.is(':checked') ? '1' : '0';
                        } else if ($input.attr('type') === 'radio') {
                            if ($input.is(':checked')) {
                                formData[name] = $input.val();
                            }
                        } else {
                            formData[name] = $input.val();
                        }
                    }
                });
                
                // Add other settings to preserve them (get values from other forms)
                $('.airs-ajax-form').not($form).each(function() {
                    var $otherForm = $(this);
                    $otherForm.find('input, textarea, select').each(function() {
                        var $input = $(this);
                        var name = $input.attr('name');
                        
                        if (name && !formData.hasOwnProperty(name)) {
                            if ($input.attr('type') === 'checkbox') {
                                formData[name] = $input.is(':checked') ? '1' : '0';
                            } else if ($input.attr('type') === 'radio') {
                                if ($input.is(':checked')) {
                                    formData[name] = $input.val();
                                }
                            } else {
                                formData[name] = $input.val();
                            }
                        }
                    });
                });
                
                // Add AJAX parameters
                formData.action = 'listeo_ai_save_settings';
                formData.nonce = listeo_ai_search_ajax.nonce;
                formData.section = section;
                
                console.log('Saving settings for section:', section, formData);
                
                // Send AJAX request
                $.post(listeo_ai_search_ajax.ajax_url, formData)
                    .done(function(response) {
                        console.log('Settings save response:', response);
                        
                        if (response.success) {
                            $message.removeClass('airs-alert-error').addClass('airs-alert-success')
                                   .html('<strong>✓ Success!</strong> ' + response.data.message)
                                   .show();
                            
                            // Hide success message after 3 seconds
                            setTimeout(function() {
                                $message.fadeOut();
                            }, 3000);
                        } else {
                            $message.removeClass('airs-alert-success').addClass('airs-alert-error')
                                   .html('<strong>✗ Error!</strong> ' + (response.data.message || 'Unknown error occurred'))
                                   .show();
                        }
                    })
                    .fail(function(xhr, status, error) {
                        console.error('AJAX Error:', xhr, status, error);
                        $message.removeClass('airs-alert-success').addClass('airs-alert-error')
                               .html('<strong>✗ Error!</strong> Connection failed: ' + error)
                               .show();
                    })
                    .always(function() {
                        // Restore button state
                        $button.prop('disabled', false);
                        $buttonText.show();
                        $buttonSpinner.hide();
                    });
            });
            
            // Test API Key Button Handler
            $('#test-api-key').on('click', function(e) {
                e.preventDefault();
                
                var $button = $(this);
                var $buttonText = $button.find('.button-text');
                var $buttonSpinner = $button.find('.button-spinner');
                var $result = $('#api-test-result');
                var apiKey = $('#listeo_ai_search_api_key').val().trim();
                
                if (!apiKey) {
                    $result.removeClass('airs-api-test-success').addClass('airs-api-test-error')
                          .html('❌ Please enter an API key first.')
                          .show();
                    return;
                }
                
                // Show loading state
                $button.prop('disabled', true);
                $buttonText.hide();
                $buttonSpinner.show();
                $result.removeClass('airs-api-test-success airs-api-test-error')
                       .html('Testing API connection...')
                       .show();
                
                // Send AJAX request
                $.post(listeo_ai_search_ajax.ajax_url, {
                    action: 'listeo_ai_test_api_key',
                    api_key: apiKey,
                    nonce: listeo_ai_search_ajax.nonce
                })
                .done(function(response) {
                    console.log('API test response:', response);
                    
                    if (response.success) {
                        $result.removeClass('airs-api-test-error').addClass('airs-api-test-success')
                              .html(response.data.message)
                              .show();
                        
                        // Show details if available
                        if (response.data.details) {
                            $result.append('<br><small>' + response.data.details + '</small>');
                        }
                    } else {
                        $result.removeClass('airs-api-test-success').addClass('airs-api-test-error')
                              .html(response.data.message || 'Test failed')
                              .show();
                    }
                })
                .fail(function(xhr, status, error) {
                    console.error('API Test AJAX Error:', xhr, status, error);
                    $result.removeClass('airs-api-test-success').addClass('airs-api-test-error')
                          .html('❌ Connection failed: ' + error)
                          .show();
                })
                .always(function() {
                    // Reset button state
                    $button.prop('disabled', false);
                    $buttonText.show();
                    $buttonSpinner.hide();
                });
            });
            
            // Clear Cache Button Handler
            $('#listeo-clear-cache-btn').on('click', function(e) {
                e.preventDefault();
                
                var $button = $(this);
                var $status = $('#listeo-clear-cache-status');
                var originalText = $button.html();
                
                // Show loading state
                $button.prop('disabled', true)
                       .html('<span class="dashicons dashicons-update spin" style="vertical-align: middle; margin-right: 5px;"></span><?php _e("Clearing...", "listeo-ai-search"); ?>');
                $status.html('').removeClass('success error');
                
                // Send AJAX request
                $.post(listeo_ai_search_ajax.ajax_url, {
                    action: 'listeo_ai_clear_cache',
                    nonce: listeo_ai_search_ajax.nonce
                })
                .done(function(response) {
                    if (response.success) {
                        $status.html(response.data.message)
                               .addClass('success')
                               .css('color', '#46b450');
                    } else {
                        $status.html(response.data.message || 'Clear cache failed')
                               .addClass('error')
                               .css('color', '#dc3232');
                    }
                })
                .fail(function(xhr, status, error) {
                    $status.html('❌ Connection failed: ' + error)
                           .addClass('error')
                           .css('color', '#dc3232');
                })
                .always(function() {
                    // Reset button state after 2 seconds
                    setTimeout(function() {
                        $button.prop('disabled', false).html(originalText);
                        $status.fadeOut(3000);
                    }, 2000);
                });
            });
            
            // Check if AJAX variables are available
            if (typeof listeo_ai_search_ajax === 'undefined') {
                console.error('AJAX variables not loaded properly');
                $('#status-content').html('<p style="color: red;">Error: AJAX configuration not loaded. Please refresh the page.</p>');
                return;
            }
            
            console.log('AJAX URL:', listeo_ai_search_ajax.ajax_url);
            console.log('Current URL:', window.location.href);
            
            // Load initial status
            refreshDatabaseStatus();
            
            // Refresh status button
            $('#refresh-status').on('click', function() {
                refreshDatabaseStatus();
            });
            
            // Test AJAX button
            $('#test-ajax').on('click', function() {
                console.log('Testing AJAX connection...');
                $.post(listeo_ai_search_ajax.ajax_url, {
                    action: 'listeo_ai_test'
                }, function(response) {
                    console.log('Test response:', response);
                    alert('AJAX Test Result: ' + JSON.stringify(response));
                }).fail(function(xhr, status, error) {
                    console.error('Test AJAX failed:', xhr, status, error);
                    alert('AJAX Test Failed: ' + error + ' (Status: ' + xhr.status + ')');
                });
            });
            
            // Database action buttons
            $('#clear-database').on('click', function() {
                if (confirm('Are you sure? This will delete all embeddings and cannot be undone.')) {
                    performDatabaseAction('clear_all', 'Clearing database...');
                }
            });
            
            // Analytics action button
            $('#clear-analytics').on('click', function() {
                performAnalyticsAction('clear_analytics', 'Clearing analytics data...');
            });
            
            // Check embedding button
            $('#check-embedding').on('click', function() {
                var listingId = $('#listing-id-input').val();
                if (!listingId) {
                    alert('Please enter a listing ID');
                    return;
                }
                
                console.log('Checking embedding for listing ID:', listingId);
                checkEmbeddingData(listingId);
            });
            
            // Regenerate embedding button
            $('#regenerate-embedding').on('click', function() {
                var listingId = $('#regenerate-listing-id-input').val();
                if (!listingId || listingId <= 0) {
                    alert('Please enter a valid listing ID');
                    return;
                }
                
                var $button = $(this);
                var $buttonText = $button.find('.button-text');
                var $buttonSpinner = $button.find('.button-spinner');
                var $result = $('#regenerate-embedding-result');
                
                // Show loading state
                $button.prop('disabled', true);
                $buttonText.hide();
                $buttonSpinner.show();
                $result.removeClass('success error').html('').hide();
                
                console.log('Regenerating embedding for listing ID:', listingId);
                
                // Send AJAX request
                $.post(listeo_ai_search_ajax.ajax_url, {
                    action: 'listeo_ai_regenerate_embedding',
                    listing_id: listingId,
                    nonce: listeo_ai_search_ajax.nonce
                })
                .done(function(response) {
                    console.log('Regenerate embedding response:', response);
                    
                    if (response.success) {
                        $result.removeClass('error').addClass('success')
                              .html(response.data.message)
                              .css('color', '#46b450')
                              .show();
                        
                        // Clear the input field on success
                        $('#regenerate-listing-id-input').val('');
                    } else {
                        $result.removeClass('success').addClass('error')
                              .html(response.data.message || 'Regeneration failed')
                              .css('color', '#dc3232')
                              .show();
                    }
                })
                .fail(function(xhr, status, error) {
                    console.error('Regenerate embedding AJAX error:', xhr, status, error);
                    $result.removeClass('success').addClass('error')
                          .html('❌ Connection failed: ' + error)
                          .css('color', '#dc3232')
                          .show();
                })
                .always(function() {
                    // Reset button state
                    $button.prop('disabled', false);
                    $buttonText.show();
                    $buttonSpinner.hide();
                });
            });
            
            // Close embedding viewer
            $('#close-embedding').on('click', function() {
                $('#embedding-viewer').hide();
            });
            
            function refreshDatabaseStatus() {
                $('#status-content').html('<p><em>Loading...</em></p>');
                
                // Test if the AJAX URL is reachable first
                console.log('Testing AJAX URL:', listeo_ai_search_ajax.ajax_url);
                
                $.post(listeo_ai_search_ajax.ajax_url, {
                    action: 'listeo_ai_manage_database',
                    database_action: 'get_stats',
                    nonce: listeo_ai_search_ajax.nonce
                }, function(response) {
                    console.log('Database status response:', response);
                    if (response.success) {
                        displayDatabaseStatus(response.data);
                        updateListingDetectionInfo(response.data);
                    } else {
                        $('#status-content').html('<p style="color: red;">Error: ' + (response.data || 'Unknown error') + '</p>');
                        $('#listing-count-text').text('Error loading listing information');
                    }
                }).fail(function(xhr, status, error) {
                    console.error('AJAX Error Details:', {
                        status: status,
                        error: error,
                        responseText: xhr.responseText,
                        url: listeo_ai_search_ajax.ajax_url,
                        readyState: xhr.readyState,
                        statusCode: xhr.status
                    });
                    $('#status-content').html('<p style="color: red;">AJAX error: ' + error + ' (Status: ' + xhr.status + ')</p>');
                });
            }
            
            function displayDatabaseStatus(data) {
                var html = '<table class="widefat">';
                
                if (data.error) {
                    html += '<tr><td colspan="2" style="color: red;"><strong>Error:</strong> ' + data.error + '</td></tr>';
                }
                
                html += '<tr><td><strong>Table Exists:</strong></td><td>' + (data.table_exists ? '✓ Yes' : '✗ No') + '</td></tr>';
                html += '<tr><td><strong>Total Listings:</strong></td><td>' + data.total_listings + '</td></tr>';
                html += '<tr><td><strong>Processed Embeddings:</strong></td><td>' + data.total_embeddings + '</td></tr>';
                html += '<tr><td><strong>Coverage:</strong></td><td>' + data.coverage_percentage + '%</td></tr>';
                html += '<tr><td><strong>Missing Embeddings:</strong></td><td>' + data.without_embeddings + '</td></tr>';
                html += '<tr><td><strong>Recent Activity (24h):</strong></td><td>' + data.recent_embeddings + '</td></tr>';
                html += '</table>';
                
                // Show recent embeddings if available
                if (data.recent_items && data.recent_items.length > 0) {
                    html += '<h4>Recent Embeddings (Last 10):</h4>';
                    html += '<table class="widefat" style="margin-top: 10px;">';
                    html += '<thead><tr><th>ID</th><th>Title</th><th>Type</th><th>Created</th></tr></thead><tbody>';
                    data.recent_items.forEach(function(item) {
                        html += '<tr>';
                        html += '<td><a href="#" class="embedding-link" data-id="' + item.listing_id + '">' + item.listing_id + '</a></td>';
                        html += '<td>' + (item.title || 'N/A') + '</td>';
                        html += '<td>' + (item.listing_type || 'N/A') + '</td>';
                        html += '<td>' + item.created_at + '</td>';
                        html += '</tr>';
                    });
                    html += '</tbody></table>';
                    html += '<p class="description"><em>Click on any ID to view its embedding data.</em></p>';
                } else {
                    html += '<p><em>No recent embeddings found.</em></p>';
                }
                
                // Show missing embeddings if available - but only if we already have some embeddings
                if (data.missing_items && data.missing_items.length > 0 && data.total_embeddings > 0) {
                    html += '<h4 style="margin-top: 30px;">Missing Embeddings:</h4>';
                    html += '<div style="margin: 10px 0;">';
                    html += '<button type="button" class="button button-secondary" id="select-all-missing">Select All</button> ';
                    html += '<button type="button" class="button button-secondary" id="deselect-all-missing">Deselect All</button> ';
                    html += '<button type="button" class="button button-primary" id="generate-selected-missing" style="margin-left: 10px;" disabled>Generate Selected</button>';
                    html += '</div>';
                    html += '<div style="max-height: 400px; overflow-y: auto; border: 1px solid #ddd; border-radius: 4px;">';
                    html += '<table class="widefat" style="margin: 0;">';
                    html += '<thead><tr><th style="width: 40px;"><input type="checkbox" id="missing-select-all"></th><th>ID</th><th>Title</th><th>Type</th><th>Last Modified</th><th>Action</th></tr></thead><tbody>';
                    data.missing_items.forEach(function(item) {
                        html += '<tr>';
                        html += '<td><input type="checkbox" class="missing-item-checkbox" data-id="' + item.listing_id + '"></td>';
                        html += '<td>' + item.listing_id + '</td>';
                        html += '<td>' + (item.title || 'N/A') + '</td>';
                        html += '<td>' + (item.listing_type || 'N/A') + '</td>';
                        html += '<td>' + item.created_at + '</td>';
                        html += '<td><button type="button" class="button button-primary button-small generate-embedding-btn" data-id="' + item.listing_id + '">Generate</button></td>';
                        html += '</tr>';
                    });
                    html += '</tbody></table>';
                    html += '</div>';
                    html += '<p class="description"><em>Click "Generate" for individual listings or select multiple and use "Generate Selected".</em></p>';
                } else if (data.without_embeddings > 0 && data.total_embeddings === 0) {
                    html += '<h4 style="margin-top: 30px;">Missing Embeddings:</h4>';
                    html += '<p><em>All ' + data.without_embeddings + ' listings are missing embeddings. Use the "Generate Structured Embeddings" tool above to process them in bulk.</em></p>';
                }
                
                $('#status-content').html(html);
                
                // Add click handlers for embedding links
                $('.embedding-link').on('click', function(e) {
                    e.preventDefault();
                    var listingId = $(this).data('id');
                    checkEmbeddingData(listingId);
                });
                
                // Add click handlers for generate embedding buttons
                $('.generate-embedding-btn').on('click', function(e) {
                    e.preventDefault();
                    var $button = $(this);
                    var listingId = $button.data('id');
                    generateSingleEmbedding(listingId, $button);
                });
                
                // Add handlers for select all/deselect all
                $('#select-all-missing, #missing-select-all').on('change click', function(e) {
                    var isChecked = $(this).prop('checked') || $(this).attr('id') === 'select-all-missing';
                    $('.missing-item-checkbox').prop('checked', isChecked);
                    updateGenerateSelectedButton();
                });
                
                $('#deselect-all-missing').on('click', function(e) {
                    $('.missing-item-checkbox').prop('checked', false);
                    $('#missing-select-all').prop('checked', false);
                    updateGenerateSelectedButton();
                });
                
                // Add handler for individual checkboxes
                $('.missing-item-checkbox').on('change', function() {
                    updateGenerateSelectedButton();
                    // Update select all checkbox state
                    var totalCheckboxes = $('.missing-item-checkbox').length;
                    var checkedCheckboxes = $('.missing-item-checkbox:checked').length;
                    $('#missing-select-all').prop('checked', checkedCheckboxes === totalCheckboxes);
                });
                
                // Add handler for generate selected
                $('#generate-selected-missing').on('click', function(e) {
                    e.preventDefault();
                    var selectedIds = [];
                    $('.missing-item-checkbox:checked').each(function() {
                        selectedIds.push($(this).data('id'));
                    });
                    
                    if (selectedIds.length > 0) {
                        generateBulkEmbeddings(selectedIds);
                    }
                });
            }
            
            function checkEmbeddingData(listingId) {
                $('#embedding-content').html('<p><em>Loading embedding data for listing ' + listingId + '...</em></p>');
                $('#embedding-viewer').show();
                
                $.post(listeo_ai_search_ajax.ajax_url, {
                    action: 'listeo_ai_manage_database',
                    database_action: 'get_embedding',
                    listing_id: listingId,
                    nonce: listeo_ai_search_ajax.nonce
                }, function(response) {
                    console.log('Embedding response:', response);
                    if (response.success) {
                        displayEmbeddingData(response.data);
                    } else {
                        $('#embedding-content').html('<p style="color: red;">Error: ' + (response.data || 'Unknown error') + '</p>');
                    }
                }).fail(function(xhr, status, error) {
                    console.error('Embedding AJAX Error:', xhr, status, error);
                    $('#embedding-content').html('<p style="color: red;">AJAX error: ' + error + '</p>');
                });
            }
            
            function displayEmbeddingData(data) {
                var html = '<h4>Listing #' + data.listing_id + ': ' + (data.title || 'Unknown Title') + '</h4>';
                
                if (data.embedding_exists) {
                    html += '<p><strong>✓ Embedding exists</strong> (Created: ' + data.created_at + ')</p>';
                    
                    if (data.processed_content) {
                        html += '<h5>Processed Content:</h5>';
                        html += '<div style="background: white; padding: 10px; border: 1px solid #ddd; white-space: pre-wrap; font-family: monospace; font-size: 12px; max-height: 200px; overflow-y: auto;">';
                        html += data.processed_content;
                        html += '</div>';
                    }
                    
                    if (data.embedding_preview) {
                        html += '<h5>Embedding Vector (first 10 dimensions):</h5>';
                        html += '<div style="background: white; padding: 10px; border: 1px solid #ddd; font-family: monospace; font-size: 12px;">';
                        html += '[' + data.embedding_preview.join(', ') + '...]';
                        html += '</div>';
                        html += '<p class="description">Full embedding vector contains ' + data.vector_dimensions + ' dimensions.</p>';
                    }
                } else {
                    html += '<p><strong>✗ No embedding found</strong></p>';
                    html += '<p>This listing has not been processed for AI search yet.</p>';
                }
                
                $('#embedding-content').html(html);
            }
            
            function performDatabaseAction(actionType, loadingMessage) {
                $('#result-message-content').text(loadingMessage).removeClass('airs-alert-error').addClass('airs-alert-success');
                $('#action-result').show();
                
                $.post(listeo_ai_search_ajax.ajax_url, {
                    action: 'listeo_ai_manage_database',
                    database_action: actionType,
                    nonce: listeo_ai_search_ajax.nonce
                }, function(response) {
                    if (response.success) {
                        $('#result-message-content').text('Action completed successfully!');
                        setTimeout(refreshDatabaseStatus, 2000);
                    } else {
                        $('#result-message-content').text('Error: ' + (response.data || 'Unknown error')).removeClass('airs-alert-success').addClass('airs-alert-error');
                    }
                }).fail(function() {
                    $('#result-message-content').text('AJAX error occurred').removeClass('airs-alert-success').addClass('airs-alert-error');
                });
            }
            
            function performAnalyticsAction(actionType, loadingMessage) {
                $('#result-message-content').text(loadingMessage).removeClass('airs-alert-error').addClass('airs-alert-success');
                $('#action-result').show();
                
                $.post(listeo_ai_search_ajax.ajax_url, {
                    action: 'listeo_ai_analytics',
                    analytics_action: actionType,
                    nonce: listeo_ai_search_ajax.nonce
                }, function(response) {
                    if (response.success) {
                        $('#result-message-content').text('Analytics data cleared successfully!');
                        setTimeout(function() { window.location.reload(); }, 2000);
                    } else {
                        $('#result-message-content').text('Error: ' + (response.data || 'Unknown error')).removeClass('airs-alert-success').addClass('airs-alert-error');
                    }
                }).fail(function() {
                    $('#result-message-content').text('AJAX error occurred').removeClass('airs-alert-success').addClass('airs-alert-error');
                });
            }
            
            // NEW: Structured embedding generation functionality
            $('#start-regeneration').on('click', function() {
                if (!confirm('This will generate embeddings for ALL listings using the structured format. This may take a while and will consume OpenAI API credits. Continue?')) {
                    return;
                }
                
                regenerationRunning = true;
                currentOffset = 0;
                totalListings = 0; // Reset total listings counter
                
                $('#start-regeneration').hide();
                $('#stop-regeneration').show();
                $('#regeneration-progress').show();
                $('#regeneration-log').show();
                
                // Initialize progress display
                $('#progress-fill').css('width', '0%');
                $('#progress-text').text('Initializing...');
                
                logMessage('🚀 Starting structured embedding generation...');
                
                runRegenerationBatch();
            });
            
            var regenerationRunning = false;
            var currentOffset = 0;
            var totalListings = 0;

            $('#stop-regeneration').on('click', function() {
                regenerationRunning = false;
                $('#start-regeneration').show();
                $('#stop-regeneration').hide();
                logMessage('⏸️ Generation stopped by user.');
            });
            
            function runRegenerationBatch() {
                if (!regenerationRunning) return;
                
                var batchSize = parseInt($('#batch-size').val()) || 20;
                
                logMessage('📦 Processing batch starting at offset ' + currentOffset + '...');
                
                $.post(listeo_ai_search_ajax.ajax_url, {
                    action: 'listeo_ai_manage_database',
                    database_action: 'start_regeneration',
                    batch_size: batchSize,
                    start_offset: currentOffset,
                    nonce: listeo_ai_search_ajax.nonce
                }, function(response) {
                    if (response.success) {
                        var data = response.data;
                        
                        // Set total listings from first response
                        if (data.total_listings && totalListings === 0) {
                            totalListings = data.total_listings;
                        }
                        
                        // Update current offset, ensure it's a valid number
                        if (typeof data.next_offset !== 'undefined') {
                            currentOffset = data.next_offset;
                        }
                        
                        // Update progress with proper validation
                        var progress = 0;
                        if (totalListings > 0 && currentOffset >= 0) {
                            progress = (currentOffset / totalListings) * 100;
                        }
                        
                        $('#progress-fill').css('width', Math.max(0, Math.min(100, progress)) + '%');
                        $('#progress-text').text('Processed: ' + (currentOffset || 0) + ' / ' + (totalListings || 0) + ' (' + Math.round(progress) + '%)');
                        
                        // Log results
                        logMessage('✅ Batch completed: ' + (data.processed || 0) + ' listings processed');
                        
                        if (data.errors && data.errors.length > 0) {
                            data.errors.forEach(function(error) {
                                logMessage('⚠️ Error: ' + error, 'error');
                            });
                        }
                        
                        // Continue or finish
                        if (data.status === 'complete' || data.processed === 0) {
                            finishRegeneration('🎉 Generation completed successfully!');
                            refreshDatabaseStatus();
                        } else if (regenerationRunning) {
                            // Continue with next batch after delay
                            setTimeout(runRegenerationBatch, 1000);
                        }
                    } else {
                        logMessage('❌ Batch failed: ' + (response.data || 'Unknown error'), 'error');
                        finishRegeneration('❌ Generation failed.');
                    }
                }).fail(function(xhr, status, error) {
                    logMessage('❌ AJAX error: ' + error, 'error');
                    finishRegeneration('❌ Generation failed due to connection error.');
                });
            }
            
            function finishRegeneration(message) {
                regenerationRunning = false;
                $('#start-regeneration').show();
                $('#stop-regeneration').hide();
                $('#progress-fill').css('width', '100%');
                logMessage(message);
            }
            
            function logMessage(message, type) {
                var timestamp = new Date().toLocaleTimeString();
                var colorClass = type === 'error' ? 'color: #ff6b6b;' : 'color: #51cf66;';
                
                $('#log-content').append(
                    '<div style="' + colorClass + '">[' + timestamp + '] ' + message + '</div>'
                );
                
                // Auto-scroll to bottom
                var logElement = $('#log-content')[0];
                logElement.scrollTop = logElement.scrollHeight;
            }
            
            function generateSingleEmbedding(listingId, $button) {
                // Update button state
                var originalText = $button.text();
                $button.text('Generating...').prop('disabled', true);
                
                $.post(listeo_ai_search_ajax.ajax_url, {
                    action: 'listeo_ai_manage_database',
                    database_action: 'generate_single',
                    listing_id: listingId,
                    nonce: listeo_ai_search_ajax.nonce
                }, function(response) {
                    if (response.success) {
                        $button.text('✓ Done').removeClass('button-primary').addClass('button-secondary');
                        // Remove the row after successful generation
                        setTimeout(function() {
                            $button.closest('tr').fadeOut(500, function() {
                                $(this).remove();
                                updateGenerateSelectedButton();
                                // Refresh the status to get updated counts
                                refreshDatabaseStatus();
                            });
                        }, 1500);
                    } else {
                        $button.text('✗ Failed').removeClass('button-primary').addClass('button-secondary');
                        alert('Failed to generate embedding: ' + (response.message || 'Unknown error'));
                        setTimeout(function() {
                            $button.text(originalText).prop('disabled', false).removeClass('button-secondary').addClass('button-primary');
                        }, 3000);
                    }
                }).fail(function(xhr, status, error) {
                    $button.text('✗ Error').removeClass('button-primary').addClass('button-secondary');
                    alert('AJAX error: ' + error);
                    setTimeout(function() {
                        $button.text(originalText).prop('disabled', false).removeClass('button-secondary').addClass('button-primary');
                    }, 3000);
                });
            }
            
            function updateGenerateSelectedButton() {
                var checkedCount = $('.missing-item-checkbox:checked').length;
                var $button = $('#generate-selected-missing');
                
                if (checkedCount > 0) {
                    $button.prop('disabled', false).text('Generate Selected (' + checkedCount + ')');
                } else {
                    $button.prop('disabled', true).text('Generate Selected');
                }
            }
            
            function updateListingDetectionInfo(data) {
                var totalListings = data.total_listings || 0;
                
                var infoText = '';
                if (totalListings === 0) {
                    infoText = 'No published listings found.';
                } else {
                    infoText = totalListings + ' listings detected.';
                }
                
                $('#listing-count-text').text(infoText);
            }
            
            function generateBulkEmbeddings(listingIds) {
                if (!listingIds || listingIds.length === 0) return;
                
                var $button = $('#generate-selected-missing');
                var originalText = $button.text();
                var totalIds = listingIds.length;
                var completedIds = 0;
                var failedIds = [];
                
                $button.text('Generating ' + totalIds + ' embeddings...').prop('disabled', true);
                
                // Process embeddings one by one to avoid API rate limits
                function processNext() {
                    if (listingIds.length === 0) {
                        // All done
                        var successCount = completedIds - failedIds.length;
                        var message = 'Completed: ' + successCount + ' success';
                        if (failedIds.length > 0) {
                            message += ', ' + failedIds.length + ' failed';
                        }
                        
                        $button.text(message).prop('disabled', false);
                        setTimeout(function() {
                            refreshDatabaseStatus();
                            $button.text('Generate Selected').prop('disabled', true);
                        }, 3000);
                        return;
                    }
                    
                    var currentId = listingIds.shift();
                    completedIds++;
                    
                    $button.text('Processing ' + completedIds + '/' + totalIds + '...');
                    
                    $.post(listeo_ai_search_ajax.ajax_url, {
                        action: 'listeo_ai_manage_database',
                        database_action: 'generate_single',
                        listing_id: currentId,
                        nonce: listeo_ai_search_ajax.nonce
                    }, function(response) {
                        if (response.success) {
                            // Remove the row for successful generation
                            $('.missing-item-checkbox[data-id="' + currentId + '"]').closest('tr').fadeOut(300, function() {
                                $(this).remove();
                            });
                        } else {
                            failedIds.push(currentId);
                        }
                        
                        // Continue with next after small delay
                        setTimeout(processNext, 1000);
                    }).fail(function() {
                        failedIds.push(currentId);
                        setTimeout(processNext, 1000);
                    });
                }
                
                processNext();
            }
        });
        </script>
        <?php
    }
    
    /**
     * Show debug mode notice
     */
    public function show_debug_mode_notice() {
        $screen = get_current_screen();
        if ($screen && $screen->id === 'toplevel_page_listeo-ai-search') {
            $debug_file = WP_CONTENT_DIR . '/debug.log';
            echo '<div class="notice notice-info">';
            echo '<p><strong>🐛 Debug Mode Active:</strong> Detailed logs are being written to <code>' . esc_html($debug_file) . '</code></p>';
            echo '<p>This is helpful for diagnosing batch processing issues. Disable debug mode when not needed to reduce log file size.</p>';
            echo '</div>';
        }
    }
}
