<?php
/**
 * Background Processing for Listeo AI Search
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Background processing class
 */
class Listeo_AI_Background_Processor {
    
    /**
     * Hook into WordPress
     */
    public static function init() {
        add_action('listeo_ai_process_listing', array(__CLASS__, 'process_single_listing'));
        add_action('listeo_ai_bulk_process_listings', array(__CLASS__, 'process_all_listings'));
    }
    
    /**
     * Process a single listing
     */
    public static function process_single_listing($listing_id) {
        global $wpdb;
        
        $listing = get_post($listing_id);
        
        if (!$listing || $listing->post_type !== 'listing' || $listing->post_status !== 'publish') {
            return false;
        }
        
        try {
            // Get API key
            $api_key = get_option('listeo_ai_search_api_key', '');
            if (empty($api_key)) {
                Listeo_AI_Search_Utility_Helper::debug_log('No API key configured', 'error');
                return false;
            }
            
            // Collect listing content
            $content = self::collect_listing_content($listing_id);
            $content_hash = md5($content);
            
            // Check if we already have current embedding
            $table_name = $wpdb->prefix . 'listeo_ai_embeddings';
            $existing = $wpdb->get_row($wpdb->prepare(
                "SELECT content_hash FROM $table_name WHERE listing_id = %d",
                $listing_id
            ));
            
            if ($existing && $existing->content_hash === $content_hash) {
                // Content hasn't changed, skip
                return true;
            }
            
            // Generate embedding
            $embedding = self::generate_embedding($content, $api_key);
            
            if (!$embedding) {
                Listeo_AI_Search_Utility_Helper::debug_log("Failed to generate embedding for listing $listing_id", 'error');
                return false;
            }
            
            // Store embedding
            $result = $wpdb->replace($table_name, array(
                'listing_id' => $listing_id,
                'embedding' => Listeo_AI_Search_Database_Manager::compress_embedding_for_storage($embedding),
                'content_hash' => $content_hash,
                'updated_at' => current_time('mysql')
            ));
            
            if ($result === false) {
                Listeo_AI_Search_Utility_Helper::debug_log("Failed to store embedding for listing $listing_id", 'error');
                return false;
            }
            
            Listeo_AI_Search_Utility_Helper::debug_log("Successfully processed listing $listing_id", 'info');
            return true;
            
        } catch (Exception $e) {
            error_log("Listeo AI Search: Error processing listing $listing_id - " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Process all existing listings
     */
    public static function process_all_listings() {
        $args = array(
            'post_type' => 'listing',
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'fields' => 'ids'
        );
        
        $listings = get_posts($args);
        
        error_log("Listeo AI Search: Starting bulk processing of " . count($listings) . " listings");
        
        $processed = 0;
        $failed = 0;
        
        foreach ($listings as $listing_id) {
            // Add delay to avoid rate limiting
            if ($processed > 0 && $processed % 3 === 0) {
                sleep(1); // 1 second delay every 3 requests
            }
            
            if (self::process_single_listing($listing_id)) {
                $processed++;
            } else {
                $failed++;
            }
            
            // Break if too many failures
            if ($failed > 10) {
                error_log("Listeo AI Search: Too many failures ($failed), stopping bulk process");
                break;
            }
        }
        
        error_log("Listeo AI Search: Bulk processing complete. Processed: $processed, Failed: $failed");
        
        // Schedule cleanup of old embeddings
        wp_schedule_single_event(time() + 300, 'listeo_ai_cleanup_embeddings');
    }
    
    /**
     * Collect content from listing
     */
    public static function collect_listing_content($listing_id) {
        $listing = get_post($listing_id);
        $content_parts = array();
        
        // Basic content
        $content_parts[] = $listing->post_title;
        $content_parts[] = $listing->post_content;
        $content_parts[] = $listing->post_excerpt;
        
        // Meta fields we discovered earlier
        $meta_fields = array(
            '_friendly_address',
            '_address',
            '_phone',
            '_website',
            '_email',
            '_listing_description',
            '_keyword',
            '_opening_hours',
            '_listing_gallery',
            '_menu',
            '_pricing_list',
            '_amenities',
            '_features',
            '_event_date',
            '_event_location',
            '_rental_period',
            '_classifieds_category',
            '_service_category'
        );
        
        foreach ($meta_fields as $field) {
            $value = get_post_meta($listing_id, $field, true);
            if (!empty($value)) {
                if (is_array($value)) {
                    $content_parts[] = implode(' ', array_filter($value));
                } else {
                    $content_parts[] = strip_tags($value);
                }
            }
        }
        
        // Get taxonomy terms
        $taxonomies = array('listing_category', 'listing_feature', 'service_category', 'rental_category', 'event_category');
        foreach ($taxonomies as $taxonomy) {
            $terms = wp_get_post_terms($listing_id, $taxonomy, array('fields' => 'names'));
            if (!is_wp_error($terms) && !empty($terms)) {
                $content_parts[] = implode(' ', $terms);
            }
        }
        
        // Clean and join
        $content_parts = array_filter(array_map('trim', $content_parts));
        $content = implode(' ', $content_parts);
        
        // Remove extra whitespace and limit length
        $content = preg_replace('/\s+/', ' ', $content);
        $content = substr($content, 0, 8000); // OpenAI limit
        
        return $content;
    }
    
    /**
     * Generate embedding using OpenAI
     */
    private static function generate_embedding($text, $api_key) {
        $start_time = microtime(true);
        
        if (get_option('listeo_ai_search_debug_mode', false)) {
            Listeo_AI_Search_Utility_Helper::debug_log("Making OpenAI API call for " . strlen($text) . " characters", 'info');
        }
        
        $response = wp_remote_post('https://api.openai.com/v1/embeddings', array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type' => 'application/json',
            ),
            'body' => json_encode(array(
                'model' => 'text-embedding-3-small',
                'input' => $text,
            )),
            'timeout' => 60,
        ));
        
        $duration = microtime(true) - $start_time;
        
        if (is_wp_error($response)) {
            $error_msg = 'OpenAI API request failed: ' . $response->get_error_message();
            if (get_option('listeo_ai_search_debug_mode', false)) {
                Listeo_AI_Search_Utility_Helper::debug_log("API Error after {$duration}s: {$error_msg}", 'error');
            }
            throw new Exception($error_msg);
        }
        
        $http_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (get_option('listeo_ai_search_debug_mode', false)) {
            Listeo_AI_Search_Utility_Helper::debug_log("API Response after {$duration}s: HTTP {$http_code}", 'info');
        }
        
        if ($http_code !== 200) {
            $error_msg = "OpenAI API HTTP error: {$http_code}";
            if (isset($body['error']['message'])) {
                $error_msg .= " - " . $body['error']['message'];
            }
            if (get_option('listeo_ai_search_debug_mode', false)) {
                Listeo_AI_Search_Utility_Helper::debug_log("API HTTP Error: {$error_msg}", 'error');
            }
            throw new Exception($error_msg);
        }
        
        if (isset($body['error'])) {
            $error_msg = 'OpenAI API error: ' . $body['error']['message'];
            if (get_option('listeo_ai_search_debug_mode', false)) {
                Listeo_AI_Search_Utility_Helper::debug_log("API Response Error: {$error_msg}", 'error');
            }
            throw new Exception($error_msg);
        }
        
        if (!isset($body['data'][0]['embedding'])) {
            $error_msg = 'OpenAI API response missing embedding data';
            if (get_option('listeo_ai_search_debug_mode', false)) {
                Listeo_AI_Search_Utility_Helper::debug_log("API Response Missing Data: " . json_encode($body), 'error');
            }
            throw new Exception($error_msg);
        }
        
        if (get_option('listeo_ai_search_debug_mode', false)) {
            $embedding_count = count($body['data'][0]['embedding']);
            Listeo_AI_Search_Utility_Helper::debug_log("Successfully received embedding with {$embedding_count} dimensions in {$duration}s", 'info');
        }
        
        return $body['data'][0]['embedding'];
    }
    
    /**
     * Cleanup old embeddings for deleted listings
     */
    public static function cleanup_embeddings() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'listeo_ai_embeddings';
        
        // Delete embeddings for non-existent listings
        $wpdb->query("
            DELETE e FROM $table_name e 
            LEFT JOIN {$wpdb->posts} p ON e.listing_id = p.ID 
            WHERE p.ID IS NULL OR p.post_status != 'publish' OR p.post_type != 'listing'
        ");
        
        error_log("Listeo AI Search: Embedding cleanup completed");
    }
}

// Initialize background processor
add_action('listeo_ai_cleanup_embeddings', array('Listeo_AI_Background_Processor', 'cleanup_embeddings'));
Listeo_AI_Background_Processor::init();
