<?php
/**
 * Plugin Name: Listeo AI Search
 * Plugin URI: https://purethemes.net
 * Description: AI-powered semantic search for Listeo directory theme with natural language queries
 * Version: 1.2
 * Author: PureThemes
 * Author URI: https://purethemes.net
 * License: GPL2
 * Text Domain: listeo-ai-search
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.3
 * Requires PHP: 7.4
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('LISTEO_AI_SEARCH_VERSION', '1.2');
define('LISTEO_AI_SEARCH_PLUGIN_URL', plugin_dir_url(__FILE__));
define('LISTEO_AI_SEARCH_PLUGIN_PATH', plugin_dir_path(__FILE__));

/**
 * Main plugin class
 */
class Listeo_AI_Search {
    
    /**
     * Single instance
     */
    private static $instance = null;
    
    /**
     * Search handler instance
     * 
     * @var Listeo_AI_Search_Search_Handler
     */
    private $search_handler;
    
    /**
     * Shortcode handler instance
     * 
     * @var Listeo_AI_Search_Shortcode_Handler
     */
    private $shortcode_handler;
    
    /**
     * Admin interface instance
     * 
     * @var Listeo_AI_Search_Admin_Interface
     */
    private $admin_interface;
    
    /**
     * Get instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Load dependencies first
        $this->load_dependencies();
        
        // Initialize AJAX handlers early (before init)
        $this->search_handler = new Listeo_AI_Search_Search_Handler();
        
        add_action('init', array($this, 'init'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }
    
    /**
     * Initialize plugin
     */
    public function init() {
        // Initialize remaining components
        $this->shortcode_handler = new Listeo_AI_Search_Shortcode_Handler();
        $this->admin_interface = new Listeo_AI_Search_Admin_Interface();
        
        // Initialize background processor if available
        if (class_exists('Listeo_AI_Background_Processor')) {
            Listeo_AI_Background_Processor::init();
        }
        
        // Auto-process listings
        add_action('save_post', array($this, 'process_listing_on_save'), 10, 2);
        
        // Background processing
        add_action('listeo_ai_process_background_batches', array($this, 'process_background_batches'));
    }
    
    /**
     * Load all required class files
     */
    private function load_dependencies() {
        // Core utility classes
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-utility-helper.php';
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-result-formatter.php';
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-embedding-manager.php';
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-database-manager.php';
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-analytics.php';
        
        // Manual batch processor for large catalogs
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-manual-batch-processor.php';
        
        // Search engines
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/search/class-fallback-engine.php';
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/search/class-ai-engine.php';
        
        // Main handlers
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-search-handler.php';
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/frontend/class-shortcode-handler.php';
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/admin/class-admin-interface.php';
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-listeo-field-integration.php';
        
        // Background processor (existing)
        if (file_exists(LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-background-processor.php')) {
            require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-background-processor.php';
        }
    }
    
    /**
     * Enqueue scripts and styles
     */
    public function enqueue_scripts() {
        wp_enqueue_script(
            'listeo-ai-search',
            LISTEO_AI_SEARCH_PLUGIN_URL . 'assets/js/search.js',
            array('jquery'),
            LISTEO_AI_SEARCH_VERSION,
            true
        );
        
        wp_localize_script('listeo-ai-search', 'listeoAiSearch', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('listeo_ai_search_nonce'),
            'ai_enabled' => true, // AI search is always enabled
            'max_results' => intval(get_option('listeo_ai_search_max_results', 10)),
            'search_url' => get_post_type_archive_link('listing') ?: home_url(), // Use proper archive URL or fallback to home
            'strings' => array(
                'searching' => __('Searching...', 'listeo-ai-search'),
                'no_results' => __('No results found.', 'listeo-ai-search'),
                'error' => __('Search error occurred.', 'listeo-ai-search'),
                'best_match' => __('Best Match', 'listeo-ai-search'),
                'type_keywords_first' => __('Type keywords first', 'listeo-ai-search'),
                'top_listing_singular' => __('Top 1 listing matching', 'listeo-ai-search'),
                'top_listings_plural' => __('Top %d listings matching', 'listeo-ai-search'),
            )
        ));
        
        wp_enqueue_style(
            'listeo-ai-search',
            LISTEO_AI_SEARCH_PLUGIN_URL . 'assets/css/search.css',
            array(),
            LISTEO_AI_SEARCH_VERSION
        );
    }
    
    /**
     * Process listing on save
     * 
     * @param int $post_id Post ID
     * @param WP_Post $post Post object
     */
    public function process_listing_on_save($post_id, $post) {
        Listeo_AI_Search_Database_Manager::process_listing_on_save($post_id, $post);
    }
    
    /**
     * Custom debug logging to debug.log
     * 
     * @param string $message Log message
     * @param string $level Log level (info, error, warning, debug)
     */
    public static function debug_log($message, $level = 'info') {
        // Only log if debug mode is enabled
        if (!get_option('listeo_ai_search_debug_mode', false)) {
            return;
        }
        
        // Use WordPress standard debug logging
        $timestamp = date('Y-m-d H:i:s');
        $formatted_message = "[{$timestamp}] [{$level}] Listeo AI Search: {$message}";
        
        error_log($formatted_message);
    }
    
    /**
     * Plugin activation
     */
    public function activate() {
        // Create database tables
        Listeo_AI_Search_Database_Manager::create_tables();
        
        // Create manual batch tracking table
        Listeo_AI_Search_Manual_Batch_Processor::create_batch_table();
        
        // Schedule bulk processing of existing listings (only if not in manual mode)
        if (!Listeo_AI_Search_Manual_Batch_Processor::should_use_manual_mode()) {
            wp_schedule_single_event(time() + 60, 'listeo_ai_bulk_process_listings');
        }
    }
    
    /**
     * Process background batches
     */
    public function process_background_batches() {
        global $wpdb;
        
        // Enhanced debug logging
        if (get_option('listeo_ai_search_debug_mode', false)) {
            self::debug_log("Background batch processing started", 'info');
        }
        
        // Get one queued batch to process
        $batch_table = Listeo_AI_Search_Manual_Batch_Processor::get_batch_table_name();
        $batch = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$batch_table} WHERE status = 'queued' ORDER BY created_at ASC LIMIT 1"
        ), ARRAY_A);
        
        if (!$batch) {
            if (get_option('listeo_ai_search_debug_mode', false)) {
                self::debug_log("No queued batches found - background processing complete", 'info');
            }
            return; // No queued batches
        }
        
        if (get_option('listeo_ai_search_debug_mode', false)) {
            self::debug_log("Processing batch {$batch['batch_id']} with {$batch['total_listings']} listings", 'info');
        }
        
        try {
            // Process this batch
            $result = Listeo_AI_Search_Manual_Batch_Processor::process_single_batch($batch['batch_id'], null);
            
            if (get_option('listeo_ai_search_debug_mode', false)) {
                $status = $result['success'] ? 'SUCCESS' : 'FAILED';
                self::debug_log("Batch {$batch['batch_id']} processing {$status}: " . $result['message'], $result['success'] ? 'info' : 'error');
            }
            
        } catch (Exception $e) {
            if (get_option('listeo_ai_search_debug_mode', false)) {
                self::debug_log("CRITICAL ERROR processing batch {$batch['batch_id']}: " . $e->getMessage(), 'error');
                self::debug_log("Stack trace: " . $e->getTraceAsString(), 'error');
            }
            
            // Mark batch as failed
            $wpdb->update(
                $batch_table,
                array(
                    'status' => 'failed',
                    'error_message' => 'Critical error: ' . $e->getMessage(),
                    'updated_at' => current_time('mysql')
                ),
                array('batch_id' => $batch['batch_id']),
                array('%s', '%s', '%s'),
                array('%s')
            );
        }
        
        // Check if there are more queued batches
        $remaining_count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$batch_table} WHERE status = 'queued'"
        ));
        
        if (get_option('listeo_ai_search_debug_mode', false)) {
            self::debug_log("Remaining queued batches: {$remaining_count}", 'info');
        }
        
        // Schedule next batch processing if there are more queued batches
        if ($remaining_count > 0) {
            wp_schedule_single_event(time() + 30, 'listeo_ai_process_background_batches');
            if (get_option('listeo_ai_search_debug_mode', false)) {
                self::debug_log("Scheduled next background batch processing in 30 seconds", 'info');
            }
        }
    }
    
    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Clean up scheduled events
        wp_clear_scheduled_hook('listeo_ai_bulk_process_listings');
        wp_clear_scheduled_hook('listeo_ai_process_listing');
        wp_clear_scheduled_hook('listeo_ai_process_background_batches');
    }
}

// Initialize plugin
Listeo_AI_Search::get_instance();
