jQuery(document).ready(function($) {

    // --- Tab Navigation (if needed for dynamic switching) ---
    $('.airs-nav-tab-wrapper .nav-tab').on('click', function(e) {
        // Since we're using PHP-based tabs with page reloads,
        // this is mainly for visual feedback. The actual tab switching
        // is handled by the PHP code based on the 'tab' URL parameter.
        
        // Remove active class from all tabs
        $('.nav-tab').removeClass('nav-tab-active');
        // Add active class to clicked tab
        $(this).addClass('nav-tab-active');
    });

    // --- Batch Processing Logic ---
    var $log = $('#airs-log');
    var $progressContainer = $('#airs-progress-container');
    var $progressBar = $('.airs-progress-bar-inner');
    var $progressText = $('.airs-progress-text');
    var $generateButton = $('#airs-start-batch');
    var $deleteSelectedButton = $('#airs-delete-selected');

    // 1. Live Search for Listings
    $('#airs-listing-search').on('keyup', function() {
        var searchTerm = $(this).val().toLowerCase();
        $('.airs-listing-list .airs-listing-item').each(function() {
            var listingTitle = $(this).find('.airs-listing-title').text().toLowerCase();
            var listingId = $(this).find('.airs-listing-id').text().toLowerCase();
            
            if (listingTitle.includes(searchTerm) || listingId.includes(searchTerm)) {
                $(this).show();
            } else {
                $(this).hide();
            }
        });
    });

    // 2. Select/Deselect All
    $('#airs-select-all').on('click', function(e) {
        e.preventDefault();
        // Only select visible listings after a search
        $('.airs-listing-list .airs-listing-item:visible input[type="checkbox"]').prop('checked', true);
    });

    $('#airs-deselect-all').on('click', function(e) {
        e.preventDefault();
        $('.airs-listing-list input[type="checkbox"]').prop('checked', false);
    });

    // 3. Generate Summaries
    $generateButton.on('click', function(e) {
        e.preventDefault();
        
        var listing_ids = $('input[name="listing_ids[]"]:checked').map(function() {
            return $(this).val();
        }).get();

        if (listing_ids.length === 0) {
            alert('Please select at least one listing to generate.');
            return;
        }

        var currentIndex = 0;
        $log.html('');
        $progressContainer.show();
        $generateButton.prop('disabled', true);
        $deleteSelectedButton.prop('disabled', true);

        function processNext() {
            if (currentIndex >= listing_ids.length) {
                $log.append('<div class="log-success"><strong>✅ Batch generation complete!</strong></div>');
                $generateButton.prop('disabled', false);
                $deleteSelectedButton.prop('disabled', false);
                updateProgress(listing_ids.length, listing_ids.length);
                return;
            }

            var post_id = listing_ids[currentIndex];
            var $listingItem = $('input[value="' + post_id + '"]').closest('.airs-listing-item');
            var post_title = $listingItem.find('.airs-listing-title').text().trim();
            
            $log.append('<div>🔄 Processing ' + (currentIndex + 1) + ' of ' + listing_ids.length + ': ' + post_title + ' (ID: ' + post_id + ')...</div>');
            $log.scrollTop($log[0].scrollHeight);

            $.post(airs_ajax.ajax_url, {
                action: 'airs_generate_summary',
                nonce: airs_ajax.nonce,
                post_id: post_id
            }).done(function(response) {
                if (response.success) {
                    $log.find('div:last-child').append(' <span class="log-success">✅ Success</span>');
                    
                    // Add status badge if it doesn't exist
                    if ($listingItem.find('.airs-status-badge').length === 0) {
                        var badge = '<span class="airs-status-badge airs-status-generated" data-listing-id="' + post_id + '" title="Summary exists">' +
                                   '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">' +
                                   '<path d="M9 16.17L4.83 12l-1.42 1.41L9 19L21 7l-1.41-1.41z" fill="currentColor"/>' +
                                   '</svg>Generated</span>';
                        $listingItem.append(badge);
                    }
                } else {
                    var errorMessage = response.data.message || 'Unknown error';
                    $log.find('div:last-child').append(' <span class="log-error">❌ Error: ' + errorMessage + '</span>');
                }
            }).fail(function() {
                $log.find('div:last-child').append(' <span class="log-error">❌ Failed (server error)</span>');
            }).always(function() {
                currentIndex++;
                updateProgress(currentIndex, listing_ids.length);
                setTimeout(processNext, 100); // Small delay to prevent overwhelming the server
            });
        }
        processNext();
    });

    function updateProgress(current, total) {
        var percentage = total > 0 ? (current / total) * 100 : 0;
        $progressBar.css('width', percentage + '%');
        $progressText.text(Math.round(percentage) + '%');
    }

    // 4. Delete Selected Summaries
    $deleteSelectedButton.on('click', function(e) {
        e.preventDefault();
        
        var post_ids = $('input[name="listing_ids[]"]:checked').map(function() {
            return $(this).val();
        }).get();

        if (post_ids.length === 0) {
            alert('Please select at least one listing to delete its summary.');
            return;
        }

        if (!confirm('Are you sure you want to delete the summaries for the ' + post_ids.length + ' selected listings? This cannot be undone.')) {
            return;
        }
        
        $log.html('');
        $progressContainer.show();
        $generateButton.prop('disabled', true);
        $deleteSelectedButton.prop('disabled', true);
        $log.append('<div>🗑️ Deleting summaries for ' + post_ids.length + ' listings...</div>');

        $.post(airs_ajax.ajax_url, {
            action: 'airs_delete_selected_summaries',
            nonce: airs_ajax.nonce,
            post_ids: post_ids
        }).done(function(response) {
            if (response.success) {
                $log.append('<div class="log-success">✅ ' + response.data.message + '</div>');
                
                // Remove status badges from UI
                response.data.deleted_ids.forEach(function(id) {
                    $('.airs-status-badge[data-listing-id="' + id + '"]').remove();
                });
            } else {
                $log.append('<div class="log-error">❌ ' + (response.data.message || 'An unknown error occurred.') + '</div>');
            }
        }).fail(function() {
            $log.append('<div class="log-error">❌ Deletion failed due to a server error.</div>');
        }).always(function() {
            $generateButton.prop('disabled', false);
            $deleteSelectedButton.prop('disabled', false);
        });
    });

    // --- Debugging Tab Logic ---
    var $debugOutputWrapper = $('#airs-debug-output-wrapper');
    var $debugOutput = $('#airs-debug-output');

    $('#airs-test-reviews').on('click', function(e) {
        e.preventDefault();
        
        var post_id = $('#airs-debug-listing-id').val();
        if (!post_id) { 
            alert('Please enter a Listing ID.'); 
            return; 
        }
        
        $debugOutputWrapper.show();
        $debugOutput.text('🔄 Fetching reviews for listing ID ' + post_id + '...');
        
        $.post(airs_ajax.ajax_url, { 
            action: 'airs_fetch_reviews', 
            nonce: airs_ajax.nonce, 
            post_id: post_id 
        }).done(function(response) { 
            if (response.success) {
                $debugOutput.text('✅ Found ' + response.data.count + ' reviews:\n\n' + JSON.stringify(response.data, null, 2));
            } else {
                $debugOutput.text('❌ Error: ' + response.data.message);
            }
        }).fail(function() { 
            $debugOutput.text('❌ Server error occurred while fetching reviews.');
        });
    });

    $('#airs-test-generation').on('click', function(e) {
        e.preventDefault();
        
        var post_id = $('#airs-debug-listing-id').val();
        if (!post_id) { 
            alert('Please enter a Listing ID.'); 
            return; 
        }
        
        $debugOutputWrapper.show();
        $debugOutput.text('🤖 Sending request to OpenAI API for listing ID ' + post_id + '...');
        
        $.post(airs_ajax.ajax_url, { 
            action: 'airs_generate_summary', 
            nonce: airs_ajax.nonce, 
            post_id: post_id 
        }).done(function(response) { 
            if (response.success) {
                $debugOutput.text('✅ Summary generated successfully:\n\n' + JSON.stringify(response.data, null, 2));
            } else {
                $debugOutput.text('❌ Error: ' + response.data.message);
            }
        }).fail(function() { 
            $debugOutput.text('❌ Server error occurred while generating summary.');
        });
    });

    // --- Enhanced UI Interactions ---
    
    // Smooth scrolling for progress log
    function smoothScrollLog() {
        $log.animate({
            scrollTop: $log[0].scrollHeight
        }, 300);
    }

    // Replace the existing scroll function calls with smooth scrolling
    if ($log.length) {
        var originalAppend = $log.append;
        $log.append = function() {
            originalAppend.apply(this, arguments);
            setTimeout(smoothScrollLog, 100);
            return this;
        };
    }

    // Add loading states for buttons
    function setButtonLoading($button, loading) {
        if (loading) {
            $button.data('original-html', $button.html());
            $button.html('<svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" class="airs-spinner">' +
                        '<path d="M12 2v4m0 12v4M4.93 4.93l2.83 2.83m8.48 8.48l2.83 2.83M2 12h4m12 0h4M4.93 19.07l2.83-2.83m8.48-8.48l2.83-2.83" stroke="currentColor" stroke-width="2" stroke-linecap="round"/>' +
                        '</svg> Processing...');
            $button.prop('disabled', true);
        } else {
            $button.html($button.data('original-html'));
            $button.prop('disabled', false);
        }
    }

    // Add spinner animation CSS
    $('<style>')
        .prop('type', 'text/css')
        .html(`
            @keyframes spin {
                from { transform: rotate(0deg); }
                to { transform: rotate(360deg); }
            }
            .airs-spinner {
                animation: spin 1s linear infinite;
            }
        `)
        .appendTo('head');

    // Enhanced form validation
    $('form').on('submit', function() {
        var isValid = true;
        var $form = $(this);
        
        // Clear previous error states
        $form.find('.airs-input').removeClass('airs-input-error');
        $form.find('.airs-error-message').remove();
        
        // Validate required fields
        $form.find('.airs-input[required]').each(function() {
            var $input = $(this);
            if (!$input.val().trim()) {
                $input.addClass('airs-input-error');
                $input.after('<div class="airs-error-message">This field is required.</div>');
                isValid = false;
            }
        });
        
        // Validate API key format (if present)
        var $apiKey = $form.find('input[name="airs_settings[openai_api_key]"]');
        if ($apiKey.length && $apiKey.val().trim() && !$apiKey.val().startsWith('sk-')) {
            $apiKey.addClass('airs-input-error');
            $apiKey.after('<div class="airs-error-message">API key should start with "sk-".</div>');
            isValid = false;
        }
        
        return isValid;
    });

    // Auto-hide success messages
    setTimeout(function() {
        $('.notice.notice-success').fadeOut(500);
    }, 3000);

    // Keyboard shortcuts
    $(document).on('keydown', function(e) {
        // Ctrl/Cmd + A to select all visible listings (when in manage tab)
        if ((e.ctrlKey || e.metaKey) && e.key === 'a' && $('.airs-manage-tab').length) {
            e.preventDefault();
            $('#airs-select-all').click();
        }
        
        // Escape to close debug output
        if (e.key === 'Escape' && $debugOutputWrapper.is(':visible')) {
            $debugOutputWrapper.hide();
        }
    });

    // Add tooltips for better UX
    function initTooltips() {
        $('[title]').each(function() {
            var $element = $(this);
            var title = $element.attr('title');
            
            $element.on('mouseenter', function() {
                var $tooltip = $('<div class="airs-tooltip">' + title + '</div>');
                $('body').append($tooltip);
                
                var offset = $element.offset();
                var elementHeight = $element.outerHeight();
                
                $tooltip.css({
                    position: 'absolute',
                    top: offset.top + elementHeight + 5,
                    left: offset.left,
                    zIndex: 9999
                });
            }).on('mouseleave', function() {
                $('.airs-tooltip').remove();
            });
            
            // Remove the default title to prevent native tooltip
            $element.removeAttr('title').data('title', title);
        });
    }

    // Initialize tooltips
    initTooltips();

    // Add tooltip styles
    $('<style>')
        .prop('type', 'text/css')
        .html(`
            .airs-tooltip {
                background: rgba(0, 0, 0, 0.9);
                color: white;
                padding: 0.5rem 0.75rem;
                border-radius: 4px;
                font-size: 0.75rem;
                white-space: nowrap;
                pointer-events: none;
                box-shadow: 0 2px 8px rgba(0,0,0,0.2);
            }
            .airs-input-error {
                border-color: #ef4444 !important;
                box-shadow: 0 0 0 3px rgba(239, 68, 68, 0.1) !important;
            }
            .airs-error-message {
                color: #ef4444;
                font-size: 0.75rem;
                margin-top: 0.25rem;
            }
        `)
        .appendTo('head');
});