<?php
if ( ! defined( 'ABSPATH' ) ) exit;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;

/**
 * AI Review Highlights Elementor Widget
 */
class AIRS_Elementor_Widget extends Widget_Base {

    /**
     * Get widget name.
     */
    public function get_name() {
        return 'airs-review-highlights';
    }

    /**
     * Get widget title.
     */
    public function get_title() {
        return __( 'AI Review Highlights', 'ai-review-summary' );
    }

    /**
     * Get widget icon.
     */
    public function get_icon() {
        return 'eicon-testimonial-carousel';
    }

    /**
     * Get widget categories.
     */
    public function get_categories() {
        return [ 'airs-widgets', 'listeo-single' ];
    }

    /**
     * Get widget keywords.
     */
    public function get_keywords() {
        return [ 'ai', 'review', 'highlights', 'summary', 'listeo', 'listing' ];
    }

    /**
     * Get style dependencies.
     */
    public function get_style_depends() {
        return [ 'airs-frontend-styles' ];
    }

    /**
     * Register widget controls.
     */
    protected function register_controls() {
        
        // Content Section
        $this->start_controls_section(
            'content_section',
            [
                'label' => __( 'Content', 'ai-review-summary' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'widget_title',
            [
                'label' => __( 'Title', 'ai-review-summary' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( '✨ AI-Powered Review Summary', 'ai-review-summary' ),
                'placeholder' => __( 'Type your title here', 'ai-review-summary' ),
            ]
        );

        $this->add_control(
            'show_description',
            [
                'label' => __( 'Show Description', 'ai-review-summary' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __( 'Show', 'ai-review-summary' ),
                'label_off' => __( 'Hide', 'ai-review-summary' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'hide_threshold',
            [
                'label' => __( 'Hide Threshold', 'ai-review-summary' ),
                'type' => Controls_Manager::NUMBER,
                'min' => 0,
                'max' => 20,
                'step' => 1,
                'default' => 3,
                'description' => __( 'Hide widget if negative points exceed this threshold', 'ai-review-summary' ),
            ]
        );

        $this->add_control(
            'columns',
            [
                'label' => __( 'Columns', 'ai-review-summary' ),
                'type' => Controls_Manager::SELECT,
                'default' => '2',
                'options' => [
                    '1' => __( '1 Column', 'ai-review-summary' ),
                    '2' => __( '2 Columns', 'ai-review-summary' ),
                    '3' => __( '3 Columns', 'ai-review-summary' ),
                    '4' => __( '4 Columns', 'ai-review-summary' ),
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Title
        $this->start_controls_section(
            'title_style_section',
            [
                'label' => __( 'Title', 'ai-review-summary' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .ai-review-summary-container h3',
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => __( 'Color', 'ai-review-summary' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .ai-review-summary-container h3' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'title_margin',
            [
                'label' => __( 'Margin', 'ai-review-summary' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'selectors' => [
                    '{{WRAPPER}} .ai-review-summary-container h3' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Cards
        $this->start_controls_section(
            'cards_style_section',
            [
                'label' => __( 'Cards', 'ai-review-summary' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'card_background_color',
            [
                'label' => __( 'Background Color', 'ai-review-summary' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .airs-summary-card' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'card_border',
                'selector' => '{{WRAPPER}} .airs-summary-card',
            ]
        );

        $this->add_control(
            'card_border_radius',
            [
                'label' => __( 'Border Radius', 'ai-review-summary' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .airs-summary-card' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'card_box_shadow',
                'selector' => '{{WRAPPER}} .airs-summary-card',
            ]
        );

        $this->add_control(
            'card_padding',
            [
                'label' => __( 'Padding', 'ai-review-summary' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'selectors' => [
                    '{{WRAPPER}} .airs-summary-card' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'card_margin',
            [
                'label' => __( 'Margin', 'ai-review-summary' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'selectors' => [
                    '{{WRAPPER}} .airs-summary-card' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Card Content
        $this->start_controls_section(
            'card_content_style_section',
            [
                'label' => __( 'Card Content', 'ai-review-summary' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'card_title_typography',
                'label' => __( 'Title Typography', 'ai-review-summary' ),
                'selector' => '{{WRAPPER}} .airs-summary-card h4',
            ]
        );

        $this->add_control(
            'card_title_color',
            [
                'label' => __( 'Title Color', 'ai-review-summary' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .airs-summary-card h4' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'card_description_typography',
                'label' => __( 'Description Typography', 'ai-review-summary' ),
                'selector' => '{{WRAPPER}} .airs-summary-card p',
            ]
        );

        $this->add_control(
            'card_description_color',
            [
                'label' => __( 'Description Color', 'ai-review-summary' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .airs-summary-card p' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Render the widget output on the frontend.
     */
    protected function render() {
        // Only show on single listing pages
        if ( ! is_singular( 'listing' ) ) {
            if ( \Elementor\Plugin::$instance->editor->is_edit_mode() ) {
                echo '<div class="elementor-alert elementor-alert-info">' . __( 'This widget is only displayed on single listing pages.', 'ai-review-summary' ) . '</div>';
            }
            return;
        }

        global $post, $wpdb;
        
        $settings = $this->get_settings_for_display();
        
        $widget_title = $settings['widget_title'];
        $show_description = $settings['show_description'] === 'yes';
        $hide_threshold = $settings['hide_threshold'];
        $columns = $settings['columns'];
        
        $table_name = $wpdb->prefix . 'ai_review_summaries';
        
        $summary_row = $wpdb->get_row( $wpdb->prepare( "SELECT summary_data FROM $table_name WHERE listing_id = %d", $post->ID ) );

        if ( ! $summary_row || empty( $summary_row->summary_data ) ) {
            if ( \Elementor\Plugin::$instance->editor->is_edit_mode() ) {
                echo '<div class="elementor-alert elementor-alert-warning">' . __( 'No AI review summary found for this listing. Make sure to generate AI summaries in the AI Review Highlights plugin.', 'ai-review-summary' ) . '</div>';
            }
            return;
        }
        
        $summary_data = json_decode( $summary_row->summary_data, true );

        if ( ! isset( $summary_data['summaries'] ) || empty( $summary_data['summaries'] ) ) {
            if ( \Elementor\Plugin::$instance->editor->is_edit_mode() ) {
                echo '<div class="elementor-alert elementor-alert-warning">' . __( 'No summary data available for this listing.', 'ai-review-summary' ) . '</div>';
            }
            return;
        }

        // Count negative points
        $negative_count = 0;
        foreach ( $summary_data['summaries'] as $item ) {
            if ( isset( $item['sentiment'] ) && $item['sentiment'] === 'negative' ) {
                $negative_count++;
            }
        }

        // Hide summary if negative points exceed threshold
        if ( $negative_count > $hide_threshold ) {
            if ( \Elementor\Plugin::$instance->editor->is_edit_mode() ) {
                echo '<div class="elementor-alert elementor-alert-info">' . sprintf( __( 'Widget hidden due to threshold settings (too many negative reviews: %d > %d).', 'ai-review-summary' ), $negative_count, $hide_threshold ) . '</div>';
            }
            return;
        }
        ?>
        <div class="ai-review-summary-container elementor-airs-widget">
            <?php if ( ! empty( $widget_title ) ) : ?>
                <h3><?php echo wp_kses_post( $widget_title ); ?></h3>
                <span class="line"></span>
                <div class="clearfix"></div>
            <?php endif; ?>
            
            <div class="airs-summary-grid airs-columns-<?php echo esc_attr( $columns ); ?>">
                <?php foreach ( $summary_data['summaries'] as $item ) : ?>
                    <?php
                    $card_class = 'airs-summary-card airs-sentiment-' . esc_attr( $item['sentiment'] );
                    if ( ! $show_description ) {
                        $card_class .= ' airs-title-only';
                    }
                    ?>
                    <div class="<?php echo esc_attr( $card_class ); ?>">
                        <div class="airs-card-icon"></div>
                        <div class="airs-card-content">
                            <h4><?php echo esc_html( $item['title'] ); ?></h4>
                            <?php if ( $show_description ) : ?>
                                <p><?php echo esc_html( $item['description'] ); ?></p>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
        
        <style>
        .elementor-airs-widget .airs-summary-grid.airs-columns-1 {
            display: grid;
            grid-template-columns: 1fr;
            gap: 15px;
        }
        .elementor-airs-widget .airs-summary-grid.airs-columns-2 {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 15px;
        }
        .elementor-airs-widget .airs-summary-grid.airs-columns-3 {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 15px;
        }
        .elementor-airs-widget .airs-summary-grid.airs-columns-4 {
            display: grid;
            grid-template-columns: repeat(4, 1fr);
            gap: 15px;
        }
        @media (max-width: 768px) {
            .elementor-airs-widget .airs-summary-grid.airs-columns-3,
            .elementor-airs-widget .airs-summary-grid.airs-columns-4 {
                grid-template-columns: repeat(2, 1fr);
            }
        }
        @media (max-width: 480px) {
            .elementor-airs-widget .airs-summary-grid {
                grid-template-columns: 1fr !important;
            }
        }
        </style>
        <?php
    }

    /**
     * Render the widget output in the editor.
     */
    protected function content_template() {
        ?>
        <#
        view.addRenderAttribute( 'wrapper', 'class', 'ai-review-summary-container elementor-airs-widget' );
        #>
        <div {{{ view.getRenderAttributeString( 'wrapper' ) }}}>
            <# if ( settings.widget_title ) { #>
                <h3>{{{ settings.widget_title }}}</h3>
                <span class="line"></span>
                <div class="clearfix"></div>
            <# } #>
            
            <div class="airs-summary-grid airs-columns-{{{ settings.columns }}}">
                <div class="airs-summary-card airs-sentiment-positive">
                    <div class="airs-card-icon"></div>
                    <div class="airs-card-content">
                        <h4><?php echo __( 'Great Service', 'ai-review-summary' ); ?></h4>
                        <# if ( settings.show_description === 'yes' ) { #>
                            <p><?php echo __( 'Customers love the excellent service quality.', 'ai-review-summary' ); ?></p>
                        <# } #>
                    </div>
                </div>
                <div class="airs-summary-card airs-sentiment-positive">
                    <div class="airs-card-icon"></div>
                    <div class="airs-card-content">
                        <h4><?php echo __( 'Amazing Location', 'ai-review-summary' ); ?></h4>
                        <# if ( settings.show_description === 'yes' ) { #>
                            <p><?php echo __( 'Perfect location with easy access to everything.', 'ai-review-summary' ); ?></p>
                        <# } #>
                    </div>
                </div>
            </div>
        </div>
        
        <style>
        .elementor-airs-widget .airs-summary-grid.airs-columns-1 {
            display: grid;
            grid-template-columns: 1fr;
            gap: 15px;
        }
        .elementor-airs-widget .airs-summary-grid.airs-columns-2 {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 15px;
        }
        .elementor-airs-widget .airs-summary-grid.airs-columns-3 {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 15px;
        }
        .elementor-airs-widget .airs-summary-grid.airs-columns-4 {
            display: grid;
            grid-template-columns: repeat(4, 1fr);
            gap: 15px;
        }
        </style>
        <?php
    }
}
